/* system includes */
#include <stdlib.h>
#include <sys/ioctl.h>

/* firmware includes */
#include <pp/base.h>
#include <pp/kvm.h>
#include <pp/usb.h>
#include <pp/cfg.h>
#include <pp/grab.h>
#include <pp/rfb.h>
#include <pp/xdefs.h>
#include <pp/hal_common.h>
#include <pp/km.h>

/* local includes */
#include "driver.h"
#include "data_conv_kbd.h"
#include "iipptr_internal.h"

#define PP_KM_USE_CIM_PROTO_DRIVER defined(PRODUCT_KX2)
#if PP_KM_USE_CIM_PROTO_DRIVER

#include <pp/portmgr.h>
#include <pp/cim_proto_lib.h>

using namespace pp;

#endif //PP_KM_USE_CIM_PROTO_DRIVER

/*
 *  hardware -> driver mapping defines
 *
 *  NOTES: - To use these defines you should use #if and _not_ #ifdef or #if defined!
 *
 *	   - More then one define may be true. In obj case the more special one
 *	     should win.
 */
#define PP_KM_USE_IPC_DRIVER		(defined(PRODUCT_SMARTIPC))
#define PP_KM_USE_RIPC_DRIVER		(defined(PRODUCT_RIPCKIMXN))
#define PP_KM_USE_KIM_DRIVER		(defined(PRODUCT_XX01IP_ANY)	\
					 || defined(PRODUCT_LARA)	\
					 || defined(PRODUCT_RC1)	\
					 || defined(PRODUCT_ERIC2)	\
					 || defined(PRODUCT_ERICXP)	\
					 || defined(PRODUCT_ERICG4)	\
					 || defined(PRODUCT_LARAXP) && !defined(OEM_LANTRONIX) \
					 || defined(PRODUCT_FORENSICCIM) \
					 || defined(PRODUCT_FLASHX4)	\
					 || defined(PRODUCT_KIMTESTERMST) \
					 || defined(PRODUCT_LARA_BMC_TEST) \
					 || defined(PRODUCT_ICPMMD)	\
					 || defined(PRODUCT_PDU)	\
					 )
#define PP_KM_USE_USBONLY_DRIVER	(defined(PRODUCT_MSIDC)		\
					 || defined(PRODUCT_SMIDC)	\
					 || defined(PRODUCT_GIGADC)	\
					 || defined(PRODUCT_AMDDC)	\
					 || defined(PRODUCT_INTELDC)	\
					 || defined(PRODUCT_ASMIDC) \
					 || defined(PRODUCT_RACKABLEDC) \
					 || defined(PRODUCT_LARAXP) && defined(OEM_LANTRONIX) \
					 )
#define DEFAULT_PS2_COMMAND_SET 2



/* FIXME: drivers should supply their own additional config keys causing reconfigure, dont use them here */
static const char * km_options[] = {
    "input_type",
    "usb_type",
    "unit.port.kbd.model",
    "unit[?].port[?].kbd.model",
#ifdef PP_FEAT_KBD_BREAK_TIMEOUT
    "unit[?].port[?].kbd.break_timeout_enabled",
    "unit[?].port[?].kbd.break_timeout_msec",
#endif /* PP_FEAT_KBD_BREAK_TIMEOUT */
    "unit[?].port[?].mouse.mode",
    "unit[?].port[?].mouse.direct_scaling",
    "unit[?].port[?].mouse.gud_eqz",
    "unit[?].port[?].mouse.gud_eqz_preset",
    "unit[?].port[?].mouse.gud_eqz_id",    
#ifdef PP_FEAT_CAT
    "unit[?].port[?].mouse.cat",    
#endif /* PP_FEAT_CAT */
    NULL /* must be present and the last */
};

pthread_mutex_t km_encoder_hook_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;
struct list_head km_encoder_hook_list;

static driver_t **driver = NULL;
static pthread_mutex_t driver_mtx;

static pthread_mutexattr_t kvm_mtx_attr;
static int ps2_possible = 1;
static int usb_possible = 0;
static u_int current_num_data_links = 1;

#if PP_KM_USE_RIPC_DRIVER
static int configure_ripc_driver(driver_t * d);
#endif

#if PP_KM_USE_KIM_DRIVER
static int configure_kim_driver(driver_t * d);
#endif

#if PP_KM_USE_CIM_PROTO_DRIVER
static int configure_cim_proto_driver(driver_t *d);
#endif

#if PP_KM_USE_IPC_DRIVER
static int configure_ipc_driver(driver_t * d);
#endif

#if PP_KM_USE_USBONLY_DRIVER
static int configure_usbonly_driver(driver_t *d);
#endif

static int init_target_helper(driver_t * obj , target_para_t* tpt);

static int cleanup_driver(driver_t* obj);
static int resume_driver(driver_t* obj);
static int reconfigure_driver(driver_t* obj);
static int suspend_driver(driver_t* obj);
static int reset_driver(driver_t* obj);
static int is_equal(driver_t* obj, driver_t* d);
#ifdef KBD_DEBUG
static void show_driver_ids(driver_t* obj);

#endif

static int reconf_ch(pp_cfg_chg_ctx_t *ctx, u_char data_link);
static int reconf_channels(pp_cfg_chg_ctx_t *ctx);
static void propchange_handler(pp_propchange_listener_t * listener, u_short prop, u_short prop_flags);
volatile int ignore_usb_propchange = 0;
pthread_mutex_t usb_propchange_mutex;
pthread_cond_t usb_resume_cond;
pthread_cond_t usb_suspend_cond;

PP_DECLARE_PROPCHANGE_LISTENER(propchange_listener, propchange_handler);

int
pp_km_init(void)
{
    u_int i;

    current_num_data_links = pp_hal_common_get_data_link_cnt();
    assert( current_num_data_links != 0 );
    driver = (driver_t **)calloc( 1, sizeof(driver_t*)*current_num_data_links);

    pthread_mutexattr_init(&kvm_mtx_attr);
    pthread_mutexattr_settype(&kvm_mtx_attr, PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init(&driver_mtx, &kvm_mtx_attr);

    /* add change handlers for settings */
    for (i = 0; km_options[i] != NULL; ++i) {
	pp_cfg_add_change_listener(reconf_channels, km_options[i]);
    }

    pthread_mutex_init(&usb_propchange_mutex, NULL);
    pthread_cond_init(&usb_resume_cond, NULL);
    pthread_cond_init(&usb_suspend_cond, NULL);

    pp_propchange_add(&propchange_listener, PP_PROP_AUTO_AUTO_ADJUSTMENT_DONE);
    pp_propchange_add(&propchange_listener, PP_PROP_USB_DEV_STATE_CHANGED);

    INIT_LIST_HEAD(&km_encoder_hook_list);

    return 0;
}

void
pp_km_cleanup(void)
{
    u_int i;

    pp_propchange_remove_all(&propchange_listener);

    /* remove change handlers for settings */
    for (i = 0; km_options[i] != NULL; ++i) {
	pp_cfg_rem_change_listener(reconf_channels, km_options[i]);
    }

    MUTEX_LOCK(&driver_mtx);
    for( i=0;i<current_num_data_links; i++){
        if (driver[i]) {
            suspend_driver(driver[i]);
            cleanup_driver(driver[i]);
            pthread_mutex_destroy(&driver[i]->state_mtx);
            free(driver[i]);
            driver[i] = NULL;
        }
    }
    MUTEX_UNLOCK(&driver_mtx);

    pthread_mutex_destroy(&usb_propchange_mutex);
    pthread_cond_destroy(&usb_resume_cond);
    pthread_cond_destroy(&usb_suspend_cond);

    pthread_mutexattr_destroy(&kvm_mtx_attr);

    free(driver);
}

int
pp_km_reset_driver(unsigned char data_link)
{
    MUTEX_LOCK(&driver_mtx);
    if (driver[data_link]) {
	if (!pp_km_reset_driver_is_implemented(data_link)) {
	    /* nothing to do */
	    KD("Driver doesn't need a reset.\n");
	    goto bail;
	}
	suspend_driver(driver[data_link]);
	reset_driver(driver[data_link]);
	resume_driver(driver[data_link]);
    }
 bail:
    MUTEX_UNLOCK(&driver_mtx);
    return 0;
}

int
pp_km_reset_driver_is_implemented(unsigned char data_link)
{
    int can_reset = 0;

    MUTEX_LOCK(&driver_mtx);

    if (driver[data_link]) {
	can_reset |= (driver[data_link]->comm_proto.reset_driver != NULL);
	can_reset |= (driver[data_link]->data_unit_mouse.reset_driver != NULL);
	can_reset |= (driver[data_link]->data_unit_kbd.reset_driver != NULL);
	can_reset |= (driver[data_link]->data_conv_mouse.reset_driver != NULL);
	can_reset |= (driver[data_link]->data_conv_kbd.reset_driver != NULL);
	KD("reset_driver_is_implemented() returned: %d\n", can_reset);
    }

    MUTEX_UNLOCK(&driver_mtx);

    return can_reset;
}

int
pp_km_connect_encoder(pp_km_encoder_desc_t* encoder)
{
    km_encoder_desc_data_t *enc_data = ( km_encoder_desc_data_t* ) malloc(sizeof(km_encoder_desc_data_t));
    if (enc_data == NULL) {
    	return PP_ERR;
    }

    enc_data->encoder_hook = encoder;
    MUTEX_LOCK(&km_encoder_hook_mtx);
    list_add(&enc_data->listnode, &km_encoder_hook_list);
    MUTEX_UNLOCK(&km_encoder_hook_mtx);

    return PP_SUC;
}

void
pp_km_disconnect_encoder(pp_km_encoder_desc_t* encoder)
{
    struct list_head *ptr, *ptr2;
    
    MUTEX_LOCK(&km_encoder_hook_mtx);
    list_for_each_safe(ptr, ptr2, &km_encoder_hook_list) {
    	km_encoder_desc_data_t *enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
    	if (enc_data->encoder_hook == encoder) {
    	    list_del(&enc_data->listnode);
    	    free(enc_data);
    	    break;
    	}
    }
    MUTEX_UNLOCK(&km_encoder_hook_mtx);
}

int
pp_km_sync_mouse(unsigned char data_link, int sync_type)
{
    MUTEX_LOCK(&driver_mtx);
    if (driver[data_link]) {
	KD("eric_km_sync_mouse()\n");
	driver[data_link]->data_conv_mouse.sync_mouse(driver[data_link], sync_type);
    }
    MUTEX_UNLOCK(&driver_mtx);
    return 0;
}

int
pp_km_send_keycodes(unsigned char data_link, u_char keys[], size_t length)
{
    int r = -1;

    assert(driver[data_link] != NULL);

    MUTEX_LOCK(&driver_mtx);
    r = driver[data_link]->data_conv_kbd.send_keycodes(
	    driver[data_link], keys, length);
    MUTEX_UNLOCK(&driver_mtx);

    return r;
}

int
pp_km_send_ptrmove(unsigned char data_link, int x, int y, int z, u_char buttonmask, u_char abs_rel)
{
    int r = -1;

    assert( driver[data_link] != NULL );
    MUTEX_LOCK(&driver_mtx);
    r = driver[data_link]->data_conv_mouse.send_ptrmove(
	    driver[data_link], x, y, z, buttonmask, abs_rel);
    MUTEX_UNLOCK(&driver_mtx);
    return r;
}

void
pp_km_reconfigure_all()
{
    u_int i;
    for( i=0; i<current_num_data_links; i++ )
	pp_km_reconfigure(i);
}

/* reconfigures the kme in respect of the current kvm port */
int
pp_km_reconfigure(unsigned char data_link)
{
    driver_t * new_driver;
    int ret = 0;

    KD("eric_km_reconfigure\n");

    MUTEX_LOCK(&driver_mtx);

    new_driver =  (driver_t*) calloc(1, sizeof(driver_t)); /* implicitely clears the memory */
    new_driver->data_link= data_link;
    new_driver->is_equal = is_equal;

    //get video_link associated with this data_link
    if( pp_kvm_get_video_link_for_data_link( data_link, &new_driver->video_link ) != 0 ){
	MUTEX_UNLOCK(&driver_mtx);
	pp_log("%s: no associated video_link with data_link(%d)\n", ___F, data_link );
	free(new_driver);
	if (driver[data_link] != NULL)
	{
	    /* remove old driver */
	    suspend_driver(driver[data_link]);

	    /* cleanup old driver */
	    cleanup_driver(driver[data_link]);
	    pthread_mutex_destroy(&driver[data_link]->state_mtx);
	    free(driver[data_link]);
	    driver[data_link] = NULL;
	}
	return -1;
    }

    pthread_mutex_init(&new_driver->state_mtx, &kvm_mtx_attr);

    /* no mutex lock required since the driver is only known to us for now */
    new_driver->state = DRIVER_NOT_INITIALIZED;
#if PP_KM_USE_IPC_DRIVER
    ret |= configure_ipc_driver(new_driver);
#elif PP_KM_USE_RIPC_DRIVER
    ret |= configure_ripc_driver(new_driver);
#elif PP_KM_USE_KIM_DRIVER
    ret |= configure_kim_driver(new_driver);
#elif PP_KM_USE_USBONLY_DRIVER
    ret |= configure_usbonly_driver(new_driver);
#elif PP_KM_USE_CIM_PROTO_DRIVER
    ret |= configure_cim_proto_driver(new_driver);
#else
# error invalid board
#endif

    /* initially the driver is suspended */
    new_driver->state = DRIVER_SUSPENDED;
    
    if (!ret) {
	if (driver[data_link] == NULL) {
	    /* no current driver */
	    KD("No old km driver, activate the new driver.\n");
	    reconfigure_driver(new_driver);
	    
#if PP_KM_USE_RIPC_DRIVER
	    reset_driver(new_driver);
#endif
	    resume_driver(new_driver);
	    driver[data_link] = new_driver;
	} else {
	    /* if new driver is used, a handover is necessary */
	    
	    /* ok, fine - we have two drivers (one actuall running, and a new one not running
	       now we'll compare them to decide, whether we have to change them */
	    
	    if (driver[data_link]->is_equal(driver[data_link], new_driver)) {
		KD("Old and new km driver are equal, reconfigure the old one.\n");
		/* they are equal - no driver change should be necessary */
		cleanup_driver(new_driver);
		pthread_mutex_destroy(&new_driver->state_mtx);
		free(new_driver);
		/* just reconfigure the old driver */
		reconfigure_driver(driver[data_link]);
	    } else {
		KD("Old and new km driver are not equal, handover necessary.\n");
		/* driver object successfully created */
		suspend_driver(driver[data_link]);

		/* reconfigure should be done in resume */
		/* reconfigure_driver(d); */
		resume_driver(new_driver);

		/* cleanup old driver */
		cleanup_driver(driver[data_link]);
		pthread_mutex_destroy(&driver[data_link]->state_mtx);
		free(driver[data_link]);
		
		/* switch to new driver */
		driver[data_link] = new_driver;
	    }
	}
    } else {
	// driver configuration failed - cleanup
	cleanup_driver(new_driver);
	pthread_mutex_destroy(&driver[data_link]->state_mtx);
	free(new_driver);
    }

#ifdef KBD_DEBUG
    show_driver_ids(driver[data_link]);
#endif
    MUTEX_UNLOCK(&driver_mtx);
    return 0;
}

int
pp_km_comm_proto_ping(unsigned char data_link, char ** info)
{
    int r = -1;

    MUTEX_LOCK(&driver_mtx);    
    if (driver[data_link]) r = driver[data_link]->comm_proto.ping(driver[data_link], info);
    MUTEX_UNLOCK(&driver_mtx);
    return r;
}

int
pp_km_is_ps2_possible(unsigned char /* data_link */)
{
    return ps2_possible;
}

int
pp_km_comm_proto_rw_pin(int is_write, int pin, int *val, unsigned char data_link)
{
    int r = -1;

    MUTEX_LOCK(&driver_mtx);    
    if (driver[data_link] && driver[data_link]->comm_proto.rw_pin)
	r = driver[data_link]->comm_proto.rw_pin(driver[data_link], is_write, pin, val);
    MUTEX_UNLOCK(&driver_mtx);
    return r;
}

int
pp_km_is_usb_possible(unsigned char /* data_link */)
{
    return usb_possible;
}
    
int
pp_km_is_usb_active(unsigned char data_link)
{
    int r = 0;

    MUTEX_LOCK(&driver_mtx);
    if (driver[data_link])  r = (strstr(driver[data_link]->data_unit_kbd.id, "usb")) ? 1 : 0;
    MUTEX_UNLOCK(&driver_mtx);
    return r;
}

int
pp_km_get_keyboard_led_state(unsigned char data_link)
{
    unsigned char r = 0;
    
    MUTEX_LOCK(&driver_mtx);
    if (driver[data_link] != NULL && driver[data_link]->data_unit_kbd.get_led_status) {
        r = driver[data_link]->data_unit_kbd.get_led_status(driver[data_link]);
    }
    MUTEX_UNLOCK(&driver_mtx);
    
    return r;
}

#if PP_KM_USE_RIPC_DRIVER
static int
configure_ripc_driver(driver_t * d)
{
    target_para_t tpt;
    
    init_data_conv_kbd(d);
    init_data_unit_kbd_ps2(d, DEFAULT_PS2_COMMAND_SET);
    init_data_conv_imouse(d);
    init_data_unit_mouse_ps2(d);
    init_comm_proto_usb(d, usb_device_ripcbelkin, peppercon_mouse_type_none);
    tpt.target_type = TARGET_TYPE_I2C;
    init_target_helper(d, &tpt);
    return 0;
}
#endif /* PP_KM_USE_RIPC_DRIVER */

#if PP_KM_USE_CIM_PROTO_DRIVER
static int
configure_cim_proto_driver(driver_t* d)
{
    int usb_use = 0; //assume PS2
    target_para_t tpt;
    int cim_type = 0;
    int cim_mode = MOUSE_TYPE_RELATIVE;

    u_char unit;
    u_short port;

#ifdef USB_TYPE_PER_PORT
    const char *mouse_usb_type_key = "unit[%u].port[%u].mouse.usb_type";
#else 
    const char *mouse_usb_type_key = "usb_type";
#endif //USB_TYPE_PER_PORT

    if(PP_FAILED(pp_kvm_get_unit_port_for_data_link(d->data_link, &unit, &port))) {
	KD("%s: data_link=%d, unit=%d, port=%d\n", ___F, d->data_link, unit, port );
	return PP_ERR;
    }

    assert( pp_portmgr != NULL );
    CPortMgr_Port *pPort = pp_portmgr->FindPort( pKVMBaseDevice, port );
    cim_type = pPort->GetType();
    if( cim_type == PP_PORTMGR_TYPE_PORT_CIM_VM ){
	usb_use = 1;
    }
    else if( cim_type != PP_PORTMGR_TYPE_PORT_CIM_PS2 ){
	KD("%s: incorrect cim_type (%d)\n", ___F, cim_type);
	return PP_ERR;
    }

    usb_possible = 1;

    KD("%s: data_link=%d, port=%d, usb=%d\n", ___F, d->data_link, port, usb_use);

    init_data_conv_kbd(d);

    //usb: can be absolute or relative
    if (usb_use) {
	char * usb_type;
	int usb_abs;

#ifdef USB_TYPE_PER_PORT
	if(PP_FAILED(pp_cfg_get(&usb_type, mouse_usb_type_key, unit, port))){
	    KD("%s: failed to obtain %s", ___F, mouse_usb_type_key);
	    return PP_ERR;
	}
#else
	if(PP_FAILED(pp_cfg_get(&usb_type, mouse_usb_type_key))){
	    KD("%s: failed to obtain %s", ___F, mouse_usb_type_key);
	    return PP_ERR;
	}
#endif //USB_TYPE_PER_PORT
	usb_abs = !pp_strcmp_safe(usb_type, PP_CD_USB_TYPE_ABSOLUTE_STR);
	free(usb_type);

	init_data_unit_kbd_usb(d);

	//TODO: remove the following line when cim supports absolute mouse
	usb_abs = 0;

	if (usb_abs) {
	    // absolute
	    cim_mode = MOUSE_TYPE_ABSOLUTE;
	    init_data_conv_mouse_ausb(d);
	    init_data_unit_mouse_ausb(d);
	} else {
	    // relative
	    init_data_conv_imouse(d);
	    init_data_unit_mouse_rusb(d);
	}
    } else {
	//use PS2
	init_data_unit_kbd_ps2(d, DEFAULT_PS2_COMMAND_SET);
	init_data_conv_imouse(d);
	init_data_unit_mouse_ps2(d);
    }
    tpt.target_type = TARGET_TYPE_DUMMY;

    datalink[d->data_link].object->CIM_SetMouseMode(cim_mode);
    init_comm_proto_cim(d, d->data_link, cim_type, port );

    init_target_helper(d,&tpt);
    return PP_SUC;
}
#endif /* PP_KM_USE_CIM_PROTO_DRIVER */

#if PP_KM_USE_KIM_DRIVER
static int
configure_kim_driver(driver_t * d)
{
    char * input_type;
    int usb_use = 0;
    target_para_t tpt;

#if defined(PP_FEAT_CAT)
    int use_cat = 0;
    u_char unit;
    u_short port;
    u_int channel;

#ifdef PRODUCT_RIPCKIMXN
# error FIXME: Determine channel inside of kvm lib
#endif
    if (pp_kvm_get_unit_and_port(0, &unit, &port) == PP_SUC) {
        pp_cfg_is_enabled(&use_cat, "unit[%u].port[%u].mouse.cat", unit, port);
    }
#endif /* PP_FEAT_CAT */

#ifdef PP_FEAT_USB
    usb_possible = 1;
#else
    usb_possible = 0;
#endif
    /* check if USB or PS2 should be used */
    pp_cfg_get(&input_type, "input_type");
    if (!strcmp(input_type, PP_CD_INPUT_TYPE_AUTO_STR)) {
	usb_use = pp_usb_get_device_state() == PP_USB_DEVSTAT_CONFIGURED;
    } else {
	usb_use = !strcmp(input_type, PP_CD_INPUT_TYPE_USB_STR);
    }
    free(input_type);

    KD("USB use: %d\n", usb_use);

    if (usb_use) {
	/*
	 * use USB
	 */
	char * usb_type;
	int usb_abs;

	pp_cfg_get(&usb_type, "usb_type");
	usb_abs = !pp_strcmp_safe(usb_type, PP_CD_USB_TYPE_ABSOLUTE_STR);
	free(usb_type);

	init_data_conv_kbd(d);
	init_data_unit_kbd_usb(d);
	if (usb_abs) {
	    // absolute
	    init_data_conv_mouse_ausb(d);
	    init_data_unit_mouse_ausb(d);
	    /* necessary for getting the size of the screen */
	    tpt.target_type = TARGET_TYPE_ERICFD;
	    init_target_helper(d, &tpt);
	    init_comm_proto_usb(d, usb_device_peppercon, peppercon_mouse_type_absolute);
	} else {
	    // relative
#if defined(PP_FEAT_CAT)
            if(use_cat) {
                init_data_conv_cat(d);
            } else 
#endif /* PP_FEAT_CAT */
            {
                init_data_conv_imouse(d);
            }
	    init_data_unit_mouse_rusb(d);
	    init_comm_proto_usb(d, usb_device_peppercon, peppercon_mouse_type_relative);
	    /* necessary for getting the size of the screen */
	    tpt.target_type = TARGET_TYPE_ERICFD;
	    init_target_helper(d, &tpt);
	}
    } else {
	/*
	 * use PS2
	 */
	init_data_conv_kbd(d);
#if defined(PP_FEAT_CAT)
        if(use_cat) {
            init_data_conv_cat(d);
        } else 
#endif /* PP_FEAT_CAT */
        {
	    init_data_conv_imouse(d);
        }
	init_data_unit_mouse_ps2(d);
	init_data_unit_kbd_ps2(d, DEFAULT_PS2_COMMAND_SET);
	
	/*
	 * use I2C interface to KME
	 */
	init_comm_proto_pppp_i2c(d, PP_GPIO_ATMEL_RST);
	tpt.target_type = TARGET_TYPE_I2C;
	init_target_helper(d, &tpt);
    }
    return 0;
}
#endif /* PP_KM_USE_KIM_DRIVER */

#if PP_KM_USE_IPC_DRIVER
static int
configure_ipc_driver(driver_t * d)
{
    target_para_t tpt;

    init_data_conv_kbd(d);
    /* FIXME: commented out until they have implemented something */

#if 0
    init_comm_proto_minicom(d);
    tpt.target_type = TARGET_TYPE_I2C;
    init_target_helper(d, &tpt);
#elif 1
    init_data_unit_kbd_usb(d);
    init_data_unit_mouse_ausb(d);
    init_data_conv_mouse_ausb(d);    
    tpt.target_type = TARGET_TYPE_ERICFD;
    init_target_helper(d, &tpt);
    init_comm_proto_usb(d, usb_device_peppercon, peppercon_mouse_type_absolute);
#else
    init_data_conv_kbd(d);
    init_data_unit_kbd_usb(d);
    // relative
    init_data_conv_imouse(d);
    init_data_unit_mouse_rusb(d);
    init_comm_proto_usb(d, usb_device_peppercon, peppercon_mouse_type_relative);
    /* necessary for getting the size of the screen */
    tpt.target_type = TARGET_TYPE_ERICFD;
    init_target_helper(d, &tpt);    
#endif
    return 0;
}
#endif /* PP_KM_USE_IPC_DRIVER */

#if PP_KM_USE_USBONLY_DRIVER
static int configure_usbonly_driver(driver_t *d) {
    int usb_abs;
    target_para_t tpt;
    char * usb_type;

    ps2_possible = 0;
    usb_possible = 1;
    KD("USB-only driver\n");
    
    /*
     * use USB
     */
    pp_cfg_get(&usb_type, "usb_type");
    usb_abs = !pp_strcmp_safe(usb_type, PP_CD_USB_TYPE_ABSOLUTE_STR);
    free(usb_type);
    
    init_data_conv_kbd(d);
    init_data_unit_kbd_usb(d);
    if (usb_abs) {
	// absolute
	init_data_conv_mouse_ausb(d);
	init_data_unit_mouse_ausb(d);
	/* necessary for getting the size of the screen */
	tpt.target_type = TARGET_TYPE_ERICFD;
	init_target_helper(d, &tpt);
	init_comm_proto_usb(d, usb_device_peppercon, peppercon_mouse_type_absolute);
    } else {
	// relative
	init_data_conv_imouse(d);
	init_data_unit_mouse_rusb(d);
	init_comm_proto_usb(d, usb_device_peppercon, peppercon_mouse_type_relative);
	/* necessary for getting the size of the screen */
	tpt.target_type = TARGET_TYPE_ERICFD;
	init_target_helper(d, &tpt);
    }
    return 0;
}
#endif /* PP_KM_USE_USBONLY_DRIVER */

static int
init_target_helper(driver_t * obj, target_para_t* tpt)
{
    assert(tpt);
    switch (tpt->target_type) {

      case TARGET_TYPE_ERICFD:
	  init_target_helper_ericfd(obj, tpt);
	  break;

      case TARGET_TYPE_I2C:
          init_target_helper_i2c(obj, tpt);
          break;

      case TARGET_TYPE_DUMMY:
          init_target_helper_dummy(obj, tpt);
          break;

      default:
	  return 1;
    }
    return 0;
}

static int
suspend_driver(driver_t * obj)
{
    int ret = 0;

    if (!obj) return 0;
    
    MUTEX_LOCK(&obj->state_mtx);
    switch (obj->state) {
      case DRIVER_SUSPENDED:
	  /* do nothing */
	  break;
      case DRIVER_RUNNING:
	  obj->data_conv_kbd.suspend(obj);
	  obj->data_conv_mouse.suspend(obj);
	  obj->data_unit_kbd.suspend(obj);
	  obj->data_unit_mouse.suspend(obj);
	  obj->comm_proto.suspend(obj);
	  obj->state = DRIVER_SUSPENDED;
	  break;
      default:
	  pp_log("driver not initialized - cannot be suspended\n");
	  ret = -1;
    }
    MUTEX_UNLOCK(&obj->state_mtx);
    return ret;
}

static int
reset_driver(driver_t * obj)
{
    int ret = 0;

    if (!obj) return 0;

    MUTEX_LOCK(&obj->state_mtx);
    switch (obj->state) {
      case DRIVER_SUSPENDED:
	  if (obj->data_conv_kbd.reset_driver) obj->data_conv_kbd.reset_driver(obj);
	  if (obj->data_conv_mouse.reset_driver) obj->data_conv_mouse.reset_driver(obj);
	  if (obj->data_unit_kbd.reset_driver) obj->data_unit_kbd.reset_driver(obj);
	  if (obj->data_unit_mouse.reset_driver) obj->data_unit_mouse.reset_driver(obj);
	  if (obj->comm_proto.reset_driver) obj->comm_proto.reset_driver(obj);
	  break;
      default:
	  pp_log("only a suspended driver can have a reset\n");
	  ret = -1;
    }
    MUTEX_UNLOCK(&obj->state_mtx);
    return ret;
}


static int
reconfigure_driver(driver_t * obj)
{
    int ret = 0;

    if (!obj) return 0;

    MUTEX_LOCK(&obj->state_mtx);
    switch (obj->state) {
      case DRIVER_SUSPENDED:
      case DRIVER_RUNNING:
	  obj->data_conv_kbd.reconfigure(obj);
	  obj->data_conv_mouse.reconfigure(obj);
	  obj->data_unit_kbd.reconfigure(obj);
	  obj->data_unit_mouse.reconfigure(obj);
	  obj->comm_proto.reconfigure(obj);
	  break;
      default:
	  pp_log("only a running or suspended driver can be reconfigured\n");
	  ret = -1;
    }
    MUTEX_UNLOCK(&obj->state_mtx);
    return ret;
}

static int
resume_driver(driver_t * obj)
{
    int ret = 0;

    if (!obj) return 0;

    MUTEX_LOCK(&obj->state_mtx);
    switch (obj->state) {
      case DRIVER_SUSPENDED:
	  reconfigure_driver(obj); /* obj needs recursive mutex */
	  obj->comm_proto.resume(obj);
	  obj->data_unit_mouse.resume(obj);
	  obj->data_unit_kbd.resume(obj);
	  obj->data_conv_mouse.resume(obj); 
	  obj->data_conv_kbd.resume(obj);
	  obj->state = DRIVER_RUNNING;
	  break;
      default:
	  pp_log("only a suspended driver can be resumed\n");
	  ret = -1;
    }
    MUTEX_UNLOCK(&obj->state_mtx);
    return ret;
}

static int
cleanup_driver(driver_t * obj)
{
    int ret = 0;

    if (!obj) return 0;

    MUTEX_LOCK(&obj->state_mtx);
    switch (obj->state) {
      case DRIVER_SUSPENDED:
	  obj->data_conv_kbd.cleanup(obj);
	  obj->data_conv_mouse.cleanup(obj);
	  obj->data_unit_kbd.cleanup(obj);
	  obj->data_unit_mouse.cleanup(obj);
	  obj->comm_proto.cleanup(obj);
	  obj->target_helper.cleanup(obj);
	  obj->state = DRIVER_NOT_INITIALIZED;
	  break;
      default:
	  pp_log("only a suspended driver can be cleanuped\n");
	  ret = -1;
    }
    MUTEX_UNLOCK(&obj->state_mtx);
    return ret;
}

#ifdef KBD_DEBUG
static void
show_driver_ids(driver_t * obj)
{
    if (!obj) return;

    MUTEX_LOCK(&obj->state_mtx);
    switch (obj->state) {
      case DRIVER_SUSPENDED:
      case DRIVER_RUNNING:
	  printf("driver consists of the following subdrivers:\n");
	  printf("%s\n", obj->data_conv_kbd.id);
	  printf("%s\n", obj->data_conv_mouse.id);
	  printf("%s\n", obj->data_unit_kbd.id);
	  printf("%s\n", obj->data_unit_mouse.id);
	  printf("%s\n", obj->comm_proto.id);
	  printf("%s\n", obj->target_helper.id);
	  break;
      default:
	  pp_log("only a suspended or running driver can be resumed\n");
    }
    MUTEX_UNLOCK(&obj->state_mtx);
}
#endif /* KBD_DEBUG */
    
static int
is_equal(driver_t * obj, driver_t * d)
{
    int ne = 0;

    MUTEX_LOCK(&obj->state_mtx);
    MUTEX_LOCK(&d->state_mtx);
    if ((obj->state == DRIVER_RUNNING || obj->state == DRIVER_SUSPENDED) &&
	(d->state == DRIVER_RUNNING || d->state == DRIVER_SUSPENDED)) {
	ne |= !obj->data_conv_kbd.is_equal(obj, d);
	ne |= !obj->data_conv_mouse.is_equal(obj, d);
	ne |= !obj->data_unit_kbd.is_equal(obj, d);
	ne |= !obj->data_unit_mouse.is_equal(obj, d);
	ne |= !obj->comm_proto.is_equal(obj, d);
	ne |= !obj->target_helper.is_equal(obj, d);
    } else {
	ne = 1;
	pp_log("one of the drivers is not initialized\n");
    }
    MUTEX_UNLOCK(&d->state_mtx);
    MUTEX_UNLOCK(&obj->state_mtx);

    return !ne;
}


static int
reconf_ch(pp_cfg_chg_ctx_t *ctx, u_char data_link)
{
    int kbd_model_changed = 0;
    int kbd_break_timeout_enabled_changed = 0;
    int kbd_reak_timeout_msec_changed = 0;

    if( driver[data_link] == NULL ){
	KD("%s: driver[%d] not created yet.\n", ___F, data_link );
	return PP_ERR;
    }
    kbd_data_t * kbd_data = (kbd_data_t*)driver[data_link]->data_conv_kbd.data;;
    u_int unit, port;

    pp_log("Reconfigure KM subsystem\n");
    /* notify clients about new kbd layout */

/* FIXME! DEADLOCK! when simultaneously intelli-syncing the mouse (lock driver, lock config to save mouse universe) and applying changes via the web frontend (lock config within tx, call reconf_ch and lock driver), obj may lead to a deadlock (tweb)  */
    MUTEX_LOCK(&driver_mtx);
    if (ctx) {
	size_t key_comp_vec_sz, i;
	key_comp_vec_sz = vector_size(ctx->key_comp_vec);
	for (i = 0; i < key_comp_vec_sz; ++i) {
	    vector_t * key_comps = (vector_t *)vector_get(ctx->key_comp_vec, i);
	    const char *key_comp[8];
	    u_int comp_no;

	    unit = UINT_MAX;
	    port = UINT_MAX;

	    if (key_comps == NULL || vector_size(key_comps) < 1) continue;

	    comp_no = 0;
	    key_comp[comp_no] = pp_cfg_get_key_comp_name_at_idx(key_comps, comp_no);
	    if (!pp_strcmp_safe(key_comp[comp_no], "unit")) {
		++comp_no;
		key_comp[comp_no] = pp_cfg_get_key_comp_name_at_idx(key_comps, comp_no);
		if (!pp_strcmp_safe(key_comp[comp_no], "_e_")) {
		    ++comp_no;
		    key_comp[comp_no] = pp_cfg_get_key_comp_name_at_idx(key_comps, comp_no);
		    unit = pp_strtoul_10(key_comp[comp_no], UINT_MAX, NULL);
		    ++comp_no;
		    key_comp[comp_no] = pp_cfg_get_key_comp_name_at_idx(key_comps, comp_no);
		}
		if (!pp_strcmp_safe(key_comp[comp_no], "port")) {
		    ++comp_no;
		    key_comp[comp_no] = pp_cfg_get_key_comp_name_at_idx(key_comps, comp_no);
		    if (!pp_strcmp_safe(key_comp[comp_no], "_e_")) {
			++comp_no;
			key_comp[comp_no] = pp_cfg_get_key_comp_name_at_idx(key_comps, comp_no);
			port = pp_strtoul_10(key_comp[comp_no], UINT_MAX, NULL);
			++comp_no;
			key_comp[comp_no] = pp_cfg_get_key_comp_name_at_idx(key_comps, comp_no);
		    }
		    if (!pp_strcmp_safe(key_comp[comp_no], "kbd")) {
			++comp_no;
			key_comp[comp_no] = pp_cfg_get_key_comp_name_at_idx(key_comps, comp_no);
			if (!pp_strcmp_safe(key_comp[comp_no], "model")) {
			    kbd_model_changed = 1;
			} else if (!pp_strcmp_safe(key_comp[comp_no], "break_timeout_enabled")) {
			    kbd_break_timeout_enabled_changed = 1;
			} else if (!pp_strcmp_safe(key_comp[comp_no], "break_timeout_msec")) {
			    kbd_reak_timeout_msec_changed = 1;
			}
		    }
		}
	    }
	}

	if (kbd_model_changed) {
	    u_int my_data_link = data_link;
#ifdef PRODUCT_RIPCKIMXN
	    if (data_link != 0) {
		assert(0);
		/* FIXME: (janb) data_link had to be 0 ! */
	    }
#endif
	    u_char current_unit;
	    u_short current_port;
	    if (pp_kvm_get_unit_port_for_data_link(my_data_link, &current_unit, &current_port) == PP_SUC) {
		char * kbd_model;
		struct list_head *ptr;
		km_encoder_desc_data_t *enc_data;
		pp_cfg_get(&kbd_model, "unit[%u].port[%u].kbd.model", current_unit, current_port);
		MUTEX_LOCK(&km_encoder_hook_mtx);
		list_for_each(ptr, &km_encoder_hook_list) {
		    enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
		    if (enc_data->encoder_hook && enc_data->encoder_hook->encoder_send_kbdlayout) {
		    	enc_data->encoder_hook->encoder_send_kbdlayout(enc_data->encoder_hook, my_data_link, kbd_model);
		    }
		}
		MUTEX_UNLOCK(&km_encoder_hook_mtx);
		free(kbd_model);
	    }
	    kbd_model_changed = 0;
	}
	if (kbd_break_timeout_enabled_changed) {
	    pp_cfg_is_enabled(&kbd_data->timeout_enabled, "unit.port.kbd.break_timeout_enabled");
	    kbd_break_timeout_enabled_changed = 0;
	}
	if (kbd_reak_timeout_msec_changed) {
	    pp_cfg_get_uint(&kbd_data->timeout, "unit.port.kbd.break_timeout_msec");
	    kbd_reak_timeout_msec_changed = 0;
	}
    }
    MUTEX_UNLOCK(&driver_mtx);

    /* reconfigure km library and notify clients about new kbd layout */
    pp_km_reconfigure(data_link);
    pp_km_sync_mouse(data_link, PP_KM_MOUSE_SYNC_ACCEL_STATE);

    return PP_SUC;
}

static int
reconf_channels(pp_cfg_chg_ctx_t *ctx)
{
    u_int i = 0;

    for( i=0; i<current_num_data_links; i++)
	reconf_ch( ctx, i );

    return 0;
}



static void
propchange_handler(UNUSED pp_propchange_listener_t * /* listener */, u_short prop, u_short prop_flags)
{
    unsigned char data_link = prop_flags;
    switch (prop) {
      case PP_PROP_AUTO_AUTO_ADJUSTMENT_DONE:
#if 0
	  /* removing obj for know since we got complaints ...
	     when a local user is sitting in front of the device
	     and changes KVM port, auto aa starts and after obj
	     the long mouse sync, which sucks. If we have a way
	     to detect activity on the local console and can avoid
	     the sync in obj case we can reenable it. */
	  // auto autoadjustment is done - so do a mouse sync
	  pp_grab_inc_clients_count();
	  pp_km_sync_mouse(0, PP_KM_MOUSE_SYNC_HARD);
	  pp_km_send_ptrmove(0, 0, 0, 0, PP_KM_PTR_MOVE_ABSOLUTE);
	  pp_grab_dec_clients_count();
#endif
	  break;

      case PP_PROP_USB_DEV_STATE_CHANGED:
#if PP_KM_USE_KIM_DRIVER
	  // usb device state may have changed -> reconfigure driver
	  {
	      static int usb_use_old = 0;
	      /* NOTE: the ?: operator is necessary to make sure that "true" is 1 and "false" is 0 */
	      int usb_use_new = pp_usb_get_device_state() == PP_USB_DEVSTAT_CONFIGURED ? 1 : 0;
	      if (usb_use_old ^ usb_use_new) {
		  usb_use_old = usb_use_new;
		  pthread_mutex_lock(&usb_propchange_mutex);
		  if (usb_use_new) {
		      pthread_cond_broadcast(&usb_resume_cond);
		  } else {
		      pthread_cond_broadcast(&usb_suspend_cond);
		  }
		  pthread_mutex_unlock(&usb_propchange_mutex);
		  if (!ignore_usb_propchange) {
		      KD("reconfigure driver because of USB device state change\n");
		      pp_km_reconfigure(0);
		  }
	      }
	  }
#endif
	  break;
      default:
	  pp_log("%s(): Unhandled property %hu received. Channel=%d\n", ___F, prop, data_link);
	  break;
    }
}
