#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <liberic_pthread.h>
#include <liberic_misc.h>
#include <liberic_config.h>
#include <pp/powerswitch.h>
#include <pp/termios.h>

#include "debug.h"
#include "power_switch.h"

static int init(power_switch_t * this, ps_data_t * ps_data);
static int get_count(power_switch_t * this, ps_data_t * ps_data, int device_id, int object);
static char* get_name(power_switch_t * this, ps_data_t * ps_data, int idx, int device_id, int object);
static int power_switch(power_switch_t * this, ps_data_t * ps_data, int port, int device, u_char on, int seq);
static int get_state(power_switch_t * this, ps_data_t * ps_data, int port, int device);

static int setdtr(int fd, int on);
#if 0
static int get_outlet_count(void);
#endif

/* -------------------- exported functions ---------------------------- */
int
power_switch_btec_init(power_switch_t *this)
{
    this->init		= init;
    this->get_state	= get_state;
    this->power_switch	= power_switch;
    this->get_name	= get_name;
    this->get_count	= get_count;
    
    this->caps = POWER_CAP_SWITCH | POWER_CAP_GET_STATE;

    return PP_SUC;
}

void
power_switch_btec_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;
    free(entry);
}

static int
init(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    u_char activate[3] = { 0x02, 0x05, 0x03 };
    int ret = PP_ERR;

    if ((ps_data->btec.fd = ps_get_fd_by_port_id(ps_data->port_id)) < 0) {
	goto error;
    }

    if (PP_FAILED(pp_base_set_tty_params(ps_data->btec.fd, 9600, "", 8, 0, 0, 0))) {
	goto error;
    }       
    
    if (PP_FAILED(setdtr(ps_data->btec.fd, 0))) {
	D(D_ERROR, "BTEC: %s(): clearing DTR failed\n", ___F);
	goto error;
    }

    usleep(300000);
    
    if (write(ps_data->btec.fd, activate, 3) != 3) {
	D(D_ERROR, "BTEC: %s(): writing activation sequence failed\n", ___F);
	setdtr(ps_data->btec.fd, 1);
	goto error;
    }
    
    usleep(300000);

    if (PP_FAILED(setdtr(ps_data->btec.fd, 1))) {
	D(D_ERROR, "BTEC: %s(): setting DTR failed\n", ___F);
	goto error;
    }
       
    usleep(300000);

    ret = PP_SUC;

 error:
    return ret;
}

static int
get_count(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	  int device_id UNUSED, int object)
{
    if (object == PWR_OBJ_PORT) return 8;
    else if (object == PWR_OBJ_DEVICE) return 1;
    else return 0; 
}

static char *
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int idx, int device_id UNUSED, int object)
{    
    char * val = NULL;

    switch (object) {
      case PWR_OBJ_PORT:
	  asprintf(&val, "Outlet %d", idx + 1);
	  return val;
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_BTEC_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_BTEC_NAME);
      default:
	  return NULL;
    }
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data,
	     int port, int device UNUSED, u_char on, int seq)
{
    ssize_t cmd_len;
    char cmd_buf[16];
    char rsp_buf[256];
    int ret = PP_ERR;

    if (seq == PWR_SWITCH_SEQ) {
	D(D_ERROR, "BTEC: %s(): sequential switch not supported!\n", ___F);
	goto error;
    }
    
    snprintf(cmd_buf, sizeof(cmd_buf), "%s %d\r", on ? "SNMPON" : "SPOFF", port + 1);

    cmd_len = strlen(cmd_buf);
    if (write(ps_data->btec.fd, cmd_buf, cmd_len) != cmd_len) {
        D(D_ERROR, "BTEC: %s(): sending command failed\n", ___F);
    }

    if (PP_FAILED(ps_read_response(ps_data->btec.fd, 0, cmd_buf, 0, 500000,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	D(D_ERROR, "BTEC: %s(): no response to command", ___F);
	goto error;
    }

    usleep(500000);

    pp_propchange_enqueue(on ? PP_PROP_PWR_ON : PP_PROP_PWR_OFF, 0);

    ret = PP_SUC;

 error:
    return ret;
}

static int
get_state(power_switch_t *this UNUSED, ps_data_t * ps_data, int port, int device UNUSED)
{
    ssize_t cmd_len;
    const char * cmd_buf = "SNMPS\r";
    char rsp_buf[256];
    char stat_buf[32];
    
    memset(stat_buf, 0, sizeof(stat_buf));

    cmd_len = strlen(cmd_buf);
    if (write(ps_data->btec.fd, cmd_buf, cmd_len) != cmd_len) {
        D(D_ERROR, "BTEC: %s(): sending command failed\n", ___F);
    }
    
    if (PP_FAILED(ps_read_response(ps_data->btec.fd, 0, NULL, 0, 500000,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_MAX_CHR))) {
	D(D_ERROR, "BTEC: %s(): no response to command", ___F);
	goto error;
    }

    if (rsp_buf && strlen(rsp_buf) > 0) {
	if (sscanf(rsp_buf, "\r\nRPC%*[^>]>SNMPS\r\n%s", stat_buf) != 1
	    || !strcmp(stat_buf, "")) {
	    sscanf(rsp_buf, "SNMPS\r\n%s", stat_buf);
	}
    }
    
    usleep(500000);

    if (stat_buf[port] == '0') {
	return 0;
    } else if (stat_buf[port] == '1') {
        return 1;
    }
 error:
    return -1;
}

/* -------------------- internal functions ---------------------------- */

/* this is currently unsued because of speed optimization */
#if 0
static int
get_outlet_count(void)
{
    char retbuf[36];
    int count;
    char action[16];
    
    snprintf(action, 16, "SNMPH\r");
    if (write(btec_fd, action, 6) != 6) {
        D(D_ERROR, "BTEC: %s(): sending command failed\n", ___F);
    }
    
    if (PP_FAILED(ps_read_response(btec_fd, 0, NULL, 0, 500000, retbuf, sizeof(retbuf),
				   NULL, TTY_READ_MAX_CHR))) {
	D(D_ERROR, "BTEC: %s(): no response to command", ___F);
	goto error;
    }
    
    if (sscanf(retbuf, "\r\nRPC%*[^>]>SNMPH\r\n%*[^,],%*[^,],%d,%*s", &count) != 1) {
	if (sscanf(retbuf, "SNMPH\r\n%*[^,],%*[^,],%d,%*s", &count) != 1) {
	    goto error;
	}
    }
            
    usleep(500000);

    return count; 
        
error:
    return -1;
}
#endif

/*
 *   Set or Clear DTR modem control line
 *
 *   Note:  TIOCMBIS:   CoMmand BIt Set
 *          TIOCMBIC:   CoMmand BIt Clear
 *
 */
static int
setdtr(int fd, int on)
{
    int controlbits = TIOCM_DTR;
    return ioctl(fd, (on ? TIOCMBIS : TIOCMBIC), &controlbits) == 0 ? PP_SUC : PP_ERR;
}
