#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <liberic_pthread.h>
#include <liberic_misc.h>
#include <liberic_config.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/powerswitch.h>
#include <pp/termios.h>

#include "debug.h"
#include "power_switch.h"

#define PWR_READ_TIMEOUT 600*1000

/*** timing constraints when sending to power switch (see manual) ***/
/* required time between commands */
#define CMD_DELAY 40*1000
/* required time between chars */
#define CHAR_DELAY 10*1000

static int init(power_switch_t *this, ps_data_t * ps_data);
static void uninit(power_switch_t *this, ps_data_t * ps_data);
static int login(power_switch_t *this, ps_data_t * ps_data);
static int get_count(power_switch_t *this, ps_data_t * ps_data, int device_id, int object);
static char* get_name(power_switch_t *this, ps_data_t * ps_data, int idx, int device_id, int object);
static int power_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);
static int get_state(power_switch_t *this, ps_data_t * ps_data, int port, int device);
static char* get_temperature(power_switch_t *this, ps_data_t * ps_data, int device);
static char* get_data(power_switch_t* this, ps_data_t * ps_data, int device, int channel, int type);
static int ext_switch(power_switch_t* this, ps_data_t * ps_data, int port, int device, u_char on);
static int get_ext_state(power_switch_t* this, ps_data_t * ps_data, int port, int device);

static int do_switch(ps_smart_t * ps_smart, int port, int device, u_char on);
static int do_seq_switch(ps_smart_t * ps_smart, int device, u_char on);
static int get_ids(ps_smart_t * ps_smart, char *addr_hbyte, char *addr_lbyte, int port_id);

static void* smart_thread_func(void* arg);

/* -------------------- exported functions ---------------------------- */
int
power_switch_smart_init(power_switch_t *this)
{
    this->init			= init;
    this->uninit		= uninit;
    this->login			= login;
    this->get_state		= get_state;
    this->power_switch		= power_switch;
    this->get_name		= get_name;
    this->get_count		= get_count;
    this->get_data		= get_data;
    this->get_temperature	= get_temperature;
    this->ext_switch		= ext_switch;
    this->get_ext_state		= get_ext_state;
   
    this->caps = POWER_CAP_SWITCH | POWER_CAP_GET_STATE | POWER_CAP_GET_TEMP |
	         POWER_CAP_SWITCH_EXT | POWER_CAP_SWITCH_SEQ;
    return PP_SUC;
}

void
power_switch_smart_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;
    free(entry);
}

static int
init(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    ps_data->smart.smart_thread_running = 0;
    if (ps_data->smart.port_id == PP_POWER_PORT_ID_SERIAL_1) {
	ps_data->smart.devaddrcnt_var_name = "ps.serial[0].smart._s_";
	ps_data->smart.devaddr_var_name = "ps.serial[0].smart[%u].dev_addr";
	ps_data->smart.devtype_var_name = "ps.serial[0].smart[%u].dev_type";
    } else if (ps_data->smart.port_id == PP_POWER_PORT_ID_SERIAL_2) {
	ps_data->smart.devaddrcnt_var_name = "ps.serial[1].smart._s_";
	ps_data->smart.devaddr_var_name = "ps.serial[1].smart[%u].dev_addr";
	ps_data->smart.devtype_var_name = "ps.serial[1].smart[%u].dev_type";
    } else {
	ps_data->smart.devaddrcnt_var_name = NULL;
	ps_data->smart.devaddr_var_name = NULL;
	ps_data->smart.devtype_var_name = NULL;
    }

    if ((ps_data->smart.fd = ps_get_fd_by_port_id(ps_data->port_id)) < 0) {
	return PP_ERR;
    }

    if (PP_FAILED(pp_base_set_tty_params(ps_data->smart.fd, 9600, "", 8, 1, 0, 0))) {
	return PP_ERR;
    }
    
    return PP_SUC;
}

static void
uninit(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    int i;
    if (ps_data->smart.smart_thread_running) {
	pthread_cancel(ps_data->smart.smart_thread);
	for (i = 0; i < 50 && ps_data->smart.smart_thread_running; ++i) {
	    usleep(200000);
	}
	if (ps_data->smart.smart_thread_running) {
	    D(D_ERROR, "SMART: %s(): canceling thread failed\n", ___F);
	}
    }
}

static int
login(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    if (ps_data->smart.smart_thread_running) return 1; /* 1 means busy */
    return PP_SUC;
}

static int
get_count(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	  int device_id UNUSED, int object)
{
    int n = 0;

    switch (object) {
      case PWR_OBJ_PORT: return 8;
      case PWR_OBJ_DEVICE:
	  if (ps_data->smart.devaddrcnt_var_name) {
	      pp_cfg_get_int_nodflt(&n, ps_data->smart.devaddrcnt_var_name);
	  }
	  return n;
      case PWR_OBJ_EXTERNAL:
	  return 2;
      default:
	  return 0;  
    }
}

static char*
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int idx, int device_id, int object)
{
    char * dev_type = NULL;
    char * dev_addr = NULL;
    char * val = NULL;

    switch (object) {
      case PWR_OBJ_PORT:
	  if (ps_data->smart.devtype_var_name
	      && PP_SUCCED(pp_cfg_get_nodflt(&dev_type, ps_data->smart.devtype_var_name, device_id))
	      && strstr(dev_type, "2x4")) {
	      asprintf(&val, "Port %c.%d", idx < 4 ? 'A' : 'B', idx + 1);
	  } else {
	      asprintf(&val, "Port %d", idx + 1);
	  }
	  free(dev_type);
	  return val;
      case PWR_OBJ_DEVICE:
	  if (ps_data->smart.devaddr_var_name
	      && PP_SUCCED(pp_cfg_get_nodflt(&dev_addr, ps_data->smart.devaddr_var_name, device_id))) {
	      if (device_id == 0) {
		  asprintf(&val, "Master Unit ( 0x%s )\n", dev_addr);
	      } else {
		  asprintf(&val, "Slave Unit %d ( 0x%s )\n", device_id, dev_addr);
	      }
	  } else {
	      if (device_id == 0) {
		  asprintf(&val, "Master Unit\n");
	      } else {
		  asprintf(&val, "Slave Unit %d\n", device_id);
	      }
	  }
	  free(dev_addr);
	  return val;
      case PWR_OBJ_EXTERNAL:
	  asprintf(&val, "External Switch %d\n", idx + 1);
	  return val;
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_SMART_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_SMART_NAME);
      default:
	  return NULL;
    }
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data,
	     int port, int device, u_char on, int seq)
{
    int ret = PP_ERR;
    if (seq == PWR_SWITCH_SEQ) {
	if (PP_SUCCED(ret = do_seq_switch(&ps_data->smart, device, on))) {
	    pp_propchange_enqueue(on ? PP_PROP_PWR_ON : PP_PROP_PWR_OFF, 0);
	}
    } else if (seq == PWR_SWITCH_NORMAL) {
	if (PP_SUCCED(ret = do_switch(&ps_data->smart, port, device, on))) {
	    pp_propchange_enqueue(on ? PP_PROP_PWR_ON : PP_PROP_PWR_OFF, 0);
	}
    } else {
	D(D_ERROR, "SMART: %s(): unknown switch mode (%d)\n", ___F, seq);
    }
    return ret;
}

static int
get_state(power_switch_t *this UNUSED, ps_data_t * ps_data,
	  int port, int device)
{
    int  i, state = -1;
    char *strptr;
    char id_buf[2];
    char rsp_buf[32];    
    char cmd_buf[8] = {  0x41,       // A
			 0x46,       // Device address high byte
			 0x46,       // Device address low byte
			 0x72,       // r
			 0x30,       // 0
			 0x00,       // Outlet ID      
			 0x0D,       // Carriage Return
			 0x00 };     

    snprintf(id_buf, sizeof(id_buf), "%1d", port + 1);

    usleep(CMD_DELAY);

    /* Read remaining data from input buffer */
    ps_read_response(ps_data->smart.fd, 0, NULL, 0, 0,
		     rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_MAX_CHR);
    
    if (PP_FAILED(get_ids(&ps_data->smart, &cmd_buf[1], &cmd_buf[2], device))) return -1;
    cmd_buf[5] = id_buf[0];
    
    for (i = 0; i < 7; i++) {
        usleep(CHAR_DELAY);
        if (write(ps_data->smart.fd, &cmd_buf[i], 1) != 1) {
	    D(D_ERROR, "SMART: %s(): set command\n", ___F);
	    goto error;
        }
    }

    *rsp_buf = 0;
    /* first try, looking for carriage return */
    if (PP_FAILED(ps_read_response(ps_data->smart.fd, 0x0D, NULL, 0, PWR_READ_TIMEOUT,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	D(D_ERROR, "SMART: %s(): set command - no response\n", ___F);
	goto error;
    }
    D(D_VERBOSE, "Resp ok: %s\n", rsp_buf);
    /* a slave device returns the sent command first before the response,
     * in this case, we must look for another CR */
    if (strstr(rsp_buf, cmd_buf)) {
	*rsp_buf = 0;
	if (PP_FAILED(ps_read_response(ps_data->smart.fd, 0x0D, NULL, 0, PWR_READ_TIMEOUT,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	    D(D_ERROR, "SMART: %s(): set command - no response on 2nd try\n", ___F);
	    goto error;
	}
    }
    cmd_buf[0] = 'U';
    cmd_buf[6] = 0;

    if ((strptr = strstr(rsp_buf, cmd_buf))) {
	if (strptr[7] == '0') {
	    state = 0;
	} else if (strptr[7] == '1') {
	    state = 1;
 	}
    }

 error:
    return state;
}

static char*
get_temperature(power_switch_t *this UNUSED, ps_data_t * ps_data, int device)
{
    int  i;
    char *strptr, *ret = NULL;
    const char *mask = "Temp\r";
    char rsp_buf[32];
    char cmd_buf[6] = {  0x41,       // A
			 0x46,       // Device address high byte
			 0x46,       // Device address low byte
			 0x74,       // t
			 0x0D,       // Carriage Return
			 0x00 };

    usleep(CMD_DELAY);

    /* Read remaining data from input buffer */
    ps_read_response(ps_data->smart.fd, 0, NULL, 0, 0,
		     rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_MAX_CHR);
    
    if (PP_FAILED(get_ids(&ps_data->smart, &cmd_buf[1], &cmd_buf[2], device))) return NULL;
    for (i = 0; i < 5; i++) {
        usleep(CHAR_DELAY);
        if (write(ps_data->smart.fd, &cmd_buf[i], 1) != 1) {
	    D(D_ERROR, "SMART: %s(): set command\n", ___F);
	    goto error;
        }
    }

    *rsp_buf = 0;
    if (PP_FAILED(ps_read_response(ps_data->smart.fd, 0, mask, 0, PWR_READ_TIMEOUT,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	D(D_ERROR, "SMART: %s(): set command - no response\n", ___F);
	goto error;
    }
    cmd_buf[0] = 'U';
    cmd_buf[4] = 0;
    
    if ((strptr = strstr(rsp_buf, cmd_buf)) ) {
	ret = strdup(strptr + 5);
    }

 error:
    return ret;
}
    
static char*
get_data(power_switch_t* this UNUSED, ps_data_t * ps_data,
	 int device, int channel, int type)
{
    int  i;
    char *strptr, *ret = NULL;
    const char *mask;
    char rsp_buf[64];    
    char cmd_buf[7] = {  0x41,       // A
			 0x46,       // Device address high byte
			 0x46,       // Device address low byte
			 0x46,       // F
			 0x00,       // Channel A(0) or B(1)
			 0x0D,       // Carriage Return
			 0x00 };

    usleep(CMD_DELAY);

    /* Read remaining data from input buffer */
    ps_read_response(ps_data->smart.fd, 0, NULL, 0, 0,
		     rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_MAX_CHR);

    if (PP_FAILED(get_ids(&ps_data->smart, &cmd_buf[1], &cmd_buf[2], device))) goto error;
    switch (type) {
      case PWR_DATA_VOLTAGE:
	  mask = "Volts\r";
	  cmd_buf[3] = 'V';
	  break;
      case PWR_DATA_CURRENT:
	  mask = "Amps\r";
	  cmd_buf[3] = 'I';
	  break;
      case PWR_DATA_FREQ:
	  mask = "HZ\r";
	  cmd_buf[3] = 'F';
	  break;
      default:
	  goto error;
    }
    cmd_buf[4] = channel ? 'B' : 'A';
        
    for (i = 0; i < 6; i++) {
        usleep(CHAR_DELAY);
        if (write(ps_data->smart.fd, &cmd_buf[i], 1) != 1) {
	    D(D_ERROR, "SMART: %s(): set command\n", ___F);
	    goto error;
        }        
    }

    *rsp_buf = 0;
    if (PP_FAILED(ps_read_response(ps_data->smart.fd, 0, mask, 0, PWR_READ_TIMEOUT,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	D(D_ERROR, "SMART: %s(): set command - no response\n", ___F);
        goto error;
    }
    cmd_buf[0] = 'U';
    cmd_buf[4] = 0;

    if ((strptr = strstr(rsp_buf, cmd_buf))) {
	ret = strdup(strptr + 5);
    }

 error:
    return ret;
}

static int
ext_switch(power_switch_t* this UNUSED, ps_data_t * ps_data,
	   int port, int device, u_char on)
{
    int i, ret = PP_ERR;
    char rsp_buf[32];    
    char cmd_buf[10] = { 0x41,       // A
			 0x46,       // Device address high byte
			 0x46,       // Device address low byte
			 0x43,       // C
			 0x00,       // External contact A or B
			 0x20,       // Space
			 0x30,       // 0
			 0x00,       // 0 = Off, 1 = On
			 0x0D,       // Carriage Return
			 0x00 };

    usleep(CMD_DELAY);

    /* Read remaining data from input buffer */
    ps_read_response(ps_data->smart.fd, 0, NULL, 0, 0,
		     rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_MAX_CHR);
    
    if (PP_FAILED(get_ids(&ps_data->smart, &cmd_buf[1], &cmd_buf[2], device))) goto error;

    if (port == 0) cmd_buf[4] = 'A';
    else if (port == 1) cmd_buf[4] = 'B';
    else goto error;

    cmd_buf[7] = on ? '1' : '0';
        
    for (i = 0; i < 9; i++) {
        usleep(CHAR_DELAY);
        if (write(ps_data->smart.fd, &cmd_buf[i], 1) != 1) {
	    D(D_ERROR, "SMART: %s(): set command\n", ___F);
	    goto error;
        }        
    }

    *rsp_buf = 0;
    /* first try, looking for carriage return */
    if (PP_FAILED(ps_read_response(ps_data->smart.fd, 0x0D, NULL, 0, PWR_READ_TIMEOUT,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	D(D_ERROR, "SMART: %s(): set command - no response\n", ___F);
	goto error;
    }
    /* a slave device returns the sent command first before the response,
     * in this case, we must look for another CR */
    if (strstr(rsp_buf, cmd_buf)) {
	*rsp_buf = 0;
	if (PP_FAILED(ps_read_response(ps_data->smart.fd, 0x0D, NULL, 0, PWR_READ_TIMEOUT,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	    D(D_ERROR, "SMART: %s(): set command - no response on 2nd try\n", ___F);
	    goto error;
	}
    }
    cmd_buf[0] = 'U';
    cmd_buf[5] = 0;
    
    ret = strstr(rsp_buf, cmd_buf) ? PP_SUC : PP_ERR;

 error:
    return ret;
}

static int
get_ext_state(power_switch_t* this UNUSED, ps_data_t * ps_data,
	      int port, int device)
{
    int i, ret = -1;
    char *retptr;
    char rsp_buf[32];    
    char cmd_buf[7] = {  0x41,       // A
			 0x46,       // Device address high byte
			 0x46,       // Device address low byte
			 0x43,       // C
			 0x00,       // External contact A or B		
			 0x0D,       // Carriage Return
			 0x00 };

    usleep(CMD_DELAY);

    /* Read remaining data from input buffer */
    ps_read_response(ps_data->smart.fd, 0, NULL, 0, 0,
		     rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_MAX_CHR);
    
    if (PP_FAILED(get_ids(&ps_data->smart, &cmd_buf[1], &cmd_buf[2], device))) goto error;

    if (port == 0) cmd_buf[4] = 'A';
    else if (port == 1) cmd_buf[4] = 'B';
    else goto error;
   
    for (i = 0; i < 6; i++) {
        usleep(CHAR_DELAY);
        if (write(ps_data->smart.fd, &cmd_buf[i], 1) != 1) {
	    D(D_ERROR, "SMART: %s(): set command\n", ___F);
	    goto error;
        }        
    }

    *rsp_buf = 0;
    /* first try, looking for carriage return */
    if (PP_FAILED(ps_read_response(ps_data->smart.fd, 0x0D, NULL, 0, PWR_READ_TIMEOUT,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	D(D_ERROR, "SMART: %s(): set command - no response\n", ___F);
	goto error;
    }
    /* a slave device returns the sent command first before the response,
     * in this case, we must look for another CR */
    if (strstr(rsp_buf, cmd_buf)) {
	*rsp_buf = 0;
	if (PP_FAILED(ps_read_response(ps_data->smart.fd, 0x0D, NULL, 0, PWR_READ_TIMEOUT,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	    D(D_ERROR, "SMART: %s(): set command - no response on 2nd try\n", ___F);
	    goto error;
	}
    }
    cmd_buf[0] = 'U';
    cmd_buf[5] = 0;
    
    if ((retptr = strstr(rsp_buf, cmd_buf)) != NULL) {
	if (retptr[7] == '0') ret = 0;
	else if ( retptr[7] == '1' ) ret = 1;
	else if ( retptr[7] == 'F' ) ret = 2;
	else if ( retptr[7] == 'D' ) ret = 3;
	else if ( retptr[7] == 'C' ) ret = 4;
    }

 error:
    return ret;
}

/* -------------------- internal functions ---------------------------- */

static int
do_switch(ps_smart_t * ps_smart, int port, int device, u_char on)
{    
    int i, ret = PP_ERR;
    char id_buf[2];
    char rsp_buf[32];    
    char cmd_buf[10] = { 0x41,       // A
			 0x46,       // F
			 0x46,       // F
			 0x72,       // r
			 0x30,       // 0
			 0x00,       // Outlet ID
			 0x20,       // Space
			 0x00,       // on / off
			 0x0D,       // Carriage Return
			 0x00 };
		       
    snprintf(id_buf, sizeof(id_buf), "%1d", port + 1);

    usleep(CMD_DELAY);

    /* Read remaining data from input buffer */
    ps_read_response(ps_smart->fd, 0, NULL, 0, 0,
		     rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_MAX_CHR);

    cmd_buf[7] =  on ? '1' : '0';
    cmd_buf[5] = id_buf[0];
    if (PP_FAILED(get_ids(ps_smart, &cmd_buf[1], &cmd_buf[2], device))) goto error;

    for (i = 0; i < 9; i++) {
        usleep(CHAR_DELAY);
        if (write(ps_smart->fd, &cmd_buf[i], 1) != 1) {
	    D(D_ERROR, "SMART: %s(): set command\n", ___F);
	    goto error;
        }
    }

    *rsp_buf = 0;
    /* first try, looking for carriage return */
    if (PP_FAILED(ps_read_response(ps_smart->fd, 0x0D, NULL, 0, PWR_READ_TIMEOUT,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	D(D_ERROR, "SMART: %s(): set command - no response\n", ___F);
	goto error;
    }
    /* a slave device returns the sent command first before the response,
     * in this case, we must look for another CR */
    if (strstr(rsp_buf, cmd_buf)) {
	*rsp_buf = 0;
	if (PP_FAILED(ps_read_response(ps_smart->fd, 0x0D, NULL, 0, PWR_READ_TIMEOUT,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	    D(D_ERROR, "SMART: %s(): set command - no response on 2nd try\n", ___F);
	    goto error;
	}
    }
    cmd_buf[0] = 'U';
    cmd_buf[6] = 0;

    ret = strstr(rsp_buf, cmd_buf) ? PP_SUC : PP_ERR;

 error:
    return ret;
}

static int
do_seq_switch(ps_smart_t * ps_smart, int device, u_char on)
{
    int i, ret = PP_ERR;
    char rsp_buf[32];
    char cmd_buf[6] = {  0x41,       // A
			 0x46,       // Device address high byte
			 0x46,       // Device address low byte
			 0x00,       // Q = sequence off, P = sequence on		
			 0x0D,       // Carriage Return
			 0x00 };

    usleep(CMD_DELAY);

    /* Read remaining data from input buffer */
    ps_read_response(ps_smart->fd, 0, NULL, 0, 0,
		     rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_MAX_CHR);
    
    if (PP_FAILED(get_ids(ps_smart, &cmd_buf[1], &cmd_buf[2], device))) goto error;
    cmd_buf[3] = on ? 'P' : 'Q';
        
    for (i = 0; i < 5; i++) {
        usleep(CHAR_DELAY);
        if (write(ps_smart->fd, &cmd_buf[i], 1) != 1) {
	    D(D_ERROR, "SMART: %s(): set command\n", ___F);
	    goto error;
        }
    }
    ps_smart->smart_thread_running = 1;
    if (eric_pthread_create(&ps_smart->smart_thread, 1, 16 * 1024,
			    smart_thread_func, (void*)ps_smart)) {
	ps_smart->smart_thread_running = 0;
    }

    ret = PP_SUC;

 error:
    return ret;
}

static void*
smart_thread_func(void * arg)
{
    ps_smart_t * ps_smart = (ps_smart_t *)arg;
    char rsp_buf[32];
    int i;

    for (i = 0; i < 10; i++) {
	if (PP_SUCCED(ps_read_response(ps_smart->fd, 0x0D, NULL, 3, 0,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	    break;
	}
    }
    ps_smart->smart_thread_running = 0;
    return NULL;
}

static int
get_ids(ps_smart_t * ps_smart, char *addr_hbyte, char *addr_lbyte, int device)
{
    char * val = NULL;
    int ret = PP_ERR;

    if (ps_smart->devaddr_var_name
	&& PP_SUCCED(pp_cfg_get_nodflt(&val, ps_smart->devaddr_var_name, device))
	&& strlen(val) >= 2) {
	*addr_hbyte = val[0];
	*addr_lbyte = val[1];
	ret = PP_SUC;
    }
    free(val);
    return ret;
}
