/**	
 * This file contains the code that manages /System/Device/.
 * It corresponds to one device object.
 * 
 * (c)2006 Raritan,  georg.hoesch@raritan.com
 * 
 * FIXME: implement reload mechanism for changes by config-system ?
 * FIXME: use seperate (reduced) parsetable for client request parsing ?
 * FIXME: acquire static device values from somewhere (in loadValues)
 * FIXME: test if it is a good idea to use same key as DeviceAccess/Name
 */


#include "Device.h"
#include "pp/cfg.h"
#include "pp/RDM_DDA_Utility.h"


#define CFG_RDM_DEVICE_NAME  "rdm.device_name"


/* ------------- Device parse table ------------- */

//-------------- <Device>
//-------------- /System/Device
//-------------- Unmodeled attributes/elements: Managed, Status, SerialNo

typedef	struct
{
    const char* pID;		// id of the device, required
    const char* pType;		// type attribute, required  (IP-Reach|Pargon|KVMSwitch, ...)
    const char* pModel;		// Model attribute, optional
    int         internal;	// internal attribute, optional
    int         baseDevice;	// !0 = base device, required if device is basedevice
    char*       pName;		// Name element, required
    const char* pBM;		// Branch Manager attribute, required internally
    // FIXME: no security ID here ?
} deviceData_t;

#define	PT_STRUCT  deviceData_t
PT_BEGIN( "Device",		deviceData_PT,		PT_UNKNOWN_OK )
PT_ATT	( "id",			pID,		0,	PT_STRING_PTR )
PT_ATT	( "Type",		pType,		0,	PT_STRING_PTR )
PT_ATT	( "Model",		pModel,		0,	PT_STRING_PTR )
PT_ATT	( "Internal",		internal,	0,	PT_INT | PT_CONDITION_1 )
PT_ATT	( "BaseDevice",		baseDevice,	0,	PT_INT | PT_CONDITION_2 )
PT_ELEM	( "Name",		pName,		0,	PT_STRING_PTR )
PT_ATT	( "BM",			pBM,		0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT



/* ------------- Device class ------------- */

Device::Device(CRDM* myRDM, const char* myDeviceId):
PP_RDM_BranchManager(myRDM)
{
    strncpy (deviceID, myDeviceId, RDM_MAX_ID);
    appendElementWithID(getSystemNode(), "Device", myDeviceId);
    
    loadValues();
}

Device::~Device() {
    // nothing to do
}
    
const char*
Device::GetName() {
    // use our ID which is already unique
    return deviceID;
}

const char*
Device::getDeviceID() {
    return deviceID;
}

int
Device::loadValues() {
    deviceData_t myData;
    int result;
    int flags;
    
    myData.pID = getDeviceID();
    myData.pType = "KVMSwitch";   // acquire from somewhere
    myData.pModel = "EricG4";     // acquire from somewhere
    myData.internal = 0;
    myData.baseDevice = 1;        // acquire from somewhere
    myData.pBM = GetName();
    myData.pName = NULL;          // load dynamically
    
    pp_cfg_get(&(myData.pName), CFG_RDM_DEVICE_NAME);
    
    flags = 0xFFFFFFFF;
    if (!myData.internal) flags &= ~PT_CONDITION_1;
    if (!myData.baseDevice) flags &= ~PT_CONDITION_2;
    result = SXDB_PT_Put( getDeviceNode(), PT_NODE_UPDATE,
                          deviceData_PT, &myData, flags );
    
    if (myData.pName) free(myData.pName); // duplicated from cfg-sys, free

    if (result <0)
	return RDM_ERROR_IO_ERROR;
    return 0;
}

int
Device::saveValues() {
    deviceData_t myData;
    int result;
    
    memset(&myData, 0, sizeof(deviceData_t));

    /* Get the device settings from RDM */
    result = SXDB_PT_Get( getDeviceNode(), deviceData_PT, &myData );
    //assert(result >= 0);
    if (result < 0) {
	return RDM_ERROR_DEVSET_PUT_DATA;
    }
    
    if ( (myData.pName == NULL) ||
         (pp_cfg_set (myData.pName, CFG_RDM_DEVICE_NAME) != PP_SUC) ||
         (pp_cfg_save(DO_FLUSH) != PP_SUC) )
    {
        return RDM_ERROR_IO_ERROR;
    }

    return 0;
}

CSXDB_Node*
Device::getDeviceNode() {
    CSXPath     xPath;
    CSXDB_Node* node;
    char        xPathStr[80];
    
    snprintf(xPathStr, 80, "/System/Device[@id=\"%s\"]", getDeviceID());
    
    xPath.Parse(xPathStr, rdm->db);
    node = xPath.Enum(0);
    
    return node;
}

CSXDB_Node*
Device::getSystemNode() {
    CSXPath     xPath;
    CSXDB_Node* node;
    xPath.Parse("/System", rdm->db);
    node = xPath.Enum(0);
    return node;
}

int
Device::Update (CSession* pUserSession, CSXDB_Node* pNode, CSXDB_Node* pData) {
    CSXDB_Node*    myNode;
    deviceData_t   myData;
    SXDB_PT*       parseTablePartial;
    void*          parseDataPartial;
    CSXDB_Node*    normalizedNode;
    CSXDB_Node*    normalizedData;
    int result;

    memset(&myData, 0, sizeof(deviceData_t));

    // get root node of our internal subtree
    myNode = getDeviceNode();
    if (myNode == NULL)
	return RDM_ERROR_DEVSET_ROOT_NOT_FOUND;

    // Determine if there is a permissions conflict
    // FIXME: (how) do we have to implement this ?
    
    printf("Device: normalizeRequest\n");
    // normalize request
    result = NormalizeRequest(myNode, deviceData_PT, &myData, pNode, pData,
                              &parseTablePartial, &parseDataPartial, &normalizedNode, &normalizedData);
    
    printf("Device: parse request\n");
    // parse data from request
    result = SXDB_PT_Get(normalizedData, parseTablePartial, parseDataPartial );
    if (result < 0)
    {
	return RDM_ERROR_DEVSET_GET_DATA;
    }

    // Validate before put in the database
    if (!myData.pName) {
        return RDM_ERROR_INVALID_DATA;
    }
    printf("Device: validate done\n");
    
    // Put the data into the database, all conditions off as conditional values are not writable anyway
    result = SXDB_PT_Put( normalizedNode, PT_NODE_UPDATE, parseTablePartial, parseDataPartial, 0 );
    if (result >= 0)
    {
	// Commit Database
	saveValues();

	// Send notifications
	if (pUserSession != NULL)
	{
	    DDA_NotifyWithName(rdm, "User",
				pUserSession->GetUserObject()->GetUserName(),
				getDeviceID(), RDM_EC_Device_Change_Request);
	}
        DDA_NotifyWithName(rdm, "Device",
			   "DeviceName",  /* FIXME: put device name here */
			   getDeviceID(), RDM_EC_Device_Changed);
    }
    else
    {
	return RDM_ERROR_DEVSET_PUT_DATA;
    }

    return 0; // everything okay
}

