/**	
 * This file contains the code the that maintains the /System/Device/DeviceSettings
 * branch. It also bridges relevant values from RDM to libpp_cfg.
 * 
 * (c)2006 Raritan,  georg.hoesch@raritan.com, scottc@raritan.com
 *	
 * FIXME: geo: implement a reload mechanism for changes made directly to the cfg-sys
 * FIXME: use a different parse table for user changeable values ?
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <pp/syms.h>
#include <pp/cfg.h>
#include <pp/xdefs.h>
#include <pp/OS_Port.h>
#include <pp/RDM.h>
#include <pp/SXDB_Parse_Table.h>
#include <pp/RDM_DDA_Utility.h>
#include <pp/Hash32.h>
#include <pp/RDM_BranchManager.h>
#include <pp/RDM_Database.h>

#include "DeviceSettings.h"


/*----------------------------------------
 *	Equates
 *--------------------------------------*/

#define	DEV_SETTINGS_MAX_IF     1     // Number of DeviceAccess/Interfaces


#define CFG_RDM_DEVICE_NAME  "rdm.device_name"
#define CFG_NETWORK_IPADDR   "network.ipaddr"
#define CFG_NETWORK_GATEWAY  "network.gateway"
#define CFG_NETWORK_NETMASK  "network.netmask"
#define CFG_NETWORK_DHCP     "network.ip_auto_config_proto"
#define CFG_RFB_PORT         "network.https_port"


/*----------------------------------------
 *	Data Types
 *--------------------------------------*/


//--------------- <Security>
//--------------- /System/Device/DeviceSettings/Security

typedef struct
{
    const char* id;
    const char* SecurityID;
} DS_Security;

// maybe add CommandCenter node here that can be allowed, required or rejected depending on conditions
#define	PT_STRUCT  DS_Security
PT_BEGIN	( "Security",    DS_Security_Table,       PT_UNKNOWN_OK )  // unknown_ok because <CommandCenter> is a child that is _not_ covered by us
PT_ATT		( "id",          id,                   0, PT_STRING_PTR )
PT_ATT		( "SecurityID",  SecurityID,           0, PT_STRING_PTR )
PT_END
#undef	PT_STRUCT


//------------ <TCPIP>
//------------ /System/Device/DeviceSettings/DeviceAccess/Interface/TCPIP

typedef struct
{
    const char* DHCP;
    const char* IPAddress;
    const char* SubnetMask;
    const char* Gateway;
} DS_TCPIP;

#define	PT_STRUCT  DS_TCPIP
PT_BEGIN	( "TCPIP",      DS_TCPIP_Table,    PT_NO_UNKNOWN )  // only our own values here
PT_ELEM		( "DHCP",       DHCP,           0, PT_STRING_PTR )
PT_ELEM		( "IPAddress",  IPAddress,      0, PT_STRING_PTR )
PT_ELEM		( "SubnetMask", SubnetMask,     0, PT_STRING_PTR )
PT_ELEM		( "Gateway",    Gateway,        0, PT_STRING_PTR )
PT_END
#undef	PT_STRUCT


//------------ <Interface>
//------------ /System/Device/DeviceSettings/DeviceAccess/Interface

typedef struct
{
    const char* id;
    DS_TCPIP    TCPIP;
} DS_Interface;

#define	PT_STRUCT  DS_Interface
PT_BEGIN	( "Interface",    DS_Interface_Table,    PT_NO_UNKNOWN )  // only our own values here
PT_ATT		( "id",           id,                 0, PT_STRING_PTR )
PT_TABLE	( DS_TCPIP_Table, TCPIP,              0 )
PT_END
#undef	PT_STRUCT


//------------ <DeviceAccess>
//------------ /System/Device/DeviceSettings/DeviceAccess

typedef struct
{
    const char* SecurityID;
    const char* Name;
    int	        interfaceCount;        //  # of DS_Interface Elements found in interfaces[]
    DS_Interface interfaces[DEV_SETTINGS_MAX_IF];
} DS_DeviceAccess;

#define	PT_STRUCT  DS_DeviceAccess
PT_BEGIN	( "DeviceAccess",   DS_DeviceAccess_Table,      PT_NO_UNKNOWN )  // only our own values here
PT_ATT		( "SecurityID",     SecurityID,              0, PT_STRING_PTR )
PT_ELEM		( "Name",           Name,                    0, PT_STRING_PTR )
PT_ARRAY	( DS_Interface_Table, interfaces, DEV_SETTINGS_MAX_IF, 0, interfaceCount )
PT_END
#undef	PT_STRUCT


//------------ <RFB>
//------------ /System/Device/DeviceSettings/Management/RFB

typedef struct
{
    const char* IPPort;
} DS_RFB;

#define	PT_STRUCT  DS_RFB
PT_BEGIN	( "RFB",     DS_RFB_Table,     PT_NO_UNKNOWN )  // only our own values here
PT_ELEM		( "IPPort",  IPPort,        0, PT_STRING_PTR )
PT_END
#undef	PT_STRUCT


//------------ <Management>
//------------ /System/Device/DeviceSettings/Management

typedef struct
{
    const char* SecurityID;
    DS_RFB      RFB;
} DS_Management;

#define	PT_STRUCT  DS_Management
PT_BEGIN	( "Management", DS_Management_Table,    PT_NO_UNKNOWN )  // only our own values here
PT_ELEM         ( "SecurityID", SecurityID,          0, PT_STRING_PTR )
PT_TABLE	( DS_RFB_Table, RFB,                 0 )
PT_END
#undef	PT_STRUCT


//------------ <ClientApplication>
//------------ /System/Device/DeviceSettings/Applications/Client/ClientApplication

typedef struct
{
    const char* id;
    const char* name;
    const char* version;
    const char* type;
    const char* fileName;
    const char* path;
} DS_ClientApplication;

#define PT_STRUCT  DS_ClientApplication
PT_BEGIN	( "ClientApplication", DS_ClientApplication_Table, PT_NO_UNKNOWN )  // only our own values here
PT_ATT		( "id",                id,        0, PT_STRING_PTR )
PT_ELEM		( "Name",              name,      0, PT_STRING_PTR )
PT_ELEM		( "Version",           version,   0, PT_STRING_PTR )
PT_ELEM		( "ApplicationType",   type,      0, PT_STRING_PTR )
PT_ELEM		( "FileName",          fileName,  0, PT_STRING_PTR )
PT_ELEM         ( "DirectoryPath",     path,      0, PT_STRING_PTR )
PT_END
#undef PT_STRUCT


//------------ <Client>
//------------ /System/Device/DeviceSettings/Applications/Client

typedef struct
{
    DS_ClientApplication ClientApplication0;   // this could be an array but we currently only have eric_applet
} DS_Client;

#define PT_STRUCT  DS_Client
PT_BEGIN	( "Client", DS_Client_Table, PT_NO_UNKNOWN)
PT_TABLE	( DS_ClientApplication_Table, ClientApplication0, 0)
PT_END
#undef PT_STRUCT


//------------ <Applications>
//------------ /System/Device/DeviceSettings/Applications/

typedef struct
{
    const char* SecurityID;
    DS_Client   Client;
} DS_Applications;

#define PT_STRUCT DS_Applications
PT_BEGIN	( "Applications",  DS_Applications_Table, PT_NO_UNKNOWN )  // only our own values here
PT_ATT		( "SecurityID",    SecurityID, 0,         PT_STRING_PTR )
PT_TABLE        ( DS_Client_Table, Client,     0 )
PT_END
#undef PT_STRUCT


//------------ <DeviceSettings>
//------------ /System/Device/DeviceSettings

typedef struct
{
    const char*      BM;
    DS_DeviceAccess  DeviceAccess;
    DS_Security      Security;
    DS_Management    Management;
    DS_Applications  Applications;
} DS_DeviceSettings;

#define	PT_STRUCT  DS_DeviceSettings
PT_BEGIN	( "DeviceSettings",       DS_DeviceSettings_Table, PT_UNKNOWN_OK )  // only our own values here
PT_ATT		( "BM",                   BM,           0, PT_STRING_PTR )
PT_TABLE	( DS_DeviceAccess_Table,  DeviceAccess, 0 )
PT_TABLE	( DS_Security_Table,      Security,     0 )
PT_TABLE        ( DS_Management_Table,    Management,   0 )
PT_TABLE	( DS_Applications_Table,  Applications, 0 )
PT_END
#undef	PT_STRUCT


/*----------------------------------------
*	Function Prototypes
*--------------------------------------*/

int RDM_InetAton(const char* pIn, unsigned long *pOut = NULL);
bool RDM_MakeBool(const char* in, bool* out = NULL);
bool RDM_MakeLong(const char* in, long& out, int base);


/*----------------------------------------
*	Static Data
*--------------------------------------*/

DeviceSettings::DeviceSettings(CRDM* myRDM, Device* myDevice): 
PP_RDM_BranchManager(myRDM)
{
    parentDevice = myDevice;
    
    pName = "DeviceSettings";
    hashCode = Hash32(0, pName);

    // create our unique branchmanager name
    strncpy(uniqueBMname, parentDevice->GetName(), RDM_MAX_ID);
    strncat(uniqueBMname, "_ds", RDM_MAX_ID);
    
    appendElement(parentDevice->getDeviceNode(), "DeviceSettings");
    
    loadDeviceSettings();
}

DeviceSettings::~DeviceSettings( )
{
    // nothing to do
}

const char* 
DeviceSettings::GetName()
{
    return uniqueBMname;
}

// FIXME: check if we really _can_ call this again to reload the values ...
int
DeviceSettings::loadDeviceSettings()
{
    CSXDB_Node* dsNode;
    DS_DeviceSettings ds;
    int result;

    memset(&ds,0,sizeof(ds));

    /* Set the DeviceSettings static data...these do not change */
    ds.BM = GetName();
    // ds.DeviceAccess.Name  // loaded dynamically
    ds.DeviceAccess.SecurityID = "DeviceAccess";  // FIXME: handle security properly
    ds.DeviceAccess.interfaceCount = 1;
    ds.DeviceAccess.interfaces[0].id = "NetworkInterface0";  // FIXME: CC currently requires this
    // ds.DeviceAccess.interfaces[0].xxx // IP,DHCP, ... are loaded dynamically
    ds.Security.SecurityID = "Security";   // FIXME: handle security properly
    ds.Security.id = "Security";
    // ds.Management.SecurityID =  // FIXME: handle security properly
    // ds.Management.RFB.IPPort = "443"; // loaded dynamically
    // ds.Applications.SecurityID = // FIXME: handle security properly
    ds.Applications.Client.ClientApplication0.id   = "eric_applet";   // FIXME: how to obtain unique id ?
    ds.Applications.Client.ClientApplication0.name = "Eric Applet";   // FIXME: make configurable
    ds.Applications.Client.ClientApplication0.version = "1.17";       // FIXME: acquire form somewhere
    ds.Applications.Client.ClientApplication0.type = "User";          // FIXME: provide better value here ?
    ds.Applications.Client.ClientApplication0.fileName = "rc.jar";    // FIXME: make configurable ?
    ds.Applications.Client.ClientApplication0.path = "/";

    /* Read dynamic (writable) values from libpp_cfg */
    // all these vars need to be freed, so don't read in DS_DeviceSettings directly to prevent obscurity:
    char* device_name = NULL;
    char* eth0_ip = NULL;
    char* eth0_gw = NULL;
    char* eth0_netmask = NULL;
    char* eth0_dhcp = NULL;
    char* rfb_port = NULL;
    
    /* load values from config system */
    // FIXME: what to do if we cannot read these values ??
    // FIXME: use cfg_get or cfg_get_nodflt here ?
    pp_cfg_get(&device_name, CFG_RDM_DEVICE_NAME);
    pp_cfg_get(&eth0_ip, CFG_NETWORK_IPADDR);
    pp_cfg_get(&eth0_gw, CFG_NETWORK_GATEWAY);
    pp_cfg_get(&eth0_netmask, CFG_NETWORK_NETMASK);
    pp_cfg_get(&eth0_dhcp, CFG_NETWORK_DHCP);
    pp_cfg_get(&rfb_port, CFG_RFB_PORT);
    
    /* prepare write to RDM */
    // FIXME: what to do if we have NULL values here due to read errors ?
    ds.DeviceAccess.Name = device_name;
    ds.DeviceAccess.interfaces[0].TCPIP.IPAddress = eth0_ip;
    ds.DeviceAccess.interfaces[0].TCPIP.Gateway = eth0_gw;
    ds.DeviceAccess.interfaces[0].TCPIP.SubnetMask = eth0_netmask;
    if (eth0_dhcp != NULL) {  // 
        if (!strcmp(eth0_dhcp, PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_DHCP_STR) || !strcmp(eth0_dhcp, PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_BOOTP_STR)) {
	    ds.DeviceAccess.interfaces[0].TCPIP.DHCP = "1";
        }
    }
    ds.Management.RFB.IPPort = rfb_port;

    /* Add data to the RDM database */
    dsNode = getDeviceSettingsNode();
    result = SXDB_PT_Put( dsNode, PT_NODE_UPDATE,
                          DS_DeviceSettings_Table, &ds );
    // FIXME: result can be -1 if GetDeviceSettingsNode == NULL
    assert(result >= 0);  // FIXME: is this assert good ?

    /* free config-sys data */
    if (device_name)  free(device_name);
    if (eth0_ip)      free(eth0_ip);
    if (eth0_gw)      free(eth0_gw);
    if (eth0_netmask) free(eth0_netmask);
    if (eth0_dhcp)    free(eth0_dhcp);
    if (rfb_port)     free(rfb_port);

    if (result <0)
	return RDM_ERROR_IO_ERROR;
    return 0;
}

int
DeviceSettings::saveDeviceSettings( )
{
    DS_DeviceSettings ds;
    int	result;
    int err = 0;

    memset(&ds,0,sizeof(ds));

    /* Get the device settings from RDM */
    result = SXDB_PT_Get( getDeviceSettingsNode(), DS_DeviceSettings_Table, &ds );
    assert(result >= 0);
    if (result < 0) {
	return RDM_ERROR_DEVSET_PUT_DATA;
    }
    
    /* Save the settings to libpp_cfg */

    if (ds.DeviceAccess.Name != NULL) {
	if (pp_cfg_set(ds.DeviceAccess.Name, CFG_RDM_DEVICE_NAME) != PP_SUC) err++;
    }
    if (ds.DeviceAccess.interfaces[0].TCPIP.DHCP != NULL) {
	if (strcmp(ds.DeviceAccess.interfaces[0].TCPIP.DHCP,"0") == 0) {
	    // dhcp disabled
	    if (pp_cfg_set(PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_NONE_STR, CFG_NETWORK_DHCP) != PP_SUC) err++;
	} else {
	    // dhcp or bootp enabled
	    // FIXME: right now this will always favor dhcp over bootp, we should do better
	    if (pp_cfg_set(PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_DHCP_STR, CFG_NETWORK_DHCP) != PP_SUC) err++;
	}
    }
    if (ds.DeviceAccess.interfaces[0].TCPIP.IPAddress != NULL) {
	if (pp_cfg_set(ds.DeviceAccess.interfaces[0].TCPIP.IPAddress, CFG_NETWORK_IPADDR) != PP_SUC) err++;
    }
    if (ds.DeviceAccess.interfaces[0].TCPIP.SubnetMask != NULL) {
	if (pp_cfg_set(ds.DeviceAccess.interfaces[0].TCPIP.SubnetMask, CFG_NETWORK_NETMASK) != PP_SUC) err++;
    }
    if (ds.DeviceAccess.interfaces[0].TCPIP.Gateway != NULL) {
	if (pp_cfg_set(ds.DeviceAccess.interfaces[0].TCPIP.Gateway, CFG_NETWORK_GATEWAY) != PP_SUC) err++;
    }
    if (ds.Management.RFB.IPPort != NULL) {
	if (pp_cfg_set(ds.Management.RFB.IPPort, CFG_RFB_PORT) != PP_SUC) err++;
    }
    

    if (pp_cfg_save(DO_FLUSH) != PP_SUC) err++;

    return ((err == 0) ? 0 : RDM_ERROR_DEVSET_PUT_DATA);
}

CSXDB_Node*
DeviceSettings::getDeviceSettingsNode() {
    CSXPath     xPath;
    CSXDB_Node* node;
    char xPathStr[100];
    
    snprintf(xPathStr, 100, "/System/Device[@id=\"%s\"]/DeviceSettings", parentDevice->getDeviceID());
    
    xPath.Parse(xPathStr, rdm->db);
    node = xPath.Enum(0);
    
    return node;
}

int
DeviceSettings::Update (CSession* pUserSession,
			CSXDB_Node* pNode,
			CSXDB_Node* pData )
{
    CSXDB_Node*        dsNode;
    DS_DeviceSettings  dsData;
    SXDB_PT*           parseTablePartial;
    void*              parseDataPartial;
    CSXDB_Node*        normalizedNode;
    CSXDB_Node*        normalizedData;
    int result;

    memset(&dsData,0,sizeof(DS_DeviceSettings));

    // get root node of our internal subtree
    dsNode = getDeviceSettingsNode();
    if (dsNode == NULL)
	return RDM_ERROR_DEVSET_ROOT_NOT_FOUND;

    // Determine if there is a permissions conflict
    // FIXME: (how) do we have to implement this ?
    
    // normalize request
    result = NormalizeRequest(dsNode, DS_DeviceSettings_Table, &dsData, pNode, pData,
                              &parseTablePartial, &parseDataPartial, &normalizedNode, &normalizedData);
    
    // parse data from request
    result = SXDB_PT_Get(normalizedData, parseTablePartial, parseDataPartial );
    if (result < 0)
    {
	return RDM_ERROR_DEVSET_GET_DATA;
    }

    // Validate before put in the database
    result = ValidateDeviceSettingsData(&dsData);
    if (result < 0) {
	return result;
    }
    
    // Put the data into the database
    result = SXDB_PT_Put( normalizedNode, PT_NODE_UPDATE, parseTablePartial, parseDataPartial );
    if (result >= 0)
    {
	// Commit Database
	saveDeviceSettings();

	// Send notifications
	if (pUserSession != NULL)
	{
	    DDA_NotifyWithName(rdm, "User",
				pUserSession->GetUserObject()->GetUserName(),
				parentDevice->getDeviceID(), RDM_EC_Device_Setting_Change_Request);
	}
	
	rdm->nm.Notify( parentDevice->getDeviceID(), RDM_EC_Device_Setting_Change, NULL );
    }
    else
    {
	return RDM_ERROR_DEVSET_PUT_DATA;
    }

    return 0; // everything okay
}


int
DeviceSettings::ValidateDeviceSettingsData(void* pUsrData)
{
    int result;
    DS_DeviceSettings* pU = (DS_DeviceSettings*)pUsrData;
    // FIXME: review whole validation stuff
    
    // General Validation
    result = ValidateDeviceAccessConfig(&(pU->DeviceAccess));

    return result;
}

//---------------------------------------------------------------------------
//
	int								// 0 or Error code
	DeviceSettings::ValidateDeviceAccessConfig
	(
		void* pUsrData		//user data to be validated
	)
//
//	Private function implemented to validate devise access configuration data
//
//---------------------------------------------------------------------------
{
	DS_DeviceAccess* pU = (DS_DeviceAccess*) pUsrData;
	int result = 0;

	int i = 0; 
	for ( i = 0; i < DEV_SETTINGS_MAX_IF; i++)
	{
		result = ValidateNetworkInterfaceConfig(&(pU->interfaces[i]));

		if (result != 0)
			return result;
	}

	return result;
}

//---------------------------------------------------------------------------
//
	int								// 0 or Error code
	DeviceSettings::ValidateNetworkInterfaceConfig
	(
		void* pUsrData		//user data to be validated
	)
//
//	Private function implemented to validate network interface configuration data
//
//---------------------------------------------------------------------------
{
	DS_Interface	*pU = (DS_Interface*) pUsrData;
	int result = 0;

	//TCPIP
	if ( ((pU->TCPIP.DHCP == NULL) || (pU->TCPIP.DHCP[0] == 0)) &&
		((pU->TCPIP.IPAddress == NULL) || (pU->TCPIP.IPAddress[0] == 0)) &&
		((pU->TCPIP.SubnetMask == NULL) || (pU->TCPIP.SubnetMask[0] == 0)) &&
		((pU->TCPIP.Gateway == NULL) || (pU->TCPIP.Gateway[0] == 0)) )
	{
		// no data. do nothing
	}
	else
	{
		if (pU->TCPIP.DHCP)
		{
			if (!RDM_MakeBool(pU->TCPIP.DHCP))
			{
				return RDM_ERROR_TCPIP_INVALID_DHCP;
			}
		}

		if (pU->TCPIP.IPAddress)
		{
			if (RDM_InetAton(pU->TCPIP.IPAddress, NULL) == 0) 
			{
				return RDM_ERROR_TCPIP_INVALID_IPADDRESS;
			}
		}

		if( atoi(pU->TCPIP.DHCP) == 0 )                  //add for CR7796 duplicate Static IP block DHCP configure  
		{
			// @todo Not sure how to check for duplicate Static IP block DHCP configure 
		}

		if (pU->TCPIP.SubnetMask)
		{
			if (RDM_InetAton(pU->TCPIP.SubnetMask, NULL) == 0) 
			{
				return RDM_ERROR_TCPIP_INVALID_SUBNETMASK;
			}
		}

		if (pU->TCPIP.Gateway)
		{
			if (RDM_InetAton(pU->TCPIP.Gateway, NULL) == 0) 
			{
				return RDM_ERROR_TCPIP_INVALID_GATEWAY;
			}
		}
	}

	return result;
}

//--------------------------------------------------------------------------------
//
    int                            //nonzero on success, otherwise zero (as returned by inet_aton) 
    RDM_InetAton
    (
        const char* pIn,            //IP address in dotted decimal form
        unsigned long *pOut        //IP address as unsigned long value
    )
//
//    return InetAddr format of input dotted decimal ip address.
//
//--------------------------------------------------------------------------------
{
    int result = 0;
    in_addr inAddr;

    result = inet_aton(pIn, &inAddr);

    if (result && pOut)
    {
        *pOut = inAddr.s_addr;
    }

    return result;
}

//--------------------------------------------------------------------------------
//
    bool                                //returns true on success, false otherwise.  
    RDM_MakeBool
    (
        const char* in,
        bool* out
    )
//
//  converts a bool from string format  
//
//--------------------------------------------------------------------------------
{
    long temp;

    if (RDM_MakeLong(in, temp, 0))
    {
        if (temp == 0)
        {
            if (out)
                *out = 0;
        }
        else if (temp == 1)
        {
            if (out)
                *out = 1;
        }
        else
        {
            return false;
        }
    }
    else
    {
        return false;
    }

    return true;
}

//--------------------------------------------------------------------------------
//
    bool                                //returns true on success, false otherwise.  
    RDM_MakeLong
    (    const char* in, 
        long& out, 
        int base
    )
//
//  converts a long from string format  
//
//--------------------------------------------------------------------------------
{
    char* endptr;

    if ( !in )
        return false;

    out = strtol(in, &endptr, base);

    if (*endptr != '\0')
        return false;

//    if (out == LONG_MAX || out == LONG_MIN)
//        return false;

    return true;
}


