/**
 * A branchmanager for peppercon objects. Base object for peppercon
 * RDM object hierarchy. Each object that represents a part of the
 * RDM database can inherit from this object to create its own
 * unique branchmanager. Objects can then overwrite Append(), 
 * Update() and Delete() to handle their very unique data, process
 * it and write it back.
 * 
 * (c) 2006 Raritan     georg.hoesch@raritan.com
 */

#include "PP_RDM_BranchManager.h"

#include <stdio.h>
#include <stdlib.h>
#include <pp/syms.h>
#include <pp/RDM_Database.h>
#include <pp/Hash32.h>

PP_RDM_BranchManager::PP_RDM_BranchManager(CRDM* myRDM): 
CRDM_BranchManager() 
{
    hashCode = 0;
    rdm = myRDM;
}


PP_RDM_BranchManager::~PP_RDM_BranchManager() {
    // nothing to do
}


CSXDB_Element*
PP_RDM_BranchManager::appendElement(CSXDB_Node* parentNode, const char* childName)
{
    CSXDB_Element* newNode;
    
    if (!parentNode) return NULL;
    
    newNode = new CSXDB_Element;
    newNode->SetName(childName);  // don't check return value for mem error, this cannot fail for peppercon FW
    parentNode->AddChild(newNode);
    
    return newNode;
}
    
CSXDB_Element*
PP_RDM_BranchManager::appendElementWithID(CSXDB_Node* parentNode,
                     const char* childName, const char* childId)
{
    CSXDB_Element* newElem;

    if (!parentNode) return NULL;

    newElem = appendElement(parentNode, childName);
    newElem->AddAttribute("id", childId);
    return newElem;
}

void
PP_RDM_BranchManager::initialize( CRDM_Database* pDBService )
{
    hashCode = Hash32(0, GetName());

    pDBService->AddBranchManager( this );
    printf("registered branchmanager %s\n", GetName());
}

int 
PP_RDM_BranchManager::GetHashCode() {
    return hashCode;
}

int
PP_RDM_BranchManager::NormalizeRequest( // input params from request / local config
					CSXDB_Node*  rootNode,
					SXDB_PT*     rootParseTable,
					void*        rootParseTableData,
					CSXDB_Node*  targetNode,
					CSXDB_Node*  targetData,
					// output params - normalized/partial versions ready for use
					SXDB_PT**    partialParseTable,
					void**       partialParseTableData,
					CSXDB_Node** normalizedTargetNode, 
					CSXDB_Node** normalizedTargetData )
{
    // FIXME: return correct (RDM) return values
    // FIXME: this code is still a little messy with father/sibling relationships from 
    //        targetNode/targetData. Either straighten this or write some comments on it.
    char* relativeXPath = NULL;
    int   partialParseTableIndex;
    int   result = -1;
    /* 
     * targetNode may be a subtree relative to our subtree (rootNode). targetData
     * is the new data relative to targetNode. In order to parse the data with a
     * parsetable (which makes validating and updating easier), we have to 'normalize'
     * it.
     * 
     * That means: 
     * 1.) calculate the part of the parsetable that corresponds to the updated
     *     data (relative to targetNode)
     * 2.) normalize targetData to fit the derived partial parsetable
     * 3.) normalize targetNode (browse DB up) to fit the derived partial parsetable
     * 
     * Now the partial parsetable can be used to get data from the normalized
     * targetData and to write it to the normalized targetNode of our database.
     */

    // make sure we have data
    targetData = targetData->Child();  // get actual data from <data>
    if (targetData == NULL)
    {
    	// no data within <data>, user error
	return RDM_ERROR_DEVSET_DATA_NOT_FOUND;
    }

    // Step 1, part 1: compute xpath from rootNode to targetNode (assume that targetNode is a child)
    relativeXPath = CreateXPath(rootNode, targetNode, 1);
    if (relativeXPath == NULL) return RDM_ERROR_FAILED;  // this can only happen if the db-service screwed things up

    // Step 1, part2: drill down in the parsetable to minimum partial parsetable that fits the request
    result = SXDB_PT_XPath( relativeXPath, rootParseTable,    rootParseTableData,
                            		   partialParseTable, partialParseTableData, &partialParseTableIndex );
    free(relativeXPath);
    if (result < 0)
    {
	return RDM_ERROR_DEVSET_NODE_NOT_FOUND;
    }
    
    printf("normalization - partialIndex = %d\n", partialParseTableIndex);
    
    // Step 2: normalize targetData to fit the partial parsetable
    *normalizedTargetData = SXDB_PT_Normalize( targetData, *partialParseTable, partialParseTableIndex, 1, 1 ); // create, include root
    if (*normalizedTargetData == NULL)
    {
	return RDM_ERROR_DEVSET_NORMALIZE_DATA;
    }

    // step 3: normalize pTargetNode 
    // Note: include_root=0 is a trick to get the root node as we want to. The reason for this is that
    // the input parameter has a father-relationship (and not a sibling relationship) to targetData.
    *normalizedTargetNode = SXDB_PT_Normalize( targetNode, *partialParseTable, partialParseTableIndex, 0, 0 ); // move_up, no_root
    if (*normalizedTargetNode == NULL)
    {
	return RDM_ERROR_DEVSET_NORMALIZE_DB;
    }
    
    printf("successfull normalization from %s, %s to %s, %s\n", targetNode->GetName(), targetData->GetName(), (*normalizedTargetNode)->GetName(), (*normalizedTargetData)->GetName());

    return 0;
}    



int
PP_RDM_BranchManager::Append ( CSession* NOTUSED(pUserSession),
                               CSXDB_Node* NOTUSED(pNode),
                               CSXDB_Node* NOTUSED(pData) )
{
    return RDM_ERROR_NOT_SUPPORTED;
}

int
PP_RDM_BranchManager::Update ( CSession* NOTUSED(pUserSession),
                               CSXDB_Node* NOTUSED(pNode),
                               CSXDB_Node* NOTUSED(pData) )
{
    return RDM_ERROR_NOT_SUPPORTED;
}

int
PP_RDM_BranchManager::Delete ( CSession* NOTUSED(pUserSession),
                               CSXDB_Node* NOTUSED(pNode) )
{
    return RDM_ERROR_NOT_SUPPORTED;
}

int
PP_RDM_BranchManager::PreProcessDelete ( CSession* NOTUSED(pUserSession),
                                         const char* NOTUSED(pXPath),
                                         CSIO* NOTUSED(pResponse) )
{
    return 0;
}

int
PP_RDM_BranchManager::AssociateOutlet ( const char* NOTUSED(port_id),
                                        const char* NOTUSED(outlet_id) )
{
    return RDM_ERROR_NOT_SUPPORTED;
}

int
PP_RDM_BranchManager::SetPowerState ( const char* NOTUSED(outlet_id),
                                      int NOTUSED(state) )
{
    return RDM_ERROR_NOT_SUPPORTED;
}

char*
PP_RDM_BranchManager::GetPowerStripStatus ( const char* NOTUSED(strip_id) )
{
    return NULL;
}

