/**
 * pp_RDM_System.cpp
 * 
 * Implements the RDM system services. Derives CRDM_System and
 * fills all the empty function calls.
 *
 * (c) 2006 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include "PP_RDM_System.h"

#include "pp/RDM.h"
#include "pp/SXDB_Parse_Table.h"
#include "pp/RDM_System.h"

extern "C" {
#include "pp/cfg.h"
}



//----------------------------------------
//		CommandCenter Parse Table
//----------------------------------------

typedef	struct 
{
    int		enable;
    char	*pClusterID;
    char	*pCCUser;
    char	*pCCPassword;
    int		ipAddress;
    int		ipPort;
    int		heartbeatTime;
    char	*pEventKey;
    int		ccTimeout;

} START_DATA; 


#define	PT_STRUCT	START_DATA
static PT_BEGIN	( "CommandCenter",	ccTable,	PT_NO_UNKNOWN )
PT_ELEM		( "Enable",		enable,		0,	PT_INT | PT_REQUIRED )
PT_ELEM		( "ClusterID",		pClusterID,	0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "CCUser",		pCCUser,	0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "CCPassword",		pCCPassword,	0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "IPAddress",		ipAddress,	0,	PT_IP | PT_REQUIRED )
PT_ELEM		( "IPPort",		ipPort,		0,	PT_INT )
PT_ELEM		( "HeartbeatTime",	heartbeatTime,	0,	PT_INT | PT_REQUIRED )
PT_ELEM		( "EventKey",		pEventKey,	0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "Timeout",		ccTimeout,	0,	PT_INT )
PT_END
#undef	PT_STRUCT


PP_RDM_System::PP_RDM_System(class CRDM* rdm, CCommandCenter* cc): CRDM_System(rdm)
{
    START_DATA cc_config;
    int err;
    this->commandCenter = cc;
    
    /* read CC persistant configuration from config system and write to RDM */
    memset(&cc_config, 0, sizeof(cc_config));
    err = 0;
    if (pp_cfg_is_enabled_nodflt(&(cc_config.enable),  "rdm.cc_config.enabled") != PP_SUC) err++;
    if (pp_cfg_get_nodflt    (&(cc_config.pClusterID), "rdm.cc_config.cluster_id") != PP_SUC) err++;
    if (pp_cfg_get_nodflt    (&(cc_config.pCCUser),    "rdm.cc_config.user") != PP_SUC) err++;
    if (pp_cfg_get_nodflt    (&(cc_config.pCCPassword),"rdm.cc_config.password") != PP_SUC) err++;
    if (pp_cfg_get_int_nodflt(&(cc_config.ipAddress),  "rdm.cc_config.int_ip_address") != PP_SUC) err++;
    if (pp_cfg_get_int_nodflt(&(cc_config.ipPort),     "rdm.cc_config.ip_port") != PP_SUC) err++;
    if (pp_cfg_get_int_nodflt(&(cc_config.heartbeatTime),"rdm.cc_config.heartbeat_time") != PP_SUC) err++;
    if (pp_cfg_get_nodflt    (&(cc_config.pEventKey),  "rdm.cc_config.event_key") != PP_SUC) err++;
    if (pp_cfg_get_int_nodflt(&(cc_config.ccTimeout),  "rdm.cc_config.cc_timeout") != PP_SUC) err++;
    
    if (err == 0) {
        CSXDB_Node	*	pSecurityNode;
        pRDM->db->BeginAccess(1);
        
        pSecurityNode = DEP_GetCCDeviceSettings();
        if (pSecurityNode == NULL) {
        	// DeviceSettings node not found, very bad, cannot recover
        	err++;
            pRDM->db->EndAccess();
            goto cleanup;
        }
        pSecurityNode = pSecurityNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, "Security" );

        if (SXDB_PT_Put(pSecurityNode, PT_CHILD_UPDATE, ccTable, &cc_config) < 0) {
            // would not write data, very bad
            err++;
            pRDM->db->EndAccess();
            goto cleanup;
        }
        pRDM->db->EndAccess();
        
        
        if (cc_config.enable == 1) {
            if (DEP_InitializeManagement() == false) {
                // FIXME: what to do ? can we recover here ?
            }
        }

    }

cleanup:
    // Free all duplicated strings returned by the pp_cfg_getxxxx methods above.
    if (cc_config.pClusterID)   free(cc_config.pClusterID);
    if (cc_config.pCCUser)      free(cc_config.pCCUser);
    if (cc_config.pCCPassword)  free(cc_config.pCCPassword);
    if (cc_config.pEventKey)    free(cc_config.pEventKey);
}

PP_RDM_System::~PP_RDM_System() {
    
}

BOOL
PP_RDM_System::DEP_InitializeManagement() {
    printf("PP_RDM_System::DEP_InitializeManagement()\n");
    
    // FIXME: close all open sessions or return -1 ?
    
    commandCenter->InitializeManagement();
    
    return 1;
}

void
PP_RDM_System::DEP_CommitCCDeviceSettings() {
    CSXDB_Node	*	pNode;
    START_DATA cc_config;
    int err=0;

    pRDM->db->BeginAccess(1);

    pNode = pRDM->db->GetNodeFromXPath("/System/Device/DeviceSettings/Security/CommandCenter",NULL);

    if (SXDB_PT_Get(pNode, ccTable, &cc_config) == 0) {
        // FIXME: this is crude 
        
        if (pp_cfg_set_enabled(cc_config.enable, "rdm.cc_config.enabled") != PP_SUC) err++;
        if (pp_cfg_set    (cc_config.pClusterID, "rdm.cc_config.cluster_id") != PP_SUC) err++;
        if (pp_cfg_set    (cc_config.pCCUser,    "rdm.cc_config.user") != PP_SUC) err++;
        if (pp_cfg_set    (cc_config.pCCPassword,"rdm.cc_config.password") != PP_SUC) err++;
        if (pp_cfg_set_int(cc_config.ipAddress,  "rdm.cc_config.int_ip_address") != PP_SUC) err++;
        if (pp_cfg_set_int(cc_config.ipPort,     "rdm.cc_config.ip_port") != PP_SUC) err++;
        if (pp_cfg_set_int(cc_config.heartbeatTime,"rdm.cc_config.heartbeat_time") != PP_SUC) err++;
        if (pp_cfg_set    (cc_config.pEventKey,  "rdm.cc_config.event_key") != PP_SUC) err++;
        if (pp_cfg_set_int(cc_config.ccTimeout,  "rdm.cc_config.cc_timeout") != PP_SUC) err++;
        if (pp_cfg_save(DO_FLUSH) != PP_SUC) err++;

    } else {
       err++;
    }

    pRDM->db->EndAccess();
    
    if (err == 0) {
        printf("CC configuration successfully written to config\n");
    } else {
        printf("CC configuration could not be written\n");
    }
}

BOOL
PP_RDM_System::DEP_AreWeManaged() {
    return (commandCenter->IsEnabled());
}

void
PP_RDM_System::DEP_StopManagement() {
    commandCenter->StopManagement();
}

void    
PP_RDM_System::DEP_SetVersion(int newVersion) {
    version = newVersion;
}   



