/**	
 * This file contains the code that manages /System/Device/Port.
 * It corresponds to one port object and must be subclassed to
 * represent different port types.
 * 
 * (c)2006 Raritan,  georg.hoesch@raritan.com
 */

#include "Port.h"

#include "pp/cfg.h"
#include "pp/RDM_DDA_Utility.h"


#define	PT_STRUCT	portData_t
PT_BEGIN( "Port",       portData_PT,		PT_NO_UNKNOWN )
PT_ATT	( "id",         pID,            0,	PT_STRING_PTR )
PT_ATT	( "Class",      pClass,         0,	PT_STRING_PTR )
PT_ATT	( "Type",       pType,          0,	PT_STRING_PTR )
PT_ATT	( "Connection", connection,          0,	PT_STRING_PTR )
PT_ATT	( "Internal",   internal,       0,	PT_INT )
PT_ATT	( "Capacity",   capacity,       0,	PT_INT )
PT_ATT	( "Status",     status,         0,	PT_INT )
PT_ELEM	( "Name",       pName,          0,	PT_STRING_PTR )
PT_ATT	( "SecurityID", pSecurityID,    0,	PT_STRING_PTR )
PT_ATT	( "BM",		pBM,		0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT


/* ------------- Port class ------------- */

Port::Port(CRDM* myRdm, Device* myParentDevice):
PP_RDM_BranchManager(myRdm)
{
    parentDevice = myParentDevice;

    // FIXME: id is not yet known
    appendElementWithID(parentDevice->getDeviceNode(), "Port", "kvmport");
    
    loadValues();
}

Port::~Port() {
    // nothing to do
}
    
const char*
Port::GetName() {
    // use our ID which is already unique
    return getPortID();
}

const char*
Port::getPortID() {
    // must be overwritten by subclass
    // FIXME: make this abstract or return something invalid
    return "kvmport";
}

int
Port::loadValues() {
    portData_t myData;
    int result;

    myData.pID = getPortID();
    myData.pClass = "KVM";      // FIXME: acquire from subclass
    myData.pType = NULL;        // unused, we have no real value here
    myData.connection = NULL;   // no connection yet
    myData.internal = 0;        // FIXME: acquire from subclass
    myData.capacity = 1;        // FIXME: acquire from subclass
    myData.status = PORT_STATUS_INACTIVE;
    myData.pName = NULL;        // dynamically loaded
    myData.pSecurityID = NULL;  // FIXME: what to use here ?
    myData.pBM = getPortID();
    
    pp_cfg_get(&(myData.pName), "rdm.port[%s].name", getPortID());
    if (myData.pName == NULL)
        { myData.pName = strdup(getPortID()); }
    
    result = SXDB_PT_Put( getPortNode(), PT_NODE_UPDATE,
                          portData_PT, &myData );
    
    if (myData.pName) free(myData.pName); // duplicated from cfg-sys, free

    if (result <0)
	return RDM_ERROR_IO_ERROR;
    return 0;
}

int
Port::saveValues() {
    portData_t myData;
    int result;
    
    memset(&myData, 0, sizeof(portData_t));

    /* Get the device settings from RDM */
    result = SXDB_PT_Get( getPortNode(), portData_PT, &myData );

    if (result < 0) {
	return RDM_ERROR_DEVSET_PUT_DATA;
    }
    
    if ( (myData.pName == NULL) ||
         (pp_cfg_set (myData.pName, "rdm.port[%s].name", getPortID()) != PP_SUC) ||
         (pp_cfg_save(DO_FLUSH) != PP_SUC) )
    {
        return RDM_ERROR_IO_ERROR;
    }

    return 0;
}

CSXDB_Node*
Port::getPortNode() {
    CSXPath     xPath;
    CSXDB_Node* node;
    char        xPathStr[120];
    
    snprintf(xPathStr, 120, "/System/Device/Port[@id=\"%s\"]", getPortID());
    
    xPath.Parse(xPathStr, rdm->db);
    node = xPath.Enum(0);
    
    return node;
}

int
Port::Update (CSession* pUserSession, CSXDB_Node* pNode, CSXDB_Node* pData) {
    CSXDB_Node*    myNode;
    portData_t     myData;
    SXDB_PT*       parseTablePartial;
    void*          parseDataPartial;
    CSXDB_Node*    normalizedNode;
    CSXDB_Node*    normalizedData;
    int result;

    memset(&myData, 0, sizeof(portData_t));

    // get root node of our internal subtree
    myNode = getPortNode();
    if (myNode == NULL)
	return RDM_ERROR_DEVSET_ROOT_NOT_FOUND;

    // Determine if there is a permissions conflict
    // FIXME: (how) do we have to implement this ?
    
    printf("Port: normalizeRequest\n");
    // normalize request
    result = NormalizeRequest(myNode, portData_PT, &myData, pNode, pData,
                              &parseTablePartial, &parseDataPartial, &normalizedNode, &normalizedData);
    
    printf("Port: parse request\n");
    // parse data from request
    result = SXDB_PT_Get(normalizedData, parseTablePartial, parseDataPartial );
    if (result < 0)
    {
	return RDM_ERROR_DEVSET_GET_DATA;
    }

    // Validate before put in the database
    
    if (validatePort(&myData)) {
        return RDM_ERROR_INVALID_DATA;
    }
    printf("Port: validate done\n");
    
    // Put the data into the database, all conditions off as conditional values are not writable anyway
    result = SXDB_PT_Put( normalizedNode, PT_NODE_UPDATE, parseTablePartial, parseDataPartial, 0 );
    if (result >= 0)
    {
	// Commit Database
	saveValues();

	// Send notifications
	if (pUserSession != NULL)
	{
	    DDA_NotifyWithName(rdm, "User",
				pUserSession->GetUserObject()->GetUserName(),
				getPortID(), RDM_EC_Port_Change_Request);
	}

	DDA_Notify_Port_Change(rdm, getPortID(), "DeviceName", "PortName", "Status");  /* FIXME: put real values here */
    }
    else
    {
	return RDM_ERROR_DEVSET_PUT_DATA;
    }

    return 0; // everything okay
}

int
Port::validatePort(portData_t* NOTUSED(portData)) {
    return 0;
}



