/**
 * RDM_OS_Dependent.cpp
 * 
 * Implements the RDM functions defined in RDM/RDM_OS_Dependent.h.
 *
 * (c) 2006 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */


#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>

#include "pp/RDM.h"
#include "pp/RDM_OS_Dependent.h"

#include "pp/PP_RDM.h"

extern "C" {
#include "pp/base.h"
#include "liberic_misc.h"
}


/**
 * Convert the given RDM_TIME structure to a string. Write output to *pString.
 */
char *
RDM_RDMTimeToString (RDM_TIME *pST, char *pString) {
	// copied from Raritan source
	sprintf(pString,"%04d-%02d-%02d %02d:%02d:%02d",
				(int) pST->year,
				(int) pST->month,
				(int) pST->day,
				(int) pST->hour,
				(int) pST->minute,
				(int) pST->second
			);

	return pString;
}

/**
 * Returns the system time in UTC as RDM_TIME structure.
 */
void
RDM_GetSystemTime (RDM_TIME *pRDMTime)
{
    time_t t;
    struct tm br_time;
    t = time(NULL);
    gmtime_r(&t, &br_time);
    
	pRDMTime->year		= br_time.tm_year + 1900;
	pRDMTime->month		= br_time.tm_mon + 1;
	pRDMTime->day		= br_time.tm_mday;
	pRDMTime->hour		= br_time.tm_hour;
	pRDMTime->minute	= br_time.tm_min;
	pRDMTime->second	= br_time.tm_sec;
}

/**
 * Set the current system time to the specified RDM_TIME in UTC.
 */
int
RDM_SetSystemTime (RDM_TIME *pRDMTime)
{
    struct tm br_time;
    time_t t;
    struct timeval reftime;
    
    br_time.tm_year  = pRDMTime->year - 1900;
    br_time.tm_mon   = pRDMTime->month - 1;
    br_time.tm_mday  = pRDMTime->day;
    br_time.tm_hour  = pRDMTime->hour;
    br_time.tm_min   = pRDMTime->minute;
    br_time.tm_sec   = pRDMTime->second;
    
    t = mktime(&br_time);
    
    if (t == -1) {
        goto bail;
    }
 
    reftime.tv_sec = t;
    reftime.tv_usec = 0;
    
    /* set the system time */
    if (settimeofday(&reftime, NULL) < 0) {
        goto bail;
    }
    /* set the hardware clock */
    pp_set_hardware_clock_exact(t, reftime);
    
	return 0;
    
bail:
    //FIXME: DEBUG only
    printf("RDM_SetSystemTime: mktime failed\n");
    return -1;
}

/** 
 * Convert UTC time to local time
 */
int
RDM_ConvertUTCToLocalTime (RDM_TIME	*pUTCTime, RDM_TIME	*pLocalTime)
{
    printf("TODO: RDM_ConvertUTCToLocalTime()\n");
    // FIXME: add up timezone here ?
    pLocalTime = pUTCTime;
    return 0;
}

/**
 * Return the node to the <DeviceSettings> element in the RDM database
 */
CSXDB_Node*
RDM_GetDeviceSettings()
{
    CSXDB_Node* result = NULL;
    CSXPath p;
    CRDM* rdm;
    char xpath[100];

    rdm = getRDM();
    if (rdm == NULL) return NULL;
    
    // FIXME: put device name here
    sprintf(xpath,"/System/Device[@id=\"IP-Reach\"]/DeviceSettings");
    p.Parse( xpath, rdm->db );
    result = p.Enum(0);
    
    return result;
}

/**
 * Commit the Device Settings branch. Save to somewhere persistent
 */
int
RDM_CommitDeviceSetting()
{
    // Replace with code to save the device settings branch of the database
    printf("TODO: RDM_CommitDeviceSettings()\n");
    return 0;
}

/**
 * Shut down the system.
 */
int
RDM_ShutDown()
{
    // cannot be applied for PP products
    return RDM_ERROR_FUNCTION_NOT_FOUND;
}

/**
 * Restart the system
 */
int
RDM_Restart()
{
    int cc = -1;
    
    // FIXME: maybe we should only enqueue this ?
    //cc = eric_misc_trigger_board_reset(0);
    //FIXME: comment in again
    
    return cc;
}

/**
 * Close all video streams.
 */
int
RDM_CloseAllVideoStreams()
{
	// Seems to be IP-Reach dependent. Not implemented.
	return RDM_ERROR_FUNCTION_NOT_FOUND;
}

/**
 * Create a unique RDM node id with the specified prefix and number.
 * ID must be stored in pOutput and returned.
 */
char *
RDM_CreateUniqueID (const char    *pPrefix,
                    int      num,
                    char    *pOutput)
{
	if (pOutput != NULL)
	{
		char macAddress[ RDM_MAX_ID ] = {0};

		memset ( pOutput, 0, RDM_MAX_ID );
		RDM_GetMACAddress( "eth0" , macAddress );
		sprintf(pOutput,"%s_%s_%d",macAddress, pPrefix,num);
	}

return pOutput;
}

/**
 * Return a pointer to a (static) string with the deviceID.
 */
const char *                      // Returns ptr to deviceID string or NULL if none
RDM_GetDeviceID() {
    
	static char deviceID[RDM_MAX_ID] = {0};

	if ( deviceID[0] == 0 ) 
		RDM_CreateUniqueID( "IPR", 0, deviceID);
	return deviceID;
}


/**
 * @brief Returns the MAC address of this ethernet device based on the interface
 *        name. The MAC address will be filled in the user supplied array otherwise
 *        it will be filled with zeros. 
 *
 * @param  ifName     - The name of the interface for example "eth0".
 * @param  macAddress - User supplied array to store MAC address. 
 *
 */
void
RDM_GetMACAddress( const char *ifName, char * MACAddress )
{
    struct ifreq ifr;  
    int          rc   = 0;
    int          sock = 0;

    MACAddress[0] = 0;
    MACAddress[1] = 0;
    MACAddress[2] = 0;
    MACAddress[3] = 0;
    MACAddress[4] = 0;
    MACAddress[5] = 0;

    // The interface name (e.g "eth0" ).
    if ( ! ifName ) {
	return;
    }

    sock = socket( PF_INET, SOCK_DGRAM, IPPROTO_IP );

    if ( sock < 0 ) {
	::perror( "RDM_GetMACAddress():socket()" );
	return;
    } 

    strcpy( ifr.ifr_name, ifName );

    rc = ::ioctl( sock, SIOCGIFHWADDR, &ifr ); 

    if ( rc < 0 ) {
	::perror( "RDM_GetMACAddress(): ioctl(SIOCGIFHWADDR)" );
	return;
    }

    unsigned char * mac = reinterpret_cast<unsigned char *>(ifr.ifr_hwaddr.sa_data);

    sprintf( MACAddress, 
             "%2X%2X%2X%2X%2X%2X", 
             mac[0],
             mac[1],
             mac[2],
             mac[3],
             mac[4],
             mac[5]
           );

}
