/*
 * hextile.c
 *
 * Routines to implement Hextile Encoding
 */

/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include <stdio.h>
#include <netinet/in.h>
#include "rfb.h"
#include "debug.h"

static int sendHextiles8(rfb_cl_t * clp, int x, int y, int w, int h);
static int sendHextiles16(rfb_cl_t * clp, int x, int y, int w, int h);
#if ENABLE_24_AND_32_BIT
static int sendHextiles32(rfb_cl_t * clp, int x, int y, int w, int h);
#else /* ENABLE_24_AND_32_BIT */
static inline int sendHextiles32(UNUSED rfb_cl_t * clp, UNUSED int x, UNUSED int y, UNUSED int w, UNUSED int h)  { ERROR_24_AND_32_BIT }
#endif /* ENABLE_24_AND_32_BIT */

/*
 * rfbSendRectEncodingHextile - send a rectangle using hextile encoding.
 */

int
rfb_send_rect_hextile(rfb_cl_t * clp, int x, int y, int w, int h)
{
    char *rect = NULL;
    size_t size = 0;

#ifdef TIME_MEASURE
    clp->time_pixel += w*h;
#endif
    
    rect = rfb_create_fb_update_rect_header(x, y, w, h, rfbEncodingHextile, &size);
    if (!rect || size == 0) {
    	return -1;
    }

    if (clp->ub_len + size > UPDATE_BUF_SIZE) {
	if (rfb_send_update_buf(clp) == -1) {
	    free(rect);
	    return -1;
	}
    }

    memcpy(&clp->update_buf[clp->ub_len], rect, size);
    free(rect);
    clp->ub_len += size;

    switch (clp->pix_fmt.bitsPerPixel_8) {
      case 8:
	  return sendHextiles8(clp, x, y, w, h);
      case 16:
	  return sendHextiles16(clp, x, y, w, h);
      case 32:
	  return sendHextiles32(clp, x, y, w, h);
    }

    pp_log("rfb_send_rect_hextile(): bpp %d?\n", clp->pix_fmt.bitsPerPixel_8);
    return -1;
}


#define PUT_PIXEL8(pix) (clp->update_buf[clp->ub_len++] = (pix))

#define PUT_PIXEL16(pix) (clp->update_buf[clp->ub_len++] = ((char*)&(pix))[0],\
			  clp->update_buf[clp->ub_len++] = ((char*)&(pix))[1])

#define PUT_PIXEL32(pix) (clp->update_buf[clp->ub_len++] = ((char*)&(pix))[0],\
			  clp->update_buf[clp->ub_len++] = ((char*)&(pix))[1],\
			  clp->update_buf[clp->ub_len++] = ((char*)&(pix))[2],\
			  clp->update_buf[clp->ub_len++] = ((char*)&(pix))[3])


#define DEFINE_SEND_HEXTILES(bpp, nextline)				      \
									      \
									      \
static int subrectEncode##bpp(rfb_cl_t * clp, u_int##bpp##_t * data, int w,   \
			      int h, u_int##bpp##_t bg, u_int##bpp##_t fg,    \
                              int mono);                                      \
static void testColours##bpp(u_int##bpp##_t * data, int size, int * mono,     \
			     int * solid, u_int##bpp##_t * bg,                \
                             u_int##bpp##_t *fg);                             \
									      \
									      \
/*									      \
 * rfbSendHextiles							      \
 */									      \
									      \
static int								      \
sendHextiles##bpp(rfb_cl_t * clp, int rx, int ry, int rw, int rh)	      \
{									      \
    int x, y, w, h;							      \
    int startUblen;							      \
    char *fb_ptr;							      \
    u_int##bpp##_t bg = 0, fg = 0, newBg, newFg;			      \
    int mono, solid;							      \
    int validBg = 0;						              \
    int validFg = 0;						              \
    u_int##bpp##_t clientPixelData[PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT *     \
				   (bpp/8)];				      \
									      \
    for (y = ry; y < ry+rh; y += PP_FB_TILE_HEIGHT) {			      \
	for (x = rx; x < rx+rw; x += PP_FB_TILE_WIDTH) {		      \
	    w = PP_FB_TILE_WIDTH;					      \
	    h = PP_FB_TILE_HEIGHT;					      \
	    if (rx+rw - x < PP_FB_TILE_WIDTH)				      \
		w = rx+rw - x;						      \
	    if (ry+rh - y < PP_FB_TILE_HEIGHT)				      \
		h = ry+rh - y;						      \
									      \
	    if ((clp->ub_len + 1 + (2 + PP_FB_TILE_WIDTH*PP_FB_TILE_HEIGHT) * \
		(bpp/8)) > UPDATE_BUF_SIZE){				      \
		if (rfb_send_update_buf(clp) == -1) {			      \
		    return -1;					              \
		}							      \
	    }								      \
									      \
	    fb_ptr = GET_PTR_INTO_FB(clp, x, y);		              \
									      \
	    clp->translate_fn(clp->translate_lookup_table, &rfb_pix_fmt,      \
			      &clp->pix_fmt, fb_ptr, (char *)clientPixelData, \
			      nextline,				              \
			      w, h);					      \
									      \
	    startUblen = clp->ub_len;					      \
	    clp->update_buf[startUblen] = 0;				      \
	    clp->ub_len++;						      \
									      \
	    testColours##bpp(clientPixelData, w * h,			      \
			     &mono, &solid, &newBg, &newFg);		      \
									      \
	    if (!validBg || (newBg != bg)) {				      \
		validBg = 1;						      \
		bg = newBg;						      \
		clp->update_buf[startUblen] |= rfbHextileBackgroundSpecified; \
		PUT_PIXEL##bpp(bg);					      \
	    }								      \
									      \
	    if (solid) {						      \
		continue;						      \
	    }								      \
									      \
	    clp->update_buf[startUblen] |= rfbHextileAnySubrects;	      \
									      \
	    if (mono) {							      \
		if (!validFg || (newFg != fg)) {			      \
		    validFg = 1;					      \
		    fg = newFg;						      \
		    clp->update_buf[startUblen] |= rfbHextileForegroundSpecified;  \
		    PUT_PIXEL##bpp(fg);					      \
		}							      \
	    } else {							      \
		validFg = 0;					              \
		clp->update_buf[startUblen] |= rfbHextileSubrectsColoured;    \
	    }								      \
									      \
	    if (subrectEncode##bpp(clp,clientPixelData,w,h,bg,fg,mono) == -1){\
		/* encoding was too large, use raw */			      \
		validBg = 0;					              \
		validFg = 0;					              \
		clp->ub_len = startUblen;				      \
		clp->update_buf[clp->ub_len++] = rfbHextileRaw; 	      \
		clp->translate_fn(clp->translate_lookup_table, &rfb_pix_fmt,  \
                                  &clp->pix_fmt, fb_ptr,                      \
                                  (char *)clientPixelData,		      \
				  nextline,				      \
				  w, h);				      \
									      \
		memcpy(&clp->update_buf[clp->ub_len], (char *)clientPixelData,\
		       w * h * (bpp/8));				      \
									      \
		clp->ub_len += w * h * (bpp/8);				      \
	    }								      \
	}								      \
    }									      \
									      \
    return 0;							              \
}									      \
									      \
									      \
static int								      \
subrectEncode##bpp(rfb_cl_t * clp, u_int##bpp##_t * data, int w, int h,	      \
		   u_int##bpp##_t bg, u_int##bpp##_t fg UNUSED, int mono)     \
{									      \
    u_int##bpp##_t cl;							      \
    int x,y;								      \
    int i,j;								      \
    int hx=0,hy,vx=0,vy;						      \
    int hyflag;								      \
    u_int##bpp##_t *seg;						      \
    u_int##bpp##_t *line;						      \
    int hw,hh,vw,vh;							      \
    int thex,they,thew,theh;						      \
    int numsubs = 0;							      \
    int newLen;								      \
    int nSubrectsUblen;							      \
									      \
    nSubrectsUblen = clp->ub_len;					      \
    clp->ub_len++;							      \
									      \
    for (y=0; y<h; y++) {						      \
	line = data+(y*w);						      \
	for (x=0; x<w; x++) {						      \
	    if (line[x] != bg) {					      \
		cl = line[x];						      \
		hy = y-1;						      \
		hyflag = 1;						      \
		for (j=y; j<h; j++) {					      \
		    seg = data+(j*w);					      \
		    if (seg[x] != cl) {break;}				      \
		    i = x;						      \
		    while ((seg[i] == cl) && (i < w)) i += 1;		      \
		    i -= 1;						      \
		    if (j == y) vx = hx = i;				      \
		    if (i < vx) vx = i;					      \
		    if ((hyflag > 0) && (i >= hx)) {			      \
			hy += 1;					      \
		    } else {						      \
			hyflag = 0;					      \
		    }							      \
		}							      \
		vy = j-1;						      \
									      \
		/* We now have two possible subrects: (x,y,hx,hy) and	      \
		 * (x,y,vx,vy).  We'll choose the bigger of the two.	      \
		 */							      \
		hw = hx-x+1;						      \
		hh = hy-y+1;						      \
		vw = vx-x+1;						      \
		vh = vy-y+1;						      \
									      \
		thex = x;						      \
		they = y;						      \
									      \
		if ((hw*hh) > (vw*vh)) {				      \
		    thew = hw;						      \
		    theh = hh;						      \
		} else {						      \
		    thew = vw;						      \
		    theh = vh;						      \
		}							      \
									      \
		if (mono) {						      \
		    newLen = clp->ub_len - nSubrectsUblen + 2;		      \
		} else {						      \
		    newLen = clp->ub_len - nSubrectsUblen + bpp/8 + 2;	      \
		}							      \
									      \
		if (newLen > (w * h * (bpp/8)))				      \
		    return -1;					              \
									      \
		numsubs += 1;						      \
									      \
		if (!mono) PUT_PIXEL##bpp(cl);				      \
									      \
		clp->update_buf[clp->ub_len++] =                              \
                        rfbHextilePackXY(thex, they);                         \
		clp->update_buf[clp->ub_len++] =                              \
                        rfbHextilePackWH(thew, theh);                         \
									      \
		/*							      \
		 * Now mark the subrect as done.			      \
		 */							      \
		for (j=they; j < (they+theh); j++) {			      \
		    for (i=thex; i < (thex+thew); i++) {		      \
			data[j*w+i] = bg;				      \
		    }							      \
		}							      \
	    }								      \
	}								      \
    }									      \
									      \
    clp->update_buf[nSubrectsUblen] = numsubs;				      \
									      \
    return 0;							              \
}									      \
									      \
									      \
/*									      \
 * testColours() tests if there are one (solid), two (mono) or more	      \
 * colours in a tile and gets a reasonable guess at the best background	      \
 * pixel, and the foreground pixel for mono.				      \
 */									      \
									      \
static void								      \
testColours##bpp(u_int##bpp##_t * data, int size, int * mono, int * solid,    \
		 u_int##bpp##_t * bg, u_int##bpp##_t * fg)		      \
{									      \
    u_int##bpp##_t colour1 = 0, colour2 = 0;				      \
    int n1 = 0, n2 = 0;							      \
    *mono = 1;							              \
    *solid = 1;							              \
									      \
    for (; size > 0; size--, data++) {					      \
									      \
	if (n1 == 0)							      \
	    colour1 = *data;						      \
									      \
	if (*data == colour1) {						      \
	    n1++;							      \
	    continue;							      \
	}								      \
									      \
	if (n2 == 0) {							      \
	    *solid = 0;						              \
	    colour2 = *data;						      \
	}								      \
									      \
	if (*data == colour2) {						      \
	    n2++;							      \
	    continue;							      \
	}								      \
									      \
	*mono = 0;							      \
	break;								      \
    }									      \
									      \
    if (n1 > n2) {							      \
	*bg = colour1;							      \
	*fg = colour2;							      \
    } else {								      \
	*bg = colour2;							      \
	*fg = colour1;							      \
    }									      \
}

#ifdef PP_VSC_HCML
DEFINE_SEND_HEXTILES(8, clp->fb_width_pd * (clp->fb_bpp / 8));
DEFINE_SEND_HEXTILES(16,clp->fb_width_pd * (clp->fb_bpp / 8));
#if ENABLE_24_AND_32_BIT
DEFINE_SEND_HEXTILES(32,clp->fb_width_pd * (clp->fb_bpp / 8));
#endif /* ENABLE_24_AND_32_BIT */
#else
DEFINE_SEND_HEXTILES(8, clp->fb_width * (clp->fb_bpp / 8));
DEFINE_SEND_HEXTILES(16,clp->fb_width * (clp->fb_bpp / 8));
#if ENABLE_24_AND_32_BIT
DEFINE_SEND_HEXTILES(32,clp->fb_width * (clp->fb_bpp / 8));
#endif /* ENABLE_24_AND_32_BIT */
#endif
