#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <liberic_pthread.h>
#include <liberic_notify.h>
#include <liberic_webs.h>
#include <pp/rfb.h>
#include "rfb.h"
#include <pp/base.h>
#include <pp/vsc.h>
#include <pp/km.h>
#include <pp/kvm.h>
#include <pp/grab.h>

/* global variables */
rfb_context_t rfb_context;
pthread_mutexattr_t err_check_mtx_attr;
pthread_mutexattr_t recursive_mtx_attr;
pthread_mutex_t rfb_session_data_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;
static int initialized = 0;
volatile int kvm_timeout_enabled = 0;

pp_vsc_encoder_desc_t vsc_encoder = {
    vsc_event:                  rfb_vsc_event,
    encoder_is_active:		pp_rfb_is_active,
};

pp_km_encoder_desc_t km_encoder = {
    km_event:				rfb_km_event,
    encoder_send_kbdlayout:		rfb_send_kbdlayout_to_all,
#if defined(PP_FEAT_SESSION_REDIRECTOR)
    encoder_send_local_key_event:	rfb_sas_send_local_key,
    encoder_send_local_mouse_event:	rfb_sas_send_local_mouse,
#endif /* PP_FEAT_SESSION_REDIRECTOR */
};

pp_grab_encoder_desc_t grab_encoder = {
    new_data_available:		rfb_schedule_update,
    grabber_available:		rfb_notify_grab_avail,
    set_color_translation:	rfb_set_color_translation,
    set_framebuffer_format:	rfb_queue_fb_format_update,
    get_excl_session:		pp_rfb_get_excl_session,
};

#if defined(PP_FEAT_SESSION_REDIRECTOR)

eric_session_encoder_desc_t session_encoder = {
    encoder_session_opened:	rfb_sas_session_opened,
    encoder_session_closed:	rfb_sas_session_closed,
};

pp_kvm_encoder_desc_t kvm_encoder = {
    encoder_send_kvm_switch_event:	rfb_sas_send_kvm_switch_event,
};

eric_webs_encoder_desc_t webs_encoder = {
    encoder_login_failure:	rfb_sas_login_failure,
};

#endif /* PP_FEAT_SESSION_REDIRECTOR */

/* function prototypes */
static int  rfb_init_contexts(void);
static void rfb_cleanup_contexts(void);

static void propchange_handler(pp_propchange_listener_t * listener, 
			       unsigned short prop, unsigned short prop_flags UNUSED);
static int kvm_timeout_enabled_ch(pp_cfg_chg_ctx_t *ctx);
static int get_kvm_timeout_enabled(void);

PP_DECLARE_PROPCHANGE_LISTENER(propchange_listener, propchange_handler);

struct list_head rfb_session_data_list;

int
pp_rfb_init(void)
{
    const char * fn = ___F;

    if (initialized) {
	return 0;
    }

    INIT_LIST_HEAD(&rfb_session_data_list);
    
    if (pthread_mutexattr_init(&err_check_mtx_attr) != 0) {
	pp_log("%s(): pthread_mutexattr_init() failed\n", fn);
	goto error;
    }

    if (pthread_mutexattr_settype(&err_check_mtx_attr,
				  PTHREAD_MUTEX_ERRORCHECK_NP) != 0) {
	pp_log("%s(): pthread_mutexattr_settype() failed\n", fn);
	goto error;
    }
    
    if (pthread_mutexattr_init(&recursive_mtx_attr) != 0) {
	pp_log("%s(): pthread_mutexattr_init() failed\n", fn);
	goto error;
    }
    
    if (pthread_mutexattr_settype(&recursive_mtx_attr,
				  PTHREAD_MUTEX_RECURSIVE_NP) != 0) {
	pp_log("%s(): pthread_mutexattr_settype() failed\n", fn);
	goto error;
    }
    
    if (rfb_init_contexts()) {
	pp_log("%s(): Cannot create pointer thread.\n", fn);
	goto error;
    }

#ifndef PP_FEAT_VSC_HW_ENCODING
    if (rfb_hwenc_init()) {
	pp_log("%s(): Cannot initialize software LRLE.\n", fn);
	goto error;
    }
#endif
    
    eric_notify_register_log_object("console", "Remote Console", PP_NOTIFY_EVENT_ALL_ENABLE);    
    
    pp_propchange_add(&propchange_listener, PP_PROP_VIDEO_MODE_CHANGED);
    pp_propchange_add(&propchange_listener, PP_PROP_VIDEO_SETTINGS_UPDATE);
    
#ifdef PRODUCT_XX01IP_ANY
    pp_propchange_add(&propchange_listener, PP_PROP_KVM_PORT_SWITCHED);
#endif
    
#if (__BYTE_ORDER == __BIG_ENDIAN)
    rfb_pix_fmt.bigEndian_8 = 1;
#else
    rfb_pix_fmt.bigEndian_8 = 0;
#endif
  
    pp_vsc_connect_encoder(&vsc_encoder);
    pp_km_connect_encoder(&km_encoder);
    pp_grab_connect_encoder(&grab_encoder);

#if defined(PP_FEAT_SESSION_REDIRECTOR)
    eric_session_connect_encoder(&session_encoder);
    pp_kvm_connect_encoder(&kvm_encoder);
    eric_webs_connect_encoder(&webs_encoder);
#endif /* PP_FEAT_SESSION_REDIRECTOR */

    kvm_timeout_enabled = get_kvm_timeout_enabled();
    pp_cfg_add_change_listener(kvm_timeout_enabled_ch, "security.idle_timeout.kvm_enabled");

    initialized = 1;
    return 0;

 error:
    pp_rfb_cleanup();
    return -1;
}

void pp_rfb_cleanup(void)
{
    struct list_head *ptr, *sd_list = &rfb_session_data_list;
    pp_rfb_session_data_t * sd;

#if !defined(PRODUCT_ERIC2)
    pp_propchange_remove_all(&propchange_listener);
#endif

    rfb_cleanup_contexts();

    for (ptr = sd_list->next; ptr != sd_list; ptr = ptr->next) {
	sd = list_entry(ptr, pp_rfb_session_data_t, listnode);
	rfb_free_session_data(sd);
    }
    rfb_session_data_list.next = rfb_session_data_list.prev = NULL;

    pthread_mutexattr_destroy(&err_check_mtx_attr);
    pthread_mutexattr_destroy(&recursive_mtx_attr);

#ifndef PP_FEAT_VSC_HW_ENCODING
    rfb_hwenc_cleanup();
#endif
    
    initialized = 0;
}

int
rfb_init_contexts()
{
    memset(&rfb_context, 0x00, sizeof(rfb_context));
    
    rfb_context.id = 0;
    rfb_context.tight_disable_gradient = 0;

    /* OSD on init */
    rfb_context.osd_message = strdup("");
    rfb_context.blank_state = PP_DONT_BLANK_SCREEN;
    rfb_context.osd_timeout_ms = 0;
	    
    pthread_mutex_init(&rfb_context.osd_state_mtx, &err_check_mtx_attr);
		
    /* init input thread */
    pthread_mutex_init(&rfb_context.input_queue_mtx, &err_check_mtx_attr);
    if (sem_init(&rfb_context.input_sem, 0, 0)) return -1;

    if ((rfb_context.input_queue = pp_queue_alloc(QUEUE_TYPE_PRIORITY)) == NULL) {
	return -1;
    }

    rfb_context.input_thd_run = 1;
    if (eric_pthread_create(&rfb_context.input_thd, 0, 128 * 1024,
			    rfb_input_thread, &rfb_context)) {
	return -1;
    }	
  
    return 0;
}

void
rfb_cleanup_contexts()
{
    rfb_context.input_thd_run = 0;
    pthread_join(rfb_context.input_thd, NULL);

    pthread_mutex_destroy(&rfb_context.osd_state_mtx);
    pthread_mutex_destroy(&rfb_context.input_queue_mtx);
    sem_destroy(&rfb_context.input_sem);

    pp_queue_free(rfb_context.input_queue);
}

#ifdef PRODUCT_XX01IP_ANY
static void disconnect_if_kvm_port_denied(eric_session_int_id_t s, 
                                          va_list args UNUSED) {
    pp_rfb_session_data_t **sd = NULL;
    int i;
    u_char unit;
    u_short port;

    /* FIXME: hardcoded video_link 0 */
    if (PP_FAILED(pp_kvm_get_unit_port_for_video_link(0, &unit, &port))) {
	pp_log("cannot determine current port from video_link, don't disconnect!\n");
	return;
    }
    if (!pp_kvm_switch_allowed(eric_session_get_user(s), unit, port)) {
	sd = rfb_get_session_data_from_session(s);
	for (i=0; sd && sd[i]; i++) {
	    sd[i]->exit_reason = rfbQuitKVMPortDenied;
	}
	free(sd);
	pp_rfb_do_disconnect(s);
    }
}
#endif

static void
propchange_handler(pp_propchange_listener_t * listener UNUSED,
		   unsigned short prop, unsigned short prop_flags UNUSED)
{
    switch (prop) {
      case PP_PROP_VIDEO_MODE_CHANGED:
      case PP_PROP_VIDEO_SETTINGS_UPDATE:
	  /* this seems to be reentrant */
	  rfb_send_video_settings(0);
	  break;
#ifdef PRODUCT_XX01IP_ANY
      case PP_PROP_KVM_PORT_SWITCHED:
	  /* this seems to be reentrant */
	  // FIXME: add channel here!
	  eric_session_for_all(disconnect_if_kvm_port_denied);
	  break;
#endif
      default:
	  pp_log("%s(): Unhandled property %hu received.\n", ___F, prop);
	  break;
    }
}

pp_rfb_session_data_t *
rfb_alloc_session_data(void)
{
    pp_rfb_session_data_t * sd;

    sd = (pp_rfb_session_data_t *)malloc(sizeof(*sd));

    if (sd == NULL) {
	pp_log_err("%s(): malloc()", ___F);
    } else {
	memset(sd, 0, sizeof(*sd));
    }

    return sd;
}

void
rfb_free_session_data(pp_rfb_session_data_t * sd)
{
    MUTEX_LOCK(&rfb_session_data_mtx);
    list_del(&sd->listnode);
    MUTEX_UNLOCK(&rfb_session_data_mtx);
    if (exclusive_session && exclusive_session == sd->session) {
	exclusive_session = 0;
	exclusive_clp = NULL;
    }
    free(sd);
}

static int kvm_timeout_enabled_ch(pp_cfg_chg_ctx_t *ctx UNUSED) {
    kvm_timeout_enabled = get_kvm_timeout_enabled();
    return PP_SUC;
}

static int get_kvm_timeout_enabled() {
    int _kvm_logout_enabled = 0;
    pp_cfg_is_enabled(&_kvm_logout_enabled, "security.idle_timeout.kvm_enabled");
    return _kvm_logout_enabled;
}
