#include <pp/rfb.h>
#include <liberic_net.h>
#include <liberic_notify.h>
#include "rfb.h"
#include <pp/intl.h>

static unsigned char aa_progress_step = 0;

/******************************************************************************
* exported pp_ functions                                                      *
******************************************************************************/

void pp_rfb_handle_connection(pp_net_conn_data_t * conn_data) {
    eric_session_int_id_t session = 0;
    rfb_cl_t * clp = NULL;
    char logmsg[PP_NOTIFY_MAX_MSG_LEN+1];
    int close_in_cleanup = 0;
    int reason;

    /* ---- preallocate client structure ---------------------------------- */
 
    clp = rfb_create_client_structure(conn_data);
    if (!clp) {
    	pp_log("Out of memory.\n");
    	goto fail;
    }

    /* ---- read the "e-RIC Auth=" string which is always the first data in RFB stream */
    if (rfb_read_connection_init_string(clp) == -1) {
    	goto fail;
    }

    /* ---- login sequence ---------------------------------- */

    if (rfb_negotiate_protocol_version(clp) == -1) {
        rfb_send_quit(clp, rfbQuitBadProtoVersion);
        goto fail;
    }

    if (rfb_authenticate(clp, &session)) {
    	goto fail;
    }
    
    /* ---- we need session to check port permission for user on xx01IP  */
    clp->session = session;

    /* ---- check whether the connection is allowed -------------------- */

    if ((reason = rfb_check_connection_allowed(clp))) {
    	rfb_send_quit(clp, reason);
    	goto fail;
    }

    /* ---- initialize client structure -------------------------------- */

    clp->session_time = time(NULL);
    pthread_cleanup_push(rfb_client_cleanup_handler, clp);
    close_in_cleanup = 1;

    if (rfb_init_client_structure(clp)) {
    	goto fail_with_thread_cleanup;
    }
    
    if (rfb_send_connection_parameters(clp)) {
    	goto fail_with_thread_cleanup;
    }
    
#if defined(PP_FEAT_SESSION_REDIRECTOR)
    rfb_sas_send_new_kvm_session(clp);
    {
    	int sas_error = rfb_sas_init(clp);
    	if (sas_error > 0) {
    	    if (rfb_send_sas_error(clp, sas_error)) {
    	    	rfb_send_quit(clp, rfbQuitInternalError);
    	    	goto fail_with_thread_cleanup;
    	    }
    	} else if (sas_error < 0) {
    	    rfb_send_quit(clp, rfbQuitInternalError);
    	    goto fail_with_thread_cleanup;
    	}
    }
#endif /* PP_FEAT_SESSION_REDIRECTOR */
    
    if ((reason = rfb_prepare_connection(clp))) {
    	rfb_send_quit(clp, reason);
    	goto fail_with_thread_cleanup;
    }

    /* ---- insert client in list and send protocol version ------------ */

    reason = rfbQuitInternalError;
    if (rfb_initialize(clp) == -1 || rfb_client_insert(clp) == -1 || (reason = rfb_start_grabbing(clp))) {
        rfb_send_quit(clp, reason);
	goto fail_with_thread_cleanup;
    }

    /* ---- start the main protocol handler ---------------------------- */    
    snprintf(logmsg, sizeof(logmsg), "Connection to client %s established.", clp->client_ip);
    eric_notify_post_event(logmsg, "console", PP_NOTIFY_EVENT_GENERIC);

    if (rfb_needs_proxy_connection(clp)) {
        rfb_proxy_run(clp);
    } else {
        rfb_main_protocol_handler(clp);
    }

fail_with_thread_cleanup:
    pthread_cleanup_pop(1);

fail:
    if (!close_in_cleanup) {
    	rfb_proxy_disconnect(clp);
    	eric_net_close(conn_data->bio, WAIT);
    	if (clp->conn_data) free(clp->conn_data);
    	if (clp) free(clp);
	if (session) eric_session_release(session);
    }
}

void pp_rfb_get_proto_version(unsigned int *major, unsigned int *minor) {
    *major = rfbProtocolMajorVersion;
    *minor = rfbProtocolMinorVersion;
}

void pp_rfb_send_command(client_index_type_t type, u_int16_t client_index, const char* command, const char* params) {
    size_t pdu_size;
    char *pdu = rfb_create_command_pdu(command, params, &pdu_size);

    if (pdu && pdu_size > 0) {
    	rfb_for_all_clients(type, client_index, rfb_enqueue_message_v, pdu, pdu_size);
    }
    
    free(pdu);
}

int pp_rfb_is_active(unsigned char channel UNUSED, eric_session_int_id_t s) {    
    pp_rfb_session_data_t **sd = NULL;
    int i;
    
    if (s != 0) {
	sd = rfb_get_session_data_from_session(s);
	for (i=0; sd && sd[i]; i++) {
	    if (sd[i]->rc_active) {
		free(sd);
		return 1;
	    }
	}
	free(sd);
    }

    return 0;
}

eric_session_int_id_t pp_rfb_get_excl_session(unsigned char channel UNUSED) {  
    return exclusive_session;
}

pp_exclusive_info_t pp_rfb_get_exclusive_session_info(eric_session_int_id_t id) {  
    if (id == 0) return PP_RFB_SESSION_NOT_EXCLUSIVE;

    if (exclusive_session == 0) return PP_RFB_NO_EXCLUSIVE_SESSION;
    
    return (exclusive_session == id) ? PP_RFB_SESSION_IS_EXCLUSIVE : PP_RFB_SESSION_NOT_EXCLUSIVE;
}

void pp_rfb_do_disconnect(eric_session_int_id_t s) {
    pp_rfb_session_data_t **sd = NULL;
    int i;
    
    if (s != 0) {
	sd = rfb_get_session_data_from_session(s);
	for (i=0; sd && sd[i]; i++) {
	    sd[i]->disconnect = 1;
	}
	free(sd);
    }    
}

/******************************************************************************
* library global functions                                                    *
******************************************************************************/

void rfb_set_color_translation(unsigned char video_link, fb_color_info_t *fb_c_info) {
    rfb_pix_fmt.bitsPerPixel_8 = fb_c_info->bpp;
    rfb_pix_fmt.depth_8        = fb_c_info->depth;
    rfb_pix_fmt.trueColour_8   = fb_c_info->bpp > 8;
    rfb_pix_fmt.redMax_be16    = (1 << fb_c_info->red_bits) - 1;
    rfb_pix_fmt.greenMax_be16  = (1 << fb_c_info->green_bits) - 1;
    rfb_pix_fmt.blueMax_be16   = (1 << fb_c_info->blue_bits) - 1;
    rfb_pix_fmt.redShift_8     = fb_c_info->blue_bits + fb_c_info->green_bits;
    rfb_pix_fmt.greenShift_8   = fb_c_info->blue_bits;
    rfb_pix_fmt.blueShift_8    = 0;
    
    rfb_for_all_clients(PP_RFB_CLIENT_TYPE_VIDEO_LINK, video_link, rfb_set_translate_function_v);
    rfb_for_all_clients(PP_RFB_CLIENT_TYPE_VIDEO_LINK, video_link, rfb_queue_fb_format_update_v);
}

void rfb_queue_fb_format_update(unsigned char video_link, fb_format_info_t *fb_f_info UNUSED) {
    rfb_for_all_clients(PP_RFB_CLIENT_TYPE_VIDEO_LINK, video_link, rfb_queue_fb_format_update_v);
}

static void
rfb_send_message(u_char data_link, const char* msg) {
    size_t pdu_size;
    char *pdu = rfb_create_rfb_message_pdu(msg, &pdu_size);

    if (pdu && pdu_size > 0) {
    	rfb_for_all_clients(PP_RFB_CLIENT_TYPE_DATA_LINK, data_link, rfb_enqueue_message_v, pdu, pdu_size);
    }

    free(pdu);
}

/* generate km messages according to km event states */
void rfb_km_event(pp_km_encoder_desc_t *enc UNUSED, unsigned char data_link, km_event_t* event) {
    if (event->type == KM_EVENT_MOUSE_SYNC) {
        switch(event->event.mouse_sync.type) {
            case KM_MOUSE_SYNC_IN_PROGRESS:
                rfb_send_message(data_link, _("Syncing mouse, please wait ..."));//dl
                break;
            case KM_MOUSE_SYNC_SUCCESS:
                rfb_send_message(data_link, _("Mouse sync successful."));//dl
                break;
            case KM_MOUSE_SYNC_FAILED:
                rfb_send_message(data_link, _("Could not sync mouse."));//dl
                break;
            case KM_MOUSE_SYNC_AMBIGUOUS_GOOD:
                rfb_send_message(data_link, _("Mouse sync ambiguous, might not work correctly."));//dl
                break;
            case KM_MOUSE_SYNC_AMBIGUOUS_BAD:
                rfb_send_message(data_link, _("Mouse sync ambiguous, please try to change mouse settings."));//dl
                break;
            default:
                pp_log("RFB received unknown mouse synchronisation event");
        }
    } else {
        if (event->type == KM_EVENT_DEBUG_MSG) {
            rfb_send_message(data_link, event->event.debug_msg.msg);//dl
        } else {
            pp_log("RFB received unknown mouse event");
        }
    }
}

static void
 rfb_set_osd_state(unsigned char video_link, const char* msg,
		   pp_blank_screen_t blank, unsigned short timeout_ms)
{
    MUTEX_LOCK(&rfb_context.osd_state_mtx);
    if (rfb_context.osd_message) free(rfb_context.osd_message);
    rfb_context.osd_message = (msg) ? strdup(msg) : strdup("");
    rfb_context.blank_state = blank;
    rfb_context.osd_timeout_ms = timeout_ms;
    MUTEX_UNLOCK(&rfb_context.osd_state_mtx);
    
    rfb_for_all_clients(PP_RFB_CLIENT_TYPE_VIDEO_LINK, video_link, rfb_send_osd_state_v);
}

/* generate osd messages according to vsc event states */
void rfb_vsc_event(unsigned char channel, pp_vsc_event_t* event) {
    unsigned char osd_message[128];
    char progress_l[AA_OSD_PROGRESS_MAX+1], progress_r[AA_OSD_PROGRESS_MAX+1];   
    unsigned char pos_l, pos_r;
    int i;
    
    if (event->type == VSC_EVENT_MODE_CHANGE) {
        // mode change events
        
        switch (event->event.mode_change.type) {
           
            case VSC_MODE_CHANGE_VALID:
#ifdef PP_FEAT_VSC_PANEL_INPUT
                snprintf(osd_message, sizeof(osd_message), "%d x %d", event->event.mode_change.res_x, event->event.mode_change.res_y);
#else
                if (event->event.mode_change.is_defaultmode) {
                    snprintf(osd_message, sizeof(osd_message), "%d x %d, %d Hz: %s", event->event.mode_change.res_x, event->event.mode_change.res_y,
                             event->event.mode_change.refresh, _("Please press the Auto-Adjust button!"));
                } else {
                    snprintf(osd_message, sizeof(osd_message), "%d x %d, %d Hz", event->event.mode_change.res_x, event->event.mode_change.res_y,
                             event->event.mode_change.refresh);
                }
#endif
                /* if the mode is not already adjusted, set a higher OSD timeout
                   to let the user read the message */
                rfb_set_osd_state(channel, osd_message, PP_DONT_BLANK_SCREEN, 
                                      (event->event.mode_change.is_defaultmode)? 3*OSD_MODE_MSG_TOUT : OSD_MODE_MSG_TOUT);
                break;
                
            case VSC_MODE_CHANGE_UNKNOWN:
                rfb_set_osd_state(channel, _("Unknown video mode"), PP_BLANK_SCREEN, 0);
                break;
                
            case VSC_MODE_CHANGE_NO_SIGNAL:
                rfb_set_osd_state(channel, _("No signal"), PP_BLANK_SCREEN, 0);
                break;
                
            case VSC_MODE_CHANGE_TO_UNSTABLE:
                rfb_set_osd_state(channel, NULL, PP_BLANK_SCREEN, 0);
                break;
                
            default: 
                pp_log("RFB received unknown vsc mode change event\n");
        }
    
    } else {
        // auto adjust events

        switch (event->event.aa_progress.type) {    
            case VSC_AA_SUCCESS:
                rfb_set_osd_state(channel, "", PP_DONT_BLANK_SCREEN, 0);
                break;
            
            case VSC_AA_IN_PROGRESS:
                /* create 'moving dots' */
                if (aa_progress_step < AA_OSD_PROGRESS_MAX) {
                    pos_l = aa_progress_step;
                } else {
                    pos_l = 2*AA_OSD_PROGRESS_MAX - aa_progress_step - 2;
                }
                pos_r = AA_OSD_PROGRESS_MAX - pos_l - 1;
                for (i=0; i<AA_OSD_PROGRESS_MAX; i++) {
                    progress_l[i] = (i == pos_l) ? '.' : ' ';
                    progress_r[i] = (i == pos_r) ? '.' : ' ';
                }   
                progress_l[AA_OSD_PROGRESS_MAX] = progress_r[AA_OSD_PROGRESS_MAX] = '\0';
                
                snprintf(osd_message, sizeof(osd_message), _("%s Auto adjustment in progress %s"), progress_l, progress_r);
                rfb_set_osd_state(channel, osd_message, PP_BLANK_SCREEN, 0);
                aa_progress_step++;
                aa_progress_step = aa_progress_step % ((AA_OSD_PROGRESS_MAX - 1)* 2);
                
                break;
                
            case VSC_AA_INTERRUPTED:
                rfb_set_osd_state(channel, _("Auto adjustment interrupted"), PP_DONT_BLANK_SCREEN, OSD_MSG_TOUT);
                break;
                
            case VSC_AA_FAILED:
                rfb_set_osd_state(channel, _("Auto adjustment failed"), PP_DONT_BLANK_SCREEN, OSD_MSG_TOUT);
                break;
                
            default:
                pp_log("RFB received unknown vsc aa event\n");
        }
    }
}

void rfb_send_kbdlayout_to_all(pp_km_encoder_desc_t *enc UNUSED, unsigned char data_link, char* kbd) {
    size_t pdu_size;
    char *pdu = rfb_create_kbd_layout_pdu(kbd, &pdu_size);

    if (pdu && pdu_size > 0) {
    	rfb_for_all_clients(PP_RFB_CLIENT_TYPE_DATA_LINK, data_link, rfb_enqueue_message_v, pdu, pdu_size);
    }
    
    free(pdu);
}

void rfb_schedule_update(unsigned char video_link) {
    rfb_for_all_clients(PP_RFB_CLIENT_TYPE_VIDEO_LINK, video_link, rfb_schedule_update_v);
}

void rfb_notify_grab_avail(unsigned char video_link) {
    rfb_for_all_clients(PP_RFB_CLIENT_TYPE_VIDEO_LINK, video_link, rfb_notify_grab_avail_v, 1);
}

void rfb_send_utf8_string(unsigned char video_link, char *msg, int len) {
    u_char unit;
    u_short port;
	
    if (PP_FAILED(pp_kvm_get_unit_port_for_video_link(video_link, &unit, &port))) {
	pp_log("cannot deterine port from video_link, don't send_utf8_string\n");
	return;
    }
    rfb_for_all_clients(PP_RFB_CLIENT_TYPE_TARGET, port, rfb_send_utf8string_v, msg, len);
}

void rfb_send_video_settings(unsigned char video_link) {
    rfb_for_all_clients(PP_RFB_CLIENT_TYPE_VIDEO_LINK, video_link, rfb_send_video_settings_v);
}

#if !defined(PP_FEAT_VSC_PANEL_INPUT)
void rfb_send_video_quality(unsigned char video_link) {
    rfb_for_all_clients(PP_RFB_CLIENT_TYPE_VIDEO_LINK, video_link, rfb_send_video_quality_v);
}
#endif

#if defined(PP_FEAT_DISABLE_KVM_OVER_IP)

int rfb_kvm_ip_disabled(void) {
    int ret;
    pp_cfg_is_enabled(&ret, "rc.disable_kvm_ip");
    return ret;
}

#endif

/* -------------------------------------------------------------------------- *
* connection parameter handling                                               *
* -------------------------------------------------------------------------- */

typedef struct {
    char *name;
    char *value;
} rfb_connection_parameter_t;

static void delete_connection_parameter(void *p) {
    rfb_connection_parameter_t *param = (rfb_connection_parameter_t *)p;
    free(param->name);
    free(param->value);
    free(param);
}

rfb_connection_parameter_list_t *rfb_create_connection_parameter_list(void) {
    return vector_new(NULL, 0, delete_connection_parameter);
}

void rfb_free_connection_parameter_list(rfb_connection_parameter_list_t *list) {
    vector_delete(list);
}

void rfb_add_connection_parameter(rfb_connection_parameter_list_t *list, const char *name, const char *value) {
    rfb_connection_parameter_t *param = malloc(sizeof(rfb_connection_parameter_t));
    param->name = name ? strdup(name) : NULL;
    param->value = value ? strdup(value) : NULL;
    vector_add(list, param);
}

size_t rfb_connection_parameter_list_size(rfb_connection_parameter_list_t *list) {
    return vector_size(list);
}

const char *rfb_connection_parameter_name(rfb_connection_parameter_list_t *list, unsigned int idx) {
    rfb_connection_parameter_t *param = vector_get(list, idx);
    return param->name;
}

const char *rfb_connection_parameter_value(rfb_connection_parameter_list_t *list, unsigned int idx) {
    rfb_connection_parameter_t *param = vector_get(list, idx);
    return param->value;
}

/* -------------------------------------------------------------------------- *
* CC-F related stuff                                                          *
* -------------------------------------------------------------------------- */

#if defined(PP_FEAT_SESSION_REDIRECTOR)

void rfb_sas_session_closed(eric_session_int_id_t id) {
    rfb_for_all_sas_clients(-1, rfb_sas_send_closed_web_session_v, id);
}

void rfb_sas_session_opened(eric_session_int_id_t id) {
    rfb_for_all_sas_clients(-1, rfb_sas_send_opened_web_session_v, id);
}

void rfb_sas_send_local_key(pp_km_encoder_desc_t *enc UNUSED, u_char key) {
    rfb_sas_send_kbd_event(key, 0, NULL);
}

void rfb_sas_send_local_mouse(pp_km_encoder_desc_t *enc UNUSED, int x, int y, int z, u_char buttonmask) {
    rfb_sas_send_mouse_event(x, y, z, buttonmask, rfbPointerRelativeEvent, 0, NULL);
}

void rfb_sas_send_kvm_switch_event(eric_session_int_id_t session,
				   u_char channel, u_char unit, u_short port) {
    rfb_for_all_sas_clients(channel, rfb_sas_send_kvm_switch_event_v, session,
    	(unsigned int)channel, (unsigned int)unit, (unsigned int)port);
}

void rfb_sas_login_failure(const char * user, const char * ip_str) {
    rfb_for_all_sas_clients(-1, rfb_sas_login_failure_v, user, ip_str);
}

#endif /* PP_FEAT_SESSION_REDIRECTOR */
