/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include <stdio.h>

#include <openssl/md5.h>
#include <pp/base.h>
#include <pp/utf.h>
#include <pp/um.h>
#include <pp/cfg.h>
#include <pp/rfb.h>
#include <pp/km.h>
#include <pp/kvm.h>
#include <pp/vsc.h>
#include <pp/intl.h>
#include <pp/hal_common.h>
#include <liberic_net.h>
#include <liberic_misc.h>
#include <liberic_notify.h>
#include <pp/xdefs.h>
#include "debug.h"
#include "rfb.h"
#include "predictor.h"
#include <pp/hal_kx2.h>
#ifdef PP_FEAT_RDM_SUPPORT
#include "pp/PP_RDM_c.h"
#endif

#define MAX_RDM_RESPONSE_LENGTH   100

#if defined (PP_FEAT_ESB2_TPT)
#include <pp/ipmi.h>
#endif

/******************************************************************************
* inline function                                                             *
******************************************************************************/

static inline int rfb_read_type(rfb_cl_t * clp) {
    u_int8_t ret;
    return rfb_read(clp, &ret, 1) ? -1 : ret;
}

/******************************************************************************
* data                                                                        *
******************************************************************************/

volatile int pp_rfb_disconnect_all = 0;

/******************************************************************************
* function prototypes                                                         *
******************************************************************************/

static int rfb_process_init_pdu(rfb_cl_t * clp);
static int rfb_authenticate_session_id(rfb_cl_t *clp, eric_session_int_id_t *session);
static int rfb_authenticate_plain(rfb_cl_t *clp, char *user, eric_session_int_id_t *session);
static int rfb_authenticate_md5(rfb_cl_t *clp, char *user, eric_session_int_id_t *session);
#ifdef PP_FEAT_RDM_SUPPORT
static int rfb_authenticate_rdm_session(rfb_cl_t *clp, eric_session_int_id_t *session);
#endif
static int rfb_authenticate_noauth(rfb_cl_t *clp, eric_session_int_id_t *session);
static int rfb_read_pdu(rfb_cl_t * clp);
static void disconnect_if_non_exclusive(eric_session_int_id_t s, va_list args);

/******************************************************************************
* library global functions                                                    *
******************************************************************************/

int rfb_authenticate(rfb_cl_t *clp, eric_session_int_id_t *session) {
    char user[32 + 1];
    int auth_method;
    u_int32_t flags;
    int ret;
    unsigned int caps = rfbAuthHttpSessionID | rfbAuthPlain | rfbAuthMD5;
    int noauth_allowed = 0;
    
#ifdef PP_FEAT_RDM_SUPPORT
    caps |= rfbAuthRDMsession;
#endif

    if (rfb_can_be_connected_by_proxy(clp)) {
        noauth_allowed = 1;
    }

    if (noauth_allowed) {
        caps |= rfbAuthNoAuth;
    }

    if (rfb_send_auth_caps(clp, caps) ||
    	(rfb_read_type(clp) != rfbLogin) ||
    	(auth_method = rfb_read_login_pdu(clp, user, &flags)) == -1) {
    	
    	return -1;
    }

    switch (auth_method) {
    	case rfbAuthHttpSessionID:
            printf("authenticating with session id\n");
            ret = rfb_authenticate_session_id(clp, session);
            break;
        case rfbAuthPlain:
            printf("authenticating plain\n");
            ret = rfb_authenticate_plain(clp, user, session);
            break;
        case rfbAuthMD5:
            printf("authenticating with md5\n");
            ret = rfb_authenticate_md5(clp, user, session);
            break;
#ifdef PP_FEAT_RDM_SUPPORT
        case rfbAuthRDMsession:
            printf("authenticating with rdm session\n");
            ret = rfb_authenticate_rdm_session(clp, session);
            break;
#endif /* PP_FEAT_RDM_SUPPORT */
        case rfbAuthNoAuth:
            if (noauth_allowed) {
                ret = rfb_authenticate_noauth(clp, session);
                break;
            }
            // else fall through
        default:
            rfb_send_quit(clp, rfbQuitProtoError);
            D(D_ERROR, "Client's authentication method not supported: 0x%.2x\n", auth_method);
            return -1;
    }
    
    if (ret) {
    	return ret;
    }
    
    /* send ack to the client */
    flags = rfb_adjust_connection_flags(clp, flags);
    
#if defined(PP_FEAT_DISABLE_KVM_OVER_IP)
    if (rfb_kvm_ip_disabled() && !(flags & rfbConnectionFlagSasHW)) {
    	pp_log("KVM-IP sessions not supported!\n");
    	rfb_send_quit(clp, rfbQuitNoPerm);
    	return -1;
    }
#endif
    
    return rfb_send_auth_successful(clp, flags);
}

int rfb_negotiate_protocol_version(rfb_cl_t * clp) {
    int major, minor;
    
    if (rfb_send_protocol_version(clp, rfbProtocolMajorVersion, rfbProtocolMinorVersion) == -1)
        return -1;

    if (rfb_read_protocol_version(clp, &major, &minor) == -1)
        return -1;

    D(D_NOTICE, "Client protocol version %d.%d\n", major, minor);

    if (major != rfbProtocolMajorVersion) {
	D(D_ERROR, "Major version mismatch\n");
	return -1;
    }

    if (minor != rfbProtocolMinorVersion) {
	/* Minor version mismatch - warn but try to continue */
	D(D_NOTICE, "Minor version mismatch. Ignored.\n");
    }

    return 0;
}

int rfb_initialize(rfb_cl_t * clp) {
    size_t utf8namelen;
    char * utf8name;

    if ((utf8name = pp_latin_to_utf8(clp->session ?
				  eric_session_get_user(clp->session) : "n/a",
				  &utf8namelen)) == NULL) {
	return -1;
    }

    if (rfb_enqueue_utf8string(clp, utf8name, utf8namelen) ||
	rfb_process_init_pdu(clp) ||
	rfb_send_osd_state(clp) ||
	rfb_send_kbd_layout(clp)) {

	free(utf8name);
	return -1;
    }
    free(utf8name);

    clp->init_handshake_done = 1;

    if (rfb_needs_proxy_connection(clp)) {
        if (rfb_enqueue_fb_format_update_params(clp, 640, 480, 0, &rfb_pix_fmt, NULL) == -1 ||
            rfb_proxy_connect_to_channel(clp) == -1) {
            return -1;
        }
    } else {
    	// FIXME: make sure that rfb_pix_fmt is current!
        if (rfb_enqueue_fb_format_update(clp, &rfb_pix_fmt) == -1) {
            return -1;
        }
    }

#if defined(PP_FEAT_SESSION_REDIRECTOR)
    if ((clp->connection_flags & (rfbConnectionFlagSasHW | rfbConnectionFlagSasSW)) && rfb_sas_send_open_sessions(clp)) {
    	return -1;
    }
#endif /* PP_FEAT_SESSION_REDIRECTOR */

    return 0;
}

int rfb_main_protocol_handler(rfb_cl_t * clp) {
    while (!rfb_read_pdu(clp)) ;
    return 0;
}

/**
 * send video settings update, may be called from writer thread
 * (if a client requested the update immediately) or triggered
 * external (using ..._to_all_clients) from tft lib
 */
int rfb_send_vs_update(rfb_cl_t * clp) {
    video_settings_t settings;
    
    if (!clp->vs_update_allowed) return 0;
    
    if (PP_FAILED(pp_vsc_get_video_settings(clp->video_link, &settings))) {
	return -1;
    }
    
    return rfb_enqueue_vs_update(clp, &settings);
}

/**
 * send a video quality update
 */
int rfb_send_vq_update(rfb_cl_t * clp) {
    int noise_filter;
    int cr;

    if ((cr = pp_cfg_get_int(&noise_filter, "video.diff_tolerance")) != PP_SUC) {
    	return -1;
    }

    return rfb_enqueue_vq_update(clp, noise_filter);
}

int rfb_send_bandwidth_request(rfb_cl_t * clp, unsigned int length) {
    clp->pred_bwidth_bytes = length;
    
    D(D_VERBOSE, "Sending Bandwidth Request Length %d\n", length);
    return rfb_enqueue_bandwidth_request(clp, length);
}

int rfb_send_osd_state(rfb_cl_t * clp) {
    int ret;
    
    MUTEX_LOCK(&rfb_context.osd_state_mtx);
    ret = rfb_enqueue_osd_state(clp, rfb_context.osd_message,
    	rfb_context.blank_state, rfb_context.osd_timeout_ms);
    MUTEX_UNLOCK(&rfb_context.osd_state_mtx);    

    return ret;
}

int rfb_send_kbd_layout(rfb_cl_t * clp) {
    size_t pdu_size;
    int ret = 0;
    char *kbd = NULL;
    char *pdu;
    u_char unit;
    u_short port;

#ifdef PP_FEAT_RPCCFG
    u_int data_link = pp_misc_get_kvm_node();
#else    
    u_int data_link = clp->data_link; 
#endif

    if (PP_SUCCED(pp_kvm_get_unit_port_for_data_link(data_link, &unit, &port))) {
	pp_cfg_get(&kbd, "unit[%u].port[%u].kbd.model", unit, port);
    }
    if (kbd == NULL) {
    	kbd = strdup("pc104");
    }
    
    pdu = rfb_create_kbd_layout_pdu(kbd, &pdu_size);

    if (pdu && pdu_size > 0) {
    	ret = rfb_writer_enqueue_pdu(clp, pdu, pdu_size);
    }
    
    free(kbd);
    free(pdu);
    
    return ret;
}

/* setup the encoding used for all framebuffer updates from now on
   (the currently running one is sent with the old encoding) */
void rfb_setup_encoding(rfb_cl_t * clp, u_long enc_word,
			rfb_send_rect_softenc_fn_t softenc_fn,
			u_char use_cache) {
    u_char enc = enc_word & rfbEncodingMask;

    MUTEX_LOCK(&clp->enc_setup_mtx);    
#ifdef PP_FEAT_VSC_HW_ENCODING
    clp->hwenc_setup_needed = 1;
	    
    if (enc & rfbEncodingIsHW) {
	clp->hwenc_subenc = (enc_word & rfbEncodingParamSubencMask) >> rfbEncodingParamSubencShift;
	clp->hwenc_zlib_level = (enc_word & rfbEncodingParamZLIBMask) >> rfbEncodingParamZLIBShift;
	D(D_NOTICE, "HW encoding - subenc: %u zlib: %u\n", clp->hwenc_subenc, clp->hwenc_zlib_level);
    }

    clp->enc_req.pref_enc = enc;
    clp->enc_req.send_rect_softenc_fn = softenc_fn;
    
    clp->enc_req.video_optimized = 0;   // video optimized doesn't make sense here
#else
    clp->enc_req.pref_enc = enc;
    clp->enc_req.send_rect_softenc_fn = softenc_fn;

    clp->enc_req.video_optimized = (enc == rfbEncodingRawVSC) ? 1 : 0;   
#endif

    /* set tight subencoding (low color depths) */
    if (enc == rfbEncodingTight) {
	clp->enc_req.tightXBitFullColor  = (enc_word & rfbEncodingParamSubencMask) >> rfbEncodingParamSubencShift;
	clp->enc_req.tightCompressLevel  = (enc_word & rfbEncodingParamZLIBMask) >> rfbEncodingParamZLIBShift;
	clp->enc_req.tightCacheEnabled   = use_cache;
    }
    
    if (clp->enc_req.video_optimized && (rfb_get_exclusive_video(clp) == 0)) {
	D(D_NOTICE, "Video optimized encoding not available, not syncing framebuffer\n");
	rfb_enqueue_message(clp, _("Video optimized encoding not available, video not synchronized."));
	clp->enc_req.video_optimized = 0;
    }

    clp->client_encoding_change_req = 1;
    MUTEX_UNLOCK(&clp->enc_setup_mtx);    
}

/* ----
   The following routines must only be called from within the writer thread! 
   ---- */

/* switch the encoding before a framebuffer update */
void rfb_prepare_encoding(rfb_cl_t * clp) {
    MUTEX_LOCK(&clp->enc_setup_mtx);
    if (clp->client_encoding_change_req) {
	clp->enc = clp->enc_req;
	
	if (clp->client_pix_fmt_change_req) {
	    clp->pix_fmt = clp->pix_fmt_req;
	    rfb_set_translate_function(clp);
	    
	    clp->client_pix_fmt_change_req = 0;
	    rfb_writer_send_ack_pixel_format(clp, &clp->pix_fmt);
	    
	    if (clp->enc.tightCacheEnabled) {
		if (!clp->cache_lists &&
		    rfb_init_cache_lists(clp, (clp->fb_width_pd/rfbTightTileDimension) * (clp->fb_height_pd/rfbTightTileDimension)) == -1) {
		    
		    rfb_deinit_cache_lists(clp);
		    D(D_ERROR, "Could not initialize CACHE lists\n");
		}
	    } else {
		rfb_deinit_cache_lists(clp);
	    }
	}

	clp->client_encoding_change_req = 0;
    }

    MUTEX_UNLOCK(&clp->enc_setup_mtx);
}

int rfb_send_fb_format_update(rfb_cl_t * clp) {
    if (clp->enc.tightCacheEnabled && ((clp->fb_width_old != clp->fb_width) || (clp->fb_height_old != clp->fb_height))) {
	rfb_deinit_cache_lists(clp);
	rfb_init_cache_lists(clp, (clp->fb_width_pd/rfbTightTileDimension) * (clp->fb_height_pd/rfbTightTileDimension));
    }
    clp->fb_width_old = clp->fb_width;
    clp->fb_height_old = clp->fb_height;

    // FIXME: make sure that rfb_pix_fmt is current !
    return rfb_writer_send_fb_format_update(clp, clp->fb_is_unsupported,
    	clp->fb_width, clp->fb_height, &rfb_pix_fmt);
}

/*
 * Send the contents of updateBuf.  Returns 1 if successful, -1 if
 * not (errno should be set).
 */
int rfb_send_update_buf(rfb_cl_t * clp) {
    int ret;
    int ub_len = clp->ub_len;

#ifdef TIME_MEASURE  
    pp_hrtime_start(&clp->time_net);
#endif
    clp->ub_len = 0;

    pp_hrtime_stop(&clp->pred_watch_real);
    clp->pred_t_ns   += pp_hrtime_read(&clp->pred_watch_real);
    clp->pred_v_real += ub_len;
    ret = rfb_writer_send_update_buf(clp, clp->update_buf, ub_len);
    pp_hrtime_start(&clp->pred_watch_real);
    
#ifdef TIME_MEASURE
    pp_hrtime_stop(&clp->time_net);
    clp->time_ns_net += pp_hrtime_read(&clp->time_net);
#endif
    
    return ret;
}

/*
 * rfbSendFramebufferUpdate - send the currently pending framebuffer update
 * to the RFB client.
 */
int rfb_send_fb_update(rfb_cl_t * clp, int clip_mod_region, int full_update) {
    int send_update = 1;
    int ret= 0;

#ifdef TIME_MEASURE
    clp->time_ns_all = clp->time_ns_net = 0;
    pp_hrtime_start(&clp->time_all);
#endif

    MUTEX_LOCK(&clp->upd_mtx);
      
    /*
     * We need to clip req_reg to current fb resolution if
     * fb resolution has changed.
     */

    if (clip_mod_region) {
	BoxRec box;
	RegionRec reg;

	box.x1 = 0;
	box.y1 = 0;
	box.x2 = clp->fb_width_pd;
	box.y2 = clp->fb_height_pd;
	REGION_INIT(&reg, &box, 0);
	REGION_INTERSECT(&clp->req_reg, &clp->req_reg, &reg);
	REGION_UNINIT(&reg);
    }  
    
    /*
     * The client is interested in the region reqRegion.
     * If it's empty then no update is needed.
     */

    if (!REGION_NOTEMPTY(&clp->req_reg)) {
	send_update = 0;
    }

    MUTEX_UNLOCK(&clp->upd_mtx);

    if (!send_update) {
	clp->fb_update_requested = 1;
	return 0;
    }
    
    /* Now send the update */
#ifdef PP_FEAT_VSC_HW_ENCODING
    ret = rfb_send_update_hwenc(clp, &clp->req_reg, full_update);
#else
    ret = rfb_send_update_softenc(clp, &clp->req_reg, full_update);
#endif

    return ret;
}


/******************************************************************************
* local protocol processing functions                                         *
******************************************************************************/

/* tell the client our capabilities and connection parameters */


/* check capabilities */
#define CHECK_PP_FEAT(feat, have, havnt) (pp_feature_supported(feat) ? (have) : (havnt))

#if defined (PRODUCT_ERIC2) || !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)
# define IS_ERIC() "yes"
#else
# define IS_ERIC() "no"
#endif

#define ADD_PARAM_IDX(name, idx, value)                         \
    do {                                                        \
        if ((int)idx >= 0) {                                    \
            char *d = NULL;                                     \
            asprintf(&d, "%s%u", name, idx);                    \
            rfb_add_connection_parameter(list, d, value);       \
            free(d);                                            \
        } else {                                                \
            rfb_add_connection_parameter(list, name, value);    \
        }                                                       \
    } while (0)
#define ADD_PARAM(name, value) ADD_PARAM_IDX(name, -1, value)

#define ADD_PARAMF_IDX(name, idx, format, args...)              \
    do {                                                        \
        char *_d = NULL;                                        \
        asprintf(&_d, format, ##args);                          \
        ADD_PARAM_IDX(name, idx, _d);                           \
        free(_d);                                               \
    } while (0)
#define ADD_PARAMF(name, format, args...) ADD_PARAMF_IDX(name, -1, format, ##args)

#define ADD_AND_FREE_IDX(name, idx, val)                        \
     do {                                                       \
        char *s = val;                                          \
        if (s) {                                                \
            ADD_PARAM_IDX(name, idx, s);                        \
            free(s);                                            \
        }                                                       \
    } while(0)
#define ADD_AND_FREE(name, val) ADD_AND_FREE_IDX(name, -1, val)

#define ADD_CFG_IDX(name, idx, key, args...)                    \
    do {                                                        \
        char *k = NULL;                                         \
        if (PP_SUCCED(pp_cfg_get_nodflt(&k, key, ##args))) {    \
            ADD_PARAM_IDX(name, idx, k);                        \
        } else {                                                \
            ADD_PARAM_IDX(name, idx, "");                       \
        }                                                       \
        free(k);                                                \
    } while (0)
#define ADD_CFG(name, key, args...) ADD_CFG_IDX(name, -1, key, ##args)

#define ADD_CFG_UID_IDX(name, idx, uid, key, args...)           \
    do {                                                        \
        if (uid >= 0) ADD_CFG_IDX(name, idx, key, ##args);      \
    } while (0)
#define ADD_CFG_UID(name, uid, key, args...) do { ADD_CFG_UID_IDX(name, -1, uid, key, ##args);  } while(0)

#define ADD_PERM(name, user, permname, permdesc)                \
    do {                                                        \
        if (user && *user != '\0' &&                            \
            PP_SUCCED(pp_um_user_has_permission(                \
                      user, permname, permdesc))) {             \
            ADD_PARAM(name, "yes");                             \
        } else {                                                \
            ADD_PARAM(name, "no");                              \
        }                                                       \
    } while (0)

/* send the table of capabilities and connection parameters */
int rfb_send_connection_parameters(rfb_cl_t * clp) {
    int ret;
    unsigned int i;
    int uid = -1;
    
    rfb_connection_parameter_list_t *list = rfb_create_connection_parameter_list();
    
    const char *user = eric_session_get_user(clp->session);
    if (user) uid = pp_um_user_get_uid(user);

    /* first, fill the parameters table */
    ADD_PARAM("hw_enc", CHECK_PP_FEAT("PP_FEAT_VSC_HW_ENCODING", "yes", "no"));
    ADD_PARAM("HWENC", CHECK_PP_FEAT("PP_FEAT_VSC_HW_ENCODING", "yes", "no"));  /* we need this twice because of old clients */
    ADD_PARAM("refresh_video", CHECK_PP_FEAT("PP_FEAT_VSC_PANEL_INPUT", "no", "yes"));
    ADD_PARAM("kvm_switch", IS_ERIC());

    /* handle video settings caps */
#ifdef PP_FEAT_VSC_PANEL_INPUT
    {
	int vsc_display_enable = pp_vsc_have_display_enable(clp->video_link);
	ADD_PARAM("video_settings", vsc_display_enable ? "no" : "yes");
	ADD_PARAM("VS_TYPE", vsc_display_enable ? "" : "offset");
    }
#else	
    ADD_PARAM("video_settings", "yes");
    ADD_PARAM("VS_TYPE", "");
#endif
    
    ADD_AND_FREE("BOARD_NAME", eric_misc_get_boardname());
    ADD_PARAM("BOARD_TYPE", "lara");
    ADD_AND_FREE("HW_ID", eric_misc_get_hardware_rev_str());
    ADD_PARAM("WLAN_ENABLED", CHECK_PP_FEAT("PP_FEAT_WLAN", "yes", "no"));

    unsigned int hotkey_no = 0;
    unsigned int hk_index = 0;
    if (uid >= 0) pp_cfg_get_uint_nodflt(&hotkey_no, "user[%u].rc.hotkey._s_", uid);
    for (i = 0; i < hotkey_no; i++) {
        ADD_CFG_UID_IDX("HOTKEY_", hk_index, uid, "user[%u].rc.hotkey[%u].key", uid, i);
        ADD_CFG_UID_IDX("HOTKEYCODE_", hk_index, uid, "user[%u].rc.hotkey[%u].code", uid, i);
        ADD_CFG_UID_IDX("HOTKEYNAME_", hk_index, uid, "user[%u].rc.hotkey[%u].name", uid, i);
        hk_index++;
    }

#ifdef PP_FEAT_RPCCFG
    u_int data_link = pp_misc_get_kvm_node();
#else    
    u_int data_link = clp->data_link; 
#endif

    unsigned int port_count = 0;
    u_char unit;
    u_short port;
    if (PP_SUCCED(pp_kvm_get_unit_port_for_data_link(data_link, &unit, &port))) {
	port_count = pp_kvm_get_unit_port_count(unit);
    }

    for (i = 0; i < port_count; i++) {
        int show = 0;
        char *key = NULL;
        char *name = NULL;
        pp_cfg_is_enabled_nodflt(&show, "unit[%u].port[%u].show_in_rc", unit, i);
        pp_cfg_get_nodflt(&key, "unit[%u].port[%u].hotkey.key", unit, i);
        pp_cfg_get_nodflt(&name, "unit[%u].port[%u].name", unit, i);
        
        if (show && key && *key != '\0') {
            if (name && *name != '\0') {
                ADD_PARAM_IDX("HOTKEY_", hk_index, name);
            } else {
#if defined(PRODUCT_XX01IP_ANY)
                ADD_PARAMF_IDX("HOTKEY_", hk_index, "%u", i+1);
#else
                ADD_PARAM_IDX("HOTKEY_", hk_index, key);
#endif
            }
            ADD_PARAMF_IDX("HOTKEYCODE_", hk_index, "kvm %u", i);
            hk_index++;
        }
        
        free(key);
        free(name);
    }

    ADD_CFG_UID("MOUSESYNC_KEY", uid, "user[%u].rc.mousesync.key", uid);
    ADD_CFG_UID("MOUSESYNC_KEYCODE", uid, "user[%u].rc.mousesync.code", uid);
    ADD_CFG_UID("EXCLUSIVE_MOUSE", uid, "user[%u].exclusive_mouse", uid);

    ADD_CFG_UID("SOFTKBD_MAPPING", uid, "user[%u].softkbd_mapping", uid);
    ADD_CFG_UID("LOCALKBD_MAPPING", uid, "user[%u].localkbd_mapping", uid);
    ADD_CFG_UID("LOCAL_CURSOR", uid, "user[%u].rc.local_cursor", uid);
    ADD_CFG("HOTKEYPAUSE", "kvm.key_pause_duration");

    // TODO: add config keys for full-screen keys
    //ADD_PARAM("FULLSCREEN_KEY", "");
    //ADD_PARAM("FULLSCREEN_KEYCODE", "");

#if defined(PP_FEAT_SESSION_REDIRECTOR)
    ADD_CFG_UID("FORENSIC_CONSOLE", uid, "user[%u].rc.forensic_enabled", uid);
#else
    ADD_PARAM("FORENSIC_CONSOLE", "no");
#endif

    int no_drvredir = 0;
#if defined(PP_FEAT_DRIVE_JAVA_REDIRECTION)
    int drv_redir_disabled = 0;
    pp_cfg_is_enabled_nodflt(&drv_redir_disabled, "vfloppy.usb_ms_msp_disabled");
    if (!drv_redir_disabled &&
        PP_SUCCED(pp_um_user_has_permission(user, "vfloppy", "yes"))) {
        no_drvredir = PP_FEAT_USB_MASS_STORAGE_NO;
    }
#endif
    ADD_PARAMF("DRIVE_REDIRECTION_NO_DRIVES", "%d", no_drvredir);
    
    int drv_redir_readonly = 0;
    pp_cfg_is_enabled_nodflt(&drv_redir_readonly, "vfloppy.usb_ms_msp_readonly");
    ADD_PARAM("drive_redirection_read_only", drv_redir_readonly ? "yes" : "no");

    ADD_PERM("VS_PERM_STD", user, "video_s", "yes");
    ADD_PERM("VS_PERM_ADV", user, "vs_adv", "yes");
    ADD_PERM("VS_ENC", user, "rc_s_enc", "yes");

    const char *monitor;
    int _monitor = 0;
    pp_cfg_is_enabled_nodflt(&_monitor, "user[%u].rc.monitor_mode_enabled", uid);
    if (!_monitor) {
       monitor = "no";
    } else if (PP_SUCCED(pp_um_user_has_permission(user, "rc_s_mm", "yes"))) {
       monitor = "yes";
    } else {
        monitor = "sticky";
    }
    ADD_PARAM("MONITOR_MODE", monitor);
    
    const char *use_iip;
#if defined(PP_FEAT_RC_MOUSEBTNS_ALWAYS_ON)
    use_iip = "yes";
#else /* !PP_FEAT_RC_MOUSEBTNS_ALWAYS_ON */
    char *input_type = NULL, *usb_type = NULL;
    pp_cfg_get_nodflt(&input_type, "input_type");
    pp_cfg_get_nodflt(&usb_type, "usb_type");
    if ((!pp_km_is_ps2_possible(unit) || (usb_type && !strcmp(input_type, "usb"))) && input_type && !strcmp(usb_type, "absolute")) {
        use_iip = "no";
    } else {
        use_iip = "yes";
    }
    free(input_type);
    free(usb_type);
#endif /* PP_FEAT_RC_MOUSEBTNS_ALWAYS_ON */
    ADD_PARAM("USE_IIP", use_iip);
    
    ADD_CFG("lang", "language");

    const char *excl;
#ifdef PP_FEAT_RC_EXCLUSIVE_MODE
    int _excl = 0;
    pp_cfg_is_enabled_nodflt(&_excl, "user[%u].rc.excl_access_enabled", uid);
    if (PP_SUCCED(pp_um_user_has_permission(user, "rc_s_excl", "yes"))) {
        excl = _excl ? "on" : "off";
    } else {
        excl = _excl ? "sticky" : "no";
    }
#else
    excl = "no";
#endif
    ADD_PARAM("EXCLUSIVE_PERM", excl);
    
    ADD_CFG_UID("SelEnc", uid, "user[%u].rc.encoding._c_", uid);
    ADD_CFG_UID("FixEnc", uid, "user[%u].rc.encoding.preconf", uid);
    ADD_CFG_UID("AdvEncCR", uid, "user[%u].rc.encoding.manual.compress", uid);
    ADD_CFG_UID("AdvEncCD", uid, "user[%u].rc.encoding.manual.colordepth", uid);

#if (defined(OEM_INTEL) || defined(OEM_LENOVO))
    ADD_PARAM("osd_bgcolor", "#00ff00");
    ADD_PARAM("osd_fgcolor", "#ff0000");
    ADD_PARAM("osd_position", "center");
    ADD_PARAM("osd_alpha", "70");
#endif

    /* send the character encoding */
    const char *charset = pp_intl_get_charset();
    if (charset) {
        ADD_PARAM("charset", charset);
    }
    
    /* then, send it */
    ret = rfb_enqueue_connection_parameters(clp, list);
    rfb_free_connection_parameter_list(list);
    return ret;
}

static int rfb_process_init_pdu(rfb_cl_t * clp) {
    BoxRec box;
    fb_format_info_t fb_f_info;
    u_char kvm_node, unit;
    u_int16_t target; 
    int ret = 0;
    u_int16_t flags;

    if (rfb_read_init_pdu(clp, &kvm_node, &unit, &target, &flags) == -1) {
	return -1;
    }
    
    /* get new link client */
    clp->link_client = pp_kvm_new_link_client(clp->session, PP_KVM_REMOTE);
    if (clp->link_client == NULL) {
	printf("couldn't get link_client - major problem\n");
	return -1;
    }

    clp->kvm_node = kvm_node;
    /* switch to port supplied by applet, 0xff means don't change target_port */
    if (flags & rfbinitMsgFlagsDoInitialSwitch) {
	if (PP_FAILED(pp_kvm_switch_port(clp->link_client, unit, target, &clp->data_link, &clp->video_link))) {
	    return -1;
#if defined(PRODUCT_KX2)
	} else  {
	    pp_hal_kx2_set_user_led_by_video_link(clp->video_link , PP_HAL_KX2_USER_LED_ON);
#endif
	}
    } else {
	if (PP_FAILED(pp_kvm_set_link_client_links(clp->link_client, kvm_node, kvm_node))) {
	    pp_log("setting links of link_client to kvm_node %d failed!\n", kvm_node);
	    return -1;
	}

	if (PP_FAILED(pp_kvm_get_video_link(clp->link_client, &clp->video_link)) ||
	    PP_FAILED(pp_kvm_get_data_link(clp->link_client, &clp->data_link)) ) {
	    pp_log("cannot determine current video_link and data_link, bailing out\n");
	    return -1;
	}
    }

    MUTEX_LOCK(&clp->s2c_queue_mtx);
    
    if (!rfb_needs_proxy_connection(clp)) {
        clp->need_fb_fmt_update = 1;
        if (pthread_cond_signal(&clp->s2c_queue_cond) != 0) {
    	    D(D_ERROR, "%s(): pthread_cond_signal() failed\n", ___F);
        }
    }
    
    MUTEX_UNLOCK(&clp->s2c_queue_mtx);

    /* get framebuffer info for this port */
    pp_grab_get_format_info(clp->video_link, &fb_f_info);
    
    clp->fb_width          = fb_f_info.g_w;    
    clp->fb_height         = fb_f_info.g_h;
    clp->fb_width_pd       = fb_f_info.g_w_pd;
    clp->fb_height_pd      = fb_f_info.g_h_pd;
    clp->fb_tiles_w        = clp->fb_tile_pitch = fb_f_info.tiles_w;
    clp->fb_tiles_h	   = fb_f_info.tiles_h;
    clp->fb_bpp            = fb_f_info.bpp;
    clp->fb_is_unsupported = fb_f_info.is_unsupported;
    
    box.x1 = box.y1 = 0;
    box.x2 = clp->fb_width_pd;
    box.y2 = clp->fb_height_pd;

    REGION_INIT(&clp->req_reg, &box, 0);
    REGION_INIT(&clp->ack_reg, NullBox, 0);

    return ret;
}

static int rfb_handle_set_pixel_format(rfb_cl_t * clp) {
    rfbPixelFormat fmt;
    
    if (rfb_read_set_pixel_format_pdu(clp, &fmt)) {
    	return -1;
    }

    MUTEX_LOCK(&clp->enc_setup_mtx);
    clp->pix_fmt_req.bitsPerPixel_8 = fmt.bitsPerPixel_8;
    clp->pix_fmt_req.depth_8        = fmt.depth_8;
    clp->pix_fmt_req.bigEndian_8    = fmt.bigEndian_8;
    clp->pix_fmt_req.trueColour_8   = fmt.trueColour_8;
    clp->pix_fmt_req.redMax_be16    = fmt.redMax_be16;
    clp->pix_fmt_req.greenMax_be16  = fmt.greenMax_be16;
    clp->pix_fmt_req.blueMax_be16   = fmt.blueMax_be16;
    clp->pix_fmt_req.redShift_8     = fmt.redShift_8;
    clp->pix_fmt_req.greenShift_8   = fmt.greenShift_8;
    clp->pix_fmt_req.blueShift_8    = fmt.blueShift_8;
    
    clp->client_pix_fmt_change_req = 1;
    MUTEX_UNLOCK(&clp->enc_setup_mtx);
    
    return 0;
}

static int rfb_handle_set_encodings(rfb_cl_t * clp) {
    u_int32_t *encodings = NULL;
    int encs, i, enc_found = 0;
    u_int32_t enc_word_request = 0;
    rfb_send_rect_softenc_fn_t softenc_request = NULL;
    u_char    cache_request = 0;
    
    if ((encs = rfb_read_set_encodings_pdu(clp, &encodings)) < 0 || !encodings) {
    	return encs;
    }

    clp->pred_type = PRED_TYPE_NONE;

    for (i = 0; i < encs; i++) {
        u_char enc;
        
        enc = encodings[i] & rfbEncodingMask;
        
        if (enc == rfbEncodingCopyRect) {
  	    D(D_NOTICE, "CopyRect encoding not supported, ignored.\n");
        } else if (enc == rfbEncodingAuto) {
#ifdef PP_FEAT_VSC_HW_ENCODING
  	    clp->pred_type = PRED_TYPE_HW;
  	    D(D_NOTICE, "Enabled automatic encoding predictor (hardware)\n");
#else
  	    clp->pred_type = PRED_TYPE_SW;
  	    D(D_NOTICE, "Enabled automatic encoding predictor (software), hw not possible\n");
#endif
        } else if (enc == rfbEncodingAutoSW) {
  	    clp->pred_type = PRED_TYPE_SW;
  	    D(D_NOTICE, "Enabled automatic encoding predictor (software)\n");
    
        } else if (enc == rfbEncodingUseTightCache) {   
  	    cache_request = 1;
  	    D(D_NOTICE, "Requesting CACHE in Tight encoding\n");
        } else if (enc_found != 0) {
  	    continue;
        } else if (enc == rfbEncodingRaw) {
  	    enc_word_request = encodings[i]; softenc_request = rfb_send_rect_raw;
  	    enc_found = 1;
  	    D(D_NOTICE, "Using RAW encoding for %s\n", clp->client_ip);
        } else if (enc == rfbEncodingRawVSC) {
  	    enc_word_request = encodings[i]; softenc_request = rfb_send_rect_rawvsc;
  	    enc_found = 1;
  	    D(D_NOTICE, "Using RAW VSC encoding for %s\n", clp->client_ip);
        } else if (enc == rfbEncodingHextile) {
  	    enc_word_request = encodings[i]; softenc_request = rfb_send_rect_hextile;
  	    enc_found = 1;
  	    D(D_NOTICE, "Using HEXTILE encoding for %s\n", clp->client_ip);
        } else if (enc == rfbEncodingTight) {
  	    enc_word_request = encodings[i]; softenc_request = rfb_send_rect_tight;  
  	    enc_found = 1;
  	    D(D_NOTICE, "Using TIGHT encoding for %s\n", clp->client_ip);
        } 
#ifdef PP_FEAT_VSC_HW_ENCODING
        else if (enc == rfbEncodingLRLEHard) {
  	    enc_word_request = encodings[i]; softenc_request = NULL;
  	    enc_found = 1;
  	    D(D_NOTICE, "Using hardware LRLE encoding for %s\n", clp->client_ip);
        }
#else
        else if (enc == rfbEncodingLRLESoft) {
  	    enc_word_request = encodings[i]; softenc_request = rfb_send_rect_lrle;
  	    enc_found = 1;
  	    D(D_NOTICE, "Using software LRLE encoding for %s\n", clp->client_ip);
        }
#endif
        else {
  	    D(D_NOTICE, "rfb_process_normal_pdu(): ignoring "
  		   "unknown encoding type (0x%x)\n",
  		   (int)enc);
        }
    }
  
    if (enc_found == 0) {
        enc_word_request = rfbEncodingRaw; softenc_request = rfb_send_rect_raw; cache_request = 0;
        D(D_NOTICE, "Falling back to RAW encoding for %s\n", clp->client_ip);
    }
  
    rfb_setup_encoding(clp, enc_word_request, softenc_request, cache_request);	    


    free(encodings);
    return 0;
}

static int rfb_handle_fb_update_req(rfb_cl_t * clp) {
    BoxRec box;
    RegionRec tmp_reg;
    int incremental;
    
#ifdef TIME_MEASURE
    if (clp->time_ns_req != 0) {
        pp_hrtime_stop(&clp->time_req);
        clp->time_ns_req = pp_hrtime_read(&clp->time_req);
        clp->time_ns += clp->time_ns_req;
        if (clp->time_ns >= 10000000000) {
  	    printf("Pixel/10s=%ld\n", clp->time_pixel);
  	    clp->time_ns = 0;
  	    clp->time_pixel = 0;
        }
    } else {
        clp->time_ns_req = 1;
    }

#endif
    if (rfb_read_fb_update_req_pdu(clp, &incremental, &box) == -1) {
    	return -1;
    }

    /* make sure requests are padded to multiples of tile size */
    if (box.x2 % PP_FB_TILE_WIDTH) box.x2 = (box.x2/PP_FB_TILE_WIDTH + 1) * PP_FB_TILE_WIDTH;
    if (box.y2 % PP_FB_TILE_HEIGHT) box.y2 = (box.y2/PP_FB_TILE_HEIGHT + 1) * PP_FB_TILE_HEIGHT;
    
    D(D_BLABLA, "FBU REQ (%4d,%4d)-(%4d,%4d), incremental=%d\n",
      box.x1, box.y1, box.x2, box.y2, incremental);
    
    SAFE_REGION_INIT(&tmp_reg, &box, 0);
    
    MUTEX_LOCK(&clp->upd_mtx);

    {
	RegionRec reg;

	box.x1 = 0;
	box.y1 = 0;
	box.x2 = clp->fb_width_pd;
	box.y2 = clp->fb_height_pd;

	REGION_INIT(&reg, &box, 0);
	REGION_INTERSECT(&tmp_reg, &tmp_reg, &reg);
	REGION_UNINIT(&reg);
    }
    
    REGION_UNION(&clp->req_reg, &clp->req_reg, &tmp_reg);
    MUTEX_UNLOCK(&clp->upd_mtx);
    
    REGION_UNINIT(&tmp_reg);
    
    /* ------------------------------------------------------------
     * now we signal the writer thread that an update is requested
     */
    MUTEX_LOCK(&clp->s2c_queue_mtx);
    if (!incremental) {
        /* a full framebuffer update must be sent everytime! */
        clp->full_fb_update = 1;
        clp->need_fb_update = 1;
    }
    
    clp->fb_update_requested = 1;
    if (pthread_cond_signal(&clp->s2c_queue_cond) != 0) {
        D(D_ERROR, "%s(): pthread_cond_signal() failed\n", ___F);
    }
    MUTEX_UNLOCK(&clp->s2c_queue_mtx);
    
    return 0;
}

static int rfb_handle_keyboard_event(rfb_cl_t * clp) {
    input_event_t event;
    
    if (rfb_read_keyboard_event_pdu(clp, &event)) {
    	return -1;
    }
    
    event.type = rfbKeyEvent;
    event.clp = clp;
    event.session = clp->session;
    
    return rfb_enqueue_input_event(&event);
}

static int rfb_handle_pointer_event(rfb_cl_t * clp, u_int8_t type) {
    input_event_t event;
    
    if (rfb_read_pointer_event_pdu(clp, &event)) {
    	return -1;
    }
    
    event.type = type;
    event.clp = clp;
    event.session = clp->session;
    
    return rfb_enqueue_input_event(&event);
}

static int rfb_handle_utf8string(rfb_cl_t * clp) {
    int len;
    char *msg = NULL;
    
    if (rfb_read_utf8string_pdu(clp, &msg, &len) == -1) {
    	if (msg) free(msg);
    	return -1;
    }
    rfb_send_utf8_string(clp->video_link, msg, len);
    free(msg);
    return 0;
}

static int rfb_handle_user_prop_change_event(rfb_cl_t * clp) {
    char *keyName = NULL, *valueName = NULL;
    const char* user = NULL;
    int uid = -1;
    int ret = PP_ERR;
    
    if (rfb_read_user_prop_change_pdu(clp, &keyName, &valueName)) goto bail;
    
    D(D_VERBOSE, "Prop change: %s is %s\n", keyName, valueName);
    
    if (!clp->session) {
	goto bail_suc; /* not critical - return SUCCESS */
    }
    
    user = eric_session_get_user(clp->session);
    if (user) uid = pp_um_user_get_uid(user);
    if (!user || uid < 0) {
	goto bail_suc; /* not critical - return SUCCESS */
    }

    // handle special options first, all others will be saved into config
    if (!strcmp(keyName, "exclusive")) {
        int exclusive_on = 0;
        int exclusive_change = 0;
#ifdef PP_FEAT_RC_ALLOW_CHANGING_EXCLUSIVE_PERM
        int exclusive_change_explicit = 0;
#endif
        
        pp_cfg_is_enabled(&exclusive_on, "user[%u].rc.excl_access_enabled", uid);
#ifdef PP_FEAT_RC_ALLOW_CHANGING_EXCLUSIVE_PERM
        pp_cfg_is_enabled(&exclusive_change_explicit, "user[%u].rc.excl_access_allowed", uid);
        exclusive_change = PP_SUC == 
            pp_um_user_has_permission(user, "rc_s_excl", 
                                      pp_acl_raasip_yes_str) ||
            exclusive_change_explicit;
#else
        exclusive_change = PP_SUC ==
            pp_um_user_has_permission(user, "rc_s_excl", pp_acl_raasip_yes_str);
#endif
	/* sanity check */
	if ( ( (strcmp(valueName, "on")==0) && (exclusive_on || exclusive_change) ) ||
             ( (strcmp(valueName, "on")!= 0) && ((!exclusive_on) || exclusive_change) ) ) {
	    
	    u_char unit;
	    u_short port;
	    if (PP_FAILED(pp_kvm_get_unit_and_port(clp->link_client, &unit, &port))) {
		pp_log("cannot determine current unit and port, don't handle propchane_event and bailing out\n");
		goto bail;
	    }

	    MUTEX_LOCK(&exclusive_mode_mtx);
	    if (!strcmp(valueName, "on") )
            {
		// switch exclusive on
		if (exclusive_session == 0) {
		    exclusive_session = clp->session;
		    exclusive_clp = clp;
		    eric_session_for_all(disconnect_if_non_exclusive);
#if defined(PP_FEAT_SESSION_REDIRECTOR)
		    rfb_sas_send_kvm_exclusive(clp, 1);
#endif /* PP_FEAT_SESSION_REDIRECTOR */
		}
		// send_command sends to all connections, for now it's ok
		// but later it would be better to have another sharemode icon
		// for "no connection"
		pp_rfb_send_command(PP_RFB_CLIENT_TYPE_TARGET, port, "exclusive_mode", "active");
	    } else {
		// switch exclusive off
		if (exclusive_session == clp->session) {
		    exclusive_session = 0;
		    exclusive_clp = NULL;
#if defined(PP_FEAT_SESSION_REDIRECTOR)
		    rfb_sas_send_kvm_exclusive(clp, 0);
#endif /* PP_FEAT_SESSION_REDIRECTOR */
		    // send exclusive_removed to the applet
		    pp_rfb_send_command(PP_RFB_CLIENT_TYPE_TARGET, port, "exclusive_mode", "inactive");
		}
	    }
	    MUTEX_UNLOCK(&exclusive_mode_mtx);
	}
    } else {
	/* ignore errors (not critical) */
	if (PP_SUCCED(pp_cfg_set(valueName, "user[%u].%s", uid, keyName))) {
	    pp_cfg_save(DONT_FLUSH);
	}
    }

bail_suc:
    ret = PP_SUC;

bail:	
    free(keyName);
    free(valueName);
    return ret;
}

#define LEVEL D_VERBOSE

static int rfb_handle_global_prop_change_event(rfb_cl_t * clp) {
    char *keyName = NULL, *valueName = NULL;
    const char* user = NULL;
    int ret = PP_ERR;
    const char *perm = NULL;
    
    /* read the PDU */
    if (rfb_read_global_prop_change_pdu(clp, &keyName, &valueName)) goto bail;
    
    D(LEVEL, "Global prop change: %s is %s\n", keyName, valueName);
    
    /* check for the user */
    if (!clp->session) {
        D(D_ERROR, "Could not change property - no session!\n");
        rfb_enqueue_message(clp, _("Could not change property - no session!"));
	goto bail_suc; /* not critical - return SUCCESS */
    }

    user = eric_session_get_user(clp->session);
    if (!user) {
        D(D_ERROR, "Could not change property - no user!\n");
        rfb_enqueue_message(clp, _("Could not change property - no user!"));
	goto bail_suc; /* not critical - return SUCCESS */
    }
    D(LEVEL, "User for prop change: %s\n", user);

    /* check for the permission */
    if (PP_FAILED(pp_cfg_get_perm(&perm, keyName)) || perm == NULL) {
        D(D_ERROR, "Could not change property - could not determine permissions for key!\n");
        rfb_enqueue_message(clp, _("Could not change property - internal error"));
	goto bail_suc; /* not critical - return SUCCESS */
    }
    D(LEVEL, "Permission for prop change: %s\n", perm);
    
    if (pp_um_user_has_permission(user, perm, pp_acl_raasip_yes_str) != PP_SUC) {
        D(D_ERROR, "Could not change property - no permission!\n");
        rfb_enqueue_message(clp, _("Could not change property - no permission"));
	goto bail_suc; /* not critical - return SUCCESS */
    }
    
    /* set the property */
    if (pp_cfg_set(valueName, keyName) != PP_ERR) {
	pp_cfg_save(DONT_FLUSH);
    } else {
        D(D_ERROR, "Could not change property - could not set config value!\n");
        rfb_enqueue_message(clp, _("Could not change property - internal error"));
	goto bail_suc; /* not critical - return SUCCESS */
    }

bail_suc:
    ret = PP_SUC;

bail:	
    free(keyName);
    free(valueName);
    return ret;
}

/**
 * request mouse synchronization
 */
static int rfb_process_mousesync_event(rfb_cl_t * clp) {
    int type;
    
    if ((type = rfb_read_mousesync_pdu(clp)) < 0) {
    	return type;
    }
    
#if defined(PP_FEAT_SESSION_REDIRECTOR)
    rfb_sas_send_mousesync_event(type, clp->session, clp);
#endif /* PP_FEAT_SESSION_REDIRECTOR */
    return pp_km_sync_mouse(clp->data_link, type);
}

/**
 * local applet cursor changes, save for this user
 */
static int rfb_process_cursor_change_event(rfb_cl_t * clp) {
    char *name;
    const char* user = NULL;
    int uid = -1;
    int ret = PP_SUC;
    
    if ((name = rfb_read_cursor_change_pdu(clp)) == NULL) {
    	return -1;
    }
    
    D(D_VERBOSE, "New cursor: %s\n", name);
    
    if (!clp->session) {
	free(name);
	return 0;
    }

    user = eric_session_get_user(clp->session);
    if (user) uid = pp_um_user_get_uid(user);

    if (!user || uid < 0) {
	free(name);
	return -1;
    }

    if (PP_ERR != pp_cfg_set(name, "user[%u].rc.local_cursor", uid)) {
	pp_cfg_save(DONT_FLUSH);
    } //else ret = PP_ERR;

    free(name);
    return ret; 
}

static int rfb_process_ping_request(rfb_cl_t * clp) {
    u_int32_t serial;
    
    if (rfb_read_ping_pdu(clp, &serial)) {
    	return -1;
    }
    
    return rfb_enqueue_ping_reply(clp, serial);
}

static int rfb_process_ping_reply(rfb_cl_t * clp) {
    u_int32_t serial;
    return rfb_read_ping_pdu(clp, &serial);
}

static int rfb_process_bandwidth_tick(rfb_cl_t * clp) {
    int stage;
    unsigned long mytime;
    
    if ((stage = rfb_read_bandwidth_tick_pdu(clp)) < 0) {
    	return stage;
    }

    if (stage == 1) {
	clp->pred_bwidth_stage1 = pp_hrtime_ns();
    } else if (stage == 2) {
	clp->pred_bwidth_stage2 = pp_hrtime_ns();
	
	mytime = (clp->pred_bwidth_stage2 - clp->pred_bwidth_stage1)/1000;

	clp->pred_bwidth = (u_int32_t) (1000000.0 / (float)mytime * clp->pred_bwidth_bytes);
	D(D_NOTICE, "Bytes = %u in %lu, ->BandWidth(b/s)=%u\n", clp->pred_bwidth_bytes, mytime,
	   	clp->pred_bwidth);

	clp->pred_bwidth_state = PRED_BWIDTH_STATE_DONE;
    }
    
    return 0;
}

#if !defined(PRODUCT_ERIC2) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)

static int rfb_process_kvm_switch_event(rfb_cl_t * clp) {
    int port;
    
    if ((port = rfb_read_kvm_switch_pdu(clp)) < 0) {
    	return port;
    }

    /* FIXME: at some point we may have to deal with kvm unit != 0 */
    u_char vl, dl;
    pp_kvm_switch_port(NULL, 0 /*unit*/, port, &dl, &vl);
    clp->video_link = vl;
    clp->data_link = dl;
    return 0;
}

#endif /* !defined(PRODUCT_ERIC2) && !PRODUCT_ERICXP */

/**
 * tell grab/diff lib to refresh whole screen
 */
static int rfb_process_video_refresh(rfb_cl_t * clp) {
    if (rfb_read_video_refresh_pdu(clp)) {
    	return -1;
    }
    
    pp_grab_refresh(clp->video_link);
    return 0;
}

/**
 * change one video/tft setting on lara
 */
static int rfb_process_video_settings_c2s(rfb_cl_t * clp) {
    u_int8_t event;
    u_int16_t value;
    
    if (rfb_read_video_settings_pdu(clp, &event, &value)) {
    	return -1;
    }

    switch (event) {
    	case VIDEO_SETTING_RESET_ALL:
	    pp_vsc_reset_video_settings(clp->video_link, VSC_SETTING_GLOBAL | VSC_SETTING_KVM_LOCAL | VSC_SETTING_MODE_ALL);
	    break;
    	case VIDEO_SETTING_RESET_CURRENT:
	    pp_vsc_reset_video_settings(clp->video_link, VSC_SETTING_MODE_LOCAL);
	    break;
    	case VIDEO_SETTING_SAVE_SETTINGS:
	    pp_vsc_save_video_settings(clp->video_link, VSC_SETTING_GLOBAL | VSC_SETTING_KVM_LOCAL | VSC_SETTING_MODE_LOCAL);
	    break;
    	case VIDEO_SETTING_CANCEL_SETTINGS:
	    pp_vsc_cancel_video_settings(clp->video_link, VSC_SETTING_GLOBAL | VSC_SETTING_KVM_LOCAL | VSC_SETTING_MODE_LOCAL);
	    break;
    	case VIDEO_SETTING_AUTO_ADJUST:
	    pp_vsc_trigger_autoadjust(clp->video_link);
	    break;
    	default:
	    pp_vsc_set_video_setting(clp->video_link, event, value);
    }
    return 0;
}

/**
 * set flag if client is interested in video setting updates
 *     or request a settings update asap
 */
static int rfb_process_video_settings_request(rfb_cl_t * clp) {
    int request;
    
    if ((request = rfb_read_video_settings_request_pdu(clp)) < 0) {
    	return request;
    }

    if (request == rfbVSUpdateNow) {
        MUTEX_LOCK(&clp->s2c_queue_mtx);
	clp->vs_update_now = clp->vs_update_allowed = 1;
        if (pthread_cond_signal(&clp->s2c_queue_cond) != 0) {
            D(D_ERROR, "%s(): pthread_cond_signal() failed\n", ___F);
        }
        MUTEX_UNLOCK(&clp->s2c_queue_mtx);
    } else if (request == rfbVSUpdateDeny) {
	clp->vs_update_allowed = 0;	
    }

    return 0;
}

#if !defined(PP_FEAT_VSC_PANEL_INPUT)

/**
 * change one video quality setting on lara
 */
static int rfb_process_video_quality_c2s(rfb_cl_t * clp) {
    u_int8_t event;
    u_int8_t value;
    int do_save = 1;
    
    if (rfb_read_video_quality_pdu(clp, &event, &value)) {
    	return -1;
    }

    switch (event) {
	case rfbVQNoiseFilter:
	    pp_cfg_set_int(value, "video.diff_tolerance");
	    break;
	default:
	    do_save = 0;
    }
    
    if (do_save) pp_cfg_save(DO_FLUSH);

    return 0;
}

/**
 * set flag if client is interested in video setting updates
 *     or request a settings update asap
 */
static int rfb_process_video_quality_request(rfb_cl_t * clp) {
    int request;
    
    if ((request = rfb_read_video_quality_request_pdu(clp)) < 0) {
    	return request;
    }

    if (request == rfbVQUpdateNow) {
	clp->vq_update_now = 1;
    }

    return 0;
}

#endif /* !PP_FEAT_VSC_PANEL_INPUT */

// the client wants to enter/leave RDP/sec. RFB mode
static int rfb_process_rc_mode_request(rfb_cl_t * clp) {
    int request;
    
    if ((request = rfb_read_rc_mode_request_pdu(clp)) < 0) {
    	return request;
    }
    
    D(D_NOTICE, "rfb_process_rc_mode_request not supported\n");
    return 0;
}

/* ------ the big protocol switch ------ */
#define HANDLE_PDU(call, string)					\
    if ((ret = call)) {							\
    	D(D_ERROR, "Error processing \"%s\" message\n", string);	\
    }									\
    return ret;

static int rfb_handle_pdu(rfb_cl_t * clp, u_int8_t pdu_type) {
    int ret;
    D(D_BLABLA, "PDU type: %d\n", pdu_type);
    
    /* touch the session */
    if (clp->session) {
	time_t now = time(NULL);
	if (now != clp->last_requested_session_touch) {
	    if (!kvm_timeout_enabled ||
	        (pdu_type != rfbFramebufferUpdateRequest &&
	         pdu_type != rfbPingReply &&
	         pdu_type != rfbPingRequestEvent)) {
		
		eric_session_touch(clp->session);
	    }
	    clp->last_requested_session_touch = now;
	}
    }

    switch (pdu_type) {
    	case rfbSetPixelFormat:
    	    HANDLE_PDU(rfb_handle_set_pixel_format(clp), "set pixel format");
    	case rfbSetEncodings:
    	    HANDLE_PDU(rfb_handle_set_encodings(clp), "set encodings");
    	case rfbFramebufferUpdateRequest:
    	    HANDLE_PDU(rfb_handle_fb_update_req(clp), "framebuffer update request");
        case rfbKeyEvent:
    	    HANDLE_PDU(rfb_handle_keyboard_event(clp), "keyboard event");        
        case rfbPointerEvent:
        case rfbPointerRelativeEvent:
    	    HANDLE_PDU(rfb_handle_pointer_event(clp, pdu_type), "pointer event");        
        case rfbUtf8String:
    	    HANDLE_PDU(rfb_handle_utf8string(clp), "utf-8 string");        
        case rfbUserPropChange:
    	    HANDLE_PDU(rfb_handle_user_prop_change_event(clp), "user prop change event");
        case rfbGlobalPropChange:
    	    HANDLE_PDU(rfb_handle_global_prop_change_event(clp), "global prop change event");
    	case rfbMouseSyncEvent:
    	    HANDLE_PDU(rfb_process_mousesync_event(clp), "mouse sync event");
    	case rfbCursorChangeEvent:
    	    HANDLE_PDU(rfb_process_cursor_change_event(clp), "cursor change event");
    	case rfbPingRequestEvent:
    	    HANDLE_PDU(rfb_process_ping_request(clp), "ping request");
    	case rfbPingReply:
    	    HANDLE_PDU(rfb_process_ping_reply(clp), "ping reply");
    	case rfbBandwidthReply:
    	    HANDLE_PDU(rfb_process_bandwidth_tick(clp), "bandwidth tick");
#if !defined(PRODUCT_ERIC2) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)
    	case rfbKVMSwitchEvent:
    	    HANDLE_PDU(rfb_process_kvm_switch_event(clp), "KVM switch event");
#endif /* !defined(PRODUCT_ERIC2) && !PRODUCT_ERICXP && !PRODUCT_ERICG4 */
    	case rfbVideoRefreshRequest:
    	    HANDLE_PDU(rfb_process_video_refresh(clp), "video refresh");
    	case rfbVideoSettingsC2SEvent:
    	    HANDLE_PDU(rfb_process_video_settings_c2s(clp), "video settings (c2s)");
    	case rfbVideoSettingsReqEvent:
    	    HANDLE_PDU(rfb_process_video_settings_request(clp), "video settings request");
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
        case rfbVideoQualityC2SEvent:
    	    HANDLE_PDU(rfb_process_video_quality_c2s(clp), "video quality (c2s)");
    	case rfbVideoQualityReqEvent:
    	    HANDLE_PDU(rfb_process_video_quality_request(clp), "video quality request");
#endif /* !PP_FEAT_VSC_PANEL_INPUT */
    	case rfbRCModeRequest:
    	    HANDLE_PDU(rfb_process_rc_mode_request(clp), "video quality request");
	  
    	default:
    	    D(D_ERROR, "rfb_handle_pdu(): unknown PDU type (0x%x). "
		       "Closing connection.\n", pdu_type);
	    return -1;
    }
}

static int rfb_read_pdu(rfb_cl_t * clp) {
    int pdu_type;

    if ((pdu_type = rfb_read_message_type(clp)) == -1) {
	return -1;
    }

    /* disconnect if requested (e.g. for firmware upgrade or exclusive mode) */
    if ((clp->sd && clp->sd->disconnect) || pp_rfb_disconnect_all) {
	return -2;
    }

    return rfb_handle_pdu(clp, pdu_type);
}

/******************************************************************************
* local helper functions                                                      *
******************************************************************************/

static int rfb_authenticate_session_id(rfb_cl_t *clp, eric_session_int_id_t *session) {
    unsigned char myrand[ERIC_ID_LENGTH / 2];
    FILE *f_rand;
    unsigned char challenge[ERIC_ID_LENGTH + 1];
    unsigned char resp[ERIC_RESPONSE_LENGTH + 1];

    /* generate a challenge for the client */
    memset(myrand, 0, sizeof(myrand));
    if ((f_rand = fopen("/dev/urandom", "r")) != NULL) {
        fread(myrand, sizeof(myrand), 1, f_rand);
        fclose(f_rand);
    }

    /* send challenge to client */
    pp_bin_to_hex_string(challenge, myrand, sizeof(myrand));
    if (rfb_send_auth_challenge(clp, challenge, sizeof(challenge)) == -1) {
        return -1;
    }

    /* get and check client's response */
    if ((rfb_read_type(clp) != rfbChallengeResponse) ||
    	(rfb_read_auth_response_pdu(clp, resp, ERIC_RESPONSE_LENGTH + 1, ERIC_RESPONSE_LENGTH + 1) < 0)) {
    	return -1;
    }

    /* ---- check applet ID and send a NOPERM Quit MSG back if wrong ---- */
    if ((*session = eric_session_get_by_challenge_response(challenge, resp)) == 0) {
        rfb_send_quit(clp, rfbQuitNoPerm);
        eric_notify_post_event("Connection to Remote Console failed: no permission.", "console", PP_NOTIFY_EVENT_GENERIC);
        D(D_ERROR, "Checking applet id failed\n");
        return -1;
    }

    return 0;
}

static int rfb_authenticate_plain(rfb_cl_t *clp, char *user, eric_session_int_id_t *session) {
    char pwd[32 + 1];
    int len, err, server_sid = 0, gid = -1;

    /* send a dummy challenge for the client */
    if (rfb_send_auth_challenge(clp, NULL, 0) == -1) {
        return -1;
    }

    /* get and check client's response: plain pwd */
    if ((rfb_read_type(clp) != rfbChallengeResponse) ||
    	(len = rfb_read_auth_response_pdu(clp, pwd, 32, 0)) < 0) {
    	return -1;
    }
    pwd[len] = '\0';

#ifndef PP_FEAT_ESB2_TPT
    // use ERIC user managament
    if (PP_FAILED(pp_um_user_authenticate_with_ip_str(user, pwd,
                                                      PP_UM_AUTH_NO_FLAGS,
                                                      clp->client_ip,
                                                      &server_sid,
                                                      &gid))) {
        rfb_send_quit(clp, rfbQuitNoPerm);
        D(D_ERROR, errno == PP_UM_ERR_AUTH_FAILED ?
                   "Wrong username/password\n" :
                   "Error checking username/password\n");
        return -1;
    }
#else // PP_FEAT_ESB2_TPT
    // use BMC user management via IPMB
    pp_ipmi_parameter_t p = { .is_cmdline = 0, };
    pp_ipmi_return_t r;
    int e;

    e = 0;
    p.data.usr_list.chan = 0x0e; // current chan
    if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_USER, PP_IPMI_USER_SUBCMD_LIST, &p, &r, &e, NULL))) {
        rfb_send_quit(clp, rfbQuitInternalError);
        D(D_ERROR, "Failed to get user list from BMC (err=%d)\n", e);
        return -1;
    }

    int i;
    for (i = 1; i < 64; i++) if (strcmp(user, r.data.usr_list.name[i]) == 0) break;
    if (i >= 64) {
        rfb_send_quit(clp, rfbQuitNoPerm);
        D(D_ERROR, "Unknown user '%s'\n", user);
        return -1;
    }

    e = 0;
    p.data.usr_test_pwd.uid = i;
    p.data.usr_test_pwd.size20 = strlen(pwd) > 16;
    strncpy(p.data.usr_test_pwd.pwd, pwd, 20);
    if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_USER, PP_IPMI_USER_SUBCMD_TEST_PASSWORD, &p, &r, &e, NULL))) {
        rfb_send_quit(clp, rfbQuitNoPerm);
        D(D_ERROR, "Failed to validate password by BMC (err=%d)\n", e);
        return -1;
    }

    e = 0;
    p.data.oem_intel_get_usr_cfg.chan = 0x0e; // current chan
    p.data.oem_intel_get_usr_cfg.uid = i;
    p.data.oem_intel_get_usr_cfg.mask = PP_IPMI_OEM_INTEL_USR_CFG_KVM_ENABLE;
    if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_OEM_INTEL_USR_CFG, PP_IPMI_OEM_INTEL_GET_USR_CFG, &p, &r, &e, NULL))
        || !(r.data.oem_intel_get_usr_cfg.mask | PP_IPMI_OEM_INTEL_USR_CFG_KVM_ENABLE)) {
        rfb_send_quit(clp, rfbQuitInternalError);
        D(D_ERROR, "Failed to get KVM access right from BMC (err=%d)\n", e);
        return -1;
    }

    if (!r.data.oem_intel_get_usr_cfg.cfg.kvm_enable) {
        rfb_send_quit(clp, rfbQuitNoPerm);
        D(D_ERROR, "KVM not enabled for user '%s'\n", user);
        return -1;
    }

#endif

    if ((*session = eric_session_create(user, clp->client_ip, "", server_sid, &err)) == 0) {
        rfb_send_quit(clp, rfbQuitNoPerm);
        D(D_ERROR, "Could not create session.\n");
        return -1;
    }
    eric_session_set_gid(*session, gid);    
    
    return 0;
}

static int rfb_authenticate_md5(rfb_cl_t *clp, char *user, eric_session_int_id_t *session) {
    char md5_hash_str[MD5_DIGEST_LENGTH * 2 + 1];
    int len, err, server_sid = 0, gid;

    /* generate a dummy challenge for the client */
    if (rfb_send_auth_challenge(clp, NULL, 0) == -1) {
        return -1;
    }

    /* get and check client's response: plain pwd */
    if ((rfb_read_type(clp) != rfbChallengeResponse) ||
    	(len = rfb_read_auth_response_pdu(clp, md5_hash_str, MD5_DIGEST_LENGTH * 2, 0)) < 0) {
    	return -1;
    }
    md5_hash_str[len] = '\0';

    // use ERIC user managament
    if (PP_FAILED(pp_um_user_authenticate_with_ip_str(user, md5_hash_str,
                                                 PP_UM_AUTH_PASSWD_AS_MD5_HASH,
                                                 clp->client_ip, &server_sid,
                                                 &gid))){
        rfb_send_quit(clp, rfbQuitNoPerm);
        D(D_ERROR, errno == PP_UM_ERR_AUTH_FAILED ?
                   "Wrong username/password\n" :
                   "Error checking username/password\n");
        return -1;
    }

    if ((*session = eric_session_create(user, clp->client_ip, "", server_sid, &err)) == 0) {
        rfb_send_quit(clp, rfbQuitNoPerm);
        D(D_ERROR, "Could not create session.\n");
        return -1;
    }
    eric_session_set_gid(*session, gid);    

    return 0;
}

#ifdef PP_FEAT_RDM_SUPPORT
/** returns PP_SUC if the rdm_session_str could be splitted in a valid *
 *  rdm_session_id and rdm_session_key. rdm_session_str is modified    */
static int authenticate_rdm_session_parse(char* rdm_session_str, 
                                          char** rdm_session_id,
                                          char** rdm_session_key)
{
    // another crude parser
    int i;

    // split
    *rdm_session_id = rdm_session_str;
    *rdm_session_key = strstr(rdm_session_str, "\":\"");
    if (*rdm_session_key == NULL) return PP_ERR;
    (*rdm_session_key)[0] = 0;
    *rdm_session_key = *rdm_session_key + 3;  // 1st character of key

    // remove 1st "
    if ((*rdm_session_id)[0] != '"') return PP_ERR;
    (*rdm_session_id) ++;

    // remove last "
    i = strlen(*rdm_session_key) - 1;
    if ((*rdm_session_key)[i] != '"') return PP_ERR;
    (*rdm_session_key)[i] = 0;

    return PP_SUC;
}

static int rfb_authenticate_rdm_session(rfb_cl_t *clp, eric_session_int_id_t *session) {
    char rdm_session_str[MAX_RDM_RESPONSE_LENGTH];
    char* rdm_session_id;
    char* rdm_session_key;
    int err, len;
    
    /* generate a dummy challenge for the client */
    if (rfb_send_auth_challenge(clp, NULL, 0) == -1) {
        return -1;
    }
    /* get and check client's response:  */
    if ((rfb_read_type(clp) != rfbChallengeResponse) ||
        (len = rfb_read_auth_response_pdu(clp, rdm_session_str, MAX_RDM_RESPONSE_LENGTH, 0)) < 0)
    {
        return -1;
    }
    
    rdm_session_str[MAX_RDM_RESPONSE_LENGTH-1] = '\0';
    if (authenticate_rdm_session_parse(rdm_session_str, &rdm_session_id, &rdm_session_key) != PP_SUC) {
        D(D_NOTICE, "RDM session str not valid\n");
        return -1;
    }

    if (pp_rdm_authenticate_session(rdm_session_id, rdm_session_key) != PP_SUC) {
        D(D_ERROR, "RDM authentication failed\n");
        return -1;
    }
    
    if ((*session = eric_session_create("admin", clp->client_ip, "CC-user (admin)", 0, &err)) == 0) {
        rfb_send_quit(clp, rfbQuitNoPerm);
        D(D_ERROR, "Could not create session\n");
        return -1;
    }

    return 0;
}
#endif

static int rfb_authenticate_noauth(rfb_cl_t *clp, eric_session_int_id_t *session) {
    int len, err;

    /* generate a dummy challenge for the client */
    if (rfb_send_auth_challenge(clp, NULL, 0) == -1) {
        return -1;
    }

    /* get and check client's response: plain pwd */
    if ((rfb_read_type(clp) != rfbChallengeResponse) ||
    	(len = rfb_read_auth_response_pdu(clp, NULL, 0, 0)) < 0) {
    	return -1;
    }

    if ((*session = eric_session_create("", clp->client_ip, "", 0, &err)) == 0) {
        rfb_send_quit(clp, rfbQuitNoPerm);
        D(D_ERROR, "Could not create session.\n");
        return -1;
    }
    
    return 0;
}

static void disconnect_if_non_exclusive(eric_session_int_id_t s, va_list args UNUSED) {
    pp_rfb_session_data_t **sd = NULL;
    int i;

    if (s != exclusive_session) {
	sd = rfb_get_session_data_from_session(s);
	for (i=0; sd && sd[i]; i++) {
	    if (sd[i]->connection_flags & rfbConnectionFlagSasHW) {
	    	free(sd);
	    	return;
	    }
	    sd[i]->exit_reason = rfbQuitExclAccess;
	}
	free(sd);
	pp_rfb_do_disconnect(s);
    }
}
