/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include <sys/time.h>
#include <pp/base.h>
#include <pp/rfb.h>
#include <liberic_net.h>
#include "rfb.h"
#include "debug.h"

static void rfb_get_current_timestamp(rfbTimestamp *ts);
static void rfb_get_timestamp_timeval(rfbTimestamp *ts, struct timeval* tv);
static void rfb_parse_timestamp_to_timeval(rfbTimestamp *ts, struct timeval* tv);

#if defined(PP_FEAT_SESSION_REDIRECTOR)

static char * rfb_create_sas_event(u_int8_t event, size_t additional_len,
				   eric_session_int_id_t session, size_t *size);
static void rfb_get_timestamp_time_t(rfbTimestamp *ts, time_t timestamp);

#endif /* PP_FEAT_SESSION_REDIRECTOR */

/******************************************************************************
* Reading RFB messages                                                        *
******************************************************************************/

/*-----------------------------------------------------------------------------
- Client -> Server                                                            -
-----------------------------------------------------------------------------*/

#define READ_PDU_BODY(clp, pdu, pdu_len) (rfb_read(clp, (char*)&pdu+1, (pdu_len)-1))

int rfb_read_connection_init_string(rfb_cl_t * clp) {
    char buf[11];
    
    return rfb_read(clp, buf, 11);
}

int rfb_read_protocol_version(rfb_cl_t * clp, int *major, int *minor) {
    char pv[rfbProtocolVersionMsgSize + 1];

    if (rfb_read(clp, pv, rfbProtocolVersionMsgSize) == -1) {
	return -1;
    }

    pv[rfbProtocolVersionMsgSize] = 0;
    if (sscanf(pv, rfbProtocolVersionFormat, major, minor) != 2) {
	D(D_ERROR, "%s(): invalid client\n", ___F);
	return -1;
    }
    
    return 0;
}

int rfb_read_message_type(rfb_cl_t * clp) {
    u_int8_t pdu_type;

    if (rfb_read_no_timeout(clp, &pdu_type, 1) == -1) {
	return -1;
    }
    
    return pdu_type;
}

int rfb_read_login_pdu(rfb_cl_t * clp, char *user, u_int32_t *flags) {
    rfbLoginMsgV1_16 lp;
    if (READ_PDU_BODY(clp, lp, sizeof(lp)) == -1
        || lp.name_len_8 > 32
        || rfb_read(clp, user, lp.name_len_8) == -1) {
        return -1;
    }
    user[lp.name_len_8] = '\0';
    *flags = be32_to_cpu(lp.connection_flags_be32);
    
    return lp.auth_method_8;
}

int rfb_read_auth_response_pdu(rfb_cl_t * clp, char *response, size_t len, size_t min_len) {
    rfbChalRespMsg crp;
    
    if (READ_PDU_BODY(clp, crp, sizeof(crp)) == -1
        || crp.data_len_8 > len
        || crp.data_len_8 < min_len) {
        return -1;
    }
    
    if (crp.data_len_8 && rfb_read(clp, response, crp.data_len_8)) {
    	return -1;
    }
    
    return crp.data_len_8;
}

int rfb_read_init_pdu(rfb_cl_t * clp, u_char *channel, u_char *unit, u_int16_t *port, u_int16_t *flags) {
    rfbClientInitMsgV1_17 ci;
    if( rfb_read(clp, &ci, sizeof(ci))== -1) {
        return -1;
    }
    *channel = ci.kvm_node_8;
    *unit = ci.unit_8;
    *flags = be16_to_cpu(ci.flags_be16);
    *port = be16_to_cpu(ci.port_be16);
    return 0;
}

int rfb_read_set_pixel_format_pdu(rfb_cl_t * clp, rfbPixelFormat *fmt) {
    rfbSetPixelFormatMsg pdu;
    if (READ_PDU_BODY(clp, pdu, sizeof(pdu))) {
    	return -1;
    }
    
    *fmt = pdu.format;
    fmt->redMax_be16 = be16_to_cpu(fmt->redMax_be16);
    fmt->greenMax_be16 = be16_to_cpu(fmt->greenMax_be16);
    fmt->blueMax_be16 = be16_to_cpu(fmt->blueMax_be16);
    
    return 0;
}

int rfb_read_set_encodings_pdu(rfb_cl_t * clp, u_int32_t **encodings) {
    rfbSetEncodingsMsg pdu;
    int i;
    
    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    pdu.nEncodings_be16 = be16_to_cpu(pdu.nEncodings_be16);
    if ((*encodings = malloc(pdu.nEncodings_be16 * sizeof(u_int32_t))) == NULL) {
    	return -1;
    }
    
    for (i = 0; i < pdu.nEncodings_be16; i++) {
    	u_int32_t enc_word;
    	if (rfb_read(clp, &enc_word, sizeof(enc_word)) == -1) {
    	    free(*encodings);
    	    *encodings = NULL;
    	    return -1;
    	}
    	(*encodings)[i] = be32_to_cpu(enc_word);
    }
    
    return pdu.nEncodings_be16;
}

int rfb_read_fb_update_req_pdu(rfb_cl_t * clp, int *incremental, BoxRec *box) {
    rfbFramebufferUpdateRequestMsg pdu;

    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    *incremental = pdu.incremental_8;
    box->x1 = be16_to_cpu(pdu.x_be16);
    box->y1 = be16_to_cpu(pdu.y_be16);	  
    box->x2 = box->x1 + be16_to_cpu(pdu.w_be16);
    box->y2 = box->y1 + be16_to_cpu(pdu.h_be16);
    
    return 0;
}

int rfb_read_keyboard_event_pdu(rfb_cl_t * clp, input_event_t *event) {
    rfbKeyEventMsg pdu;
    
    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    event->data.keycode = pdu.key_8;
    
    return 0;
}

int rfb_read_pointer_event_pdu(rfb_cl_t * clp, input_event_t *event) {
    rfbPointerEventMsgV1_3 pdu;
    
    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    event->data.mouse_event.x = be16_to_cpu(pdu.x_be16);
    event->data.mouse_event.y = be16_to_cpu(pdu.y_be16);
    event->data.mouse_event.z = be16_to_cpu(pdu.z_be16);
    event->data.mouse_event.button_mask = pdu.buttonMask_8;
    
    return 0;
}

int rfb_read_utf8string_pdu(rfb_cl_t * clp, char **string, int *len) {
    rfbUtf8StringMsg pdu;
    
    *string = NULL;
    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    *len = be16_to_cpu(pdu.len_be16);
    if (*len == 0) {
        return 0;   // everything okay
    }
    
    if ((*string = malloc(*len)) == NULL) {
        return -1;
    }
    if (rfb_read(clp, *string, *len) == -1) {
	free(*string);
	*string = NULL;
	return -1;
    }
    return 0;
}

#define RFB_READ_PROP_CHANGE_PDU(pdutype)                           \
    pdutype pdu;                                                    \
                                                                    \
    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {               \
    	return -1;                                                  \
    }                                                               \
                                                                    \
    if ((*key = malloc(pdu.keyLength_8 + 1)) == NULL) return -1;    \
    if (rfb_read(clp, *key, pdu.keyLength_8) == -1) {               \
	free(*key);                                                 \
	return -1;                                                  \
    }                                                               \
    (*key)[pdu.keyLength_8]='\0';                                   \
                                                                    \
    if ((*value = malloc(pdu.valueLength_8 + 1)) == NULL) {         \
    	free(*key);                                                 \
    	return -1;                                                  \
    }                                                               \
                                                                    \
    if (rfb_read(clp, *value, pdu.valueLength_8) == -1) {           \
	free(*key);                                                 \
	free(*value);                                               \
	return -1;                                                  \
    }                                                               \
    (*value)[pdu.valueLength_8]='\0';                               \
                                                                    \
    return 0

int rfb_read_user_prop_change_pdu(rfb_cl_t * clp, char ** key, char ** value) {
    RFB_READ_PROP_CHANGE_PDU(rfbUserPropChangeMsg);
}

int rfb_read_global_prop_change_pdu(rfb_cl_t * clp, char ** key, char ** value) {
    RFB_READ_PROP_CHANGE_PDU(rfbGlobalPropChangeMsg);
}

int rfb_read_mousesync_pdu(rfb_cl_t * clp) {
    rfbMousesyncEventMsg pdu;

    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    return pdu.sync_type_8;
}

char* rfb_read_cursor_change_pdu(rfb_cl_t * clp) {
    rfbCursorChangeEventMsg pdu;
    char *cursor;
    
    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return NULL;
    }
    
    if ((cursor = malloc(pdu.len_8 + 1)) == NULL) { return NULL; }
    if (rfb_read(clp, cursor, pdu.len_8) == -1) {
	free(cursor);
	return NULL;
    }
    cursor[pdu.len_8]='\0';
    
    return cursor;
}

int rfb_read_ping_pdu(rfb_cl_t * clp, u_int32_t *serial) {
    rfbPingReplyMsg pdu;

    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    *serial = be32_to_cpu(pdu.serial_be32);
    return 0;
}

int rfb_read_bandwidth_tick_pdu(rfb_cl_t * clp) {
    rfbBandwidthReplyMsg pdu;

    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    return pdu.stage_8;
}

int rfb_read_kvm_switch_pdu(rfb_cl_t * clp) {
    rfbKVMSwitchEventMsg pdu;

    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    return be16_to_cpu(pdu.port_be16);
}

int rfb_read_video_refresh_pdu(rfb_cl_t * clp) {
    rfbVideoRefreshRequestMsg pdu;

    return READ_PDU_BODY(clp, pdu, sizeof(pdu));
}

int rfb_read_video_settings_pdu(rfb_cl_t * clp, u_int8_t *event, u_int16_t *value) {
    rfbVideoSettingsC2SMsg pdu;
    
    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    *event = pdu.event_8;
// FIXME: because some settings are 8 bit, we should use a struct here
    *value = be16_to_cpu(pdu.value_be16);
    return 0;
}

int rfb_read_video_settings_request_pdu(rfb_cl_t * clp) {
    rfbVideoSettingsReqMsg pdu;

    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    return pdu.request_8;
}

int rfb_read_video_quality_pdu(rfb_cl_t * clp, u_int8_t *event, u_int8_t *value) {
    rfbVideoQualityC2SMsg pdu;
    
    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    *event = pdu.event_8;
    *value = pdu.value_8;
    return 0;
}

int rfb_read_video_quality_request_pdu(rfb_cl_t * clp) {
    rfbVideoQualityReqMsg pdu;

    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    return pdu.request_8;
}

int rfb_read_rc_mode_request_pdu(rfb_cl_t * clp) {
    rfbRCModeRequestMsg pdu;

    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    return pdu.mode_8;
}

/*-----------------------------------------------------------------------------
- Server -> Client                                                            -
-----------------------------------------------------------------------------*/

int rfb_read_auth_caps(rfb_cl_t * clp, unsigned int *caps) {
    rfbAuthCapsMsg acp;
    
    if (rfb_read(clp, &acp, sizeof(acp)) == -1) {
        return -1;
    }
    
    *caps = acp.auth_caps_8;
    
    return 0;
}

int rfb_read_auth_challenge(rfb_cl_t * clp, char *challenge, size_t expected_len) {
    rfbSessChalMsg msg;
    if (rfb_read(clp, &msg, sizeof(msg)) == -1) {
        return -1;
    }
    
    if (msg.data_len_8 > expected_len) {
	D(D_ERROR, "Protocol error: bad challenge size\n");
	return -1;
    }
    if (msg.data_len_8 > 0) {
	if (rfb_read(clp, challenge, msg.data_len_8) == -1) {
	    return -1;
	}
    }

    return msg.data_len_8;
}

int rfb_read_osd_state(rfb_cl_t * clp, char **msg, u_int8_t *blank, u_int16_t *timeout, struct timeval *tv) {
    rfbOSDStateMsgV1_16 pdu;

    *msg = NULL;
    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    pdu.length_be16 = be16_to_cpu(pdu.length_be16);
    *timeout = pdu.timeout_be16 = be16_to_cpu(pdu.timeout_be16);
    *blank = pdu.blank_8;
    rfb_parse_timestamp_to_timeval(&pdu.timestamp, tv);
    
    if(pdu.length_be16 > 0) {
	*msg = malloc(pdu.length_be16 + 1);
	if (*msg == NULL || rfb_read(clp, *msg, pdu.length_be16) == -1) {
	    free(*msg);
	    *msg = NULL;
	    return -1;
	}
	(*msg)[pdu.length_be16] = '\0';
    }

    return 0;
}

int rfb_read_auth_successful(rfb_cl_t * clp, u_int32_t *flags) {
    rfbAuthSuccessfulMsgV1_16 msg;
    if (READ_PDU_BODY(clp, msg, sizeof(msg)) == -1) {
        return -1;
    }
    
    *flags = be32_to_cpu(msg.connection_flags_be32);
    return 0;
}

int rfb_read_quit(rfb_cl_t * clp, u_int8_t *reason) {
    rfbQuitMsg msg;
    if (READ_PDU_BODY(clp, msg, sizeof(msg)) == -1) {
        return -1;
    }
    
    *reason = msg.reason_8;
    return 0;
}

int rfb_read_kbd_layout(rfb_cl_t * clp, char** kbd) {
    rfbKeyboardLayoutEventMsg pdu;

    *kbd = NULL;
    if (READ_PDU_BODY(clp, pdu, sizeof(pdu)) == -1) {
    	return -1;
    }
    
    pdu.length_be16 = be16_to_cpu(pdu.length_be16);
    
    if(pdu.length_be16 > 0) {
	*kbd = malloc(pdu.length_be16 + 1);
	if (*kbd == NULL || rfb_read(clp, *kbd, pdu.length_be16) == -1) {
	    free(*kbd);
	    *kbd = NULL;
	    return -1;
	}
	(*kbd)[pdu.length_be16] = '\0';
    }

    return 0;
}

rfb_connection_parameter_list_t *rfb_read_connection_parameters(rfb_cl_t * clp) {
    rfbConnectionParameterListMsg pdu;
    unsigned int i;
    rfb_connection_parameter_list_t *list = rfb_create_connection_parameter_list();

    if (rfb_read(clp, &pdu, sizeof(pdu)) == -1) {
        goto fail;
    }
    
    for (i = 0; i < pdu.parameters_8; i++) {
        char cap_name[256];
        char cap_value[256];
        
        rfbConnectionParameter pdu2;
        
        if (rfb_read(clp, &pdu2, sizeof(pdu2)) == -1) {
            goto fail;
        }
        
        if(pdu2.name_length_8 > 0 && rfb_read(clp, cap_name, pdu2.name_length_8) == -1) {
            goto fail;
        }
        if(pdu2.value_length_8 > 0 && rfb_read(clp, cap_value, pdu2.value_length_8) == -1) {
            goto fail;
        }
        
        cap_name[pdu2.name_length_8] = '\0';
        cap_value[pdu2.value_length_8] = '\0';
        
        rfb_add_connection_parameter(list, cap_name, cap_value);
    }
    
    return list;
    
 fail:
    if (list) rfb_free_connection_parameter_list(list);
    return NULL;
}

int rfb_read_fb_format_update(rfb_cl_t * clp, int *fb_width, int *fb_height, u_int8_t *unsupported, rfbPixelFormat *fmt, struct timeval *tv) {
    rfbServerFBFormatMsgV1_16 sfbf;

    if (READ_PDU_BODY(clp, sfbf, sizeof(sfbf)) == -1) {
        return -1;
    }

    *unsupported                = sfbf.unsupported_8;
    *fb_width                   = cpu_to_be16(sfbf.framebufferWidth_be16);
    *fb_height                  = cpu_to_be16(sfbf.framebufferHeight_be16);
    sfbf.format.redMax_be16     = cpu_to_be16(sfbf.format.redMax_be16);
    sfbf.format.greenMax_be16   = cpu_to_be16(sfbf.format.greenMax_be16);
    sfbf.format.blueMax_be16    = cpu_to_be16(sfbf.format.blueMax_be16);
    *fmt = sfbf.format;
    rfb_parse_timestamp_to_timeval(&sfbf.timestamp, tv);

    return 0;
}

/******************************************************************************
* Write RFB messages (send directly)                                          *
******************************************************************************/

/*-----------------------------------------------------------------------------
- Server -> Client                                                            -
-----------------------------------------------------------------------------*/

int rfb_send_protocol_version(rfb_cl_t * clp, int major, int minor) {
    char pv[rfbProtocolVersionMsgSize + 1];
    snprintf((char*)&pv, sizeof(pv), rfbProtocolVersionFormat, major, minor);
    return rfb_write(clp, &pv, rfbProtocolVersionMsgSize);
}

int rfb_send_quit(rfb_cl_t * clp, int reason) {
    rfbQuitMsg pdu;

    pdu.type_8 = rfbQuit;
    pdu.reason_8 = (u_int8_t)reason;
    
    return rfb_write(clp, &pdu, sizeof(pdu));
}

int rfb_send_auth_caps(rfb_cl_t * clp, unsigned int caps) {
    rfbAuthCapsMsg acp;
    
    acp.type_8 = rfbAuthCaps;
    acp.auth_caps_8 = caps;
    
    return rfb_write(clp, &acp, sizeof(acp));
}

int rfb_send_auth_successful(rfb_cl_t * clp, u_int32_t flags) {
    rfbAuthSuccessfulMsgV1_16 success;
    success.type_8 = rfbAuthSuccessful;
    success.connection_flags_be32 = cpu_to_be32(flags);
    return rfb_write(clp, &success, sizeof(success));
}

int rfb_send_auth_challenge(rfb_cl_t * clp, char *challenge, size_t len) {
    unsigned char chal_pkt[sizeof(rfbSessChalMsg) + len];
    rfbSessChalMsg *scp = (rfbSessChalMsg*)chal_pkt;
    
    scp->type_8 = rfbSessionChallenge;
    scp->data_len_8 = len;
    if (challenge && len) {
    	memcpy(&chal_pkt[sizeof(rfbSessChalMsg)], challenge, len);
    }
    
    return rfb_write(clp, chal_pkt, sizeof(chal_pkt));
}

#if defined(PP_FEAT_SESSION_REDIRECTOR)

int rfb_send_sas_error(rfb_cl_t * clp, int errorcode) {
    rfbSasErrorMsg pdu;
    
    pdu.type_8 = rfbSasError;
    pdu.errorcode_8 = errorcode;
    
    return rfb_write(clp, &pdu, sizeof(pdu));
}

#endif /* PP_FEAT_SESSION_REDIRECTOR */

/*-----------------------------------------------------------------------------
- Client -> Server                                                            -
-----------------------------------------------------------------------------*/

int rfb_send_connection_init_string(rfb_cl_t * clp) {
    char buf[rfbProtocolInitStringLen] = rfbProtocolInitString;
    
    return rfb_write(clp, buf, rfbProtocolInitStringLen);
}

int rfb_send_init_pdu(rfb_cl_t * clp, u_char kvm_node, u_char unit, u_int16_t port, u_int16_t flags) {
    rfbClientInitMsgV1_17 ci;
    
    ci.flags_be16 = cpu_to_be16(flags);
    ci.kvm_node_8 = kvm_node;
    ci.unit_8 = unit;
    ci.port_be16 = cpu_to_be16(port); 
    return rfb_write(clp, &ci, sizeof(ci));
}

int rfb_send_login_pdu(rfb_cl_t * clp, const char *username, u_int8_t method) {
    if (username == NULL) username = "";
    size_t len = strlen(username) + 1;

    rfbLoginMsgV1_16 msg;

    msg.type_8 = rfbLogin;
    msg.auth_method_8 = method;
    msg.name_len_8 = len;
    msg.connection_flags_be32 = 0;
    
    return rfb_write(clp, &msg, sizeof(msg)) || rfb_write(clp, username, len);
}

int rfb_send_auth_response_pdu(rfb_cl_t * clp, char *response, size_t len) {
    rfbChalRespMsg msg;
    msg.type_8 = rfbChallengeResponse;
    msg.data_len_8 = len;
    
    return (rfb_write(clp, &msg, sizeof(msg)) == -1 ||
        (len > 0 && rfb_write(clp, response, len)) == -1) ? -1 : 0;
}

/******************************************************************************
* Write RFB messages (called from writer thread)                              *
******************************************************************************/

/*-----------------------------------------------------------------------------
- Server -> Client                                                            -
-----------------------------------------------------------------------------*/

int rfb_writer_send_ack_pixel_format(rfb_cl_t * clp, rfbPixelFormat *fmt) {
    rfbAckPixelFormatMsg apf;

    apf.type_8 = rfbAckPixelFormat;
    apf.pix_fmt = *fmt;
    apf.pix_fmt.redMax_be16   = cpu_to_be16(fmt->redMax_be16);
    apf.pix_fmt.greenMax_be16 = cpu_to_be16(fmt->greenMax_be16);
    apf.pix_fmt.blueMax_be16  = cpu_to_be16(fmt->blueMax_be16);

    return rfb_write_writer(clp, &apf, sizeof(apf));
}

int rfb_writer_send_fb_format_update(rfb_cl_t * clp, u_int8_t unsupported,
				     u_int16_t width, u_int16_t height,
				     rfbPixelFormat *fmt) {
    rfbServerFBFormatMsgV1_16 sfbf;

    D(D_VERBOSE, "Sending FB format update, w: %d, h: %d\n", width, height);

    sfbf.type_8                 = rfbServerFBFormat;
    sfbf.unsupported_8          = unsupported;
    sfbf.framebufferWidth_be16  = cpu_to_be16(width);
    sfbf.framebufferHeight_be16 = cpu_to_be16(height);
    sfbf.format                 = *fmt;
    sfbf.format.redMax_be16     = cpu_to_be16(sfbf.format.redMax_be16);
    sfbf.format.greenMax_be16   = cpu_to_be16(sfbf.format.greenMax_be16);
    sfbf.format.blueMax_be16    = cpu_to_be16(sfbf.format.blueMax_be16);
    rfb_get_current_timestamp(&sfbf.timestamp);

    return rfb_write_writer(clp, &sfbf, sizeof(sfbf));
}

int rfb_writer_send_update_buf(rfb_cl_t * clp, char *buf, size_t len) {
    return rfb_write_writer(clp, buf, len);
}

/******************************************************************************
* Write RFB messages (enqueue in writer thread)                               *
******************************************************************************/

/*-----------------------------------------------------------------------------
- Server -> Client                                                            -
-----------------------------------------------------------------------------*/

void rfb_enqueue_message_v(rfb_cl_t * clp, va_list args) {
    void*  pdu;
    size_t len;

    pdu = va_arg(args, char*);
    len = va_arg(args, size_t);
    if (pdu && len > 0) {
	rfb_writer_enqueue_pdu(clp, pdu, len);
    }    
}

int rfb_enqueue_utf8string(rfb_cl_t* clp, const void* msg, u_int16_t length) {
    int ret;
    size_t pdulength = sizeof(rfbUtf8StringMsg) + length;
    char _pdu[pdulength];
    rfbUtf8StringMsg *pdu;

    pdu = (rfbUtf8StringMsg *)_pdu;

    pdu->type_8 = rfbUtf8String;
    pdu->len_be16 = cpu_to_be16(length);

    memcpy(_pdu + sizeof(rfbUtf8StringMsg), msg, length);

    ret = rfb_writer_enqueue_pdu(clp, _pdu, pdulength);

    return ret;
}

static int rfb_enqueue_conn_params_header(rfb_cl_t * clp, unsigned int no_caps) {
    rfbConnectionParameterListMsg pdu;
    pdu.type_8		= rfbConnectionParameterList;
    pdu.parameters_8	= no_caps;
    return rfb_writer_enqueue_pdu(clp, &pdu, sizeof(pdu));
}

static int rfb_enqueue_conn_param(rfb_cl_t * clp, const char* name, const char* value) {
    int ret;
    int name_len = name ? min((size_t)255, strlen(name)) : 0;
    int value_len = value ? min((size_t)255, strlen(value)) : 0;
    size_t pdulength = sizeof(rfbConnectionParameter) + name_len + value_len;
    rfbConnectionParameter *pdu;
    char _pdu[pdulength];
    
    pdu = (rfbConnectionParameter *)_pdu;
    pdu->name_length_8	= name_len;
    pdu->value_length_8	= value_len;
    
    if (name) memcpy(_pdu + sizeof(rfbConnectionParameter), name, name_len);
    if (value) memcpy(_pdu + sizeof(rfbConnectionParameter) + name_len, value, value_len);
    
    ret = rfb_writer_enqueue_pdu(clp, pdu, pdulength);
    return ret;
}

int rfb_enqueue_connection_parameters(rfb_cl_t * clp, rfb_connection_parameter_list_t *params) {
    int ret;
    unsigned int i;
    size_t no_caps = rfb_connection_parameter_list_size(params);
    
    /* then, send it */
    if ((ret = rfb_enqueue_conn_params_header(clp, no_caps))) {
    	return ret;
    }
    
    for (i = 0; i < no_caps; i++) {
        const char *name = rfb_connection_parameter_name(params, i);
        const char *value = rfb_connection_parameter_value(params, i);
    	if ((ret = rfb_enqueue_conn_param(clp, name, value))) {
    	    return ret;
    	}
    }
    
    return 0;
}

int rfb_enqueue_fb_format_update(rfb_cl_t * clp, rfbPixelFormat *fmt) {
    return rfb_enqueue_fb_format_update_params(clp, clp->fb_width, clp->fb_height, clp->fb_is_unsupported, fmt, NULL);
}

int rfb_enqueue_fb_format_update_params(rfb_cl_t * clp, int fb_width, int fb_height, u_int8_t unsupported,
                                        rfbPixelFormat *fmt, struct timeval *tv) {
    rfbServerFBFormatMsgV1_16 sfbf;

    D(D_VERBOSE, "Enqueueing FB format update, w: %d, h: %d\n", fb_width, fb_height);

    sfbf.type_8                 = rfbServerFBFormat;
    sfbf.unsupported_8          = unsupported;
    sfbf.framebufferWidth_be16  = cpu_to_be16(fb_width);
    sfbf.framebufferHeight_be16 = cpu_to_be16(fb_height);
    sfbf.format = *fmt;
    sfbf.format.redMax_be16     = cpu_to_be16(sfbf.format.redMax_be16);
    sfbf.format.greenMax_be16   = cpu_to_be16(sfbf.format.greenMax_be16);
    sfbf.format.blueMax_be16    = cpu_to_be16(sfbf.format.blueMax_be16);
    if (tv == NULL) {
        rfb_get_current_timestamp(&sfbf.timestamp);
    } else {
        rfb_get_timestamp_timeval(&sfbf.timestamp, tv);
    }

    return rfb_writer_enqueue_pdu(clp, &sfbf, sizeof(sfbf));
}

int rfb_enqueue_ping_request(rfb_cl_t * clp, u_int32_t serial) {
    rfbPingRequestEventMsg pdu;
    
    pdu.type_8    = rfbPingRequestEvent;
    pdu.serial_be32 = cpu_to_be32(serial);
    return rfb_writer_enqueue_pdu(clp, &pdu, sizeof(pdu));
}

int rfb_enqueue_ping_reply(rfb_cl_t * clp, u_int32_t serial) {
    rfbPingReplyMsg pdu;
    
    pdu.type_8    = rfbPingReply;
    pdu.serial_be32 = cpu_to_be32(serial);
    return rfb_writer_enqueue_pdu(clp, &pdu, sizeof(pdu));
}

int rfb_enqueue_vs_update(rfb_cl_t * clp, video_settings_t *settings) {
    rfbVideoSettingsS2CMsgV1_8 settings_pdu;

    settings_pdu.type_8			= rfbVideoSettingsS2CEvent;
    settings_pdu.brightness_8		= settings->brightness;
    settings_pdu.contrast_red_8		= settings->contrast_red;
    settings_pdu.contrast_green_8	= settings->contrast_green;
    settings_pdu.contrast_blue_8	= settings->contrast_blue;
    settings_pdu.clock_be16		= cpu_to_be16(settings->clock);
    settings_pdu.phase_be16		= cpu_to_be16(settings->phase);
    settings_pdu.x_offset_be16		= cpu_to_be16(settings->offset_x);
    settings_pdu.y_offset_be16		= cpu_to_be16(settings->offset_y);
    settings_pdu.x_res_be16		= cpu_to_be16(settings->res_x);
    settings_pdu.y_res_be16		= cpu_to_be16(settings->res_y);
    settings_pdu.refresh_be16		= cpu_to_be16(settings->refresh);
    settings_pdu.y_max_offset_be16	= cpu_to_be16(settings->y_max_offset);
    
    return rfb_writer_enqueue_pdu(clp, &settings_pdu, sizeof(settings_pdu));
}

int rfb_enqueue_vq_update(rfb_cl_t * clp, int noise_filter) {
    rfbVideoQualityS2CMsgV1_12 settings_pdu;
    
    settings_pdu.type_8		= rfbVideoQualityS2CEvent;
    settings_pdu.noise_filter_8	= (u_int8_t)noise_filter;

    return rfb_writer_enqueue_pdu(clp, &settings_pdu, sizeof(settings_pdu));
}

int rfb_enqueue_bandwidth_request(rfb_cl_t * clp, unsigned int length) {
    rfbBandwidthRequestEventMsg* pdu;
    size_t pdulength = sizeof(rfbBandwidthRequestEventMsg) + length;
    char _pdu[pdulength];
    int ret;

    pdu = (rfbBandwidthRequestEventMsg *)_pdu;

    pdu->type_8 = rfbBandwidthRequestEvent;
    pdu->length_be16 = cpu_to_be16(clp->pred_bwidth_bytes);

    ret = rfb_writer_enqueue_pdu(clp, pdu, pdulength);

    return ret;
}

int rfb_enqueue_message(rfb_cl_t *clp, const char *msg) {
    int ret = 0;
    size_t pdu_size = 0;
    char *pdu = NULL;

    pdu = rfb_create_rfb_message_pdu(msg, &pdu_size);
    
    if (pdu && pdu_size > 0) {
	ret = rfb_writer_enqueue_pdu(clp, pdu, pdu_size);
    }

    free(pdu);
    return ret;
}

int rfb_enqueue_osd_state(rfb_cl_t * clp, const char *msg, u_int8_t blank, u_int16_t timeout) {
    return rfb_enqueue_osd_state_time(clp, msg, blank, timeout, NULL);
}

int rfb_enqueue_osd_state_time(rfb_cl_t * clp, const char *msg, u_int8_t blank, u_int16_t timeout, struct timeval *tv) {
    rfbOSDStateMsgV1_16* pdu;
    if (msg == NULL) msg = "";
    size_t msgsize = strlen(msg);
    size_t pdulength = sizeof(rfbOSDStateMsgV1_16) + msgsize;
    char _pdu[pdulength];
    int ret;

    pdu = (rfbOSDStateMsgV1_16 *)_pdu;

    pdu->type_8     = rfbOSDState;
    pdu->blank_8    = blank;
    pdu->timeout_be16 = cpu_to_be16(timeout);
    pdu->length_be16  = cpu_to_be16(msgsize);
    if (tv == NULL) {
        rfb_get_current_timestamp(&pdu->timestamp);
    } else {
        rfb_get_timestamp_timeval(&pdu->timestamp, tv);
    }
   
    memcpy(_pdu + sizeof(rfbOSDStateMsgV1_16), msg, msgsize);

    ret = rfb_writer_enqueue_pdu(clp, pdu, pdulength);   

    return ret;
}

int rfb_enqueue_kbd_layout(rfb_cl_t * clp, char* kbd) {
    size_t pdu_size;
    char *pdu = rfb_create_kbd_layout_pdu(kbd, &pdu_size);

    int ret = rfb_writer_enqueue_pdu(clp, pdu, pdu_size);
    
    free(pdu);
    return ret;
}

#if defined(PP_FEAT_SESSION_REDIRECTOR)

int rfb_enqueue_sas_error(rfb_cl_t * clp, int errorcode) {
    rfbSasErrorMsg pdu;
    
    pdu.type_8 = rfbSasError;
    pdu.errorcode_8 = errorcode;
    
    return rfb_writer_enqueue_pdu(clp, &pdu, sizeof(pdu));
}

int rfb_enqueue_sas_existing_session(rfb_cl_t * clp, const char *user, const char *ip,
				     int self, time_t login, eric_session_int_id_t session_id) {
    char *pdu, *pdu_end;
    size_t user_size = 0, ip_size = 0;
    size_t size, session_size;
    rfbSasEventExistingSessionMsg *session;
    int ret = 0;
    
    if (user) user_size = min(strlen(user), (size_t)255);
    if (ip) ip_size += min(strlen(ip), (size_t)255);
    
    session_size = sizeof(rfbSasEventExistingSessionMsg) + user_size + ip_size;

    pdu = rfb_create_sas_event(rfbSasEventExistingSession,
    	session_size, session_id, &size);
    if (!pdu) {
    	return -1;
    }
    
    session = (rfbSasEventExistingSessionMsg *)(pdu + size);
    pdu_end = pdu + size + sizeof(rfbSasEventExistingSessionMsg);
    
    session->session_info.user_len_8 = user_size;
    session->session_info.ip_len_8 = ip_size;
    session->self_8 = self ? 1 : 0;
    rfb_get_timestamp_time_t(&session->login_time, login);

    if (user) memcpy(pdu_end, user, user_size);
    if (ip) memcpy(pdu_end + user_size, ip, ip_size);
    
    ret = rfb_writer_enqueue_pdu(clp, pdu, size + session_size);
    
    free(pdu);
    return ret;
}

int rfb_enqueue_sas_existing_kvm_session(rfb_cl_t * clp, eric_session_int_id_t session_id,
					 rfb_cl_t *new_clp, int exclusive, time_t login_time) {
    char *pdu;
    size_t size;
    int ret;
    rfbSasEventExistingKvmSessionMsg *kvm;

    pdu = rfb_create_sas_event(rfbSasEventExistingKvmSession,
    	sizeof(rfbSasEventExistingKvmSessionMsg), session_id, &size);
    if (!pdu) {
    	return -1;
    }
    
    kvm = (rfbSasEventExistingKvmSessionMsg *)(pdu + size);
    kvm->exclusive_8 = exclusive ? 1 : 0;
    kvm->kvm.kvm_session_id_be32 = cpu_to_be32((u_int32_t)new_clp);
    rfb_get_timestamp_time_t(&kvm->login_time, login_time);
    
    ret = rfb_writer_enqueue_pdu(clp, pdu, size + sizeof(rfbSasEventExistingKvmSessionMsg));
    
    free(pdu);
    return ret;
}

int rfb_enqueue_sas_login_failure_event(rfb_cl_t * clp, const char *user, const char *ip) {
    char *pdu, *pdu_end;
    size_t user_size = 0, ip_size = 0;
    size_t size, info_size;
    rfbSasUserInfo *info;
    int ret = 0;
    
    if (user) user_size = min(strlen(user), (size_t)255);
    if (ip) ip_size += min(strlen(ip), (size_t)255);

    info_size = sizeof(rfbSasUserInfo) + user_size + ip_size;

    pdu = rfb_create_sas_event(rfbSasEventUserLoginFailure, info_size, 0, &size);
    if (!pdu) {
    	return -1;
    }
    
    info = (rfbSasUserInfo *)(pdu + size);
    pdu_end = pdu + size + sizeof(rfbSasUserInfo);

    info->user_len_8 = user_size;
    info->ip_len_8 = ip_size;

    if (user) memcpy(pdu_end, user, user_size);
    if (ip) memcpy(pdu_end + user_size, ip, ip_size);
    
    ret = rfb_writer_enqueue_pdu(clp, pdu, size + info_size);
    
    free(pdu);
    return ret;
}

int rfb_enqueue_sas_new_session_event(rfb_cl_t * clp, const char *user, const char *ip,
				      time_t login, eric_session_int_id_t session) {
    char *pdu, *pdu_end;
    size_t user_size = 0, ip_size = 0;
    size_t size, newsess_size;
    rfbSasEventNewSessionMsg *newsess;
    int ret = 0;
    
    if (user) user_size = min(strlen(user), (size_t)255);
    if (ip) ip_size += min(strlen(ip), (size_t)255);

    newsess_size = sizeof(rfbSasEventNewSessionMsg) + user_size + ip_size;

    pdu = rfb_create_sas_event(rfbSasEventUserSessionOpened, newsess_size, session, &size);
    if (!pdu) {
    	return -1;
    }
    
    newsess = (rfbSasEventNewSessionMsg *)(pdu + size);
    pdu_end = pdu + size + sizeof(rfbSasEventNewSessionMsg);

    rfb_get_timestamp_time_t(&newsess->login_time, login);
    newsess->session_info.user_len_8 = user_size;
    newsess->session_info.ip_len_8 = ip_size;

    if (user) memcpy(pdu_end, user, user_size);
    if (ip) memcpy(pdu_end + user_size, ip, ip_size);
    
    ret = rfb_writer_enqueue_pdu(clp, pdu, size + newsess_size);
    
    free(pdu);
    return ret;
}

int rfb_enqueue_sas_generic_event(rfb_cl_t * clp, int type, eric_session_int_id_t session_id) {
    char *pdu;
    size_t size;
    int ret;

    pdu = rfb_create_sas_event(type, 0, session_id, &size);
    if (!pdu) {
    	return -1;
    }
    
    ret = rfb_writer_enqueue_pdu(clp, pdu, size);
    
    free(pdu);
    return ret;
}

int rfb_enqueue_sas_new_kvm_session(rfb_cl_t * clp, eric_session_int_id_t session_id, rfb_cl_t *causing_clp, time_t session_time) {
    char *pdu;
    size_t size;
    int ret;
    rfbSasNewKvmSessionMsg *kvm;

    pdu = rfb_create_sas_event(rfbSasEventKvmSessionOpened, sizeof(rfbSasNewKvmSessionMsg), session_id, &size);
    if (!pdu) {
    	return -1;
    }
    
    kvm = (rfbSasNewKvmSessionMsg *)(pdu + size);
    kvm->info.kvm_session_id_be32 = cpu_to_be32((u_int32_t)causing_clp);
    rfb_get_timestamp_time_t(&kvm->session_time, session_time);
    
    ret = rfb_writer_enqueue_pdu(clp, pdu, size + sizeof(rfbSasNewKvmSessionMsg));
    
    free(pdu);
    return ret;
}

int rfb_enqueue_sas_kvm_session_event(rfb_cl_t * clp, int type, eric_session_int_id_t session_id, rfb_cl_t *causing_clp) {
    char *pdu;
    size_t size;
    int ret;
    rfbSasKvmInfo *si;

    pdu = rfb_create_sas_event(type, sizeof(rfbSasKvmInfo), session_id, &size);
    if (!pdu) {
    	return -1;
    }
    
    si = (rfbSasKvmInfo *)(pdu + size);
    si->kvm_session_id_be32 = cpu_to_be32((u_int32_t)causing_clp);
    
    ret = rfb_writer_enqueue_pdu(clp, pdu, size + sizeof(rfbSasKvmInfo));
    
    free(pdu);
    return ret;
}

int rfb_enqueue_sas_input_event(rfb_cl_t * clp, input_event_t *ie, eric_session_int_id_t session_id, rfb_cl_t *causing_clp) {
    size_t add_size = 0, size;
    char *_pdu;
    rfbSasEventMsg *pdu;
    int ret = 0;
    
    if (ie->type == rfbKeyEvent) {
    	add_size += sizeof(rfbSasKeyboardEventMsg);
    } else if (ie->type == rfbPointerEvent || ie->type == rfbPointerRelativeEvent) {
    	add_size += sizeof(rfbSasPointerEventMsg);
    } else if (ie->type == rfbMouseSyncEvent) {
    	add_size += sizeof(rfbSasMousesyncEventMsg);
    } else {
    	return -1;
    }
    
    _pdu = rfb_create_sas_event(rfbSasEventInput, add_size, session_id, &size);
    if (!_pdu) {
    	return -1;
    }
    
    /* replace timestamp */
    pdu = (rfbSasEventMsg *)_pdu;
    rfb_get_timestamp_timeval(&pdu->timestamp, &ie->arrived);
    
    if (ie->type == rfbKeyEvent) {
    	rfbSasKeyboardEventMsg *kbd = (rfbSasKeyboardEventMsg *)(_pdu + size);
    	kbd->kvm.kvm_session_id_be32 = cpu_to_be32((u_int32_t)causing_clp);
    	kbd->kbd.type_8 = ie->type;
    	kbd->kbd.key_8 = ie->data.keycode;
    } else if (ie->type == rfbMouseSyncEvent) {
    	rfbSasMousesyncEventMsg *msync = (rfbSasMousesyncEventMsg *)(_pdu + size);
    	msync->kvm.kvm_session_id_be32 = cpu_to_be32((u_int32_t)causing_clp);
    	msync->sync.type_8 = ie->type;
    	msync->sync.sync_type_8 = ie->data.sync_type;
    } else {
    	rfbSasPointerEventMsg *ptr = (rfbSasPointerEventMsg *)(_pdu + size);
    	ptr->kvm.kvm_session_id_be32 = cpu_to_be32((u_int32_t)causing_clp);
    	ptr->pointer.type_8 = ie->type;
    	ptr->pointer.buttonMask_8 = ie->data.mouse_event.button_mask;
    	ptr->pointer.x_be16 = cpu_to_be16(ie->data.mouse_event.x);
    	ptr->pointer.y_be16 = cpu_to_be16(ie->data.mouse_event.y);
    	ptr->pointer.z_be16 = cpu_to_be16(ie->data.mouse_event.z);
    }
    
    ret = rfb_writer_enqueue_pdu(clp, _pdu, size + add_size);
    
    free(_pdu);
    return ret;
}

int rfb_enqueue_sas_kvm_switch_event(rfb_cl_t * clp, eric_session_int_id_t session_id,
				     u_char channel, u_char unit, u_short port) {
    char *pdu;
    size_t size;
    int ret;
    rfbSasKvmSwitchEventMsg *kvm;

    pdu = rfb_create_sas_event(rfbSasEventKvmSwitch, sizeof(rfbSasKvmSwitchEventMsg), session_id, &size);
    if (!pdu) {
    	return -1;
    }
    
    kvm = (rfbSasKvmSwitchEventMsg *)(pdu + size);
    kvm->channel_8 = channel;
    kvm->unit_8 = unit;
    kvm->port_be16 = cpu_to_be16(port);
    
    ret = rfb_writer_enqueue_pdu(clp, pdu, size + sizeof(rfbSasKvmSwitchEventMsg));
    
    free(pdu);
    return ret;
}

#endif /* PP_FEAT_SESSION_REDIRECTOR */

/******************************************************************************
* Creating PDUs for further handling                                          *
******************************************************************************/

char * rfb_create_rfb_message_pdu(const char* msg, size_t *size) {
    char *text, *buf;
    size_t text_size, pdu_size;
    rfbMessageV1_16 *pdu;

    text_size = strlen(msg);
    pdu_size = sizeof(rfbMessageV1_16) + text_size; 
    buf = malloc(pdu_size);
    if (!buf) return 0;

    pdu		   = (rfbMessageV1_16*)buf;
    text	   = buf + sizeof(rfbMessageV1_16);
    pdu->type_8	   = rfbServerRCMessage;
    pdu->len_be32  = cpu_to_be32((u_int32_t)text_size);
    rfb_get_current_timestamp(&pdu->timestamp);
    memcpy(text, msg, text_size);

    *size = pdu_size;

    return buf;
}

char * rfb_create_command_pdu(const char* command, const char* params, size_t *size) {
    char *buf, *command_text, *params_text;
    size_t pdu_size, command_size, params_size;
    rfbCommandMsg *pdu;

    command_size = strlen(command);
    params_size = strlen(params);
    pdu_size = sizeof(rfbCommandMsg) + command_size + params_size; 
    buf = malloc(pdu_size);
    if (!buf) return NULL;

    pdu		           = (rfbCommandMsg*)buf;
    command_text           = buf + sizeof(rfbCommandMsg);
    params_text            = command_text + command_size;
    pdu->type_8	           = rfbServerCommand;
    pdu->length_command_be16 = cpu_to_be16(command_size);
    pdu->length_params_be16  = cpu_to_be16(params_size);
    memcpy(command_text, command, command_size);
    memcpy(params_text, params, params_size);

    *size = pdu_size;
    
    return buf;
}

char * rfb_create_kbd_layout_pdu(char* kbd, size_t *size) {
    char *buf, *text;
    size_t pdu_size, text_size;
    rfbKeyboardLayoutEventMsg *pdu;

    assert(kbd);
    text_size = strlen(kbd);
    pdu_size = sizeof(rfbKeyboardLayoutEventMsg) + text_size; 
    buf = malloc(pdu_size);
    if (!buf) return NULL;

    pdu			= (rfbKeyboardLayoutEventMsg*)buf;
    text		= buf + sizeof(rfbKeyboardLayoutEventMsg);
    pdu->type_8		= rfbKeyboardLayoutEvent;
    pdu->length_be16	= cpu_to_be16((u_int16_t)text_size);
    memcpy(text, kbd, text_size);

    *size = pdu_size;
    
    return buf;
}

char * rfb_create_fb_update_pdu(int n_upd_reg_rects, int with_timestamp,
				int full_update, size_t *size) {
    rfbFramebufferUpdateMsg *fbu;
    //with_timestamp = 0;
    
    D(D_BLABLA, "Creating update header, no_rects: %d\n", n_upd_reg_rects);
    
    *size = sizeof(rfbFramebufferUpdateMsg);
    if (with_timestamp) {
    	*size += sizeof(rfbTimestamp);
    }
    fbu = malloc(*size);
    if (!fbu) return NULL;
    
    fbu->type_8 = rfbFramebufferUpdate;
    fbu->flags_8 = 0;
    fbu->nRects_be16 = cpu_to_be16((u_int16_t)n_upd_reg_rects);

    if (with_timestamp) {
    	fbu->flags_8 |= rfbFramebufferUpdateFlagTimestamp;
    	rfb_get_current_timestamp((rfbTimestamp *)((char *)fbu + sizeof(rfbFramebufferUpdateMsg)));
    }
    
    if (full_update) {
    	fbu->flags_8 |= rfbFramebufferUpdateFlagFullFrame;
    }
    
    return (char *)fbu;
}

char * rfb_create_fb_update_rect_header(int x, int y, int w, int h,
					u_int32_t enc, size_t *size) {
    rfbFramebufferUpdateRectHeader *rect;

    D(D_BLABLA, "Creating update rect header, x: %d, y: %d, w: %d, h: %d\n", x, y, w, h);

    rect = malloc(sizeof(rfbFramebufferUpdateRectHeader));
    if (!rect) return NULL;
    
    rect->r.x_be16 = cpu_to_be16((u_int16_t)x);
    rect->r.y_be16 = cpu_to_be16((u_int16_t)y);
    rect->r.w_be16 = cpu_to_be16((u_int16_t)w);
    rect->r.h_be16 = cpu_to_be16((u_int16_t)h);
    rect->encoding_be32 = cpu_to_be32(enc);
    
    *size = sizeof(rfbFramebufferUpdateRectHeader);
    return (char *)rect;
}

char * rfb_create_raw_vsc_hdr(u_int8_t flags, size_t *size) {
    rfbRawVscHdr *hdr;

    hdr = malloc(sizeof(rfbRawVscHdr));
    if (!hdr) return NULL;
    
    hdr->flags_8 = flags;
    *size = sizeof(rfbRawVscHdr);
    return (char *)hdr;
}

char * rfb_create_hwenc_hdr(u_int32_t hw_size, size_t *size) {
    rfbHwencHdr *hdr;

    hdr = malloc(sizeof(rfbHwencHdr));
    if (!hdr) return NULL;
    
    hdr->size_be32 = cpu_to_be32(hw_size);
    *size = sizeof(rfbHwencHdr);
    return (char *)hdr;
}

#if defined(PP_FEAT_SESSION_REDIRECTOR)

static char * rfb_create_sas_event(u_int8_t event, size_t additional_len,
				   eric_session_int_id_t session, size_t *size) {
    char *ret;
    rfbSasEventMsg *pdu;
    
    *size = sizeof(rfbSasEventMsg);
    if ((ret = malloc(*size + additional_len)) == NULL) {
    	return NULL;
    }
    
    pdu = (rfbSasEventMsg *)ret;
    pdu->type_8 = rfbSasEvent;
    pdu->event_type_8 = event;
    pdu->add_len_be16 = cpu_to_be16((u_int16_t)additional_len);
    pdu->session_id_be32 = cpu_to_be32((u_int32_t)session);
    rfb_get_current_timestamp(&pdu->timestamp);
        
    return ret;
}

#endif /* PP_FEAT_SESSION_REDIRECTOR */


/******************************************************************************
* Misc stuff                                                                  *
******************************************************************************/

static void rfb_get_current_timestamp(rfbTimestamp *ts) {
    struct timeval t;

    gettimeofday(&t, NULL);
    
    rfb_get_timestamp_timeval(ts, &t);
}

static void rfb_get_timestamp_timeval(rfbTimestamp *ts, struct timeval* tv) {
    ts->timestamp_be32 = cpu_to_be32(tv->tv_sec);
    ts->microseconds_be32 = cpu_to_be32(tv->tv_usec);
}

static void rfb_parse_timestamp_to_timeval(rfbTimestamp *ts, struct timeval* tv) {
    tv->tv_sec = cpu_to_be32(ts->timestamp_be32);
    tv->tv_usec = cpu_to_be32(ts->microseconds_be32);
}

#if defined(PP_FEAT_SESSION_REDIRECTOR)

static void rfb_get_timestamp_time_t(rfbTimestamp *ts, time_t timestamp) {
    ts->timestamp_be32 = cpu_to_be32(timestamp);
    ts->microseconds_be32 = 0;
}

#endif /* PP_FEAT_SESSION_REDIRECTOR */
