/*
 * This file was extracted from TightVNC 1.2.1 and modified by Peppercon AG.
 */

/*
 * tight.c
 *
 * Routines to implement Tight Encoding
 */

/*
 *  Copyright (C) 2000, 2001 Const Kaplinsky.  All Rights Reserved.
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include <stdio.h>
#include <netinet/in.h>
#include "rfb.h"
#include "debug.h"

/* Note: The following constant should not be changed. */
#define TIGHT_MIN_TO_COMPRESS	12

/* The parameters below may be adjusted. */
#define MIN_SPLIT_RECT_SIZE     4096
#define MIN_SOLID_SUBRECT_SIZE  2048
#define MAX_SPLIT_TILE_SIZE       16

/* Compression level stuff. The following array contains various
   encoder parameters for each of 10 compression levels (0..9).
   Last three parameters correspond to JPEG quality levels (0..9). */

typedef struct TIGHT_CONF_s {
    int maxRectSize, maxRectWidth;
    int monoMinRectSize, gradientMinRectSize;
    int idxZlibLevel, monoZlibLevel, rawZlibLevel, gradientZlibLevel;
    unsigned long gradientThreshold, gradientThreshold24;
    int idxMaxColorsDivisor;
} TIGHT_CONF;

/* for HCML make sure that maxRectSize/maxRectWidth won't lead
   to single subrects with a height < PP_FB_TILE_HEIGHT (16),
   else we get misaligned rect corners */
static TIGHT_CONF tightConf[10] = {
    {   512,   32,   6, 65536, 0, 0, 0, 0,   0,   0,   4 },
    {  2048,  128,   6, 65536, 1, 1, 1, 0,   0,   0,   8 },
    {  6144,  256,   8, 65536, 3, 3, 2, 0,   0,   0,  24 },
    {  8192,  512,  12, 65536, 5, 5, 3, 0,   0,   0,  32 },
    { 16384, 1024,  12, 65536, 6, 6, 4, 0,   0,   0,  32 },
    { 32768, 2048,  12,  4096, 7, 7, 5, 4, 150, 380,  32 },
    { 65536, 2048,  16,  4096, 7, 7, 6, 4, 170, 420,  48 },
    { 65536, 2048,  16,  4096, 8, 8, 7, 5, 180, 450,  64 },
    { 65536, 2048,  32,  8192, 9, 9, 8, 6, 190, 475,  64 },
    { 65536, 2048,  32,  8192, 9, 9, 9, 6, 200, 500,  96 }
};

/* Prototypes for static functions. */

static int SendRectSimple(rfb_cl_t * clp, int x, int y, int w, int h);
static int SendSubrect(rfb_cl_t * clp, int x, int y, int w, int h);
static int SendTightHeader(rfb_cl_t * clp, int x, int y, int w, int h, u_int32_t encoding);

static int SendSolidRect(rfb_cl_t * clp);
static int SendMonoRect(rfb_cl_t * clp, int w, int h);
static int SendIndexedRect(rfb_cl_t * clp, int w, int h);
static int SendFullColorRect(rfb_cl_t * clp, int w, int h);
static int TranslateXBitFullColorRect(rfb_cl_t * clp, int w, int h);
static int SendXBitFullColorRect(rfb_cl_t * clp, int dataSize);
static int SendGradientRect(rfb_cl_t * clp, int w, int h);
static u_int8_t ComputeCachetile(u_int8_t * hash_buf, rfb_cache_lists_t * cache_lists, int tileSize);
static int SendCacheMetarect(rfb_cl_t * clp, int x, int y, int w, int h, int * dataSize);

static int CompressData(rfb_cl_t * clp, int streamId, int dataLen,
			int zlibLevel, int zlibStrategy);
static int SendCompressedData(rfb_cl_t * clp, int compressedLen);

static void FillPalette8(rfb_cl_t * clp, int count);
static void FillPalette16(rfb_cl_t * clp, int count);
#if ENABLE_24_AND_32_BIT
static void FillPalette32(rfb_cl_t * clp, int count);
#else /* ENABLE_24_AND_32_BIT */
#define FillPalette32(...) ERROR_24_AND_32_BIT
#endif /* ENABLE_24_AND_32_BIT */

static void PaletteReset(rfb_cl_t * clp);
static int PaletteInsert(rfb_cl_t * clp, u_int32_t rgb, int numPixels, int bpp);

#if ENABLE_24_AND_32_BIT
static void Pack24(char * buf, rfb_pixel_format_t * fmt, int count);
#else /* ENABLE_24_AND_32_BIT */
#define Pack24(...) ERROR_24_AND_32_BIT
#endif /* ENABLE_24_AND_32_BIT */

static void EncodeIndexedRect16(rfb_cl_t * clp, u_int8_t * buf, int count);
#if ENABLE_24_AND_32_BIT
static void EncodeIndexedRect32(rfb_cl_t * clp, u_int8_t * buf, int count);
#else /* ENABLE_24_AND_32_BIT */
#define EncodeIndexedRect32(...) ERROR_24_AND_32_BIT
#endif /* ENABLE_24_AND_32_BIT */

static void EncodeMonoRect8(rfb_cl_t * clp, u_int8_t * buf, int w, int h);
static void EncodeMonoRect16(rfb_cl_t * clp, u_int8_t * buf, int w, int h);
#if ENABLE_24_AND_32_BIT
static void EncodeMonoRect32(rfb_cl_t * clp, u_int8_t * buf, int w, int h);
#else /* ENABLE_24_AND_32_BIT */
#define EncodeMonoRect32(...) ERROR_24_AND_32_BIT
#endif /* ENABLE_24_AND_32_BIT */

static void FilterGradient16(rfb_cl_t * clp, u_int16_t * buf, int w, int h);
#if ENABLE_24_AND_32_BIT
static void FilterGradient24(rfb_cl_t * clp, char * buf, int w, int h);
static void FilterGradient32(rfb_cl_t * clp, u_int32_t * buf, int w, int h);
#else /* ENABLE_24_AND_32_BIT */
#define FilterGradient24(...) ERROR_24_AND_32_BIT
#define FilterGradient32(...) ERROR_24_AND_32_BIT
#endif /* ENABLE_24_AND_32_BIT */

static int DetectSmoothImage(rfb_cl_t * clp, int w, int h);
static unsigned long DetectSmoothImage16(rfb_cl_t * clp, int w, int h);
#if ENABLE_24_AND_32_BIT
static unsigned long DetectSmoothImage24(rfb_cl_t * clp, int w, int h);
static unsigned long DetectSmoothImage32(rfb_cl_t * clp, int w, int h);
#endif /* ENABLE_24_AND_32_BIT */

/*
 * Tight encoding implementation.
 */

int
rfb_num_coded_rects_tight(rfb_cl_t * clp, int x UNUSED, int y UNUSED, int w,
			  int h)
{
    int maxRectSize, maxRectWidth;
    int subrectMaxWidth, subrectMaxHeight;

    maxRectSize = tightConf[clp->enc.tightCompressLevel].maxRectSize;
    maxRectWidth = tightConf[clp->enc.tightCompressLevel].maxRectWidth;

    if (w > maxRectWidth || w * h > maxRectSize) {
        subrectMaxWidth = (w > maxRectWidth) ? maxRectWidth : w;

	if ((subrectMaxHeight = maxRectSize / subrectMaxWidth) % PP_FB_TILE_HEIGHT) {
	    subrectMaxHeight = subrectMaxHeight / PP_FB_TILE_HEIGHT * PP_FB_TILE_HEIGHT;
	}

	assert(subrectMaxHeight);
	
        return (((w - 1) / maxRectWidth + 1) *
                ((h - 1) / subrectMaxHeight + 1));
    } else {
        return 1;
    }
}

int
rfb_send_rect_tight(rfb_cl_t * clp, int x, int y, int w, int h)
{
#ifdef TIME_MEASURE
    clp->time_pixel += w*h;
#endif
   
    if ( clp->pix_fmt.depth_8 == 24 && clp->pix_fmt.redMax_be16 == 0xFF &&
         clp->pix_fmt.greenMax_be16 == 0xFF && clp->pix_fmt.blueMax_be16 == 0xFF ) {
	clp->usePixelFormat24 = 1;
    } else {
        clp->usePixelFormat24 = 0;
    }

    return SendRectSimple(clp, x, y, w, h);
}

static int
SendRectSimple(rfb_cl_t * clp, int x, int y, int w, int h)
{
    int maxBeforeSize, maxAfterSize;
    int maxRectSize, maxRectWidth;
    int subrectMaxWidth, subrectMaxHeight;
    int dx, dy;
    int rw, rh;

    maxRectSize = tightConf[clp->enc.tightCompressLevel].maxRectSize;
    maxRectWidth = tightConf[clp->enc.tightCompressLevel].maxRectWidth;

    maxBeforeSize = maxRectSize * (clp->pix_fmt.bitsPerPixel_8 / 8);
    maxAfterSize = maxBeforeSize + (maxBeforeSize + 99) / 100 + 12;

    if (clp->tightBeforeBufSize < maxBeforeSize) {
        clp->tightBeforeBufSize = maxBeforeSize;
        if (clp->tightBeforeBuf == NULL) {
            clp->tightBeforeBuf = (char *)malloc(clp->tightBeforeBufSize);
            clp->tightXBitFullColorBuf = (char *)malloc(clp->tightBeforeBufSize);
        }
        else {
            clp->tightBeforeBuf = (char *)realloc(clp->tightBeforeBuf, clp->tightBeforeBufSize);
            clp->tightXBitFullColorBuf = (char *)realloc(clp->tightBeforeBuf, clp->tightBeforeBufSize);
        }
    }

    if (clp->tightAfterBufSize < maxAfterSize) {
        clp->tightAfterBufSize = maxAfterSize;
        if (clp->tightAfterBuf == NULL)
            clp->tightAfterBuf = (char *)malloc(clp->tightAfterBufSize);
        else
            clp->tightAfterBuf = (char *)realloc(clp->tightAfterBuf,
						 clp->tightAfterBufSize);
    }

    if (w > maxRectWidth || w * h > maxRectSize) {
        subrectMaxWidth = (w > maxRectWidth) ? maxRectWidth : w;

	if ((subrectMaxHeight = maxRectSize / subrectMaxWidth) % PP_FB_TILE_HEIGHT) {
	    subrectMaxHeight = subrectMaxHeight / PP_FB_TILE_HEIGHT * PP_FB_TILE_HEIGHT;
	}

	assert(subrectMaxHeight);
	    
        for (dy = 0; dy < h; dy += subrectMaxHeight) {
            for (dx = 0; dx < w; dx += maxRectWidth) {
                rw = (dx + maxRectWidth < w) ? maxRectWidth : w - dx;
                rh = (dy + subrectMaxHeight < h) ? subrectMaxHeight : h - dy;
                if (SendSubrect(clp, x+dx, y+dy, rw, rh) == -1) {
                    return -1;
		}
            }
        }
    } else {
	if (SendSubrect(clp, x, y, w, h) == -1) {
            return -1;
	}
    }

    return 0;
}

static int
SendSubrect(rfb_cl_t * clp, int x, int y, int w, int h)
{
    char * fb_ptr;
    int success = 0;
    int rectSize = w * h;

    assert(x % 16 == 0); assert(y % 16 == 0);
    assert(clp != NULL);
    assert(clp->translate_fn != NULL);

    /* Send pending data if there is more than 128 bytes. */
    if (clp->ub_len > 128) {
	if (rfb_send_update_buf(clp) == -1) {
            return -1;
	}
    }
   
    fb_ptr = GET_PTR_INTO_FB(clp, x, y);

    clp->translate_fn(clp->translate_lookup_table, &rfb_pix_fmt,
		      &clp->pix_fmt, fb_ptr, clp->tightBeforeBuf,
		      clp->fb_tile_pitch, w, h);

    clp->paletteMaxColors = rectSize / tightConf[clp->enc.tightCompressLevel].idxMaxColorsDivisor;
    if ( clp->paletteMaxColors < 2 &&
         rectSize >= tightConf[clp->enc.tightCompressLevel].monoMinRectSize ) {
        clp->paletteMaxColors = 2;
    }
    switch (clp->pix_fmt.bitsPerPixel_8) {
      case 8:
	  FillPalette8(clp, rectSize);
	  break;
      case 16:
	  FillPalette16(clp, rectSize);
	  break;
      default:
	  FillPalette32(clp, rectSize);
    }

    switch (clp->paletteNumColors) {
      case 0:
	  /* Truecolor image */
	  if (DetectSmoothImage(clp, w, h)) {
	      if (SendTightHeader(clp, x, y, w, h, rfbEncodingTight) == -1) {
		  return -1;
	      }
	      success = SendGradientRect(clp, w, h);
	  } else {
	      int sendRect = 0;
	      int dataSize = TranslateXBitFullColorRect(clp, w, h);
	      if (clp->useTightCache && h >= 16 && !((x % rfbTightTileDimension) || (w % rfbTightTileDimension))) {
		    if (SendTightHeader(clp, x, y, w, h, rfbEncodingUseTightCache) == -1) {
		    	return -1;
		    }
		    sendRect = SendCacheMetarect(clp, x, y, w, h, &dataSize);
	      }
	      else { // no cache
	          sendRect = 1;
	          if (SendTightHeader(clp, x, y, w, h, rfbEncodingTight) == -1) {
	              return -1;
	          }
	      }
	      if (sendRect) { // finally send the rectangle if necessary
		  if (clp->enc.tightXBitFullColor < rfbTight8Bit256Colors) {
		      success = SendXBitFullColorRect(clp, dataSize);
		  }
		  else {
		      success = SendFullColorRect(clp, w, h);
		  }
	      }
	  }
	  break;
      case 1:
	  if (SendTightHeader(clp, x, y, w, h, rfbEncodingTight) == -1) {
	      return -1;
	  }
	  /* Solid rectangle */
	  success = SendSolidRect(clp);
	  break;
      case 2:
	  if (SendTightHeader(clp, x, y, w, h, rfbEncodingTight) == -1) {
	      return -1;
	  }
	  /* Two-color rectangle */
	  success = SendMonoRect(clp, w, h);
	  break;
      default:
	  if (SendTightHeader(clp, x, y, w, h, rfbEncodingTight) == -1) {
	      return -1;
	  }
	  /* Up to 256 different colors */
	  success = SendIndexedRect(clp, w, h);
    }
    return success;
}

static int
SendTightHeader(rfb_cl_t * clp, int x, int y, int w, int h, u_int32_t encoding)
{
    char *rect;
    size_t size = 0;

    rect = rfb_create_fb_update_rect_header(x, y, w, h, encoding, &size);
    if (!rect || size == 0) {
    	return -1;
    }

    if (clp->ub_len + size > UPDATE_BUF_SIZE) {
	if (rfb_send_update_buf(clp) == -1) {
            return -1;
	}
    }

    memcpy(&clp->update_buf[clp->ub_len], rect, size);
    free(rect);
    clp->ub_len += size;

    return 0;
}

/*
 * Subencoding implementations.
 */

static int
SendSolidRect(rfb_cl_t * clp)
{
    int len;

    if (clp->usePixelFormat24) {
        Pack24(clp->tightBeforeBuf, &clp->pix_fmt, 1);
        len = 3;
    } else
        len = clp->pix_fmt.bitsPerPixel_8 / 8;

    if (clp->ub_len + 1 + len > UPDATE_BUF_SIZE) {
	if (rfb_send_update_buf(clp) == -1) {
            return -1;
	}
    }

    if (clp->enc.tightXBitFullColor < 8 ) {
        u_int8_t * lut = NULL;
        switch (clp->enc.tightXBitFullColor) {
            case rfbTight1BitBlackWhite: lut = tightLUT1BitFullColorBlackWhite; break;
            case rfbTight2BitGrayscale:  lut = tightLUT2BitFullColorGrayscale;  break;
            case rfbTight4BitGrayscale:  lut = tightLUT4BitFullColorGrayscale;  break;
            case rfbTight4Bit16Colors:   lut = tightLUT4BitFullColor16Colors;   break;
        }
        clp->update_buf[clp->ub_len++] = (char)(rfbTightFillXBit << 4);
        clp->update_buf[clp->ub_len++] = clp->enc.tightXBitFullColor;
        clp->update_buf[clp->ub_len++] = lut[((u_int8_t *)clp->tightBeforeBuf)[0]];
    }
    else {
        clp->update_buf[clp->ub_len++] = (char)(rfbTightFill << 4);
        memcpy(&clp->update_buf[clp->ub_len], clp->tightBeforeBuf, len);
        clp->ub_len += len;
    }

    return 0;
}

static int
SendMonoRect(rfb_cl_t * clp, int w, int h)
{
    int streamId = 1;
    int paletteLen, dataLen;

    if ( (clp->ub_len + TIGHT_MIN_TO_COMPRESS + 6 +
          2 * clp->pix_fmt.bitsPerPixel_8 / 8) > UPDATE_BUF_SIZE ) {
	if (rfb_send_update_buf(clp) == -1) {
            return -1;
	}
    }

    /* Prepare tight encoding header. */
    dataLen = (w + 7) / 8;
    dataLen *= h;

    /* Prepare palette, convert image. */
    switch (clp->pix_fmt.bitsPerPixel_8) {

      case 32:
          clp->update_buf[clp->ub_len++] = (streamId | rfbTightExplicitFilter) << 4;
          clp->update_buf[clp->ub_len++] = rfbTightFilterPalette;
          clp->update_buf[clp->ub_len++] = 1;

	  EncodeMonoRect32(clp, (u_int8_t *)clp->tightBeforeBuf, w, h);

	  ((u_int32_t *)clp->tightAfterBuf)[0] = clp->monoBackground;
	  ((u_int32_t *)clp->tightAfterBuf)[1] = clp->monoForeground;
	  if (clp->usePixelFormat24) {
	      Pack24(clp->tightAfterBuf, &clp->pix_fmt, 2);
	      paletteLen = 6;
	  } else
	      paletteLen = 8;

	  memcpy(&clp->update_buf[clp->ub_len], clp->tightAfterBuf, paletteLen);
	  clp->ub_len += paletteLen;
	  break;

      case 16:
          clp->update_buf[clp->ub_len++] = (streamId | rfbTightExplicitFilter) << 4;
          clp->update_buf[clp->ub_len++] = rfbTightFilterPalette;
          clp->update_buf[clp->ub_len++] = 1;

	  EncodeMonoRect16(clp, (u_int8_t *)clp->tightBeforeBuf, w, h);

	  ((u_int16_t *)clp->tightAfterBuf)[0] = (u_int16_t)clp->monoBackground;
	  ((u_int16_t *)clp->tightAfterBuf)[1] = (u_int16_t)clp->monoForeground;

	  memcpy(&clp->update_buf[clp->ub_len], clp->tightAfterBuf, 4);
	  clp->ub_len += 4;
	  break;

      default:
	  EncodeMonoRect8(clp, (u_int8_t *)clp->tightBeforeBuf, w, h);

          if (clp->enc.tightXBitFullColor < 8 ) {
              u_int8_t b = 0;
              switch (clp->enc.tightXBitFullColor) {
                  case rfbTight1BitBlackWhite:
                      b = (tightLUT1BitFullColorBlackWhite[(u_int8_t)clp->monoBackground] << 1) | 
          	          (tightLUT1BitFullColorBlackWhite[(u_int8_t)clp->monoForeground]);
          	        break;
                  case rfbTight2BitGrayscale:
                      b = (tightLUT2BitFullColorGrayscale[(u_int8_t)clp->monoBackground] << 2) | 
          	          (tightLUT2BitFullColorGrayscale[(u_int8_t)clp->monoForeground]);
          	        break;
                  case rfbTight4BitGrayscale:
                      b = (tightLUT4BitFullColorGrayscale[(u_int8_t)clp->monoBackground] << 4) | 
          	          (tightLUT4BitFullColorGrayscale[(u_int8_t)clp->monoForeground]);
          	        break;
                  case rfbTight4Bit16Colors:
                      b = (tightLUT4BitFullColor16Colors[(u_int8_t)clp->monoBackground] << 4) | 
          	          (tightLUT4BitFullColor16Colors[(u_int8_t)clp->monoForeground]);
          	        break;
              }
              clp->update_buf[clp->ub_len++] = (streamId | rfbTightExplicitFilter) << 4;
              clp->update_buf[clp->ub_len++] = (clp->enc.tightXBitFullColor << 4) | rfbTightFilterPalette;
              clp->update_buf[clp->ub_len++] = 1;
              clp->update_buf[clp->ub_len++] = b;
          }
          else {
              clp->update_buf[clp->ub_len++] = (streamId | rfbTightExplicitFilter) << 4;
              clp->update_buf[clp->ub_len++] = rfbTightFilterPalette;
              clp->update_buf[clp->ub_len++] = 1;
              clp->update_buf[clp->ub_len++] = (char)clp->monoBackground;
              clp->update_buf[clp->ub_len++] = (char)clp->monoForeground;
          }
    }

    return CompressData(clp, streamId, dataLen,
                        tightConf[clp->enc.tightCompressLevel].monoZlibLevel,
                        Z_DEFAULT_STRATEGY);
}

static int
SendIndexedRect(rfb_cl_t * clp, int w, int h)
{
    int streamId = 2;
    int i, entryLen;

    if ( (clp->ub_len + TIGHT_MIN_TO_COMPRESS + 6 +
          clp->paletteNumColors * clp->pix_fmt.bitsPerPixel_8 / 8) > UPDATE_BUF_SIZE ) {
	if (rfb_send_update_buf(clp) == -1) {
            return -1;
	}
    }

    /* Prepare tight encoding header. */
    clp->update_buf[clp->ub_len++] =
	(streamId | rfbTightExplicitFilter) << 4;
    clp->update_buf[clp->ub_len++] = rfbTightFilterPalette;
    clp->update_buf[clp->ub_len++] = (char)(clp->paletteNumColors - 1);

    /* Prepare palette, convert image. */
    switch (clp->pix_fmt.bitsPerPixel_8) {

      case 32:
	  EncodeIndexedRect32(clp, (u_int8_t *)clp->tightBeforeBuf, w * h);

	  for (i = 0; i < clp->paletteNumColors; i++) {
	      ((u_int32_t *)clp->tightAfterBuf)[i] =
		  clp->palette.entry[i].listNode->rgb;
	  }
	  if (clp->usePixelFormat24) {
	      Pack24(clp->tightAfterBuf, &clp->pix_fmt, clp->paletteNumColors);
	      entryLen = 3;
	  } else
	      entryLen = 4;

	  memcpy(&clp->update_buf[clp->ub_len], clp->tightAfterBuf,
		 clp->paletteNumColors * entryLen);
	  clp->ub_len += clp->paletteNumColors * entryLen;
	  break;

      case 16:
	  EncodeIndexedRect16(clp, (u_int8_t *)clp->tightBeforeBuf, w * h);

	  for (i = 0; i < clp->paletteNumColors; i++) {
	      ((u_int16_t *)clp->tightAfterBuf)[i] =
		  (u_int16_t)clp->palette.entry[i].listNode->rgb;
	  }

	  memcpy(&clp->update_buf[clp->ub_len], clp->tightAfterBuf,
		 clp->paletteNumColors * 2);
	  clp->ub_len += clp->paletteNumColors * 2;
	  break;

      default:
	  return -1;           /* Should never happen. */
    }

    return CompressData(clp, streamId, w * h,
                        tightConf[clp->enc.tightCompressLevel].idxZlibLevel,
                        Z_DEFAULT_STRATEGY);
}

static int
TranslateXBitFullColorRect(rfb_cl_t * clp, int w, int h)
{
    int i, x, y, wDiv = w/8, wMod = w%8;
    u_int16_t idx = 0, idx2 = 0;
    u_int8_t * clientPixelData, * outBuf, * lut;

    if (clp->enc.tightXBitFullColor >= rfbTight8Bit256Colors) return w * h;

    clientPixelData = (u_int8_t *)clp->tightBeforeBuf;
    outBuf = (u_int8_t *)clp->tightXBitFullColorBuf;

    switch (clp->enc.tightXBitFullColor) {
    	case rfbTight1BitBlackWhite:
	    wDiv = w/8;
	    wMod = w%8;
	    lut = tightLUT1BitFullColorBlackWhite;
	    for (y = 0; y < h; y++) {
	    	for (x = 0; x < wDiv; x++) {
	            outBuf[idx2++] = (lut[clientPixelData[idx+0]] << 7) | 
	    	                       (lut[clientPixelData[idx+1]] << 6) | 
	    	                       (lut[clientPixelData[idx+2]] << 5) | 
	    	                       (lut[clientPixelData[idx+3]] << 4) | 
	    	                       (lut[clientPixelData[idx+4]] << 3) | 
	    	                       (lut[clientPixelData[idx+5]] << 2) | 
	    	                       (lut[clientPixelData[idx+6]] << 1) | 
	    	                       (lut[clientPixelData[idx+7]]);
		    idx += 8;
	    	}
	    	if (wMod != 0) {
	    	    outBuf[idx2++] = 0;
	    	    for (i = wMod; i > 0; i--) {
	    	    	outBuf[idx2] |= (lut[clientPixelData[idx++]] << i);
	    	    }
	    	}
	    }
    	    break;
    	case rfbTight2BitGrayscale:
	    wDiv = w/4;
	    wMod = w%4;
	    lut = tightLUT2BitFullColorGrayscale;
	    for (y = 0; y < h; y++) {
	    	for (x = 0; x < wDiv; x++) {
	            outBuf[idx2++] = (lut[clientPixelData[idx+0]] << 6) | 
	    	                       (lut[clientPixelData[idx+1]] << 4) | 
	    	                       (lut[clientPixelData[idx+2]] << 2) | 
	    	                       (lut[clientPixelData[idx+3]]);
		    idx += 4;
	    	}
	    	if (wMod != 0) {
	    	    outBuf[idx2++] = 0;
	    	    for (i = wMod; i > 0; i--) {
	    	    	outBuf[idx2] |= (lut[clientPixelData[idx++]] << 2*i);
	    	    }
	    	}
	    }
    	    break;
    	case rfbTight4BitGrayscale:
    	case rfbTight4Bit16Colors:
	    wDiv = w/2;
	    wMod = w%2;
	    if (clp->enc.tightXBitFullColor == rfbTight4Bit16Colors) lut = tightLUT4BitFullColor16Colors;
	    else lut = tightLUT4BitFullColorGrayscale;
	    for (y = 0; y < h; y++) {
	    	for (x = 0; x < wDiv; x++) {
	            outBuf[idx2++] = (lut[clientPixelData[idx+0]] << 4) | 
	    	                       (lut[clientPixelData[idx+1]]);
		    idx += 2;
	    	}
	    	if (wMod != 0) {
	    	    outBuf[idx2++] = lut[clientPixelData[idx++]] << 4;
	    	}
	    }
    	    break;
    	default:
    	    return -1;           /* Should never happen. */
    }

    clp->tightBeforeBuf = (char *) outBuf;
    clp->tightXBitFullColorBuf = (char *) clientPixelData;

    return idx2;
}

static int
SendXBitFullColorRect(rfb_cl_t * clp, int dataSize)
{
    int streamId = 0;

    if (clp->ub_len + TIGHT_MIN_TO_COMPRESS + 1 > UPDATE_BUF_SIZE) {
	if (rfb_send_update_buf(clp) == -1) {
            return -1;
	}
    }

    switch (clp->enc.tightXBitFullColor) {
    	case rfbTight1BitBlackWhite:
    	    clp->update_buf[clp->ub_len++] = (streamId | rfbTight1BitFullColorBlackWhite) << 4;
    	    break;
    	case rfbTight2BitGrayscale:
    	    clp->update_buf[clp->ub_len++] = (streamId | rfbTight2BitFullColorGrayscale) << 4;
    	    break;
    	case rfbTight4BitGrayscale:
    	    clp->update_buf[clp->ub_len++] = (streamId | rfbTight4BitFullColorGrayscale) << 4;
    	    break;
    	case rfbTight4Bit16Colors:
    	    clp->update_buf[clp->ub_len++] = (streamId | rfbTight4BitFullColor16Colors) << 4;
    	    break;
    	default:
    	    return -1;           /* Should never happen. */
    }

    return CompressData(clp, streamId, dataSize,
                        tightConf[clp->enc.tightCompressLevel].rawZlibLevel,
                        Z_DEFAULT_STRATEGY);
}

static int
SendFullColorRect(rfb_cl_t * clp, int w, int h)
{
    int streamId = 0;
    int len;

    if (clp->ub_len + TIGHT_MIN_TO_COMPRESS + 1 > UPDATE_BUF_SIZE) {
	if (rfb_send_update_buf(clp) == -1) {
            return -1;
	}
    }

    /* stream id = 0, no flushing, no filter */
    clp->update_buf[clp->ub_len++] = 0x00;

    if (clp->usePixelFormat24) {
        Pack24(clp->tightBeforeBuf, &clp->pix_fmt, w * h);
        len = 3;
    } else
        len = clp->pix_fmt.bitsPerPixel_8 / 8;

    return CompressData(clp, streamId, w * h * len,
                        tightConf[clp->enc.tightCompressLevel].rawZlibLevel,
                        Z_DEFAULT_STRATEGY);
}

static int
SendGradientRect(rfb_cl_t * clp, int w, int h)
{
    int streamId = 3;
    int len;

    if (clp->pix_fmt.bitsPerPixel_8 == 8)
        return SendFullColorRect(clp, w, h);

    if (clp->ub_len + TIGHT_MIN_TO_COMPRESS + 2 > UPDATE_BUF_SIZE) {
	if (rfb_send_update_buf(clp) == -1) {
            return -1;
	}
    }

    if (clp->prevRowBuf == NULL)
        clp->prevRowBuf = (int *)malloc(2048 * 3 * sizeof(int));

    clp->update_buf[clp->ub_len++] =
	(streamId | rfbTightExplicitFilter) << 4;
    clp->update_buf[clp->ub_len++] = rfbTightFilterGradient;

    if (clp->usePixelFormat24) {
        FilterGradient24(clp, clp->tightBeforeBuf, w, h);
        len = 3;
    } else if (clp->pix_fmt.bitsPerPixel_8 == 32) {
        FilterGradient32(clp, (u_int32_t *)clp->tightBeforeBuf, w, h);
        len = 4;
    } else {
        FilterGradient16(clp, (u_int16_t *)clp->tightBeforeBuf, w, h);
        len = 2;
    }

    return CompressData(clp, streamId, w * h * len,
                        tightConf[clp->enc.tightCompressLevel].gradientZlibLevel,
                        Z_FILTERED);
}

static int
CompressData(rfb_cl_t * clp, int streamId, int dataLen,
	     int zlibLevel, int zlibStrategy)
{
    z_streamp pz;
    int err;

    if (dataLen < TIGHT_MIN_TO_COMPRESS) {
        memcpy(&clp->update_buf[clp->ub_len], clp->tightBeforeBuf, dataLen);
        clp->ub_len += dataLen;
        return 0;
    }

    pz = &clp->zsStruct[streamId];

    /* Initialize compression stream if needed. */
    if (!clp->zsActive[streamId]) {
        pz->zalloc = Z_NULL;
        pz->zfree = Z_NULL;
        pz->opaque = Z_NULL;

        err = deflateInit2 (pz, zlibLevel, Z_DEFLATED, MAX_WBITS,
                            MAX_MEM_LEVEL, zlibStrategy);
        if (err != Z_OK) {
            return -1;
	}

        clp->zsActive[streamId] = 1;
        clp->zsLevel[streamId] = zlibLevel;
    }

    /* Prepare buffer pointers. */
    pz->next_in = (Bytef *)clp->tightBeforeBuf;
    pz->avail_in = dataLen;
    pz->next_out = (Bytef *)clp->tightAfterBuf;
    pz->avail_out = clp->tightAfterBufSize;

    /* Change compression parameters if needed. */
    if (zlibLevel != clp->zsLevel[streamId]) {
        if (deflateParams (pz, zlibLevel, zlibStrategy) != Z_OK) {
            return -1;
        }
        clp->zsLevel[streamId] = zlibLevel;
    }

    /* Actual compression. */
    if ( deflate (pz, Z_SYNC_FLUSH) != Z_OK ||
         pz->avail_in != 0 || pz->avail_out == 0 ) {
        return -1;
    }

    return SendCompressedData(clp, clp->tightAfterBufSize - pz->avail_out);
}

static u_int8_t
ComputeCachetile(u_int8_t * hash_buf, rfb_cache_lists_t * cache_lists, int tileSize)
{
    u_int8_t ret;
    u_int32_t hash_v_new, hash_v_sdbm, col;
    int8_t client_id;
    rfb_cache_list_value_t * lt_2l;
    rfb_cache_list_value_t * st_2l;
    rfb_cache_list_value_t * hl_2l;

    // calculate current hash values
    hash_v_new = do_hash_new(hash_buf, tileSize);
    hash_v_sdbm = do_hash_sdbm(hash_buf, tileSize);
    col = ((u_int32_t *)hash_buf)[0];

    if ((lt_2l = rfb_search_cache_list_lt(&client_id, cache_lists, RFB_NUM_RECTS_LONG_TERM_CACHE, hash_v_new, hash_v_sdbm, col))) {
	if ((st_2l = rfb_search_cache_list_st(&client_id, cache_lists, RFB_NUM_RECTS_SHORT_TERM_CACHE, hash_v_new, hash_v_sdbm, col, lt_2l))) {
	    if ((hl_2l = rfb_search_cache_list_hl(&client_id, cache_lists, RFB_NUM_RECTS_HASH_LIST, hash_v_new, hash_v_sdbm, col, lt_2l, st_2l))) {
		rfb_insert_short_term(&client_id, cache_lists, hash_v_new, hash_v_sdbm, col, st_2l, hl_2l);
	    }
	    ret = rfbTightTileNot_Cached | client_id;
	}
	else {
	    ret = rfbTightTileCached | client_id;
	}
    }
    else {
	ret = rfbTightTileCached | client_id;
    }

    return ret;
}

static int
SendCacheMetarect(rfb_cl_t * clp, int x, int y, int w, int h, int * dataSize)
{
    int streamId = 3;
    int i, j, k;
    int mc_t;
    int mc_t_cached = 0;
    int mc_t_not_cached = 0;
    int mc_pos = 0;
    int cache_tag;
    int sendRect;
    int tile_dim = 16;
    int tile_size = tile_dim * tile_dim;
    int mc_t_size;
    int pix_red_fac;
    int y_s, h_e;

    u_int8_t * hash_buf, * memcpy_tag;
    u_int8_t * src, * dst;

    if (y % 16) {
    	y_s = ((y / 16 + 1) * 16) - y;
    }
    else {
    	y_s = 0;
    }
    if ((y+h) % 16) {
    	h_e = ((y+h) / 16) * 16 - y;
    }
    else {
    	h_e = h;
    }
	
    mc_t_size = (w/tile_dim) * (h_e/tile_dim);

    hash_buf   = (u_int8_t *)malloc(tile_size);
    memcpy_tag = (u_int8_t *)malloc(mc_t_size);

    switch (clp->enc.tightXBitFullColor) {
    	case rfbTight1BitBlackWhite:
    	    pix_red_fac = 8;
    	    break;
    	case rfbTight2BitGrayscale:
    	    pix_red_fac = 4;
    	    break;
    	case rfbTight4BitGrayscale:
    	case rfbTight4Bit16Colors:
    	    pix_red_fac = 2;
    	    break;
    	case rfbTight8Bit256Colors:
    	    pix_red_fac = 1;
    	    break;
    	default:
    	    return -1;           /* Should never happen. */
    }

    // src -> pixel-data of whole rect
    src = (u_int8_t *)clp->tightBeforeBuf;
    dst = (u_int8_t *)clp->tightXBitFullColorBuf;

    // get cache meta-data
    for (j = y_s; j < h_e; j+=tile_dim) {
	for (i = 0; i < w; i+=tile_dim) {
	    for (k = 0; k < tile_dim; k++) {
		memcpy(&hash_buf[k * (tile_dim/pix_red_fac)], &src[(((j+k)*w) + i)/pix_red_fac], tile_dim/pix_red_fac);
	    }
	    mc_t = ComputeCachetile(hash_buf, clp->cache_lists[(((y+j)/tile_dim)*(clp->fb_width/tile_dim)) + ((x+i)/tile_dim)], tile_size/pix_red_fac);
	    memcpy_tag[((j/tile_dim)*(w/tile_dim)) + (i/tile_dim)] = mc_t;
	    if (mc_t & rfbTightTileCached) mc_t_cached = 1;
	    else  mc_t_not_cached = 1;
	}
    }

    // set correct cache_tag
    if (mc_t_not_cached && !mc_t_cached) cache_tag = rfbTightCacheNone;
    else if (mc_t_cached && !mc_t_not_cached) {
    	if (y_s || h_e!=h) cache_tag = rfbTightCacheMix;
    	else cache_tag = rfbTightCacheAll;
    }
    else cache_tag = rfbTightCacheMix;

    // send cache meta-data
    clp->update_buf[clp->ub_len++] = ((streamId | cache_tag) << 4) | clp->enc.tightXBitFullColor;
    clp->tightBeforeBuf = (char *) memcpy_tag;
    if ((CompressData(clp, streamId, mc_t_size, tightConf[clp->enc.tightCompressLevel].rawZlibLevel, Z_DEFAULT_STRATEGY)) == -1) {
	clp->tightBeforeBuf = (char *) src;
	free(hash_buf);
	free(memcpy_tag);
        return -1;
    }

    if (cache_tag == rfbTightCacheMix) { // some tiles are cached, some not
	if (y_s) {
	    memcpy(dst, src, (y_s*w)/pix_red_fac);
	    mc_pos += (y_s*w)/pix_red_fac;
	}
	// get pixel-data without cached tiles
	for (j = y_s; j < h_e; j+=tile_dim) {
	    for (k = 0; k < tile_dim; k++) {
		for (i = 0; i < w; i+=tile_dim) {
		    if (!(memcpy_tag[((j/tile_dim)*(w/tile_dim)) + (i/tile_dim)] & rfbTightTileCached)) {
			memcpy(&dst[mc_pos], &src[(((j+k)*w) + i)/pix_red_fac], tile_dim/pix_red_fac);
			mc_pos += tile_dim/pix_red_fac;
		    }
		}
	    }
	}
	if (h_e!=h) {
	    memcpy(&dst[mc_pos], &src[(j*w)/pix_red_fac], ((h-h_e)*w)/pix_red_fac);
	    mc_pos += ((h-h_e)*w)/pix_red_fac;
	}
	// set the new datasize
	*dataSize = mc_pos;
	// set the pointers to the right buffers
	clp->tightBeforeBuf = (char *) dst;
	clp->tightXBitFullColorBuf = (char *) src;
	// rect must be send
	sendRect = 1;
    }
    else if  (cache_tag == rfbTightCacheNone) { // no tile is cached
	// set the pointers to the right buffers
	clp->tightBeforeBuf = (char *) src;
	// rect must be send
	sendRect = 1;
    }
    else /* rfbTightCacheAll */ { // all tiles are cached
	// set the pointers to the right buffers
	clp->tightBeforeBuf = (char *) src;
	// rect must not be send
	sendRect = 0;
    }

    free(hash_buf);
    free(memcpy_tag);

    return sendRect;
}

static int
SendCompressedData(rfb_cl_t * clp, int compressedLen)
{
    int i, portionLen;

    clp->update_buf[clp->ub_len++] = compressedLen & 0x7F;
    if (compressedLen > 0x7F) {
        clp->update_buf[clp->ub_len-1] |= 0x80;
        clp->update_buf[clp->ub_len++] = compressedLen >> 7 & 0x7F;
        if (compressedLen > 0x3FFF) {
            clp->update_buf[clp->ub_len-1] |= 0x80;
            clp->update_buf[clp->ub_len++] = compressedLen >> 14 & 0xFF;
        }
    }

    portionLen = UPDATE_BUF_SIZE;
    for (i = 0; i < compressedLen; i += portionLen) {
        if (i + portionLen > compressedLen) {
            portionLen = compressedLen - i;
        }
        if (clp->ub_len + portionLen > UPDATE_BUF_SIZE) {
	    if (rfb_send_update_buf(clp) == -1) {
                return -1;
	    }
        }
        memcpy(&clp->update_buf[clp->ub_len], &clp->tightAfterBuf[i], portionLen);
        clp->ub_len += portionLen;
    }
    return 0;
}

/*
 * Code to determine how many different colors used in rectangle.
 */

static void
FillPalette8(rfb_cl_t * clp, int count)
{
    u_int8_t *data = (u_int8_t *)clp->tightBeforeBuf;
    u_int8_t c0, c1;
    int i, n0, n1;

    clp->paletteNumColors = 0;

    c0 = data[0];
    for (i = 1; i < count && data[i] == c0; i++);
    if (i == count) {
        clp->paletteNumColors = 1;
        return;                 /* Solid rectangle */
    }

    if (clp->paletteMaxColors < 2)
        return;

    n0 = i;
    c1 = data[i];
    n1 = 0;
    for (i++; i < count; i++) {
        if (data[i] == c0) {
            n0++;
        } else if (data[i] == c1) {
            n1++;
        } else
            break;
    }
    if (i == count) {
        if (n0 > n1) {
            clp->monoBackground = (u_int32_t)c0;
            clp->monoForeground = (u_int32_t)c1;
        } else {
            clp->monoBackground = (u_int32_t)c1;
            clp->monoForeground = (u_int32_t)c0;
        }
        clp->paletteNumColors = 2;   /* Two colors */
    }
}

#define DEFINE_FILL_PALETTE_FUNCTION(bpp)                               \
                                                                        \
static void                                                             \
FillPalette##bpp(rfb_cl_t * clp, int count)                             \
{                                                                       \
    u_int##bpp##_t *data = (u_int##bpp##_t *)clp->tightBeforeBuf;       \
    u_int##bpp##_t c0, c1, ci = 0;                                      \
    int i, n0, n1, ni;                                                  \
                                                                        \
    c0 = data[0];                                                       \
    for (i = 1; i < count && data[i] == c0; i++);                       \
    if (i >= count) {                                                   \
        clp->paletteNumColors = 1;   /* Solid rectangle */              \
        return;                                                         \
    }                                                                   \
                                                                        \
    if (clp->paletteMaxColors < 2) {                                    \
        clp->paletteNumColors = 0;   /* Full-color encoding preferred */\
        return;                                                         \
    }                                                                   \
                                                                        \
    n0 = i;                                                             \
    c1 = data[i];                                                       \
    n1 = 0;                                                             \
    for (i++; i < count; i++) {                                         \
        ci = data[i];                                                   \
        if (ci == c0) {                                                 \
            n0++;                                                       \
        } else if (ci == c1) {                                          \
            n1++;                                                       \
        } else                                                          \
            break;                                                      \
    }                                                                   \
    if (i >= count) {                                                   \
        if (n0 > n1) {                                                  \
            clp->monoBackground = (u_int32_t)c0;                        \
            clp->monoForeground = (u_int32_t)c1;                        \
        } else {                                                        \
            clp->monoBackground = (u_int32_t)c1;                        \
            clp->monoForeground = (u_int32_t)c0;                        \
        }                                                               \
        clp->paletteNumColors = 2;   /* Two colors */                   \
        return;                                                         \
    }                                                                   \
                                                                        \
    PaletteReset(clp);                                                  \
    PaletteInsert(clp, c0, (u_int32_t)n0, bpp);                         \
    PaletteInsert(clp, c1, (u_int32_t)n1, bpp);                         \
                                                                        \
    ni = 1;                                                             \
    for (i++; i < count; i++) {                                         \
        if (data[i] == ci) {                                            \
            ni++;                                                       \
        } else {                                                        \
            if (!PaletteInsert (clp, ci, (u_int32_t)ni, bpp))           \
                return;                                                 \
            ci = data[i];                                               \
            ni = 1;                                                     \
        }                                                               \
    }                                                                   \
    PaletteInsert(clp, ci, (u_int32_t)ni, bpp);                         \
}

DEFINE_FILL_PALETTE_FUNCTION(16)
#if ENABLE_24_AND_32_BIT
DEFINE_FILL_PALETTE_FUNCTION(32)
#endif /* ENABLE_24_AND_32_BIT */

    /*
     * Functions to operate with palette structures.
     */

#define HASH_FUNC16(rgb) ((int)((((rgb) >> 8) + (rgb)) & 0xFF))
#define HASH_FUNC32(rgb) ((int)((((rgb) >> 16) + ((rgb) >> 8)) & 0xFF))

static void
PaletteReset(rfb_cl_t * clp)
{
    clp->paletteNumColors = 0;
    memset(clp->palette.hash, 0, 256 * sizeof(COLOR_LIST *));
}

static int
PaletteInsert(rfb_cl_t * clp, u_int32_t rgb, int numPixels, int bpp)
{
    COLOR_LIST *pnode;
    COLOR_LIST *prev_pnode = NULL;
    int hash_key, idx, new_idx, count;

    hash_key = (bpp == 16) ? HASH_FUNC16(rgb) : HASH_FUNC32(rgb);

    pnode = clp->palette.hash[hash_key];

    while (pnode != NULL) {
        if (pnode->rgb == rgb) {
            /* Such palette entry already exists. */
            new_idx = idx = pnode->idx;
            count = clp->palette.entry[idx].numPixels + numPixels;
            if (new_idx && clp->palette.entry[new_idx-1].numPixels < count) {
                do {
                    clp->palette.entry[new_idx] = clp->palette.entry[new_idx-1];
                    clp->palette.entry[new_idx].listNode->idx = new_idx;
                    new_idx--;
                }
                while (new_idx && clp->palette.entry[new_idx-1].numPixels < count);
                clp->palette.entry[new_idx].listNode = pnode;
                pnode->idx = new_idx;
            }
            clp->palette.entry[new_idx].numPixels = count;
            return clp->paletteNumColors;
        }
        prev_pnode = pnode;
        pnode = pnode->next;
    }

    /* Check if palette is full. */
    if (clp->paletteNumColors == 256 ||
	clp->paletteNumColors == clp->paletteMaxColors) {
        clp->paletteNumColors = 0;
        return 0;
    }

    /* Move palette entries with lesser pixel counts. */
    for ( idx = clp->paletteNumColors;
          idx > 0 && clp->palette.entry[idx-1].numPixels < numPixels;
          idx-- ) {
        clp->palette.entry[idx] = clp->palette.entry[idx-1];
        clp->palette.entry[idx].listNode->idx = idx;
    }

    /* Add new palette entry into the freed slot. */
    pnode = &clp->palette.list[clp->paletteNumColors];
    if (prev_pnode != NULL) {
        prev_pnode->next = pnode;
    } else {
        clp->palette.hash[hash_key] = pnode;
    }
    pnode->next = NULL;
    pnode->idx = idx;
    pnode->rgb = rgb;
    clp->palette.entry[idx].listNode = pnode;
    clp->palette.entry[idx].numPixels = numPixels;

    return (++(clp->paletteNumColors));
}

#if ENABLE_24_AND_32_BIT

/*
 * Converting 32-bit color samples into 24-bit colors.
 * Should be called only when redMax, greenMax and blueMax are 255.
 * Color components assumed to be byte-aligned.
 */

static void
Pack24(char * buf, rfb_pixel_format_t * fmt, int count)
{
    u_int32_t * buf32;
    u_int32_t pix;
    int r_shift, g_shift, b_shift;

    buf32 = (u_int32_t *)buf;

    if (!rfb_pix_fmt.bigEndian_8 == !fmt->bigEndian_8) {
        r_shift = fmt->redShift_8;
        g_shift = fmt->greenShift_8;
        b_shift = fmt->blueShift_8;
    } else {
        r_shift = 24 - fmt->redShift_8;
        g_shift = 24 - fmt->greenShift_8;
        b_shift = 24 - fmt->blueShift_8;
    }

    while (count--) {
        pix = *buf32++;
        *buf++ = (char)(pix >> r_shift);
        *buf++ = (char)(pix >> g_shift);
        *buf++ = (char)(pix >> b_shift);
    }
}

#endif /* ENABLE_24_AND_32_BIT */

/*
 * Converting truecolor samples into palette indices.
 */

#define DEFINE_IDX_ENCODE_FUNCTION(bpp)                                 \
                                                                        \
static void                                                             \
EncodeIndexedRect##bpp(rfb_cl_t * clp, u_int8_t * buf, int count)       \
{                                                                       \
    COLOR_LIST *pnode;                                                  \
    u_int##bpp##_t *src;                                                \
    u_int##bpp##_t rgb;                                                 \
    int rep = 0;                                                        \
                                                                        \
    src = (u_int##bpp##_t *) buf;                                       \
                                                                        \
    while (count--) {                                                   \
        rgb = *src++;                                                   \
        while (count && *src == rgb) {                                  \
            rep++, src++, count--;                                      \
        }                                                               \
        pnode = clp->palette.hash[HASH_FUNC##bpp(rgb)];                 \
        while (pnode != NULL) {                                         \
            if ((u_int##bpp##_t)pnode->rgb == rgb) {                    \
                *buf++ = (u_int8_t)pnode->idx;                          \
                while (rep) {                                           \
                    *buf++ = (u_int8_t)pnode->idx;                      \
                    rep--;                                              \
                }                                                       \
                break;                                                  \
            }                                                           \
            pnode = pnode->next;                                        \
        }                                                               \
    }                                                                   \
}

DEFINE_IDX_ENCODE_FUNCTION(16)
#if ENABLE_24_AND_32_BIT
DEFINE_IDX_ENCODE_FUNCTION(32)
#endif /* ENABLE_24_AND_32_BIT */

#define DEFINE_MONO_ENCODE_FUNCTION(bpp)                                \
                                                                        \
static void                                                             \
EncodeMonoRect##bpp(rfb_cl_t * clp, u_int8_t * buf, int w, int h)       \
{                                                                       \
    u_int##bpp##_t *ptr;                                                \
    u_int##bpp##_t bg;                                                  \
    unsigned int value, mask;                                           \
    int aligned_width;                                                  \
    int x, y, bg_bits;                                                  \
                                                                        \
    ptr = (u_int##bpp##_t *) buf;                                       \
    bg = (u_int##bpp##_t) clp->monoBackground;                          \
    aligned_width = w - w % 8;                                          \
                                                                        \
    for (y = 0; y < h; y++) {                                           \
        for (x = 0; x < aligned_width; x += 8) {                        \
            for (bg_bits = 0; bg_bits < 8; bg_bits++) {                 \
                if (*ptr++ != bg)                                       \
                    break;                                              \
            }                                                           \
            if (bg_bits == 8) {                                         \
                *buf++ = 0;                                             \
                continue;                                               \
            }                                                           \
            mask = 0x80 >> bg_bits;                                     \
            value = mask;                                               \
            for (bg_bits++; bg_bits < 8; bg_bits++) {                   \
                mask >>= 1;                                             \
                if (*ptr++ != bg) {                                     \
                    value |= mask;                                      \
                }                                                       \
            }                                                           \
            *buf++ = (u_int8_t)value;                                   \
        }                                                               \
                                                                        \
        mask = 0x80;                                                    \
        value = 0;                                                      \
        if (x >= w)                                                     \
            continue;                                                   \
                                                                        \
        for (; x < w; x++) {                                            \
            if (*ptr++ != bg) {                                         \
                value |= mask;                                          \
            }                                                           \
            mask >>= 1;                                                 \
        }                                                               \
        *buf++ = (u_int8_t)value;                                       \
    }                                                                   \
}

DEFINE_MONO_ENCODE_FUNCTION(8)
DEFINE_MONO_ENCODE_FUNCTION(16)
#if ENABLE_24_AND_32_BIT
DEFINE_MONO_ENCODE_FUNCTION(32)
#endif /* ENABLE_24_AND_32_BIT */

#if ENABLE_24_AND_32_BIT

/*
 * ``Gradient'' filter for 24-bit color samples.
 * Should be called only when redMax, greenMax and blueMax are 255.
 * Color components assumed to be byte-aligned.
 */

static void
FilterGradient24(rfb_cl_t * clp, char * buf, int w, int h)
{
    u_int32_t *buf32;
    u_int32_t pix32;
    int *prevRowPtr;
    int shiftBits[3];
    int pixHere[3], pixUpper[3], pixLeft[3], pixUpperLeft[3];
    int prediction;
    int x, y, c;

    buf32 = (u_int32_t *)buf;
    memset (clp->prevRowBuf, 0, w * 3 * sizeof(int));

    if (!rfb_pix_fmt.bigEndian_8 == !clp->pix_fmt.bigEndian_8) {
        shiftBits[0] = clp->pix_fmt.redShift_8;
        shiftBits[1] = clp->pix_fmt.greenShift_8;
        shiftBits[2] = clp->pix_fmt.blueShift_8;
    } else {
        shiftBits[0] = 24 - clp->pix_fmt.redShift_8;
        shiftBits[1] = 24 - clp->pix_fmt.greenShift_8;
        shiftBits[2] = 24 - clp->pix_fmt.blueShift_8;
    }

    for (y = 0; y < h; y++) {
        for (c = 0; c < 3; c++) {
            pixUpper[c] = 0;
            pixHere[c] = 0;
        }
        prevRowPtr = clp->prevRowBuf;
        for (x = 0; x < w; x++) {
            pix32 = *buf32++;
            for (c = 0; c < 3; c++) {
                pixUpperLeft[c] = pixUpper[c];
                pixLeft[c] = pixHere[c];
                pixUpper[c] = *prevRowPtr;
                pixHere[c] = (int)(pix32 >> shiftBits[c] & 0xFF);
                *prevRowPtr++ = pixHere[c];

                prediction = pixLeft[c] + pixUpper[c] - pixUpperLeft[c];
                if (prediction < 0) {
                    prediction = 0;
                } else if (prediction > 0xFF) {
                    prediction = 0xFF;
                }
                *buf++ = (char)(pixHere[c] - prediction);
            }
        }
    }
}

#endif /* ENABLE_24_AND_32_BIT */

/*
 * ``Gradient'' filter for other color depths.
 */

#define DEFINE_GRADIENT_FILTER_FUNCTION(bpp)                             \
                                                                         \
static void                                                              \
FilterGradient##bpp(rfb_cl_t * clp, u_int##bpp##_t *buf,                 \
		    int w, int h)                                        \
{                                                                        \
    u_int##bpp##_t pix, diff;                                            \
    int endianMismatch;                                                  \
    int *prevRowPtr;                                                     \
    int maxColor[3], shiftBits[3];                                       \
    int pixHere[3], pixUpper[3], pixLeft[3], pixUpperLeft[3];            \
    int prediction;                                                      \
    int x, y, c;                                                         \
                                                                         \
    memset (clp->prevRowBuf, 0, w * 3 * sizeof(int));                    \
                                                                         \
    endianMismatch = (!rfb_pix_fmt.bigEndian_8 != !clp->pix_fmt.bigEndian_8);\
                                                                         \
    maxColor[0] = clp->pix_fmt.redMax_be16;                              \
    maxColor[1] = clp->pix_fmt.greenMax_be16;                            \
    maxColor[2] = clp->pix_fmt.blueMax_be16;                             \
    shiftBits[0] = clp->pix_fmt.redShift_8;                              \
    shiftBits[1] = clp->pix_fmt.greenShift_8;                            \
    shiftBits[2] = clp->pix_fmt.blueShift_8;                             \
                                                                         \
    for (y = 0; y < h; y++) {                                            \
        for (c = 0; c < 3; c++) {                                        \
            pixUpper[c] = 0;                                             \
            pixHere[c] = 0;                                              \
        }                                                                \
        prevRowPtr = clp->prevRowBuf;                                    \
        for (x = 0; x < w; x++) {                                        \
            pix = *buf;                                                  \
            if (endianMismatch) {                                        \
                pix = bswap_##bpp(pix);                                  \
            }                                                            \
            diff = 0;                                                    \
            for (c = 0; c < 3; c++) {                                    \
                pixUpperLeft[c] = pixUpper[c];                           \
                pixLeft[c] = pixHere[c];                                 \
                pixUpper[c] = *prevRowPtr;                               \
                pixHere[c] = (int)(pix >> shiftBits[c] & maxColor[c]);   \
                *prevRowPtr++ = pixHere[c];                              \
                                                                         \
                prediction = pixLeft[c] + pixUpper[c] - pixUpperLeft[c]; \
                if (prediction < 0) {                                    \
                    prediction = 0;                                      \
                } else if (prediction > maxColor[c]) {                   \
                    prediction = maxColor[c];                            \
                }                                                        \
                diff |= ((pixHere[c] - prediction) & maxColor[c])        \
                    << shiftBits[c];                                     \
            }                                                            \
            if (endianMismatch) {                                        \
                diff = bswap_##bpp(diff);                                \
            }                                                            \
            *buf++ = diff;                                               \
        }                                                                \
    }                                                                    \
}

DEFINE_GRADIENT_FILTER_FUNCTION(16)
#if ENABLE_24_AND_32_BIT
DEFINE_GRADIENT_FILTER_FUNCTION(32)
#endif /* ENABLE_24_AND_32_BIT */


    /*
     * Code to guess if given rectangle is suitable for smooth image
     * compression (by applying "gradient" filter).
     */

#define DETECT_SUBROW_WIDTH    7
#define DETECT_MIN_WIDTH       8
#define DETECT_MIN_HEIGHT      8

static int
DetectSmoothImage(rfb_cl_t * clp, int w, int h)
{
    unsigned long avgError;

    if ( rfb_pix_fmt.bitsPerPixel_8 == 8 || clp->pix_fmt.bitsPerPixel_8 == 8 ||
         w < DETECT_MIN_WIDTH || h < DETECT_MIN_HEIGHT ) {
        return 0;
    }

    if ( rfb_context.tight_disable_gradient ||
         w * h < tightConf[clp->enc.tightCompressLevel].gradientMinRectSize ) {
        return 0;
    }

#if ENABLE_24_AND_32_BIT
    if (clp->pix_fmt.bitsPerPixel_8 == 32) {
        if (clp->usePixelFormat24) {
            avgError = DetectSmoothImage24(clp, w, h);
            return (avgError < tightConf[clp->enc.tightCompressLevel].gradientThreshold24);
        } else {
            avgError = DetectSmoothImage32(clp, w, h);
        }
    } else {
        avgError = DetectSmoothImage16(clp, w, h);
    }
#else /* ENABLE_24_AND_32_BIT */
    if (clp->pix_fmt.bitsPerPixel_8 == 32) {
        ERROR_24_AND_32_BIT;
    } else {
        avgError = DetectSmoothImage16(clp, w, h);
    }
#endif /* ENABLE_24_AND_32_BIT */
    return (avgError < tightConf[clp->enc.tightCompressLevel].gradientThreshold);
}

#if ENABLE_24_AND_32_BIT

static unsigned long
DetectSmoothImage24 (rfb_cl_t * clp, int w, int h)
{
    int off;
    int x, y, d, dx, c;
    int diffStat[256];
    int pixelCount = 0;
    int pix, left[3];
    unsigned long avgError;

    /* If client is big-endian, color samples begin from the second
       byte (offset 1) of a 32-bit pixel value. */
    off = (clp->pix_fmt.bigEndian_8 != 0);

    memset(diffStat, 0, 256*sizeof(int));

    y = 0, x = 0;
    while (y < h && x < w) {
        for (d = 0; d < h - y && d < w - x - DETECT_SUBROW_WIDTH; d++) {
            for (c = 0; c < 3; c++) {
                left[c] = (int)clp->tightBeforeBuf[((y+d)*w+x+d)*4+off+c] & 0xFF;
            }
            for (dx = 1; dx <= DETECT_SUBROW_WIDTH; dx++) {
                for (c = 0; c < 3; c++) {
                    pix = (int)clp->tightBeforeBuf[((y+d)*w+x+d+dx)*4+off+c] & 0xFF;
                    diffStat[abs(pix - left[c])]++;
                    left[c] = pix;
                }
                pixelCount++;
            }
        }
        if (w > h) {
            x += h;
            y = 0;
        } else {
            x = 0;
            y += w;
        }
    }

    if (diffStat[0] * 33 / pixelCount >= 95)
        return 0;

    avgError = 0;
    for (c = 1; c < 8; c++) {
        avgError += (unsigned long)diffStat[c] * (unsigned long)(c * c);
        if (diffStat[c] == 0 || diffStat[c] > diffStat[c-1] * 2)
            return 0;
    }
    for (; c < 256; c++) {
        avgError += (unsigned long)diffStat[c] * (unsigned long)(c * c);
    }
    avgError /= (pixelCount * 3 - diffStat[0]);

    return avgError;
}

#endif /* ENABLE_24_AND_32_BIT */

#define DEFINE_DETECT_FUNCTION(bpp)                                          \
                                                                             \
static unsigned long                                                         \
DetectSmoothImage##bpp (rfb_cl_t * clp, int w, int h)                        \
{                                                                            \
    int endianMismatch;                                                      \
    u_int##bpp##_t pix;                                                      \
    int maxColor[3], shiftBits[3];                                           \
    int x, y, d, dx, c;                                                      \
    int diffStat[256];                                                       \
    int pixelCount = 0;                                                      \
    int sample, sum, left[3];                                                \
    unsigned long avgError;                                                  \
                                                                             \
    endianMismatch = (!rfb_pix_fmt.bigEndian_8 != !clp->pix_fmt.bigEndian_8);  \
                                                                             \
    maxColor[0] = clp->pix_fmt.redMax_be16;                                  \
    maxColor[1] = clp->pix_fmt.greenMax_be16;                                \
    maxColor[2] = clp->pix_fmt.blueMax_be16;                                 \
    shiftBits[0] = clp->pix_fmt.redShift_8;                                  \
    shiftBits[1] = clp->pix_fmt.greenShift_8;                                \
    shiftBits[2] = clp->pix_fmt.blueShift_8;                                 \
                                                                             \
    memset(diffStat, 0, 256*sizeof(int));                                    \
                                                                             \
    y = 0, x = 0;                                                            \
    while (y < h && x < w) {                                                 \
        for (d = 0; d < h - y && d < w - x - DETECT_SUBROW_WIDTH; d++) {     \
            pix = ((u_int##bpp##_t *)clp->tightBeforeBuf)[(y+d)*w+x+d];      \
            if (endianMismatch) {                                            \
                pix = bswap_##bpp(pix);                                      \
            }                                                                \
            for (c = 0; c < 3; c++) {                                        \
                left[c] = (int)(pix >> shiftBits[c] & maxColor[c]);          \
            }                                                                \
            for (dx = 1; dx <= DETECT_SUBROW_WIDTH; dx++) {                  \
                pix = ((u_int##bpp##_t *)clp->tightBeforeBuf)[(y+d)*w+x+d+dx];\
                if (endianMismatch) {                                        \
                    pix = bswap_##bpp(pix);                                  \
                }                                                            \
                sum = 0;                                                     \
                for (c = 0; c < 3; c++) {                                    \
                    sample = (int)(pix >> shiftBits[c] & maxColor[c]);       \
                    sum += abs(sample - left[c]);                            \
                    left[c] = sample;                                        \
                }                                                            \
                if (sum > 255)                                               \
                    sum = 255;                                               \
                diffStat[sum]++;                                             \
                pixelCount++;                                                \
            }                                                                \
        }                                                                    \
        if (w > h) {                                                         \
            x += h;                                                          \
            y = 0;                                                           \
        } else {                                                             \
            x = 0;                                                           \
            y += w;                                                          \
        }                                                                    \
    }                                                                        \
                                                                             \
    if ((diffStat[0] + diffStat[1]) * 100 / pixelCount >= 90)                \
        return 0;                                                            \
                                                                             \
    avgError = 0;                                                            \
    for (c = 1; c < 8; c++) {                                                \
        avgError += (unsigned long)diffStat[c] * (unsigned long)(c * c);     \
        if (diffStat[c] == 0 || diffStat[c] > diffStat[c-1] * 2)             \
            return 0;                                                        \
    }                                                                        \
    for (; c < 256; c++) {                                                   \
        avgError += (unsigned long)diffStat[c] * (unsigned long)(c * c);     \
    }                                                                        \
    avgError /= (pixelCount - diffStat[0]);                                  \
                                                                             \
    return avgError;                                                         \
}

DEFINE_DETECT_FUNCTION(16)
#if ENABLE_24_AND_32_BIT
DEFINE_DETECT_FUNCTION(32)
#endif /* ENABLE_24_AND_32_BIT */
