#include <predictor.h>
#include <sys/time.h>
#include "debug.h"

#define PP_KEEP_ALIVE_TIMEOUT 30

static void rfb_writer_cleanup_handler(void * _clp);

void *
rfb_writer(void * _clp)
{
    rfb_cl_t * clp = (rfb_cl_t *)_clp;
    pp_queue_entry_t * qe;
    int fb_resolution_changed = 0;
    int fb_update_requested = 0;
    int need_fb_fmt_update = 0;
    int need_fb_update = 0;
    int full_fb_update = 0;
    int is_first_update = 1;

    pthread_cleanup_push(rfb_writer_cleanup_handler, clp);

    while (1) {
	qe = NULL;
	MUTEX_LOCK(&clp->s2c_queue_mtx);
	/*
	 * rfb_writer_dequeue_pdu() must be called before checking
	 * clp->need_update since it has priority over client updates
	 */
	while(!clp->do_exit) {
	    int retcode;
	    struct timeval now;
	    struct timespec timeout;
	    
	    fb_update_requested |= clp->fb_update_requested;
	    need_fb_fmt_update |= clp->need_fb_fmt_update;
	    need_fb_update |= clp->need_fb_update;
	    full_fb_update |= clp->full_fb_update;

	    if ((qe = pp_queue_dequeue(clp->s2c_queue)) != NULL){
		break;
	    }

	    MUTEX_LOCK(&clp->fbu_possible_mtx);
	    if (clp->fbu_possible_release) {
		clp->fbu_possible_release = 0;
		clp->fbu_possible = 1;
	    }
	    MUTEX_UNLOCK(&clp->fbu_possible_mtx);
		
	    if (clp->init_handshake_done) {
		if (need_fb_fmt_update ||
		    (clp->fbu_possible && fb_update_requested && need_fb_update) ||
		    clp->vs_update_now) {
		    clp->fb_update_requested = 0;
		    clp->need_fb_fmt_update = 0;
		    clp->need_fb_update = 0;
		    clp->full_fb_update = 0;
		    break;
		}
	    }

	    gettimeofday(&now, NULL);
	    timeout.tv_sec = now.tv_sec + PP_KEEP_ALIVE_TIMEOUT;
	    timeout.tv_nsec = now.tv_usec * 1000;
	    retcode = pthread_cond_timedwait(&clp->s2c_queue_cond,
					     &clp->s2c_queue_mtx,
					     &timeout);
	    if (retcode == ETIMEDOUT) {
		/* timeout occurred */
		rfb_enqueue_ping_request(clp, 0);
	    }
	}

	MUTEX_UNLOCK(&clp->s2c_queue_mtx);	
	if (clp->do_exit) { /* exit */
	    if (qe != NULL && qe->len > 0) {
		free(qe->data);
	    }
	    free(qe);
	    break;
	} else if (qe != NULL) { /* send a pdu */
	    if (qe->data != NULL) {
		if (rfb_write_writer(clp, qe->data, qe->len) == -1) {
		    pp_log("%s(): Sending PDU failed. Closing connection.\n", ___F);
		    if (qe->len > 0) {
			free(qe->data);
		    }
		    free(qe);
		    break;
		}
	    }
	    if (qe->len > 0) {
		free(qe->data);
	    }
	    free(qe);
	} else {
	    if (need_fb_fmt_update) {
		fb_format_info_t fb_f_info;
		pp_grab_get_format_info(clp->video_link, &fb_f_info);
		if (clp->fb_width != fb_f_info.g_w) {
		    fb_resolution_changed = 1;
		    clp->fb_width = fb_f_info.g_w;
		    clp->fb_width_pd = fb_f_info.g_w_pd;
		    clp->fb_tiles_w = clp->fb_tile_pitch = fb_f_info.tiles_w;
		}
		if (clp->fb_height != fb_f_info.g_h) {
		    fb_resolution_changed = 1;
		    clp->fb_height = fb_f_info.g_h;
		    clp->fb_height_pd = fb_f_info.g_h_pd;
		    clp->fb_tiles_h = fb_f_info.tiles_h;
		}

		clp->fb_bpp = fb_f_info.bpp;
		clp->fb_is_unsupported = fb_f_info.is_unsupported;
	    }

	    if (clp->fb_width != 0 && clp->fb_height != 0) {
		/* send frame buffer format update if necessary */
		if (need_fb_fmt_update &&
		    rfb_send_fb_format_update(clp) == -1) {
		    pp_log("%s(): Sending FB format PDU failed. Closing connection.\n", ___F);
		    break;
		}
		need_fb_fmt_update = 0;
		
		/* send frame buffer update if requested and necessary */
		if (fb_update_requested && need_fb_update) {
		    if (rfb_send_fb_update(clp, fb_resolution_changed || is_first_update, full_fb_update) == -1) {
			pp_log("%s(): Sending FB update PDU failed. Closing connection.\n", ___F);
			break;
		    }
		    fb_resolution_changed = 0;
		    need_fb_update = full_fb_update = 0;
		    fb_update_requested = 0;
		    is_first_update = 0;
		}
	    }

	    if (clp->pred_bwidth_state == PRED_BWIDTH_STATE_REQ) {
		rfb_send_bandwidth_request(clp, 16384);
		clp->pred_bwidth_state = PRED_BWIDTH_STATE_MEASURING;
	    }
	    if (clp->vs_update_now) {
		rfb_send_vs_update(clp);
		clp->vs_update_now = 0;
	    }
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
	    if (clp->vq_update_now) {
		rfb_send_vq_update(clp);
		clp->vq_update_now = 0;
	    }
#endif /* !PP_FEAT_VSC_PANEL_INPUT */
	}
    }

    D(D_VERBOSE, "rfb writer cleanup\n");
    pthread_cleanup_pop(1);
    pthread_exit(NULL);
}

static void
rfb_writer_cleanup_handler(void * _clp)
{
    rfb_cl_t * clp = (rfb_cl_t *)_clp;

    clp->do_exit = 1;
}

int
rfb_writer_enqueue_pdu(rfb_cl_t * clp, void * data, size_t data_len)
{
    int ret = 0;

    MUTEX_LOCK(&clp->s2c_queue_mtx);

    if (pp_queue_enqueue(clp->s2c_queue, data, data_len, 0) == -1) {
	ret = -1;
    }

    if (pthread_cond_signal(&clp->s2c_queue_cond) != 0) {
	pp_log("%s(): pthread_cond_signal() failed\n", ___F);
    }

    MUTEX_UNLOCK(&clp->s2c_queue_mtx);

    return ret;
}
