#include <fcntl.h>
#include <termios.h>
#include <pp/cfg.h>
#include <pp/serial.h>
#include "serial_internal.h"
#include "debug.h"

static int serial_reconf_ch(pp_cfg_chg_ctx_t * ctx);

void
serial_reconf_init(void)
{
    /* NOTE(miba): using the post_tx version here because some power switch may
       want to set and save config options in its (un)init routine (ipm_tty), 
       which would hang in the middle of a transaction */

    pp_cfg_add_post_tx_change_listener(serial_reconf_ch, "serialport[?]");
    pp_cfg_add_post_tx_change_listener(serial_reconf_ch, "ps.serial[?]");
}

void
serial_reconf_cleanup(void)
{
    pp_cfg_rem_change_listener(serial_reconf_ch, "serialport[?]");
    pp_cfg_rem_change_listener(serial_reconf_ch, "ps.serial[?]");
}

int
pp_serial_reconfigure(char ** errstr_p)
{
    serial_port_t * sp;
    struct termios tty_data; 
    pp_strstream_t errstream;
    int ret = PP_SUC;
    int newfd, n;
    int trigger_gettyswitch = 0;

    pp_strstream_init(&errstream);

    MUTEX_LOCK(&serial_lock);
    for (sp = (serial_port_t *)pp_hash_get_first_entry(serial_ports_hash);
	 sp != NULL;
	 sp = (serial_port_t *)pp_hash_get_next_entry(serial_ports_hash)) {
	if (sp->need_reconfigure) {
	    pp_serial_uninit_func_t uninit_func;
	    pp_serial_uninit_func_t init_func;
	    char * new_usage_str;
	    pp_serial_usage_t new_usage;
	    char * errstr;

	    D(D_VERBOSE, "Reconfigure serial port %u\n", sp->id);
	    trigger_gettyswitch = 1;

	    pp_cfg_get(&new_usage_str, "serialport[%u]._c_", sp->id);
	    new_usage = str_to_usage(new_usage_str);
	    free(new_usage_str);
	    uninit_func = serial_port_callbacks[sp->usage].uninit;
	    init_func = serial_port_callbacks[new_usage].init;

	    if (uninit_func) {
		if (PP_FAILED(uninit_func(sp->device_path, sp->id, sp->fd, &errstr))) {
		    if (errstr) {
			if (pp_strstream_pos(&errstream) > 0) pp_strappendf(&errstream, "<br><br>");
			pp_strappendf(&errstream, errstr);
			free(errstr);
		    }
		    ret = PP_ERR;
		    continue;
		}
	    }

	    /* reopen terminal device (getty does weird things to it) */
	    if ((newfd = open(sp->device_path, O_RDWR | O_NDELAY)) < 0
		|| (n = fcntl(newfd, F_GETFL, 0)) < 0
		|| fcntl(newfd, F_SETFL, n & ~O_NDELAY) < 0) {
		pp_log_err("%s(): open() or fcntl() failed", ___F);
		ret = PP_ERR;
	    } else {
		dup2(newfd, sp->fd);
	    }
	    close(newfd);

	    if ((sp->usage != new_usage) && (tcgetattr(sp->fd, &tty_data) == 0)) {	
		/* if the usage mode has changed, we might want to switch the operation mode */
                if (sp->usage == PP_SERIAL_USAGE_DISABLED) {
		    /* we do not want to echo input on disabled serial ports, so switch off canonical mode and echo */
		    tty_data.c_lflag &= ~(ICANON | ECHO);
	        } else {
		    /* canonical mode and echo are the default */
		    tty_data.c_lflag |= (ICANON | ECHO);
	        }
		tcsetattr(sp->fd, TCSANOW, &tty_data);
	    }
	    sp->usage = new_usage;

	    sp->need_reconfigure = 0;

	    if (init_func) {
		if (PP_FAILED(init_func(sp->device_path, sp->id, sp->fd, &errstr))) {
		    if (errstr) {
			if (pp_strstream_pos(&errstream) > 0) pp_strappendf(&errstream, "<br><br>");
			pp_strappendf(&errstream, errstr);
			free(errstr);
		    }
		    ret = PP_ERR;
		}
	    }
	}
    }

    /* trigger getty_switch.sh restart */
    if (trigger_gettyswitch) {
	pp_system("killall getty_switch.sh.sleep >/dev/null 2>&1");
    }

    MUTEX_UNLOCK(&serial_lock);

    if (pp_strstream_pos(&errstream) && errstr_p) {
        *errstr_p = pp_strstream_buf(&errstream);
    } else {
        pp_strstream_free(&errstream);
    }

    return ret;
}

static int
serial_reconf_ch(pp_cfg_chg_ctx_t * ctx)
{
    serial_port_t * sp;
    size_t key_comp_vec_sz, i;
    int ret;

    D(D_VERBOSE, "Reconfigure serial subsystem\n");
    
    MUTEX_LOCK(&serial_lock);

    /* mark serial ports with changed configuration */
    key_comp_vec_sz = vector_size(ctx->key_comp_vec);
    for (i = 0; i < key_comp_vec_sz; ++i) {
        vector_t * key_comps = (vector_t *)vector_get(ctx->key_comp_vec, i);
	const char *key_comp_0, *key_comp_1, *key_comp_2, *key_comp_3;
	int sp_id;

	if (key_comps == NULL || vector_size(key_comps) < 1) continue;
        
	key_comp_0 = pp_cfg_get_key_comp_name_at_idx(key_comps, 0);

	if (!strcmp("serialport", key_comp_0) && vector_size(key_comps) >= 3) {
	    key_comp_1 = pp_cfg_get_key_comp_name_at_idx(key_comps, 1);
	    key_comp_2 = pp_cfg_get_key_comp_name_at_idx(key_comps, 2);
	    if (!strcmp(pp_cfg_vect_elems_comp, key_comp_1)) {
		sp_id = pp_strtol_10(key_comp_2, -1, NULL);
		sp = (serial_port_t *)pp_hash_get_entry_i(serial_ports_hash, sp_id);
		if (sp) sp->need_reconfigure = 1;
	    }
	} else if (!strcmp("ps", key_comp_0) && vector_size(key_comps) >= 4) {
	    key_comp_1 = pp_cfg_get_key_comp_name_at_idx(key_comps, 1);
	    key_comp_2 = pp_cfg_get_key_comp_name_at_idx(key_comps, 2);
	    key_comp_3 = pp_cfg_get_key_comp_name_at_idx(key_comps, 3);
	    if (!strcmp("serial", key_comp_1) && !strcmp(pp_cfg_vect_elems_comp, key_comp_2)) {
		/* don't trigger reconfiguration on power switch state change */
		if ( vector_size(key_comps) < 6
		     || strcmp("state", pp_cfg_get_key_comp_name_at_idx(key_comps, 5)) ) {
		    sp_id = pp_strtol_10(key_comp_3, -1, NULL);
		    sp = (serial_port_t *)pp_hash_get_entry_i(serial_ports_hash, sp_id);
		    if (sp) sp->need_reconfigure = 1;
		}
	    }
	}
    }

    /* reconfigure marked serial ports */
    ret = pp_serial_reconfigure(ctx->retstr);

    MUTEX_UNLOCK(&serial_lock);

    return ret;
}
