#include <stdlib.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <linux/netfilter.h>
#include <libipq.h>
#include <netpacket/packet.h>
#include <net/ethernet.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <openssl/md5.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/um.h>
#include <pp/ipc.h>
#include <pp/firmware.h>
#if defined(PP_FEAT_MASS_STORAGE)
#include <pp/usb.h>
#endif /* PP_FEAT_MASS_STORAGE */
#include <liberic_config.h>
#include <liberic_misc.h>
#include <liberic_net.h>
#include <liberic_pthread.h>

#include "setup_proto_internal.h"


#define PACKET_BUFSIZE 2048

struct ip_pseudo_hdr {
    u_int32_t saddr;
    u_int32_t daddr;
    u_int8_t zero;
    u_int8_t protocol;
    u_int16_t udp_len;
};

static int initialized = 0;
static int sp_worker_running = 0;
static unsigned short ip_id;
static struct ipq_handle * nl_handle = NULL;
static int sock_fd = -1;
static pthread_t sp_thread;
static int sp_worker_should_die = 0;
static char * if_name = NULL;

static int setup_proto_start(void);
static int setup_proto_stop(void);
static int setup_proto_reconf_ch(pp_cfg_chg_ctx_t *ctx);
static void setup_proto_ipc_handler(pp_ipc_req_ctx_t *ctx,
	pp_ipc_req_type_t type, int length, unsigned char *data);

static void * sp_worker(void * arg);
static int generate_udp_ip_pdu(void ** pdu, size_t * pdu_len,
			       in_addr_t src_in_addr, in_addr_t dst_in_addr,
			       unsigned short src_port, unsigned short dst_port,
			       void * data, size_t data_len);
static u_int16_t in_cksum(u_int16_t * addr, size_t len);
static int send_via_udp(pp_setup_rsp_t * rsp, void* sp_pdu, ssize_t sp_pdu_len);
#ifdef PP_FEAT_MASS_STORAGE
static int send_via_usb(int ms_index, pp_setup_rsp_t * rsp, void* sp_pdu, ssize_t sp_pdu_len);
#endif /* PP_FEAT_MASS_STORAGE */

int
pp_setup_proto_init()
{
    if (initialized) return 0;

    _pp_setup_proto_register_errnos();

    if (setup_proto_start() == 0) {
	pp_cfg_add_change_listener(setup_proto_reconf_ch, "network.disable_setup_proto");
	pp_ipc_register_handler(PP_IPC_REQ_SETUP_PROTO, setup_proto_ipc_handler);
	initialized = 1;
	return 0;
    } else {
	pp_setup_proto_cleanup();
	return -1;
    }
}

void
pp_setup_proto_cleanup()
{
    pp_cfg_rem_change_listener(setup_proto_reconf_ch, "network.disable_setup_proto");
    pp_ipc_unregister_handler(PP_IPC_REQ_SETUP_PROTO);

    setup_proto_stop();

    /* rwa: this will not work and is not really necessary:
     * _pp_setup_proto_unregister_errnos();
     */

    initialized = 0;
}

static int
setup_proto_start(void)
{
    struct sockaddr_ll sa;

    pp_cfg_get(&if_name, "network.eth_interface");

    ip_id = time(NULL) & 0xffff;
	
    /* get netlink queue handle */
    if ((nl_handle = ipq_create_handle(0, PF_INET)) == NULL) {
	pp_log("%s(): ipq_create_handle() failed (%s).\n", ___F, ipq_errstr());
	goto bail;
    }

    /* set queue mode */
    if (ipq_set_mode(nl_handle, IPQ_COPY_PACKET, PACKET_BUFSIZE) < 0) {
	pp_log("%s(): ipq_set_mode(IPQ_COPY_PACKET) failed (%s).\n", ___F, ipq_errstr());
	goto bail;
    }

    /* create send socket */
    if ((sock_fd = socket(PF_PACKET, SOCK_DGRAM, htons(ETH_P_IP))) < 0) {
	pp_log_err("%s(): socket(PF_PACKET) failed", ___F);
	goto bail;
    }

    /* bind send socket */
    memset(&sa, 0, sizeof(sa));
    sa.sll_family = AF_PACKET;
    sa.sll_protocol = htons(ETH_P_IP);
    sa.sll_ifindex = if_nametoindex(if_name);
    
    if (bind(sock_fd, (struct sockaddr*)&sa, sizeof(sa)) < 0) {
	pp_log_err("%s(): bind() of send socket failed", ___F);
	goto bail;
    }

    /* start setup thread */
    if (eric_pthread_create(&sp_thread, 0, 65536, sp_worker, NULL) != 0) {
	pp_log("%s(): Cannot create sp-thread.\n", ___F);
	goto bail;
    }
    sp_worker_running = 1;

#ifdef PP_FEAT_MASS_STORAGE
    pp_usb_enable_setup_proto(1);
#endif /* PP_FEAT_MASS_STORAGE */

    return PP_SUC;

 bail:
    setup_proto_stop();
    return PP_ERR;
}

static int
setup_proto_stop(void)
{
#ifdef PP_FEAT_MASS_STORAGE
    pp_usb_enable_setup_proto(0);
#endif /* PP_FEAT_MASS_STORAGE */

    if (sp_worker_running) {
	sp_worker_should_die = 1;
	pthread_join(sp_thread, NULL);
	sp_worker_should_die = 0;
	sp_worker_running = 0;
	pp_log("sp_worker finished\n");
    }
    if (nl_handle) {
	ipq_destroy_handle(nl_handle);
	nl_handle = NULL;
    }
    if (sock_fd != -1) {
	close(sock_fd);
	sock_fd = -1;
    }
    free(if_name);
    if_name = NULL;
    return PP_SUC;
}

static int
setup_proto_reconf_ch(pp_cfg_chg_ctx_t *ctx UNUSED)
{
    int disabled;

    pp_cfg_is_enabled(&disabled, "network.disable_setup_proto");
    return disabled ? setup_proto_stop() : setup_proto_start();
}

static pp_setup_rsp_t *
handle_request(pp_setup_req_t * req) {
    const char * fn = ___F;
    pp_setup_rsp_t * rsp = NULL;
    u_int8_t own_mac[6];
    char *su, *supw, *supwnew, *ipacp, *ip, *nm, *gw;
    
#if 0 && defined(PP_FEAT_WLAN)
    char *essid, *wep, *key, *wmode;
#endif
    int server_sid = 0;
    int param_error = 0;
    int unspec_error = 0;
    int super_needs_save = 0;
    int global_needs_save = 0;
    int gid;
    
    assert(req);
    
    su = pp_prop_get(&req->props, PP_SP_PROP_SUPER_USER_NAME);
    supw = pp_prop_get(&req->props, PP_SP_PROP_SUPER_USER_PASSWORD);
    supwnew = pp_prop_get(&req->props, PP_SP_PROP_SUPER_USER_PASSWORD_NEW);
    ipacp = pp_prop_get(&req->props, PP_SP_PROP_IP_AUTO_CONFIG_PROTO);
    ip = pp_prop_get(&req->props, PP_SP_PROP_IP_ADDRESS);
    nm = pp_prop_get(&req->props, PP_SP_PROP_NETMASK);
    gw = pp_prop_get(&req->props, PP_SP_PROP_GATEWAY);
    /* eth is only here for reference, it will actually not be set by the setup protocol */
    // const char * eth = pp_prop_get(&req->props, PP_SP_PROP_INTERFACE);
#if 0 && defined(PP_FEAT_WLAN)
    essid = pp_prop_get(&req->props, PP_SP_PROP_WLAN_ESSID);
    wep = pp_prop_get(&req->props, PP_SP_PROP_WLAN_WEP_ENABLED);
    key = pp_prop_get(&req->props, PP_SP_PROP_WLAN_WEP_KEY);
    wmode = pp_prop_get(&req->props, PP_SP_PROP_WLAN_MODE);
#endif
   
    pp_log("%s: received valid request.\n", fn);
    
    /* get own mac */
    if (pp_get_mac(if_name, own_mac) == -1) {
	pp_log("%s: get_own_mac() failed.\n", fn);
	/* create response */
    } else if (memcmp(own_mac, pp_setup_proto_req_get_device_id(req), sizeof(own_mac))
	       && !(*pp_setup_proto_req_get_device_id(req) & 0x80)) {
	/* not for us - ignore */
    } else if ((rsp = pp_setup_proto_rsp_create(NULL, 0)) == NULL) {
	pp_log("%s: pp_setup_proto_rsp_create() failed.\n", fn);

	/* check if we got pinged */
    } else if (req->command == PP_SP_CMD_PING) {
	pp_setup_proto_rsp_set_code(rsp, PP_SP_RSP_SUCCESS);
	/*
	 * Wait a random time (<1s) to allow the client some time for
	 * processing all the replies. Otherwise UDP packets may get
	 * lost.
	 */
	usleep((unsigned long)random() % 1000000);
	/* no authentication is needed for query */
    } else if (req->command == PP_SP_CMD_QUERY) {
	pp_setup_proto_rsp_set_code(rsp, PP_SP_RSP_SUCCESS);
	/* check if the specified user is the super user */
    } else if (su == NULL) {
	pp_setup_proto_rsp_set_code(rsp, PP_SP_RSP_PERM_DENIED);
    } else if (PP_ERR == pp_um_user_get_uid(su)) {
	pp_setup_proto_rsp_set_code(rsp, PP_SP_RSP_AUTH_FAILED);
	/* check if the password is valid */
    } else if (PP_ERR == pp_um_user_authenticate_with_ip_str(su, supw, 
					         PP_UM_AUTH_IGNORE_BLOCKING,
                                                 ip, &server_sid,
                                                 &gid)) {
	pp_setup_proto_rsp_set_code(rsp, errno == PP_UM_ERR_AUTH_FAILED ? 
                                         PP_SP_RSP_AUTH_FAILED :
                                         PP_SP_RSP_UNSPEC_ERR);
    } else {
	/* auth ok -> do query/setup stuff */
	if (req->command == PP_SP_CMD_SETUP) {
	    if (supwnew) {
		char *code = pp_um_passwd_encrypt(supwnew);
		if (PP_SUCCED(pp_cfg_set(code, "user[0].pw_encr"))) {
		    super_needs_save = 1;
		} else unspec_error = 1;
		free(code);
	    }
	    if (supwnew) {
		char md5_hash[MD5_DIGEST_LENGTH];
		char md5_hash_str[MD5_DIGEST_LENGTH*2+1];
		MD5(supwnew, strlen(supwnew), md5_hash);
		pp_md5_hash_to_str(md5_hash, md5_hash_str);
		if (PP_SUCCED(pp_cfg_set(md5_hash_str, "user[0].pw_hash"))) {
		    super_needs_save = 1;
		} else unspec_error = 1;
	    }
	    if (!param_error && !unspec_error && ipacp && strcmp(ipacp, "none")) {
		if (!strcmp(ipacp, "dhcp") || !strcmp(ipacp, "bootp")) {
		    if (PP_SUCCED(pp_cfg_set(ipacp, "network.ip_auto_config_proto"))) {
			global_needs_save = 1;
		    } else unspec_error = 1;
		} else param_error = 1;
	    } else {
		if (PP_SUCCED(pp_cfg_set(ipacp ? ipacp : "none", "network.ip_auto_config_proto"))) {
		    global_needs_save = 1;
		} else unspec_error = 1;
		if (!param_error && !unspec_error && ip) {
		    if (*ip == '\0' || pp_check_ip_syntax(ip) == 0) {
			if (PP_SUCCED(pp_cfg_set(ip, "network.ipaddr"))) {
			    global_needs_save = 1;
			} else unspec_error = 1;
		    } else param_error = 1;
		}
		if (!param_error && !unspec_error && nm) {
		    if (*nm == '\0' || pp_check_ip_syntax(nm) == 0) {
			if (PP_SUCCED(pp_cfg_set(nm, "network.netmask"))) {
			    global_needs_save = 1;
			} else unspec_error = 1;
		    } else param_error = 1;
		}
		if (!param_error && !unspec_error && gw) {
		    if (*gw == '\0' || pp_check_ip_syntax(gw) == 0) {
			if (PP_SUCCED(pp_cfg_set(gw, "network.gateway"))) {
			    global_needs_save = 1;
			} else unspec_error = 1;
		    } else param_error = 1;
		}
	    }
#if 0 && defined(PP_FEAT_WLAN)
	    /* check dependencies */
	    if (!param_error && !unspec_error && wep) {
		if (!strcmp(wep, "yes")) {
		    /* check if there is a wep key, too */
		    if (!key || *key == '\0') {
			param_error = 1;
		    }
		}
	    }
	    
	    if (!param_error && !unspec_error && essid) {
		if (*essid != '\0') {
		    if (PP_SUCCED(pp_cfg_set(essid, "wlan.essid"))) {
			global_needs_save = 1;
		    } else unspec_error = 1;
		} else param_error = 1;
	    }

	    if (!param_error && !unspec_error && wep) {
		if (!strcmp(wep, "yes") || !strcmp(wep, "no")) {
		    if (PP_SUCCED(pp_cfg_set(wep, "wlan.wep_enabled"))) {
			global_needs_save = 1;
		    } else unspec_error = 1;
		} else param_error = 1;
	    }

	    if (!param_error && !unspec_error && key) {
		/* set the key regardless if it is empty,
		   the check is done above */
		if (PP_SUCCED(pp_cfg_set(key, "wlan.wep_key"))) {
		    global_needs_save = 1;
		} else unspec_error = 1;
	    }

	    if (!param_error && !unspec_error && wmode) {
#if 0
		/* disable until driver is adhoc capable */
		if (!strcmp(wmode, "managed") || !strcmp(wmode, "adhoc"))
#else
	        if (!strcmp(wmode, "managed"))
#endif
		{
		    if (PP_SUCCED(pp_cfg_set(wmode, "wlan.mode"))) {
			global_needs_save = 1;
		    } else unspec_error = 1;
		} else param_error = 1;
	    }
#endif /* 0 && PP_FEAT_WLAN */
	    if (!param_error && !unspec_error && super_needs_save) {
		if (PP_FAILED(pp_cfg_save(!global_needs_save))) {
		    unspec_error = 1;
		}
	    }
	    if (!param_error && !unspec_error && global_needs_save) {
		if (PP_FAILED(pp_cfg_save(DO_FLUSH))) {
		    unspec_error = 1;
		}
	    }
	    if (param_error) {
		pp_setup_proto_rsp_set_code(rsp, PP_SP_RSP_PARAM_ERR);
	    } else if (unspec_error) {
		pp_setup_proto_rsp_set_code(rsp, PP_SP_RSP_UNSPEC_ERR);
	    } else {
		pp_setup_proto_rsp_set_code(rsp, PP_SP_RSP_SUCCESS);
	    }
	} else {
	    pp_setup_proto_rsp_set_code(rsp, PP_SP_RSP_UNSPEC_ERR);
	}
    }
    if (rsp) {
	char * opt;
	pp_log("%s: sending response.\n", fn);
	pp_setup_proto_rsp_set_dst_hw_addr(rsp, pp_setup_proto_req_get_src_hw_addr(req));
	pp_setup_proto_rsp_set_dst_in_addr(rsp, pp_setup_proto_req_get_src_in_addr(req));
	pp_setup_proto_rsp_set_device_id(rsp, own_mac);
	pp_setup_proto_rsp_set_id(rsp, pp_setup_proto_req_get_id(req));
	/* return the settings for all commands except "ping" */
	if (req->command != PP_SP_CMD_PING
	    && pp_setup_proto_rsp_get_code(rsp) == PP_SP_RSP_SUCCESS) {
	    
#define CFG_2_SETUP_PROP(cfgprop, setupprop)				\
	    if (PP_SUCCED(pp_cfg_get_nodflt(&opt, cfgprop))) {		\
		pp_setup_proto_rsp_set_property(rsp, setupprop, opt);	\
		free(opt);						\
	    }

	    CFG_2_SETUP_PROP("serial", PP_SP_PROP_SERIAL);
	    CFG_2_SETUP_PROP("network.ip_auto_config_proto", PP_SP_PROP_IP_AUTO_CONFIG_PROTO);
	    CFG_2_SETUP_PROP("network.ipaddr", PP_SP_PROP_IP_ADDRESS);
	    CFG_2_SETUP_PROP("network.netmask", PP_SP_PROP_NETMASK);
	    CFG_2_SETUP_PROP("network.gateway", PP_SP_PROP_GATEWAY);
#if 0 && defined(PP_FEAT_WLAN)
#error FIXME - this is old and not functional
	    CFG_2_SETUP_PROP("wlan.essid", PP_SP_PROP_WLAN_ESSID);
	    CFG_2_SETUP_PROP("wlan.wep_enabled", PP_SP_PROP_WLAN_WEP_ENABLED);
	    CFG_2_SETUP_PROP("wlan.mode", PP_SP_PROP_WLAN_MODE);

	    /* now sends the actual seen ESSIDs */
	    {
		char** essids = eric_net_wlan_get_essids();
		if (essids != NULL) {
		    int wi;
		    for (wi = 0; essids[wi] != NULL; ++wi) {
			char es[12];
			snprintf(es, sizeof(es), "sessid%d", wi);
			pp_setup_proto_rsp_set_property(rsp, es, essids[wi]);
		    }
		}
	    }
#endif /* 0 && PP_FEAT_WLAN */
	    pp_setup_proto_rsp_set_property(rsp, PP_SP_PROP_INTERFACE, if_name);
	    opt = eric_misc_get_boardname();
	    pp_setup_proto_rsp_set_property(rsp, PP_SP_PROP_BOARD_NAME, opt);
	    free(opt);
	    asprintf(&opt, "%d.%d.%d.%d",
	        pp_firmware_erla_version_major, pp_firmware_erla_version_minor,
	        pp_firmware_erla_version_subminor, pp_firmware_erla_build_nr);
	    pp_setup_proto_rsp_set_property(rsp, PP_SP_PROP_FW_VERSION, opt);
	    free(opt);
	}
    }
    return rsp;
}

void pp_setup_proto_answer_request(int ms_index, pp_setup_req_t * req, pp_setup_proto_comm_type_t comm_type) {
    pp_setup_rsp_t * rsp = handle_request(req);
    if (!rsp) return;
    pp_setup_proto_rsp_send(ms_index, rsp, comm_type);
    pp_setup_proto_rsp_free(rsp);
}

static void *
sp_worker(void * arg UNUSED)
{
    pp_setup_req_t * req;
    
    while (!sp_worker_should_die) {
	if ((req = pp_setup_proto_req_recv(1000000, NULL)) != NULL) {
	    pp_setup_proto_answer_request(0, req, PP_SETUP_PROTO_NETLINK);
	    pp_setup_proto_req_free(req);
	}
    }
    pthread_exit(NULL);
}

void pp_setup_proto_req_handle_usb(int ms_index, char* sp_pdu, ssize_t sp_pdu_len)
{
    pp_setup_req_t * req = NULL;
    in_addr_t ip;
    char hw_addr[6];

    memset(&ip, 0, sizeof(ip));
    memset(hw_addr, 0, sizeof(hw_addr));
    
    req = pp_setup_proto_req_create(NULL, sp_pdu, sp_pdu_len,
				    hw_addr, ip);
    if (req != NULL) {
	pp_setup_proto_answer_request(ms_index, req, PP_SETUP_PROTO_USB);
	pp_setup_proto_req_free(req);
    }
}

pp_setup_req_t *
pp_setup_proto_req_recv(int timeout, int * timedout)
{
    const char * fn = ___F;
    unsigned char buf[PACKET_BUFSIZE];
    ipq_packet_msg_t * msg = NULL;
    int status;
    pp_setup_req_t * req = NULL;
    struct iphdr * ip_hdr;
    void * pdu;
    void * sp_pdu;
    ssize_t pdu_len, sp_pdu_len;

    if (timedout) *timedout = 0;

    if ((status = ipq_read(nl_handle, buf, PACKET_BUFSIZE, timeout)) < 0) {
	pp_log("%s(): ipq_read() failed.\n", fn);
	return NULL;
    }

    if (status == 0) {
	if (timedout) *timedout = 1;
	return NULL; /* timeout */
    }

    switch (ipq_message_type(buf)) {
      case NLMSG_ERROR:
	  pp_log("%s(): ipq_read() returned error message with code %d (%s).\n", fn, ipq_get_msgerr(buf), strerror(ipq_get_msgerr(buf)));
	  return NULL;
      case IPQM_PACKET:
	  msg = ipq_get_packet(buf);
	  if (ipq_set_verdict(nl_handle, msg->packet_id, NF_DROP, 0, NULL) < 0) {
	      pp_log("%s(): ipq_set_verdict() failed.\n", fn);
	      return NULL;
	  }
	  break;
      default:
	  pp_log("%s(): ipq_read() returned unknown message type.\n", fn);
	  return NULL;
    }

    pdu = msg->payload;
    pdu_len = msg->data_len;

    ip_hdr = (void *)pdu;
    sp_pdu = (void *)ip_hdr + ip_hdr->ihl * 4 + sizeof(struct udphdr);

    sp_pdu_len = pdu_len - ip_hdr->ihl * 4 - sizeof(struct udphdr);

    req = pp_setup_proto_req_create(if_name, sp_pdu, sp_pdu_len,
				    msg->hw_addr, ip_hdr->saddr);

    return req;
}

int
pp_setup_proto_rsp_send(int ms_index, pp_setup_rsp_t * rsp, pp_setup_proto_comm_type_t comm_type)
{
    void * sp_pdu;
    setup_rsp_hdr_t * sp_hdr;
    void * sp_payload;
    ssize_t sp_pdu_len;
    int ret = -1;

    (void)ms_index; /* avoid warning */

    /* allocate pdu */
    if ((sp_pdu = malloc(PP_SP_RSP_MTU)) == NULL) goto bail;

    sp_hdr = (setup_rsp_hdr_t *)sp_pdu;
    sp_payload = sp_pdu + sizeof(setup_rsp_hdr_t);

    /* write payload into pdu */
    if ((sp_pdu_len = pp_prop_write(&rsp->props, sp_payload,
				    PP_SP_RSP_MTU - sizeof(setup_rsp_hdr_t))) < 0) {
	goto bail;
    }

    /* write header into pdu */
    sp_hdr->magic = htonl(PP_SP_MAGIC);
    sp_hdr->version = PP_SP_VERSION;
    memcpy(sp_hdr->dev_id, rsp->dev_id, sizeof(sp_hdr->dev_id));
    sp_hdr->id = htonl(rsp->id);
    sp_hdr->response_code = rsp->response_code;
    sp_pdu_len += sizeof(setup_rsp_hdr_t);

    if (comm_type == PP_SETUP_PROTO_NETLINK) {
	if (send_via_udp(rsp, sp_pdu, sp_pdu_len) < 0) {
	    goto bail;
	}
#ifdef PP_FEAT_MASS_STORAGE
    } else {
	if (send_via_usb(ms_index, rsp, sp_pdu, sp_pdu_len) < 0) {
	    goto bail;
	}
#endif /* PP_FEAT_MASS_STORAGE */
    }
    ret = 0;
    
 bail:
    free(sp_pdu);
    return ret;
}

static int
send_via_udp(pp_setup_rsp_t * rsp, void* sp_pdu, ssize_t sp_pdu_len) {
    void * ip_udp_pdu;
    size_t ip_udp_pdu_len;
    int ret = -1;
    struct sockaddr_ll sa;
    
    /* generate pdu with UDP and IP header */
    if ((generate_udp_ip_pdu(&ip_udp_pdu, &ip_udp_pdu_len,
			     inet_addr("0.0.0.0"), 
			     inet_addr("255.255.255.255"),
			     PP_SP_SRV_PORT, PP_SP_CLI_PORT,
			     sp_pdu, sp_pdu_len)) <  0) {
	goto bail;
    }

    /* send pdu */
    memset(&sa, 0, sizeof(sa));
    sa.sll_family = AF_PACKET;
    sa.sll_protocol = htons(ETH_P_IP);
    sa.sll_ifindex = if_nametoindex(if_name);
    sa.sll_halen = ETH_ALEN;
    memcpy(sa.sll_addr, rsp->dst_hw_addr, ETH_ALEN);
    if (sendto(sock_fd, ip_udp_pdu, ip_udp_pdu_len, 0,
	       (struct sockaddr *)&sa, sizeof(sa)) < 0) {
	if (sendto(sock_fd, ip_udp_pdu, ip_udp_pdu_len, 0,
	       (struct sockaddr *)&sa, sizeof(sa)) < 0) {
	    pp_log_err("%s(): sendddto() failed\n", ___F);
	    goto bail;
	}
    }

    ret = 0;

 bail:
    free(ip_udp_pdu);
    return ret;
}

#ifdef PP_FEAT_MASS_STORAGE
static int
send_via_usb(int ms_index, pp_setup_rsp_t * rsp UNUSED, void* sp_pdu, ssize_t sp_pdu_len) {
    if (sp_pdu_len < 0) return -1;
    return pp_usb_set_setup_rsp_packet(ms_index, sp_pdu, (size_t)sp_pdu_len);
}
#endif /* PP_FEAT_MASS_STORAGE */

static int
generate_udp_ip_pdu(void ** pdu, size_t * pdu_len,
		    in_addr_t src_in_addr, in_addr_t dst_in_addr,
		    unsigned short src_port, unsigned short dst_port,
		    void * data, size_t data_len)
{
    struct ip_pseudo_hdr * ip_pseudo_hdr;
    struct iphdr * ip_hdr;
    struct udphdr * udp_hdr;
    void * payload;

    /* allocate pdu */
    *pdu_len = sizeof(struct iphdr) + sizeof(struct udphdr) + data_len;
    if ((*pdu = malloc(*pdu_len)) == NULL) return -1;
    memset(*pdu, 0, *pdu_len);

    ip_pseudo_hdr = (struct ip_pseudo_hdr *)*pdu;
    ip_hdr = (struct iphdr *)*pdu;
    udp_hdr = (struct udphdr *)(*pdu + sizeof(struct iphdr));
    payload = (void *)udp_hdr + sizeof(struct udphdr);

    ip_pseudo_hdr->saddr = src_in_addr;
    ip_pseudo_hdr->daddr = dst_in_addr;
    ip_pseudo_hdr->protocol = IPPROTO_UDP;
    ip_pseudo_hdr->udp_len = htons(sizeof(struct udphdr) + data_len);

    memcpy(payload, data, data_len);

    udp_hdr->source = htons(src_port);
    udp_hdr->dest = htons(dst_port);
    udp_hdr->len = htons(sizeof(struct udphdr) + data_len);
    udp_hdr->check = 0;
    udp_hdr->check = in_cksum((unsigned short *)ip_pseudo_hdr,
			      sizeof(struct iphdr) + sizeof(struct udphdr) + data_len);
    if (udp_hdr->check == 0) udp_hdr->check = 0xffff;

    ip_hdr->ihl = 5;
    ip_hdr->version = IPVERSION;
    ip_hdr->tos = 0;
    ip_hdr->tot_len = htons(*pdu_len);
    ip_hdr->id = htons(ip_id++);
    ip_hdr->frag_off = 0;
    ip_hdr->ttl = IPDEFTTL;
    ip_hdr->protocol = IPPROTO_UDP;
    ip_hdr->saddr = src_in_addr;
    ip_hdr->daddr = dst_in_addr;
    ip_hdr->check = 0;
    ip_hdr->check = in_cksum((unsigned short *)ip_hdr, sizeof(struct iphdr));

    return 0;
}

static u_int16_t
in_cksum(u_int16_t * addr, size_t len)
{
    int32_t sum = 0;
    u_int16_t * w = addr;
    size_t nleft = len;
    while (nleft > 1) {
	sum += *w++;
	nleft -= 2;
    }
    if (nleft == 1) {
	/*
	 * Make sure that the left-over byte is added correctly both
	 * with little and big endian hosts
	 */
	u_int16_t tmp = 0;
	*(u_int8_t *)&tmp = *(u_int8_t *)w;
	sum += tmp;
    }
    /*  Fold 32-bit sum to 16 bits */
    while (sum >> 16)
	sum = (sum & 0xffff) + (sum >> 16);
    return ~sum;
}

static void
setup_proto_ipc_handler(pp_ipc_req_ctx_t *ctx,
	pp_ipc_req_type_t type UNUSED, int length, unsigned char *data)
{
    pp_setup_req_t *req = NULL;
    pp_setup_rsp_t *rsp;
    in_addr_t ip;
    char hw_addr[6];

    unsigned char *rsp_buffer = NULL;
    int rsp_length = 0;
    pp_ipc_rsp_status_t rsp_status = PP_IPC_RSP_ERROR;

    memset(&ip, 0, sizeof(ip));
    memset(hw_addr, 0, sizeof(hw_addr));

    req = pp_setup_proto_req_create(NULL, data, length, hw_addr, ip);
    if (!req) goto bail;

    if ((rsp = handle_request(req)) != NULL) {
	/* allocate pdu */
	rsp_buffer = malloc(PP_IPC_SOCKET_MTU);
	setup_rsp_hdr_t *sp_hdr = (setup_rsp_hdr_t *)rsp_buffer;
	void *sp_payload = rsp_buffer + sizeof(setup_rsp_hdr_t);

	/* write payload into pdu */
	rsp_length = pp_prop_write(&rsp->props, sp_payload,
		PP_IPC_SOCKET_MTU - sizeof(setup_rsp_hdr_t));
	if (rsp_length < 0) goto bail;

	/* write header into pdu */
	sp_hdr->magic = htonl(PP_SP_MAGIC);
	sp_hdr->version = PP_SP_VERSION;
	memcpy(sp_hdr->dev_id, rsp->dev_id, sizeof(sp_hdr->dev_id));
	sp_hdr->id = htonl(rsp->id);
	sp_hdr->response_code = rsp->response_code;
	rsp_length += sizeof(setup_rsp_hdr_t);
	rsp_status = PP_IPC_RSP_SUCCESS;
    }

bail:
    pp_setup_proto_req_free(req);
    if (rsp_status == PP_IPC_RSP_SUCCESS) {
	pp_ipc_send_response(ctx, rsp_status, rsp_length, rsp_buffer);
    } else {
	pp_ipc_send_response(ctx, rsp_status, 0, NULL);
    }
    free(rsp_buffer);
}

