/**
 * User manangement routines
 * This library implements RAASIP compatible usermanagement and replaces
 * the old libpp_um (that was based on permission inheritance).
 * 
 * All functions in this library operate (unless otherwise noted)
 * on the config system. That means they operate either on the local
 * flashdisk (fully writable) or on a remote ldap server (mostly
 * readonly).
 *
 * (c) 2006 Peppercon AG, geo@peppercon.de, tweb@raritan.com
 */

#ifndef __UM_INTERN_H__
#define __UM_INTERN_H__
 
#define str_compare (int(*)(const void*, const void*))strcmp

/**
 * username to UID mapping
 */
//extern pp_hash_t *um_uid_by_username;

/**
 * user handling mutex
 */
//extern pthread_mutex_t um_user_mtx;

/**
 * Handles blocking for user, i.e. increase login fails counter if
 * authentication failed or unblock user if authentication succeeded
 *
 * @param name                  name of user
 * @param auth_result           result of (previous) authentication request
 *
 * @return                      PP_ERR on internal failure, PP_SUC otherwhise
 */
int um_user_handle_blocking(const char* name, int auth_result);

/**
 * Initialize UID / username mapping
 */
void um_load_all_users(void);

/**
 * Initialize user specific settings.
 * @return                      PP_SUC or errno
 */
int um_user_init(void);

/**
 * Cleanup user specific settings.
 */
void um_user_cleanup(void);

/**
 * Creates a new group ('@username' in most cases) for user
 * 
 * @param username           user to create group for
 * @param groupname          return value, name of group actually created
 *                           ('@username' in most cases)
 *
 * @return                   PP_SUC and groupname if the group was successfully
 *                           created, PP_ERR and errno otherwhise
 */
int um_user_create_individual_group(const char *username, char **groupname);

/**
 * Initialize GID / groupname mapping
 */
void um_load_all_groups(void);

/**
 * Initialize group specific settings.
 * @return                      PP_SUC or errno
 */
int um_group_init(void);

/**
 * Cleanup group specific settings.
 */
void um_group_cleanup(void);

/**
 * Set a permission for a group to the specified value.
 */
int um_group_set_permission(int gid, 
                            const char* permission_name,
                            const char* permission_value);

/**
 * Check a permission for a group. Technically permissions are not restricted
 * to the (Y/N) or (none/view/control) permissions defined in RAASIP. Although
 * it is possible to use abritrary permission, it is highly recommended to stick
 * to the RAASIP permission scheme to unify user experience.
 * 
 * Existing permissions and their possible values are configured in the config
 * system and retrieved with 'acl.h'.
 * 
 * @param gid                group ID
 * @param permission_name    the name of the permission (e.g. 'port_0')
 * @param permission_value   the desired permission value, 
 *                           (eg. 'view' or 'control')
 * 
 * @return                   PP_SUC if the group has the permission
 *                           errno with verbose string else (no permission,
 *                           internal error, ...)
 */
int um_group_has_permission(int gid,
                            const char* permission_name,
                            const char* permission_value);

/**
 * Get a permission for a user.
 * Remember to free returned permission_value!
 * 
 * @param gid                group ID
 * @param permission_name    the name of the permission (e.g. 'port_0')
 * @param permission_value   return value, not set in PP_ERR case
 * 
 * @return                   PP_SUC if the groups permission could be retrived,
 *                           errno with verbose string else (no permission,
 *                           internal error, ...)
 */
int um_group_get_permission(int gid,
                            const char* permission_name,
                            char** permission_value);

/**
 * Callback for updating user pw change timestamp on pw change
 *
 * @param ctx               config change context
 *
 * @return                  PP_SUC on success, PP_ERR otherwhise
 */
int um_user_set_pw_change_timestamp_cb(pp_cfg_chg_ctx_t *ctx);

/**
 * Check a permission of a group to connect from given IP. 
 * Magic as follows:
 * - access counter is initialized with default group ACL policy
 *   (1 for ACCEPT, 0 for DENY)
 * - each ACCEPT-rule that matches group and ip increments access counter
 * - each DENY-rule that matches group and ip decrements access counter
 * - return PP_SUC for group ACL disabled or access counter > 0,
 *          PP_ERR on error or access counter <= 0
 * 
 * @param gid                group reference
 * @param ip                 IP address xxx.xxx.xxx.xxx, the ACL is checked for
 * 
 * @return                   PP_SUC if the group has the permission
 *                           PP_ERR (and errno) else
 */
int um_group_is_allowed_from_ip_str(int gid, const char* ip);

/**
 * external authentication service is active for user
 * -> get gid from remote AS
 * -> if gid is valid -> gid
 * -> if gid is invalid -> <Unknown>
 * -> if no gid could be retrieved -> <None>
 *
 * @param  groupname -- groupname got from remote AS, could be NULL.
 * @return validated group id.
 */
int um_group_validate_remote(const char* groupname );

/**
 * decrypts and base64decodes a string
 *
 * @param code                  encrypted password
 *
 * @return                      plain text password string
 */
char* um_passwd_decrypt(const char* code);

/**
 * Set the local password of a user.
 * This function performs strong password checks if the option is activated.
 * 
 * @param uid                 the target user id
 * @param password            the new password
 * @param flags               flags to control password storage, 
 *                            e.g. password is already MD5-hash or plain
 * 
 * @return                    PP_SUC if the password could be set, errno else
 */
int um_user_set_password(int uid, const char* password, int flags);

/**
 * Get a list of IDs of all users assigned to a specified group. Caller is
 * responsible for freeing the returned vector.
 * 
 * @see pp_um_get_all_users_by_gid
 */
vector_t* um_get_all_uids_by_gid(int gid);

/**
 * Checks if transformation is available for a user / group with id
 *
 * @param type                "user" or "group"
 * @param id                  ID of user / group
 * @param transformation      transformation to check, e.g. "rename"
 *
 * @return                    PP_SUC if transformation is available, PP_ERR else
 */
int um_is_transformable(const char *type, int id, const char* transformation);

#define um_group_is_renameable(__id__) \
    um_is_transformable("group", __id__, "rename")
#define um_group_is_deleteable(__id__) \
    um_is_transformable("group", __id__, "delete")
#define um_group_is_modifiable(__id__) \
    um_is_transformable("group", __id__, "modify")
#define um_user_is_renameable(__id__) \
    um_is_transformable("user", __id__, "rename")
#define um_user_is_deleteable(__id__) \
    um_is_transformable("user", __id__, "delete")
#define um_user_is_modifiable(__id__) \
    um_is_transformable("user", __id__, "modify")

#if defined(PP_FEAT_STRONG_PASSWORDS)
/**
 * Callback for notification on strong password settings change
 *
 * @see um_user_strong_pw_cb
 */
int um_user_strong_pw_notify_cb(pp_cfg_chg_ctx_t *ctx);

/**
 * Callbacks for setting "need_pw_change" flags on strong_pw change
 * Special handling for keys, that are switched from disabled to enabled
 *
 * @see um_user_strong_pw_cb
 */
int um_user_strong_pw_got_enabled_cb(pp_cfg_chg_ctx_t *ctx);

/**
 * Callback for setting "need_pw_change" flags on strong_pw change
 *
 * @param ctx               config change context
 *
 * @return                  PP_SUC on success, PP_ERR otherwhise
 */
int um_user_strong_pw_cb(pp_cfg_chg_ctx_t *ctx);

/**
 * Checks whether a the password meets Raritan common security requirements.
 * @see pp_um_strong_pw_check_for_user
 */
int um_strong_pw_check_for_user_core(const char *username, const char *password,
                                     const char *old_password, char **errstr);
#endif /* PP_FEAT_STRONG_PASSWORDS */

#endif /* __UM_INTERN_H__ */

