/* system includes */
#include <sys/ioctl.h>
#include <fcntl.h>
#include <poll.h>
#include <sys/mman.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <sys/time.h>

/* firmware includes */
#include <pp/base.h>
#include <pp/bio.h>
#include <liberic_pthread.h>
#include <liberic_session.h>
#include <pp/setup_proto.h>
#include <liberic_net.h>
#include <pp/cfg.h>
#include <pp/um.h>
#include <msp_adapter.h>

/* our public interface */
#include <pp/usb.h>
#include <pp_msp_proto.h>
#include "usb.h"

/* debugging */
#define noMSP_DEBUG
#ifdef MSP_DEBUG
#define DBG(fmt, x...)	pp_log(fmt, ##x)
#else
#define DBG(fmt, x...)	((void)0)
#endif

/* return values for ping-pong protocol function */
#define PING_PONG_DATA_AVAILABLE	0
#define PING_PONG_NO_DATA		1
#define PING_PONG_NO_RESPONSE		2
#define PING_PONG_CLOSE			3
#define PING_PONG_ERROR			4

#define POLL_TIMEOUT			1	/* seconds */
#define PING_PONG_TIMEOUT		30	/* seconds */
#define MSP_DATA_TIMEOUT		30	/* seconds */

#define MAX_BE_DESCRIPTORS		1

/* global variables */
msp_data_t msp_data[PP_FEAT_USB_MASS_STORAGE_NO];

typedef struct {
    int use_connection_request_v_1_2;
} msp_connection_capability_t;

typedef struct {
    int index;
    
    BIO * conn_bio;
    
    char clt_ip[INET6_ADDRSTRLEN];
    
    eric_session_int_id_t session;
    time_t last_requested_session_touch;
    
    volatile int client_connected;
    pthread_mutex_t client_connected_mutex;
    
    int close_connection;
    int kill_thread;
    
    /* version and supported stuff */
    msp_connection_capability_t caps;
    
    /* data read variables */
    pthread_cond_t msp_read_data_cond;
    pthread_mutex_t msp_read_data_mtx;
    
    u_int8_t *cd_read_data;
    u_int8_t cd_read_data_finished;
    u_int8_t cd_read_code;
    u_int16_t cd_read_sectors;
    
    file_request_t last_read_request;
    file_response_t last_read_response;
    
    /* data write variables */
    pthread_cond_t msp_write_data_cond;
    pthread_mutex_t msp_write_data_mtx;
    
    u_int8_t *cd_write_data;
    u_int8_t cd_write_data_finished;
    u_int8_t cd_write_ack;
} msp_connection_t;

static msp_connection_t msp_connections[PP_FEAT_USB_MASS_STORAGE_NO];

static volatile int vmedia_timeout_enabled = 0;

/* function prototypes */

/* functions to establish the connection */
static int msp_init_connection(BIO *clt_bio, eric_session_int_id_t *session, const char *clt_ip, int * old_connection_status, int * ms_index);
static int msp_read_init_string(BIO *clt_bio);
static int negotiate_protocol_version(BIO *clt_bio, msp_connection_capability_t * caps);
static int msp_process_authentication(BIO *clt_bio, const char *clt_ip, eric_session_int_id_t *session);
static int msp_process_login(BIO *clt_bio, const char *clt_ip, eric_session_int_id_t *session);
static int msp_process_session_id(BIO *clt_bio, eric_session_int_id_t *session);
static int msp_answer_login(BIO *clt_bio, int block_connection, int reason);
static int msp_read_connection_info(BIO *clt_bio, int * ms_index, msp_connection_capability_t * caps);

/* functions for established connections */
static void msp_process_protocol(msp_connection_t * conn);
static int msp_test_alive(msp_connection_t * conn);
static int msp_process_ping_message(msp_connection_t * conn);
static int msp_process_pong_message(msp_connection_t * conn);
static int msp_process_quit_message(msp_connection_t * conn);
static int msp_send_data_request_message(msp_connection_t * conn, u_int16_t count, u_int32_t sector);
static int msp_process_send_data(msp_connection_t * conn);
static int msp_process_data_ack(msp_connection_t * conn);
static int msp_process_medium_change(msp_connection_t * conn);

/* other stuff */
static int vmedia_timeout_enabled_ch(pp_cfg_chg_ctx_t *ctx);
static int get_vmedia_timeout_enabled(void);
static void msp_close_callback(eric_session_int_id_t session_id);

/* msp adapter file connection and handling */
#ifdef MSP_USE_ADAPTER
static int msp_fd;
static int msp_thread_started = 0;
static int msp_thread_should_die = 0;
static pthread_t msp_thread;
static void* msp_thread_func(void* data);
#endif /* MSP_USE_ADAPTER */

/* functions */
inline static int
msp_read(BIO *clt_bio, void * buf, size_t len)
{
    return eric_net_read_exact(clt_bio, buf, len, 0);
}

inline static int
msp_write(BIO *clt_bio, void * buf, size_t len)
{
    return eric_net_write_exact(clt_bio, buf, len, 0);
}

int
usb_msp_init()
{
    u_int i;
    int ret = -1;
    
#ifdef MSP_USE_ADAPTER
    if ((msp_fd = open(MSP_ADAPTER_BACKEND_FILE, O_RDWR | O_EXCL)) < 0) {
	pp_log_err("%s() cannot open '%s'\n", ___F, MSP_ADAPTER_BACKEND_FILE);
	goto bail;
    }

    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
        usb_msp_cleanup_file_info(i);
    }
    
    msp_thread_started = 1;
    msp_thread_should_die = 0;
    if (eric_pthread_create(&msp_thread, 0, 65536, msp_thread_func, NULL)) {
	pp_log_err("%s() cannot create MSP thread\n", ___F);
	msp_thread_started = 0;
	goto bail;
    }
#endif /* MSP_USE_ADAPTER */

    memset(msp_connections, 0, sizeof(msp_connections));
    
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
    	msp_connection_t * conn = &msp_connections[i];
    	
    	conn->index = i;
    	
    	if (pthread_mutex_init(&conn->client_connected_mutex, NULL) ||
    	    pthread_cond_init(&conn->msp_read_data_cond, NULL) ||
    	    pthread_mutex_init(&conn->msp_read_data_mtx, NULL) ||
    	    pthread_cond_init(&conn->msp_write_data_cond, NULL) ||
    	    pthread_mutex_init(&conn->msp_write_data_mtx, NULL)) {
    	    goto bail;
    	}
    }

    vmedia_timeout_enabled = get_vmedia_timeout_enabled();
    pp_cfg_add_change_listener(vmedia_timeout_enabled_ch, "security.idle_timeout.vmedia_enabled");

    ret = 0;
    
 bail:
    return ret;
}

void
usb_msp_cleanup(void)
{
#ifdef MSP_USE_ADAPTER

    if (msp_thread_started) {
	msp_thread_should_die = 1;
	pthread_join(msp_thread, NULL);
	msp_thread_started = 0;
    }

    if (msp_fd != -1) {
	close(msp_fd);
	msp_fd = -1;
    }
#endif /* MSP_USE_ADAPTER */
}

const char *
usb_msp_get_connected_ip(u_int ms_index)
{
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) return "";
    return msp_connections[ms_index].clt_ip;
}

static int
msp_disabled(void)
{
    int enabled;
    pp_cfg_is_enabled(&enabled, "vfloppy.usb_ms_msp_disabled");
    return enabled;
}

static int
msp_force_readonly(void)
{
    int readonly = 0;    
    pp_cfg_is_enabled(&readonly, "vfloppy.usb_ms_msp_readonly");
    return readonly;
}

const char *
usb_msp_get_disc_type(u_int ms_index)
{
    if (msp_data[ms_index].msp_disc_type == MSP_DISC_TYPE_CDROM) {
    	return "CD-ROM";
    } else if (msp_data[ms_index].msp_disc_type == MSP_DISC_TYPE_REMOVABLE) {
    	return "Removable disk";
    } else if (msp_data[ms_index].msp_disc_type == MSP_DISC_TYPE_FLOPPY) {
    	return "Floppy disk";
    } else {
    	return "Harddisk";
    }
}

#define READ_PDU_REMAINDER(pdu_len) ((pdu_len) > 1 ? msp_read(clt_bio, (char*)&pdu+1, (pdu_len)-1) : 0)

void
pp_usb_msp_handle_connection(pp_net_conn_data_t * conn_data)
{
    eric_session_int_id_t session = 0;
    char clt_ip[INET6_ADDRSTRLEN];
    msp_connection_t * conn = NULL;
    u_int ms_index = UINT_MAX;
    int old_connection_status;
    int error = 0;

    pp_bio_get_peer_name(conn_data->bio, clt_ip, sizeof(clt_ip));
    
    pp_log("MSP connection from %s established.\n", clt_ip);
    
    // initialize the connection
    if ((msp_init_connection(conn_data->bio, &session, clt_ip, &old_connection_status, &ms_index) != 0)
	|| (ms_index == UINT_MAX)) {
    	pp_log("MSP connection cancelled.\n");
    	goto bail;
    }
    
    conn = &msp_connections[ms_index];
    memcpy(conn->clt_ip, clt_ip, sizeof(clt_ip));
    conn->session = session;
    conn->last_requested_session_touch = 0;
    
    if (conn->session) {
        eric_session_register_close_cb(conn->session, msp_close_callback);
    }
    
    // we have a useful connection
    if (usb_ms_set_msp(ms_index, &error, 1)) {
    	pp_log("Setting msp image failed.\n");
    	pp_usb_ms_unset_image(ms_index, &error);
    	goto bail;
    }
    
    // initialize some variables
    memset(&conn->last_read_request, 0, sizeof(conn->last_read_request));
    memset(&conn->last_read_response, 0, sizeof(conn->last_read_response));
    conn->cd_read_data_finished = 0;
    conn->cd_write_data_finished = 0;
    conn->cd_read_data = NULL;
    conn->cd_write_data = NULL;
    
    // save the bio for the other thread
    conn->conn_bio = conn_data->bio;
    
    // don't close the connection if no ping pong data was received
    conn->close_connection = 0;
    conn->kill_thread = 0;
    
    // run the protocol
    msp_process_protocol(conn);
    
    // bio no longer needed    
    conn->conn_bio = NULL;
    
    // image is gone
    pp_usb_ms_unset_image(ms_index, &error);
    
 bail:
    if (conn != NULL) conn->session = 0;
    eric_net_close(conn_data->bio, NOWAIT);
    free(conn_data);
    
    pp_log("MSP connection closed.\n");
    
    // reset the client connection state
    // we must not use the conn variable here because it might not be set!
    if (ms_index != UINT_MAX) {
    	msp_connections[ms_index].client_connected = old_connection_status;
    }
    
    if (session) eric_session_release(session);
}

usb_device_type_t
usb_msp_get_image_type(u_int ms_index)
{
    usb_device_type_t rv;
    
    switch(msp_data[ms_index].msp_disc_type) {
      case MSP_DISC_NONE:
	DBG("Image type is PP_USB_IMAGE_TYPE_NONE\n");
    	rv = PP_USB_IMAGE_TYPE_NONE;
	break;
      case MSP_DISC_TYPE_CDROM:
    	DBG("Image type is PP_USB_IMAGE_TYPE_ISO\n");
    	rv = PP_USB_IMAGE_TYPE_ISO;
	break;
      case MSP_DISC_TYPE_REMOVABLE:
	DBG("Image type is PP_USB_IMAGE_TYPE_REMOVABLE\n");
	rv = PP_USB_IMAGE_TYPE_REMOVABLE;
	break;
      case MSP_DISC_TYPE_FLOPPY:
    	DBG("Image type is PP_USB_IMAGE_TYPE_FLOPPY\n");
    	rv = PP_USB_IMAGE_TYPE_FLOPPY;
	break;
      default:
#ifdef PRODUCT_SMIDC
	// Supermicro needs this type as default
	//   because otherwise W2003 won't assign a drive letter
	//   (for all other products we accept this W2003 bug)
    	DBG("Image type is PP_USB_IMAGE_TYPE_REMOVABLE\n");
    	rv = PP_USB_IMAGE_TYPE_REMOVABLE;
#else
    	DBG("Image type is PP_USB_IMAGE_TYPE_FIXED\n");
    	rv = PP_USB_IMAGE_TYPE_FIXED;
#endif
    }
    return rv;
}

void
usb_msp_disconnect(u_int ms_index)
{
    DBG("Disconnecting msp connection %u...\n", ms_index);
    
    if (ms_index < PP_FEAT_USB_MASS_STORAGE_NO) {
	msp_connections[ms_index].kill_thread = 1;
    }
}

static int
usb_msp_setup_file_info_internal(u_int ms_index, int do_cleanup)
{
#ifdef MSP_USE_ADAPTER
    pp_msp_file_info_t file_info = {
	.dev_no = ms_index,
	.block_length = do_cleanup ? 0 : msp_data[ms_index].msp_sector_size,
	.block_count  = do_cleanup ? 0 : msp_data[ms_index].msp_last_sectorno + 1
    };
    int ret = -1;

    if (ioctl(msp_fd, PP_MSP_IOC_SETFILEINFO, &file_info) == -1) {
	pp_log_err("%s(): ioctl(PP_MSP_IOC_SETFILEINFO)", ___F);
	goto bail;	
    }

    ret = 0;
    
 bail:
    return ret;
#else /* !MSP_USE_ADAPTER */
    (void)ms_index;
    (void)do_cleanup;
    return 0;
#endif /* !MSP_USE_ADAPTER */
}

int
usb_msp_setup_file_info(u_int ms_index)
{
    return usb_msp_setup_file_info_internal(ms_index, 0);
}

int
usb_msp_cleanup_file_info(u_int ms_index)
{
    return usb_msp_setup_file_info_internal(ms_index, 1);
}

/* serve a data request from the kernel */
#ifdef MSP_USE_ADAPTER
void
msp_request_data(u_int ms_index, pp_msp_file_request_t *request, pp_msp_file_response_t *file_response)
#else
void /* !MSP_USE_ADAPTER */
usb_msp_request_data(u_int ms_index, file_request_t *request, file_response_t *file_response)    
#endif /* !MSP_USE_ADAPTER */
{
    struct timeval now;
    struct timespec timeout;
    msp_connection_t * conn;
    
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) return;
    
    conn = &msp_connections[ms_index];
    
    int retcode;

#if !defined(MSP_USE_ADAPTER)
    // check for a repeated request
    if (conn->last_read_request.id == request->id &&
        conn->last_read_request.length == request->length &&
        conn->last_read_request.block_address == request->block_address &&
        conn->last_read_request.block_length == request->block_length) {
        
        DBG("detected repeated request, using last received data.\n");
        
        memcpy(file_response, &conn->last_read_response, sizeof(conn->last_read_response));
        goto bail_nolock;
    }
#endif /* MSP_USE_ADAPTER */
    
    DBG("requesting for %d: %d sectors, beginning from %d\n",
	ms_index, request->length, request->block_address);
    
    file_response->dev_no = ms_index;
    
    // request data from the connected msp client
    if (conn->conn_bio == NULL || 
        msp_send_data_request_message(conn, request->length, request->block_address)) {
        
    	pp_log("Cannot send CD-ROM data request to the client.\n");
    	file_response->response = FILE_DATA_BAD;
    	return;
    }
    
    DBG("data request sent.\n");
    
    // now wait for the data or an error/timeout
    gettimeofday(&now, NULL);
    timeout.tv_sec = now.tv_sec + MSP_DATA_TIMEOUT;
    timeout.tv_nsec = now.tv_usec * 1000;
    
    MUTEX_LOCK(&conn->msp_read_data_mtx);
    
    if (!conn->cd_read_data_finished) {
    	DBG("Waiting...\n");
    	retcode = pthread_cond_timedwait(&conn->msp_read_data_cond,
                                         &conn->msp_read_data_mtx,
                                         &timeout);
    	DBG("...done\n");
    
    	if (retcode == ETIMEDOUT) {
    	    pp_log("timeout waiting for CD-ROM sectors from client.\n");
    	    file_response->response = FILE_DATA_BAD;
    	    goto bail;
    	}
    } else {
    	DBG("Waiting not needed, already got the data.\n");
    }
    
    DBG("got the sectors.\n");
    
    conn->cd_read_data_finished = 0;
    
    // now hand over the sectors to the usb kernel thread
    if (conn->cd_read_data && conn->cd_read_code == MSP_DATA_OKAY && conn->cd_read_sectors == request->length) {
    	file_response->response = FILE_DATA_OK;
    	file_response->length = request->block_length * request->length;
	file_response->data = conn->cd_read_data;
    } else {
	pp_log("Read data failed (cd_read_sectors=%d, request->length=%d).\n",
	       conn->cd_read_sectors, request->length);
    	file_response->response = FILE_DATA_BAD;
    }
    
 bail:
    MUTEX_UNLOCK(&conn->msp_read_data_mtx);

#if !defined(MSP_USE_ADAPTER)
 bail_nolock:
    memcpy(&conn->last_read_request, request, sizeof(conn->last_read_request));
    memcpy(&conn->last_read_response, file_response, sizeof(conn->last_read_response));
#endif /* !MSP_USE_ADAPTER */
}

/* serve a write request from the kernel */
#ifdef MSP_USE_ADAPTER
void
msp_save_write_data(u_int ms_index, pp_msp_file_request_t *request, pp_msp_file_response_t *file_response)
#else /* !MSP_USE_ADAPTER */
void
usb_msp_save_write_data(u_int ms_index, file_request_t *request, file_response_t *file_response)
#endif /* !MSP_USE_ADAPTER */
{
    msp_connection_t * conn;
    size_t len;
    
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) return;
    
    conn = &msp_connections[ms_index];

    DBG("Writing for %d: %d sectors, beginning from %d\n",
	ms_index, request->length, request->block_address);
    
    file_response->dev_no = ms_index;
    
    len = request->block_length * request->length;
    
    if (len > file_response->length) {
        free(conn->cd_write_data);
        conn->cd_write_data = NULL;
    }
    if (conn->cd_write_data == NULL) {
    	conn->cd_write_data = malloc(len + sizeof(msp_send_data_pdu_t));
    }
    
    if (conn->cd_write_data == NULL) {
    	file_response->response = FILE_DATA_BAD;
    } else {
     	file_response->response = FILE_DATA_OK;
    	file_response->length = len;
    	// reserve space for the send data pdu header */
    	file_response->data = conn->cd_write_data + sizeof(msp_send_data_pdu_t);
    }
}

/* serve a write done request from the kernel
   write the previously written data to the network */
#ifdef MSP_USE_ADAPTER
void
msp_do_write_data(u_int ms_index, pp_msp_file_request_t *request)
#else /* !MSP_USE_ADAPTER */
void
usb_msp_do_write_data(u_int ms_index, file_request_t *request)
#endif /* !MSP_USE_ADAPTER */
{
    struct timeval now;
    struct timespec timeout;
    
    int retcode;
    
    msp_connection_t * conn;
    
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) return;
    
    conn = &msp_connections[ms_index];

    msp_send_data_pdu_t *pdu = (msp_send_data_pdu_t*) conn->cd_write_data;
    
    DBG("Finally writing for %d: %d sectors, beginning from %d\n",
	ms_index, request->length, request->block_address);
    
    if (conn->cd_write_data == NULL) {
    	pp_log("Cannot write MSP data, pointer is NULL!\n");
    	goto bail_nolock;
    }
    
    // fill out header
    pdu->type_8 = MSP_TYPE_SEND_DATA;
    pdu->code_8 = MSP_DATA_OKAY;
    pdu->actual_count_be16 = cpu_to_be16(request->length);
    pdu->sector_size_be16 = cpu_to_be16(request->block_length);
    pdu->sector_be32 = cpu_to_be32(request->block_address);
    
    // send the data
    if (conn->conn_bio == NULL || 
        msp_write(conn->conn_bio, conn->cd_write_data, request->length * request->block_length + sizeof(msp_send_data_pdu_t))) {
        
    	pp_log("Cannot send CD-ROM data to the client.\n");
    	goto bail_nolock;
    }
    
    // wait for the reply
    gettimeofday(&now, NULL);
    timeout.tv_sec = now.tv_sec + MSP_DATA_TIMEOUT;
    timeout.tv_nsec = now.tv_usec * 1000;
    
    MUTEX_LOCK(&conn->msp_write_data_mtx);
    
    if (!conn->cd_write_data_finished) {
    	DBG("Waiting...\n");
    	retcode = pthread_cond_timedwait(&conn->msp_write_data_cond,
                                         &conn->msp_write_data_mtx,
                                         &timeout);
    	DBG("...done\n");
    
    	if (retcode == ETIMEDOUT) {
    	    pp_log("timeout waiting for data ack from client.\n");
    	    goto bail;
    	}
    } else {
    	DBG("Waiting not needed, already got the ack.\n");
    }
    
    DBG("got the ack from the client.\n");
    
    conn->cd_write_data_finished = 0;

 bail:
    MUTEX_UNLOCK(&conn->msp_write_data_mtx);
    
 bail_nolock:
    if (conn->cd_write_data) {
    	free(conn->cd_write_data);
    	conn->cd_write_data = NULL;
    }
}

/* MSP protocol processing functions */
static int
msp_init_connection(BIO *clt_bio, eric_session_int_id_t *session, const char *clt_ip, int * old_connection_status, int * ms_index)
{
    int reason = MSP_CONNECTION_RESP_OKAY;
    int block_connection = 0;
    msp_connection_capability_t caps;
    
    int login_result, rq_result;
    
    if (msp_disabled()) {
    	pp_log("MSP connection not allowed.\n");
    	reason = MSP_CONNECTION_RESP_NOT_AVAILABLE;
    	block_connection = 1;
    }
    
    // read the init string
    if (msp_read_init_string(clt_bio) < 0) {
    	goto fail;
    }
    
    // negotiate protocol version
    if (negotiate_protocol_version(clt_bio, &caps) < 0) {
    	goto fail;
    }
    
    // okay, we know that we have an MSP protocol client
    // from now on, we can send an error code to the client in case of an error
    
    // process the login
    login_result = msp_process_authentication(clt_bio, clt_ip, session);
    if (login_result < 0) {
    	goto fail;
    }
    if (login_result != MSP_CONNECTION_RESP_OKAY) {
    	block_connection = 1;
    	reason = login_result;
    }
    
    // answer the login
    if (msp_answer_login(clt_bio, block_connection, reason) < 0) {
    	goto fail;
    }
    
    // continue only if the connection is acknowledged
    if (block_connection) {
    	goto fail;
    }
    
    // now read the image information
    if ((rq_result = msp_read_connection_info(clt_bio, ms_index, &caps)) < 0) {
    	goto fail;
    }

    if (rq_result != MSP_CONNECTION_RESP_OKAY) {
    	block_connection = 1;
    	reason = rq_result;
        // answer the image information
        if (msp_answer_login(clt_bio, block_connection, reason) < 0) {
            goto fail;
        }
    } else {
        msp_connection_t * conn = &msp_connections[*ms_index];
        memcpy(&conn->caps, &caps, sizeof(msp_connection_capability_t));
        
        // save the current state
        MUTEX_LOCK(&conn->client_connected_mutex);
        *old_connection_status = conn->client_connected;
        block_connection = conn->client_connected;
        conn->client_connected = 1;
        MUTEX_UNLOCK(&conn->client_connected_mutex);
        
        if (block_connection && reason == MSP_CONNECTION_RESP_OKAY) {
            pp_log("Another MSP connection is already established.\n");
            reason = MSP_CONNECTION_RESP_ALREADY_CONNECTED;
        } else if (pp_usb_ms_get_image_type(*ms_index) != PP_USB_MS_IMAGE_NONE) {
            pp_log("Another image is already in use.\n");
            reason = MSP_CONNECTION_RESP_ALREADY_IMAGE;
            block_connection = 1;
        }
        
        // answer the image information
        if (msp_answer_login(clt_bio, block_connection, reason) < 0) {
            goto fail;
        }
    }
        
    return block_connection ? -1 : 0;
    
 fail:
    return -1;
}

static void
msp_process_protocol(msp_connection_t * conn)
{
    int ret = 0;
    
    msp_c2s_pdu_t pdu;
    
    int alive;
    
    while(1) {
 test_alive:
    	alive = msp_test_alive(conn);
    	if (alive == PING_PONG_NO_DATA) {
    	    goto test_alive;
    	} else if (alive == PING_PONG_ERROR) {
    	    pp_log("Error waiting for pong message.\n");
    	    break;
    	} else if (alive == PING_PONG_CLOSE) {
    	    pp_log("Connection closed from outside.\n");
    	    break;
    	} else if (alive != PING_PONG_DATA_AVAILABLE) {
    	    // this means PING_PONG_NO_RESPONSE or unknown value
    	    pp_log("No response to ping pong message, quitting.\n");
    	    break;
    	}

    	if (eric_net_read_exact(conn->conn_bio, &pdu, 1, 1)) {
    	    DBG("error in read.\n");
    	    break;
    	}
    	
        /* touch the session */
        if (conn->session) {
            time_t now = time(NULL);
            if (now != conn->last_requested_session_touch) {
        	if (!vmedia_timeout_enabled || (pdu.type != MSP_TYPE_PING && pdu.type != MSP_TYPE_PONG)) {
        	    eric_session_touch(conn->session);
        	}
                conn->last_requested_session_touch = now;
            }
        }
    	
	switch (pdu.type) {
	    case MSP_TYPE_PING:
	    	ret = msp_process_ping_message(conn);
	    	break;
	    
	    case MSP_TYPE_PONG:
	    	ret = msp_process_pong_message(conn);
	    	break;
	    
	    case MSP_TYPE_QUIT_CONNECTION:
	    	ret = msp_process_quit_message(conn);
	    	break;
	    	
	    case MSP_TYPE_SEND_DATA:
	    	ret = msp_process_send_data(conn);
	    	break;
	    
	    case MSP_TYPE_DATA_ACK:
	    	ret = msp_process_data_ack(conn);
	    	break;
	    
	    case MSP_TYPE_RQ_CONNECTION:
	    	ret = msp_process_medium_change(conn);
	    	break;
	    
	    default:
		pp_log("Received unknown MSP message: 0x%02x\n", pdu.type);
		break;
    	}
    	
    	if (ret != 0) {
    	    break;
    	}
    }
    
    // inform the other thread that the connection has died
    if (pthread_cond_signal(&conn->msp_read_data_cond) != 0) {
	pp_log("%s(): pthread_cond_signal() failed\n", ___F);
    }
    if (pthread_cond_signal(&conn->msp_write_data_cond) != 0) {
	pp_log("%s(): pthread_cond_signal() failed\n", ___F);
    }
    
}

// protocol processing helper functions

static int
msp_read_init_string(BIO *clt_bio)
{
    char init_pdu[msp_prot_init_pdu_size + 1];
    
    if (msp_read(clt_bio, &init_pdu, msp_prot_init_pdu_size) < 0) {
    	pp_log("Cannot read MSP init string.\n");
    	return -1;
    }
    
    init_pdu[msp_prot_init_pdu_size] = '\0';
    if (strcmp(init_pdu, MSP_PROTOCOL_INIT_MESSAGE)) {
    	pp_log("MSP protocol error.\n");
    	DBG("expected: %s\ngot: %s\n", MSP_PROTOCOL_INIT_MESSAGE, init_pdu);
    	return -1;
    }
    
    return 0;
}

static int
negotiate_protocol_version(BIO * clt_bio, msp_connection_capability_t * caps)
{
    int my_version;
    msp_prot_ver_pdu_t ver_pdu;
    int clt_version_major, clt_version_minor;
    
    sprintf(ver_pdu, MSP_PROTOCOL_VERSION_FORMAT, MSP_PROTOCOL_MAJOR_VERSION, MSP_PROTOCOL_MINOR_VERSION);
    
    if (msp_write(clt_bio, &ver_pdu, msp_prot_ver_pdu_size) < 0) {
    	pp_log("Cannot write protocol version!\n");
    	return -1;
    }
    
    if (msp_read(clt_bio, &ver_pdu, msp_prot_ver_pdu_size) < 0) {
    	pp_log("Cannot read client protocol version!\n");
    	return -1;
    }
    
    if (sscanf(ver_pdu, MSP_PROTOCOL_VERSION_FORMAT, &clt_version_major, &clt_version_minor) != 2) {
    	pp_log("Invalid protocol!\n");
    	return -1;
    }
    
    pp_log("Client supports protocol version %d.%d.\n", clt_version_major, clt_version_minor);
    
    /* set some parameters */
    memset(caps, 0, sizeof(msp_connection_capability_t));
    
    my_version = clt_version_major * 1000 + clt_version_minor;
    if (my_version >= 1002) {
    	caps->use_connection_request_v_1_2 = 1;
    }
    
    return 0;
}

static int
msp_process_authentication(BIO *clt_bio, const char *clt_ip, eric_session_int_id_t *session)
{
    msp_c2s_pdu_t pdu;
    
    if (eric_net_read_exact(clt_bio, &pdu, 1, 1)) {
    	pp_log("Error in reading authentication.\n");
    	return -1;
    }
    
    switch (pdu.type) {
    	case MSP_TYPE_LOGIN:
    	    return msp_process_login(clt_bio, clt_ip, session);
    	case MSP_TYPE_SESSION_ID:
    	    return msp_process_session_id(clt_bio, session);
    	default:
    	    DBG("Got wrong pdu type in authentication: 0x%02x\n", pdu.type);
    	    return -1;
    }
}

static int
msp_process_login(BIO * clt_bio, const char *clt_ip, eric_session_int_id_t *session)
{
    msp_login_pdu_t pdu;
    char *user = NULL;
    char *password = NULL;
    int len;
    int server_sid;
    int gid;
    int error;
    
    int ret = MSP_CONNECTION_RESP_OKAY;
    
    if (READ_PDU_REMAINDER(sizeof(pdu)) < 0) {
    	pp_log("Cannot read login pdu!\n");
    	ret = -1;
    	goto bail;
    }
    
    // read the username
    len = be16_to_cpu(pdu.username_len_be16);
    user = malloc(len + 1);
    if (user == NULL) {
    	pp_log("Error allocating memory.\n");
    	ret = -1;
    	goto bail;
    }
    if (msp_read(clt_bio, user, len) < 0) {
    	pp_log("Cannot read login username!\n");
    	ret = -1;
    	goto bail;
    }
    user[len] = '\0';
    
    // read the password
    len = be16_to_cpu(pdu.password_len_be16);
    password = malloc(len + 1);
    if (password == NULL) {
    	pp_log("Error allocating memory.\n");
    	ret = -1;
    	goto bail;
    }
    if (msp_read(clt_bio, password, len) < 0) {
    	pp_log("Cannot read login password!\n");
    	ret = -1;
    	goto bail;
    }
    password[len] = '\0';
    
    DBG("Connection username: %s password: %s\n", user, password);

    // verify the username/password combination
    /* TODO! where to get IP from? do we need to authenticate with IP here? */
    if (PP_ERR == pp_um_user_authenticate_with_ip_str(user, password, PP_UM_AUTH_NO_FLAGS,
                                                      clt_ip, &server_sid, &gid)) {
    	pp_log("Authentication failed!\n");
    	ret = MSP_CONNECTION_RESP_AUTH_FAILED;
    	goto bail;
    }
    
    // verify the permission
    if (PP_ERR == pp_um_user_has_permission(user, "vfloppy",
                                            pp_acl_raasip_yes_str)) {
    	pp_log("Permission denied!\n");
    	ret = MSP_CONNECTION_RESP_NO_PERMISSION;
    	goto bail;
    }

    *session = eric_session_create(user, clt_ip, "", 0, &error);
    if (*session == 0) {
    	ret = MSP_CONNECTION_RESP_NO_PERMISSION;
    	goto bail;
    }

 bail:
    if (user)     free(user);
    if (password) free(password);
    
    return ret;
}

static int
msp_process_session_id(BIO * clt_bio, eric_session_int_id_t *session)
{
    int ret = MSP_CONNECTION_RESP_OKAY;
    msp_session_id_pdu_t pdu;
    int dont_check_applet_id;
    int err;
    const char *user;
    size_t i;
    FILE *f_rand;
    char buf[ERIC_ID_LENGTH + 11 + 1];
    char rand_data[ERIC_ID_LENGTH / 2];
    char challenge[ERIC_ID_LENGTH + 1];
    char resp[ERIC_RESPONSE_LENGTH + 1];
    
    if (READ_PDU_REMAINDER(sizeof(pdu)) < 0) {
    	pp_log("Cannot read session id pdu!\n");
    	ret = -1;
    	goto bail;
    }
    
    // generate a challenge for the client
    memset(rand_data, 0, sizeof(rand_data));
    if ((f_rand = fopen("/dev/urandom", "r")) != NULL) {
    	fread(rand_data, sizeof(rand_data), 1, f_rand);
    	fclose(f_rand);
    }
    for (i = 0; i < sizeof(rand_data); i++) {
    	snprintf(&challenge[i * 2], sizeof(challenge) - i * 2, "%02X", rand_data[i]);
    }
    challenge[ERIC_ID_LENGTH] = '\0';
    snprintf(buf, sizeof(buf), "MSP CHAL=%s", challenge);

    // send the challenge to the client and read the response
    if (eric_net_write_exact(clt_bio, buf, ERIC_ID_LENGTH + 9, 0) ||
        eric_net_read_exact(clt_bio, buf, ERIC_RESPONSE_LENGTH + 9, 0)) {
	
	eric_net_close(clt_bio, WAIT);
	return -1;
    }
    buf[ERIC_RESPONSE_LENGTH + 9] = '\0';

    // check applet id
    pp_cfg_is_enabled(&dont_check_applet_id, "dont_check_applet_id"); 
    if (dont_check_applet_id) {
	/* if opt is set, we create a new dummy session for super user */
	if ((*session = eric_session_create("super", "0.0.0.0", "", 0, &err)) == 0) {
	    pp_log("Could not create dummy session.\n");
	    ret = MSP_CONNECTION_RESP_NO_PERMISSION;
	    goto bail;
	}
    } else {
	sscanf(buf, "MSP RESP=%32[0-9A-F]", resp);
	
	if ((*session = eric_session_get_by_challenge_response(challenge, resp)) == 0) {
	    ret = MSP_CONNECTION_RESP_NO_PERMISSION;
	    pp_log("Checking session id failed\n");
	    goto bail;
	}
    }
    
    // check the username conencted to this session
    user = eric_session_get_user(*session);
    if (PP_ERR == pp_um_user_has_permission(user, "vfloppy",
                                            pp_acl_raasip_yes_str)) {
    	pp_log("Permission denied!\n");
    	ret = MSP_CONNECTION_RESP_NO_PERMISSION;
    	goto bail;
    }
    
 bail:
    return ret;
}

static int
msp_answer_login(BIO *clt_bio, int block_connection, int reason)
{
    msp_response_pdu_t pdu;
    
    pdu.type_8 = MSP_TYPE_RSP_CONNECTION;
    pdu.ack_8 = block_connection ? 0 : 1;
    pdu.reason_8 = reason;
    
    DBG("sending connection response.\n");
    
    if (msp_write(clt_bio, &pdu, sizeof(pdu)) < 0) {
    	pp_log("Cannot write login response.\n");
    	return -1;
    }
    
    return 0;
}

static int
msp_read_connection_info(BIO *clt_bio, int * ms_index_p, msp_connection_capability_t * caps)
{
    int ret = MSP_CONNECTION_RESP_OKAY;
    
    if (caps->use_connection_request_v_1_2) {
    	msp_rq_connection_pdu_v_1_2_t pdu;
    
    	if (msp_read(clt_bio, &pdu, sizeof(pdu)) < 0) {
    	    pp_log("Cannot read connection info pdu!\n");
    	    return -1;
    	}
    	
    	if (pdu.ms_index_8 >= PP_FEAT_USB_MASS_STORAGE_NO) {
    	    pp_log("Requested mass storage device number (%d, max=%d) does not exist.\n", pdu.ms_index_8, PP_FEAT_USB_MASS_STORAGE_NO);
    	    ret = MSP_CONNECTION_RESP_NO_SUCH_MS_INDEX;
    	    goto bail;
    	}

#if (defined(OEM_INTEL) || defined(OEM_LENOVO))
    	if (msp_force_readonly() && !pdu.ro_8) {
    	    pp_log("Write-support in Drive Redirection not allowed.\n");
    	    ret = MSP_CONNECTION_RESP_NO_PERMISSION;
    	    goto bail;
    	}
#else
    	if (msp_force_readonly()) {
    	    pp_log("Forcing Drive Redirection to read-only.\n");
    	    msp_data[*ms_index_p].msp_disc_ro = 1;
    	} else {
    	    msp_data[*ms_index_p].msp_disc_ro = pdu.ro_8;
    	}
#endif

    	*ms_index_p = pdu.ms_index_8;

        msp_data[*ms_index_p].msp_disc_ro = pdu.ro_8;
    	msp_data[*ms_index_p].msp_disc_type = pdu.disc_type_8;
    	msp_data[*ms_index_p].msp_sector_size = be16_to_cpu(pdu.sector_size_be16);
    	msp_data[*ms_index_p].msp_last_sectorno = be32_to_cpu(pdu.last_sector_no_be32);
    } else {
    	msp_rq_connection_pdu_v_1_1_t pdu;
    
    	if (msp_read(clt_bio, &pdu, sizeof(pdu)) < 0) {
    	    pp_log("Cannot read connection info pdu!\n");
    	    return -1;
    	}
    	
    	*ms_index_p = 0;
    
#if (defined(OEM_INTEL) || defined(OEM_LENOVO))
    	if (msp_force_readonly() && !pdu.ro_8) {
    	    pp_log("Write-support in Drive Redirection not allowed.\n");
    	    ret = MSP_CONNECTION_RESP_NO_PERMISSION;
    	    goto bail;
    	}
#else
    	if (msp_force_readonly()) {
    	    pp_log("Forcing Drive Redirection to read-only.\n");
    	    msp_data[*ms_index_p].msp_disc_ro = 1;
    	} else {
    	    msp_data[*ms_index_p].msp_disc_ro = pdu.ro_8;
    	}
#endif

    	msp_data[*ms_index_p].msp_disc_ro = pdu.ro_8;
    	msp_data[*ms_index_p].msp_disc_type = pdu.disc_type_8;
    	msp_data[*ms_index_p].msp_sector_size = be16_to_cpu(pdu.sector_size_be16);
    	msp_data[*ms_index_p].msp_last_sectorno = be32_to_cpu(pdu.last_sector_no_be32);
    }
    
    DBG("MSP image info: ro=%d, type=0x%02x, sector_size=%d, last_sectorno=%d\n",
    	msp_data[*ms_index_p].msp_disc_ro, msp_data[*ms_index_p].msp_disc_type,
    	msp_data[*ms_index_p].msp_sector_size, msp_data[*ms_index_p].msp_last_sectorno);

 bail:
    return ret;
}

static int
msp_test_alive(msp_connection_t * conn)
{
    int fd;
    int i;
    
    struct timeval to;
    fd_set aliveset;
    
    msp_ping_pdu_t ping;

    if (!conn->conn_bio || (fd = BIO_get_fd(conn->conn_bio, NULL)) == -1) {
    	DBG("Internal error.\n");
    	return PING_PONG_ERROR;
    }

    for (i = 0; i < PING_PONG_TIMEOUT; i++) {
    	FD_ZERO(&aliveset);
    	FD_SET(fd, &aliveset);
    	
    	to.tv_sec = POLL_TIMEOUT;
    	to.tv_usec = 0;
    	
    	if (select(fd + 1, &aliveset, NULL, NULL, &to) < 0) {
	    perror("THREAD: msp_test_alive: select");
	    return PING_PONG_ERROR;
    	}
    	
    	if(FD_ISSET(fd, &aliveset)) {
	    // data is available, everything is okay
	    conn->close_connection = 0;
	    //DBG("data available.\n");
	    return PING_PONG_DATA_AVAILABLE;
    	}
    
    	if (conn->kill_thread) {
    	    return PING_PONG_CLOSE;
    	}
    }
    
    // no data available, send a ping message or kill the connection
    // if the ping message has already been sent
    if (!conn->close_connection) {
    	//DBG("no data available, sending ping\n");
    	ping.type_8 = MSP_TYPE_PING;
    	if (msp_write(conn->conn_bio, &ping, sizeof(ping))) {
    	    DBG("Cannot write pong message.\n");
    	    return PING_PONG_ERROR;
    	}
    	conn->close_connection = 1;
    	return PING_PONG_NO_DATA;
    } else {
    	return PING_PONG_NO_RESPONSE;
    }
}

static int
msp_process_ping_message(msp_connection_t * conn)
{
    msp_ping_pdu_t pdu;
    msp_pong_pdu_t pong;
    BIO * clt_bio = conn->conn_bio;
    
    int ret;
    
    // receive the ping
    if ((ret = READ_PDU_REMAINDER(sizeof(pdu))) < 0) {
    	return ret;
    }
    
    // and send a pong
    pong.type_8 = MSP_TYPE_PONG;
    
    return msp_write(clt_bio, &pong, sizeof(pong));
}

static int
msp_process_pong_message(msp_connection_t * conn)
{
    msp_pong_pdu_t pdu;
    BIO * clt_bio = conn->conn_bio;
    
    // just read the pong
    return READ_PDU_REMAINDER(sizeof(pdu));
}

static const char*
get_quit_message(u_int8_t reason)
{
    switch(reason) {
    	case MSP_QUIT_USER_CANCELLED:
    	    return "User cancelled connection";
    	case MSP_QUIT_DEVICE_CANCELLED:
    	    return "User cancelled connection";
    }
    
    return "Unknown quit reason";
}

static int
msp_process_quit_message(msp_connection_t * conn)
{
    msp_quit_pdu_t pdu;
    BIO * clt_bio = conn->conn_bio;
    int ret;
    
    if ((ret = READ_PDU_REMAINDER(sizeof(pdu)))) {
    	pp_log("Error reading MSP quit message.\n");
    	return ret;
    }
    
    pp_log("MSP connection quit: %s.\n", get_quit_message(pdu.reason_8));
    
    // return -1 here to signalize that the connection should be closed
    return -1;
}

static int
msp_send_data_request_message(msp_connection_t * conn, u_int16_t count, u_int32_t sector)
{
    msp_rq_data_pdu_t pdu;
    BIO * clt_bio = conn->conn_bio;
    
    pdu.type_8 = MSP_TYPE_RQ_DATA;
    pdu.count_be16 = cpu_to_be16(count);
    pdu.sector_be32 = cpu_to_be32(sector);

    return msp_write(clt_bio, &pdu, sizeof(pdu));
}

static int
msp_process_send_data(msp_connection_t * conn)
{
    msp_send_data_pdu_t pdu;
    BIO * clt_bio = conn->conn_bio;
    int ret = 0;
    
    if ((ret = READ_PDU_REMAINDER(sizeof(pdu)))) {
    	pp_log("Error reading MSP data header!\n");
    	conn->cd_read_data_finished = 1;
    	return ret;
    }
    
    conn->cd_read_code = pdu.code_8;
    conn->cd_read_sectors = be16_to_cpu(pdu.actual_count_be16);
    msp_data[conn->index].msp_sector_size = be16_to_cpu(pdu.sector_size_be16);
    
    MUTEX_LOCK(&conn->msp_read_data_mtx);
    
    if (conn->cd_read_code != MSP_DATA_OKAY) {
    	pp_log("Error during data transfer! (code=0x%02x)\n", conn->cd_read_code);
    	ret = -1;
    	goto bail;
    }
    
    DBG("cd_read_sectors=%d, msp_sector_size=%d\n", conn->cd_read_sectors, msp_data[conn->index].msp_sector_size);
    if (conn->cd_read_data) {
    	free(conn->cd_read_data);
    	conn->cd_read_data = NULL;
    }
    conn->cd_read_data = malloc(conn->cd_read_sectors * msp_data[conn->index].msp_sector_size);
    if (!conn->cd_read_data) {
    	pp_log("%s(): Error allocating memory for CD-ROM data!\n", ___F);
    	conn->cd_read_code = MSP_DATA_ERROR;
    	goto bail;
    }
    
    if ((ret = msp_read(clt_bio, conn->cd_read_data, conn->cd_read_sectors * msp_data[conn->index].msp_sector_size))) {
    	pp_log("Error reading MSP data!\n");
    	goto bail;
    }
    
    DBG("read %d bytes of data.\n", conn->cd_read_sectors * msp_data[conn->index].msp_sector_size);
    
bail:    
    conn->cd_read_data_finished = 1;
    
    // wake up the other thread
    if (pthread_cond_signal(&conn->msp_read_data_cond) != 0) {
	pp_log("%s(): pthread_cond_signal() failed\n", ___F);
    }
    
    MUTEX_UNLOCK(&conn->msp_read_data_mtx);

    return ret;
}

static int
msp_process_data_ack(msp_connection_t * conn)
{
    msp_data_ack_pdu_t pdu;
    BIO * clt_bio = conn->conn_bio;
    int ret = 0;
    
    if ((ret = READ_PDU_REMAINDER(sizeof(pdu)))) {
    	pp_log("Error reading MSP data ack message!\n");
    	conn->cd_write_data_finished = 1;
    	return ret;
    }
    
    conn->cd_write_ack = pdu.ack_8;
    
    if (conn->cd_write_ack) {
    	DBG("Data successfully written.\n");
    } else {
    	DBG("Data write not successful!\n");
    }

    MUTEX_LOCK(&conn->msp_write_data_mtx);
    
    conn->cd_write_data_finished = 1;
    
    // wake up the other thread
    if (pthread_cond_signal(&conn->msp_write_data_cond) != 0) {
	pp_log("%s(): pthread_cond_signal() failed\n", ___F);
    }
    
    MUTEX_UNLOCK(&conn->msp_write_data_mtx);

    return ret;
}

static int
msp_process_medium_change(msp_connection_t * conn)
{
    int ret = 0;
    BIO * clt_bio = conn->conn_bio;
    int error = 0;
    // this is only supported since MSP 1.02
    msp_rq_connection_pdu_v_1_2_t pdu;

    if ((ret = READ_PDU_REMAINDER(sizeof(pdu)))) {
    	pp_log("Error reading MSP connection request message!\n");
    	return ret;
    }
    
    msp_data[conn->index].msp_disc_type = pdu.disc_type_8;
    msp_data[conn->index].msp_sector_size = be16_to_cpu(pdu.sector_size_be16);
    msp_data[conn->index].msp_last_sectorno = be32_to_cpu(pdu.last_sector_no_be32);
    
    ret = usb_ms_set_msp(conn->index, &error, 0);
    
    return ret;
}

#ifdef MSP_USE_ADAPTER
/**
 * the MSP thread
 * handles requests from the MSP adapter backend
 */
static void*
msp_thread_func(void* data UNUSED)
{
    int i;
    struct pollfd fds[MAX_BE_DESCRIPTORS];

    assert(msp_fd != -1);

    for (i = 0; i < MAX_BE_DESCRIPTORS; i++) {
        fds[i].fd = -1;
    }

    fds[0].fd = msp_fd;
    fds[0].events = POLLIN;

    while (!msp_thread_should_die) {
	pp_msp_file_request_t request;
	pp_msp_file_response_t response;

	if ((i = poll(fds, MAX_BE_DESCRIPTORS, 1000)) == 0 || (i < 0 && errno == EINTR)) {
            continue;
        } else if (i < 0) {
            pp_log_err("%s(): poll()", ___F);
            break;
        }

	DBG("MSP: We have a request\n");
	
	if (ioctl(msp_fd, PP_MSP_IOC_GETREQUEST, &request) == -1) {
	    if (errno != ENODATA) {
		pp_log_err("%s(): ioctl(PP_MSP_IOC_GETREQUEST)", ___F);
		break;
	    } else {
		continue;
	    }
	}

	DBG("MSP: Got request %d for %d\n", request.cmd, request.dev_no);
	
	switch (request.cmd) {
	  case PP_MSP_READ:
	      msp_request_data(request.dev_no, &request, &response);
	      break;
	  case PP_MSP_WRITE:
	      msp_save_write_data(request.dev_no, &request, &response);
	      break;
	  case PP_MSP_WRITE_DONE:
	      msp_do_write_data(request.dev_no, &request);
	      break;
	  default:
	      pp_log("%s(): unknown command %d", ___F, request.cmd);
	      break;
	}

	if (ioctl(msp_fd, PP_MSP_IOC_SETRESPONSE, &response) < 0) {
	    pp_log_err("%s(): ioctl(PP_MSP_IOC_SETRESPONSE)", ___F);
	    break;
	}
    }

    return NULL;
}
#endif /* MSP_USE_ADAPTER */

static int vmedia_timeout_enabled_ch(pp_cfg_chg_ctx_t *ctx UNUSED) {
    vmedia_timeout_enabled = get_vmedia_timeout_enabled();
    return PP_SUC;
}

static int get_vmedia_timeout_enabled() {
    int _vmedia_logout_enabled = 0;
    pp_cfg_is_enabled(&_vmedia_logout_enabled, "security.idle_timeout.vmedia_enabled");
    return _vmedia_logout_enabled;
}

static void msp_close_callback(eric_session_int_id_t session_id) {
    int i;
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
        if (msp_connections[i].session == session_id) {
            int error;
            pp_usb_ms_unset_image(i, &error);
        }
    }
}
