/*********************************************************************
 * libpp_vsc: adc.c
 *
 * contains functions for ADC (AD9888) control and pll settings
 *
 ********************************************************************/

#include <math.h>
#include <pp/base.h>
#include "adc.h"
#include "adc_io.h"
#include "debug.h"
#include "mode.h"
#include "vsc_funcs.h"

/**************************************************************
 * Common ADC functions
 **************************************************************/

void adc_set_sync_input(vsc_context_t *context) {
    adc_tx_byte_masked(context, ADC_REG_GENERAL_MODE, 
	                ADC_GENERAL_MODE_INP_CHANNEL1,
			(vsc_use_2nd_sync(context) ? ADC_GENERAL_MODE_INP_CHANNEL1 : 0));
}

int
adc_init(vsc_context_t *context)
{
    DCH(D_BLABLA, "adc_init port\n");
    
    adc_tx_byte(context, ADC_REG_TEST_1, ADC_TEST_1_DEFAULT);
    adc_tx_byte(context, ADC_REG_TEST_2, ADC_TEST_2_DEFAULT);

    adc_tx_byte(context, ADC_REG_HSYNC_O_PULSE, ADC_HSYNC_O_PULSE_VALUE);

    // setup sync control
    adc_tx_byte(context, ADC_REG_SYNC_CONTROL,
		ADC_SYNC_CONTROL_ACT_HSYNC_SET	|
		ADC_SYNC_CONTROL_ACT_VSYNC_SET);
    
    // setup clamp/coast control
    adc_tx_byte(context, ADC_REG_CLAMP_CONTROL,
		ADC_CLAMP_CONTROL_STOP_PWRDOWN);
  
    // set general mode
#if defined(LARA_KIMSMI) || defined(PP_FEAT_VSC_SINGLE_PIXEL)
    adc_tx_byte(context, ADC_REG_GENERAL_MODE,
		ADC_GENERAL_MODE_BW_150MHZ);
#else
    adc_tx_byte(context, ADC_REG_GENERAL_MODE,
		ADC_GENERAL_MODE_DUAL_CHANNEL |
# ifdef PP_BOARD_KIRA
		/* remove this ifdef when we have new FPGA binaries
		   which work with the the parallel mode */
		ADC_GENERAL_MODE_PARALLEL     |
# endif
		ADC_GENERAL_MODE_BW_150MHZ);
#endif
    adc_set_sync_input(context);
    
    // coast default
    adc_set_coast_env(context, PP_FALSE, 0, 0);
    
    adc_sync_mode(context, SEPARATE_SYNC);
	
    return 0;
}

void
adc_set_clamp(vsc_context_t *context, u_int8_t placement,
	      u_int8_t duration)
{
    D(D_VERBOSE, "adc_set_clamp placement %02x, duration %02x\n",
      placement, duration);

    adc_tx_byte(context, ADC_REG_CLAMP_PLACEMENT, placement);
    adc_tx_byte(context, ADC_REG_CLAMP_DURATION, duration);
}

void
adc_set_bandwidth(vsc_context_t *context, u_int8_t bandwidth)
{
    u_int8_t val;

    D(D_VERBOSE, "adc_set_bandwidth %02x\n", bandwidth);
    
    val =  adc_rx_byte(context, ADC_REG_GENERAL_MODE);
    val &= 0xF9;
    val |= bandwidth;
    adc_tx_byte(context, ADC_REG_GENERAL_MODE, val);
}

pp_bool_t
adc_hsync_on(vsc_context_t *context)
{
    return (adc_rx_byte(context, ADC_REG_SYNC_DETECT) &
	    ADC_SYNC_DETECT_HSYNC_DETECT) ? PP_TRUE : PP_FALSE;
}

pp_bool_t
adc_vsync_on(vsc_context_t *context)
{
    return (adc_rx_byte(context, ADC_REG_SYNC_DETECT) &
	    ADC_SYNC_DETECT_VSYNC_DETECT) ? PP_TRUE : PP_FALSE;
}
  
void
adc_sync_mode(vsc_context_t *context, adc_sync_mode_t mode)
{
    D(D_VERBOSE, "adc_sync_mode %02x -> %02x\n", context->adc_sync_mode, mode);
    
    switch (mode) {
      case SEPARATE_SYNC:
	  if (context->adc_sync_mode != SEPARATE_SYNC) {
	      adc_tx_byte(context, ADC_REG_SYNC_CONTROL,
			  adc_rx_byte(context, ADC_REG_SYNC_CONTROL) & ~ADC_SYNC_CONTROL_ACT_VSYNC_SSV);

	      context->adc_sync_mode = SEPARATE_SYNC;
	  }
	  break;
      case COMPOSITE_SYNC:
	  if (context->adc_sync_mode != COMPOSITE_SYNC) {	      
	      adc_tx_byte(context, ADC_REG_SYNC_CONTROL,
			  adc_rx_byte(context, ADC_REG_SYNC_CONTROL) | ADC_SYNC_CONTROL_ACT_VSYNC_SSV);

	      context->adc_sync_mode = COMPOSITE_SYNC;
	  }
	  break;
      default:
	  D(D_ERROR, "Unknown adc_sync_mode %d specified\n", mode);
	  break;
    }
}

/**************************************************************
 * PLL Functions
 **************************************************************/

/**
 * set the PLL according to the required horizontal frequency
 * in consideration of the current total line length
 */
int
adc_set_pll(vsc_context_t *context, u_int16_t freqH, u_int16_t clocks)
{
#define Ct_nF               (39)            // Ct in [nF]
#define PLL_STABILITY       (19.5)
    u_int16_t divider = clocks;
    u_int8_t  vco, charge_pump_idx;
    u_int16_t charge_pump_vals[8] = { 50, 100, 150, 250, 350, 500, 750, 1500 };
    float pixel_freq;	// pixel frequency in MHz
    float h_freq_hz;    // horizontal frequency in Hz  
    float vco_gain;     // calculation of vco_gain constant
    float charge_pump;
    float tmp;

    /**
     * Calculation steps (see AD9888 manual page 12)
     *
     * divider  = pixel_freq / h_freq
     * pll freq = h_freq * divider
     */
      
    pixel_freq = (float) freqH * divider / 100000.0; // MHz

    // check whether pixel freq is out of range
    if ((pixel_freq < 10.0) || (pixel_freq > 170.0)) {
	return -1;
    }

    /**
     * Settings dependend on the pixel frequence
     *
     * Vco1	Vco0	Pixel clock range (MHz)	VCO gain (MHz/V)
     *	0	 0	10 .. 45		22,5
     *	0	 1	45 .. 90		45
     *  1	 0	90 .. 150		90
     *	1	 1	150.. 205		205
     */

    if (pixel_freq <= 45.0)   {
        vco = 0; vco_gain = 22.5;
    } else if (pixel_freq <= 90.0) {
	vco = 1; vco_gain = 45.0;
    } else if (pixel_freq <= 150.0) {
	vco = 2; vco_gain = 90.0;
    } else {
	vco = 3; vco_gain = 180.0;
    }

    /**
     *   Calculate the charge pump current by the relation:
     *
     *   charge_pump <= ((h_freq * 2 * PI / PLL_STABILITY) ^ 2 ) * ((Ct * divider) / vco_gain)
     *
     */

    h_freq_hz = freqH * 10;

    charge_pump = (h_freq_hz * 2 * M_PI) / PLL_STABILITY;
    charge_pump = charge_pump * charge_pump;
    
    tmp = (Ct_nF * divider) / vco_gain;
    
    charge_pump = (charge_pump * tmp) / 1000000000;

    for (charge_pump_idx = 7; charge_pump_idx > 0; charge_pump_idx--){
    	if ((u_int16_t)charge_pump > charge_pump_vals[charge_pump_idx]) break;
    }
    if (charge_pump_idx <= 7) charge_pump_idx++;

    // program the calculated divider, vco, charge_pump values
    DCH(D_VERBOSE, "adc_set_pll Div=%d, VCO=%02x, ChargePump=%02x\n",
      divider, vco, charge_pump_idx);

    divider -= 1;
    adc_tx_byte(context, ADC_REG_PLL_DIV_MSB, (divider >> 4) & 0xFF);
    adc_tx_byte(context, ADC_REG_PLL_DIV_LSB, (divider << 4) & 0xF0);

    vco			= (vco & 0x03) << 6;
    charge_pump_idx	= (charge_pump_idx & 0x07) << 3;

    adc_tx_byte(context, ADC_REG_VCO_CPMP, vco | charge_pump_idx);

    return 0;
}

void
adc_set_coast_env(vsc_context_t *context, pp_bool_t activate_coast,
		  unsigned char pre, unsigned char post)
{
    unsigned char val;
    
    D(D_VERBOSE, "adc_set_coast_env: state=%d, pre=%d, post=%d\n",
      activate_coast, pre, post);

    val = adc_rx_byte(context, ADC_REG_CLAMP_CONTROL) & ~ADC_CLAMP_CONTROL_COAST_VSYNC;
        
    adc_tx_byte(context, ADC_REG_CLAMP_CONTROL,
		val | ((activate_coast) ? ADC_CLAMP_CONTROL_COAST_VSYNC : 0));
		    
    adc_tx_byte(context, ADC_REG_PRE_COAST, pre);
    adc_tx_byte(context, ADC_REG_POST_COAST, post);
}

/**************************************************************
 * set functions for settings struct with common interface
 **************************************************************/

int
adc_set_brightness(vsc_context_t *context, unsigned short brightness)
{
    D(D_BLABLA, "adc_set_brightness %d\n", brightness);
    
    adc_tx_byte(context, ADC_REG_RED_OFFSET,   (127-brightness)<<1);
    adc_tx_byte(context, ADC_REG_GREEN_OFFSET, (127-brightness)<<1);
    adc_tx_byte(context, ADC_REG_BLUE_OFFSET,  (127-brightness)<<1);
    
    return 0;
}

int
adc_set_brightness_r(vsc_context_t *context, unsigned short brightness)
{
    D(D_BLABLA, "adc_set_brightness_r %d\n", brightness);

    adc_tx_byte(context, ADC_REG_RED_OFFSET, (127-brightness)<<1);
    
    return 0;
}

int
adc_set_brightness_g(vsc_context_t *context, unsigned short brightness)
{
    D(D_BLABLA, "adc_set_brightness_g %d\n", brightness);

    adc_tx_byte(context, ADC_REG_GREEN_OFFSET, (127-brightness)<<1);
    
    return 0;
}

int
adc_set_brightness_b(vsc_context_t *context, unsigned short brightness)
{
    D(D_BLABLA, "adc_set_brightness_b %d\n", brightness);

    adc_tx_byte(context, ADC_REG_BLUE_OFFSET, (127-brightness)<<1);
    
    return 0;
}

int
adc_set_contrast_r(vsc_context_t *context, unsigned short contrast)
{
    D(D_BLABLA, "adc_set_contrast_r %d\n", contrast);
	
    adc_tx_byte(context, ADC_REG_RED_GAIN, 255-contrast);
    
    return 0;
}

int
adc_set_contrast_g(vsc_context_t *context, unsigned short contrast)
{
    D(D_BLABLA, "adc_set_contrast_g %d\n", contrast);
	
    adc_tx_byte(context, ADC_REG_GREEN_GAIN, 255-contrast);

    return 0;
}

int
adc_set_contrast_b(vsc_context_t *context, unsigned short contrast)
{
    D(D_BLABLA, "adc_set_contrast_b %d\n", contrast);
	
    adc_tx_byte(context, ADC_REG_BLUE_GAIN, 255-contrast);

    return 0;
}

int
adc_set_blacklevel(vsc_context_t *context, unsigned short blacklevel)
{
    adc_set_clamp(context, blacklevel, ADC_CLAMP_DURATION_DEFAULT);
    
    return 0;
}

int
adc_set_clocks(vsc_context_t *context, unsigned short clocks)
{
    u_int16_t freqH;
    
    if (mode_get_res_info(context, NULL, NULL, &freqH, NULL) != 0) {
	return -1;
    }

    if (freqH == 0) return -1;
    
    D(D_BLABLA, "adc_set_clock %d\n", clocks);
    
    return adc_set_pll(context, freqH, clocks);
}

int
adc_set_phase(vsc_context_t *context, unsigned short phase)
{
    DCH(D_BLABLA, "adc_set_phase %d\n", phase);
    phase = (phase & 0x1F) << 3;

    adc_tx_byte(context, ADC_REG_PHASE, phase);

    return 0;
}

/**************************************************************
 * get functions for settings struct with common interface
 **************************************************************/

unsigned short
adc_get_brightness(vsc_context_t *context)
{        
    return 127-(adc_rx_byte(context, ADC_REG_RED_OFFSET)>>1);
}

unsigned short
adc_get_brightness_r(vsc_context_t *context)
{        
    return 127-(adc_rx_byte(context, ADC_REG_RED_OFFSET)>>1);
}

unsigned short
adc_get_brightness_g(vsc_context_t *context)
{
    return 127-(adc_rx_byte(context, ADC_REG_GREEN_OFFSET)>>1);
}

unsigned short
adc_get_brightness_b(vsc_context_t *context)
{
    return 127-(adc_rx_byte(context, ADC_REG_BLUE_OFFSET)>>1);
}

unsigned short
adc_get_contrast_r(vsc_context_t *context)
{
    return 255-adc_rx_byte(context, ADC_REG_RED_GAIN);
}

unsigned short
adc_get_contrast_g(vsc_context_t *context)
{
    return 255-adc_rx_byte(context, ADC_REG_GREEN_GAIN);
}

unsigned short
adc_get_contrast_b(vsc_context_t *context)
{
    return 255-adc_rx_byte(context, ADC_REG_BLUE_GAIN);
}

unsigned short
adc_get_blacklevel(vsc_context_t *context)
{
    return adc_rx_byte(context, ADC_REG_CLAMP_PLACEMENT);
}

unsigned short
adc_get_clocks(vsc_context_t *context)
{
    u_int16_t msb = adc_rx_byte(context, ADC_REG_PLL_DIV_MSB);
    u_int16_t lsb = adc_rx_byte(context, ADC_REG_PLL_DIV_LSB);

    return ((msb << 4) | (lsb >> 4)) + 1;
}

unsigned short
adc_get_phase(vsc_context_t *context)
{
    u_int8_t reg = adc_rx_byte(context, ADC_REG_PHASE);

    return (reg >> 3);
}
