#ifndef _VSC_TYPES_H
#define _VSC_TYPES_H

#include <stdio.h>
#include <sys/types.h>
#include <pp/base.h>
#include <pp/vsc_types_external.h>

typedef struct _vsc_context_s vsc_context_t;

/**************************************************************
 * VSC types
 **************************************************************/

/* switch sync input for vsc, for mode change detection */
typedef enum {
    HSYNC_BUF,
    HSYNC_ADC,
} vsc_hsync_input_t;

typedef enum {
    VSYNC_BUF,
    VSYNC_ADC,
} vsc_vsync_input_t;

typedef struct {
    const short id;			// id (index in array)
    const vsc_setting_type_t type;	// global, kvm local or mode specific
    const unsigned short min;
    const unsigned short max;
    const unsigned short def;		// default value
    const unsigned short increment;	// enforce a certain increment/step size
    const unsigned char diff_mask_off;	// disable diff mask when this settings is changed
    const unsigned char adjust_vsc_ofs; // vsc offset has to be corrected after changing this
    const char * cfg_key;		// name in config fs for not mode specific opts

    // getter/setter functions for hardware access
    unsigned short (*get_function)(vsc_context_t *context);
    int (*set_function)(vsc_context_t *context, unsigned short value);
} vsc_setting_info_t;

struct vsc_mode_s;
typedef struct vsc_mode_s vsc_mode_t;

/**
 * data from a mode which gets saved,
 * differs from mode descriptor to save a little bit space
 */
typedef struct {
    u_int16_t freqH;
    u_int16_t freqV;
    u_int8_t polH;
    u_int8_t polV;

    u_int16_t ofsX;
    u_int16_t ofsY;
    u_int16_t clocks;
    u_int16_t phase;

    u_int8_t custom;
} vsc_mode_data_t;

/* data which is shared by all contexts (channels) */
typedef struct {
    pp_hash_i_t *mode_hash;
    unsigned char custom_modes_invalid;
} common_data_t;

/* custom mode usage, set by the user through config fs */
typedef enum {
    CUSTOM_MODE_OFF,
    CUSTOM_MODE_USE,
    CUSTOM_MODE_FORCE
} custom_mode_handling_t;

typedef enum {
    RESET_REQUEST_NONE,
    RESET_REQUEST_SOFT,
    RESET_REQUEST_HARD
} vsc_reset_request_t;

/* vsc feature set information */
typedef struct {
    u_char extd_sync_switch : 1; /* switch ADC/VGA syncs separately for reg, irq, and sampling */
    u_char sdram_initword   : 1; /* VSC is able to init SDRAM with a custom data word */
    u_char display_enable   : 1; /* VSC has a working display_enable signal for panel input */
} vsc_features_t;

/**************************************************************
 * ADC/Sync types
 **************************************************************/

/* (de)activate sync separater in ADC */
typedef enum {
    UNDEFINED_SYNC,
    SEPARATE_SYNC,
    COMPOSITE_SYNC
} adc_sync_mode_t;

/* currently existing sync signals on ADC input */
typedef enum {
    NONE,
    H,
    V,
    HV,
    UNDETERMINED
} sync_signals_t;

/* states for our main state machine */
typedef enum {
    INITIAL,	// (re)initializing controllers
    RESET,	// VSC in reset state
    RESET_RAM,  // VSC in reset state after SDRAM init
    UNSTABLE,	// state is changing (after IRQ)
    VFIFO_RESET,// VSC video fifo reset on resolution change
    PRESTABLE,	// on its way to stable, some preparations
    STABLE,	// stable state reached, programming mode
    POSTSTABLE,	// cleanups after mode programming
    POWERDOWN,	// no sync signal
    UNKNOWN,	// current mode unknown
    IDLE,	// operations done, waiting for events
    AUTOADJUST, // auto adjustment in progress
    FINAL	// finalize, destroy
} sync_state_t;

typedef enum {
    SYNC_CHANGED,
    SYNC_NO_CHANGE
} sync_detect_result_t;

typedef enum {
    AA_IDLE,
    AA_CLOCK_PRE,
    AA_PHASE,
    AA_CLOCK_POST,
    AA_OFFSET,
    AA_IRE,
} aa_state_t;

typedef struct {
    aa_state_t	state;
    pp_bool_t	request;
    pp_bool_t	ire_request;
    pp_bool_t	ire_process;
    pp_bool_t	do_phase;
    pp_bool_t	brute_force;
    u_int8_t	clock_retry;
    u_int8_t	phase_retry;
    u_int16_t	clock;
    u_int16_t	clock_max;
    u_int16_t	clock_opt;
    u_int8_t	phase;
    u_int8_t	phase_max;
    u_int8_t	phase_opt;
    u_int32_t	min_ire;
    u_int8_t    grab_suspend;
    
    u_int16_t	clock_bak;
    u_int8_t	phase_bak;
    u_int16_t	ofsx_bak;
    u_int16_t	ofsy_bak;    
    
    pp_stopwatch_t time;

} aa_data_t;

typedef enum {
    MODE_DET_OK,
    MODE_DET_FAILED,
    MODE_DET_TRY_AGAIN,
} mode_det_res_t;

/**************************************************************
 * timer types
 **************************************************************/

#define MAX_TIMER_COUNT 16
typedef struct {
    pp_bool_t expired;
    pp_bool_t active;
    long remaining_ms;
} timer_entry_t;

typedef enum {
    TICK_STANDARD,
    TICK_FAST
} timer_tick_t;

#endif /* _VSC_TYPES_H */
