#include <stdio.h>

#include <pp/base.h>
#include <pp/wsman.h>

#include "wsman.h"
#include "debug.h"
#include "uuid.h"
#include "fault.h"
#include "xml_parser.h"
#include "xml_writer.h"
#include "requests.h"
#include "resource.h"
#include "identify.h"

/******************************************************************************
* function definitions                                                        *
******************************************************************************/

static void create_response_header(xml_writer_t response,
				   wsman_request_t* request,
				   char *action, int fault);
static int process_action(xml_writer_t response, wsman_request_t* request,
			  int *fault_code, int *fault_detail, char **fault_detail_string,
			  char **action_string, char *auth_string);
static int process_fault(xml_writer_t response, xml_writer_t *header, xml_writer_t *body,
			 wsman_request_t *req, int fault_code, int fault_subcode,
			 char *fault_detail);

/******************************************************************************
* main request handling function                                              *
******************************************************************************/

/* parse the request */
int process_wsman_request(char * request_string, size_t req_len, char ** response, size_t *resp_len, char *auth_string) {
    int ret = PP_WSMAN_NO_ERROR;
    
    int parse_result;
    int encoding;
    wsman_request_t req;
    int fault_code = 0, fault_detail = 0;
    char *fault_detail_string = NULL;
    char *action_string = NULL;
    
    xml_writer_t response_envelope = NULL;
    xml_writer_t response_header;
    xml_writer_t response_body;
    
    init_wsman_request(&req);
    
    if (!request_string) {
    	D(D_ERROR, "No string to parse!.\n");
    	ret = PP_WSMAN_ERROR_INTERNAL;
    	goto bail;
    }
    
    /* parse the XML request */
    parse_result = parse_wsman_request(&req, request_string, req_len, &encoding, &fault_code, &fault_detail, &fault_detail_string);
    
    if (req.is_identify_request) {
        /* create envelope */
        response_envelope = print_xml_tag(NULL, "s", "Envelope", NULL,
    	    "xmlns", "s", "", namespace_soap_env,
    	    "xmlns", "wsmid", "", namespace_identity);
        response_header = print_xml_tag(response_envelope, "s", "Header", NULL, NULL);
        response_body = print_xml_tag(response_envelope, "s", "Body", NULL, NULL);

        ret = create_identify(response_body, &req);
    } else {
        /* create envelope */
        response_envelope = print_xml_tag(NULL, "s", "Envelope", NULL,
    	    "xml", "lang", "", "en-us",
    	    "xmlns", "xs", "", namespace_xmlschema,
    	    "xmlns", "xsi", "", namespace_xmlscm_inst,
    	    "xmlns", "s", "", namespace_soap_env,
    	    "xmlns", "wsa", "", namespace_addressing,
    	    "xmlns", "wsman", "", namespace_management,
    	    "xmlns", "wfx", "", namespace_transfer,
    	    "xmlns", "wsen", "", namespace_enumeration);
        response_header = print_xml_tag(response_envelope, "s", "Header", NULL, NULL);
        response_body = print_xml_tag(response_envelope, "s", "Body", NULL, NULL);
    
        /* create header and body - use fault message or result of action call */
        if (!parse_result &&
            !process_action(response_body, &req, &fault_code, &fault_detail,
        		    &fault_detail_string, &action_string, auth_string)) {
    	    create_response_header(response_header, &req, action_string, 0);
        } else {
    	    ret = process_fault(response_envelope, &response_header, &response_body,
    		    &req, fault_code, fault_detail, fault_detail_string);
        }
    }
    
    /* build XML text from xml_writer_t */
    *response = xml_writer_to_string(response_envelope, encoding, resp_len);
    
    /* check for envelope size overflow */
    if (req.max_envelope_size > 0 && *resp_len > (size_t)req.max_envelope_size) {
    	D(D_NOTICE, "MaximumEnvelopeSize exceeded!\n");
    	
    	ret = process_fault(response_envelope, &response_header, &response_body,
    		&req, FAULT_ENCODING_LIMIT, FAULT_DETAIL_MAX_ENVELOPE_SIZE, NULL);

    	if (*response) {
    	    free(*response);
    	    *response = NULL;
    	}
    	*response = xml_writer_to_string(response_envelope, encoding, resp_len);
    }
    
 bail:
    if (action_string) free(action_string);
    free_wsman_request(&req);
    free_xml(response_envelope);
    
    return ret;
}

/******************************************************************************
* helper functions                                                            *
******************************************************************************/

static int process_fault(xml_writer_t response, xml_writer_t *header, xml_writer_t *body,
			 wsman_request_t *req, int fault_code, int fault_subcode,
			 char *fault_detail) {
    char *fault_string;

    remove_xml(response, *header);
    remove_xml(response, *body);
    *header = print_xml_tag(response, "s", "Header", NULL, NULL);
    *body = print_xml_tag(response, "s", "Body", NULL, NULL);
    generate_fault_message(*body, &fault_string, fault_code, fault_subcode, fault_detail);
    create_response_header(*header, req, fault_string, 1);
    
    return PP_WSMAN_FAULT;
}

static void create_response_header(xml_writer_t response, wsman_request_t* req, char *action, int fault) {
    char message_id[UUID_LENGTH];
    wsman_epr_t *epr;
    
    build_uuid(message_id);
    if (fault && req->fault_to.address) {
    	epr = &req->fault_to;
    } else {
    	epr = &req->reply_to;
    }
    
    print_xml_tag(response, "wsa", "To", epr->address, NULL);
    if (epr->ref_params) {
    	copy_xml(response, epr->ref_params, 1);
    }
    if (epr->ref_props) {
    	copy_xml(response, epr->ref_props, 1);
    }
    if (action) {
    	print_xml_tag(response, "wsa", "Action", action, "s", "mustUnderstand", "", "true");
    }
    print_xml_tag(response, "wsa", "MessageID", message_id, "s", "mustUnderstand", "", "true");
    if (req->message_id) {
    	print_xml_tag(response, "wsa", "RelatesTo", req->message_id, "s", "mustUnderstand", "", "true");
    }
    
    if (!fault && req->fragment.fragment_full) {
    	print_xml_tag(response, "wsman", "FragmentTransfer", req->fragment.fragment_full, NULL);
    }
}

#define RETURN_ERROR(code, detail, det_str)	\
    do {					\
    	ret = -1;				\
    	*fault_code = code;			\
    	*fault_detail = detail;			\
    	*fault_detail_string = det_str;		\
    	goto bail;				\
    } while(0)

static int process_action(xml_writer_t response, wsman_request_t* req,
			  int *fault_code, int *fault_detail, char **fault_detail_string,
			  char **action_string, char *auth_string) {
    int ret = 0;
    wsman_resource_t * res;
    action_t action;
    
    /* try to find the requested resource */
    D(D_BLABLA, "Searching resource %s\n", req->resource_uri);
    if (!(res = find_resource_by_uri(req->resource_uri, req->resource_uri_xml))) {
    	D(D_ERROR, "Resource URI %s not found.\n", req->resource_uri);
    	RETURN_ERROR(FAULT_DESTINATION_UNREACHABLE,
    		     FAULT_DETAIL_INVALID_RESOURCE_URI, NULL);
    }
    
    /* try to find the requested action */
    D(D_BLABLA, "Searching action %s\n", req->action_uri);
    if (!(action = find_action_by_name(res, req->action_uri))) {
    	D(D_ERROR, "Action URI %s not found.\n", req->action_uri);
    	RETURN_ERROR(FAULT_ACTION_NOT_SUPPORTED, 0, req->action_uri);
    }
    
    /* process the action */
    ret = action(response, res, req, action_string, fault_code, fault_detail,
    		 fault_detail_string, auth_string);
    
 bail:
    return ret;
}
