#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>

#include <pp/tokenizer.h>

#include "wsman.h"
#include "debug.h"
#include "ezxml.h"
#include "xml_parser.h"
#include "fault.h"
#include "uuid.h"

/******************************************************************************
* local definitions                                                           *
******************************************************************************/

#define RETURN_ERROR(code, detail, det_str)	\
    do {					\
    	ret = -1;				\
    	*fault_code = code;			\
    	*fault_detail = detail;			\
    	*fault_detail_string = det_str;		\
    	goto bail;				\
    } while(0)

static char* remove_spaces(char *string);
static int parse_header(wsman_request_t *request, ezxml_t header,
			int *fault_code, int *fault_detail,
			char **fault_detail_string);
static int parse_header_element(wsman_request_t *request, ezxml_t elem,
				int *fault_code, int *fault_detail,
				char **fault_detail_string);
static int check_header(wsman_request_t *request,
			int *fault_code, int *fault_detail,
			char **fault_detail_string);

static int parse_body_element(wsman_request_t *request, ezxml_t elem,
			      int *fault_code, int *fault_detail,
			      char **fault_detail_string);
static int parse_body(wsman_request_t *request, ezxml_t body,
		      int *fault_code, int *fault_detail,
		      char **fault_detail_string);

static ezxml_t ezxml_child_namespace(ezxml_t xml,
				     const char *namespace_uri,
				     const char *name);
static int strcasecmp_namespace(ezxml_t xml,
				const char *s1,
				const char *namespace,
				const char *s2);
static int strncasecmp_namespace(ezxml_t xml,
				 const char *s1,
				 const char *namespace,
				 const char *s2, size_t len);

static char *get_tag_name_wo_namespace(char *tagname);

/******************************************************************************
* library global functions                                                    *
******************************************************************************/

/* initialize a wsman_request_t structure */
void init_wsman_request(wsman_request_t *request) {
    memset(request, 0, sizeof(wsman_request_t));
}

/* cleanup a wsman_request_t structure */
void free_wsman_request(wsman_request_t *request) {
    if (request->parsed_xml) {
    	ezxml_free(request->parsed_xml);
    }
    if (request->selectors) {
    	free(request->selectors);
    }
    if (request->fragment.fragment_full) {
    	free(request->fragment.fragment_full);
    }
    if (request->fragment.fragments) {
    	vector_delete(request->fragment.fragments);
    }
    // reset everything in case someone re-uses the structure
    init_wsman_request(request);
}

/* parse a WS-Management request and fill a wsman_request_t structure */
int parse_wsman_request(wsman_request_t *request,
			char *string, int len, int *encoding,
			int *fault_code, int *fault_detail,
			char **fault_detail_string) {
    static int request_no = 0;
    int ret = 0;
    ezxml_t header;
    char temp[100];
    const char *err;
    
    assert(request && string && fault_code && fault_detail);
    
    memset(temp, 0, sizeof(temp));
    
    /* check for an empty string */
    if (!strlen(string)) {
    	RETURN_ERROR(FAULT_INVALID_MESSAGE, 0, "Empty message received!");
    }
    
    /* first, parse the string */
    request->parsed_xml = ezxml_parse_str(string, len, encoding);
    if (!request->parsed_xml || !request->parsed_xml->name) {
    	RETURN_ERROR(FAULT_INTERNAL_ERROR, 0, "Could not parse XML request");
    }
    
    err = ezxml_error(request->parsed_xml);
    if (err && strlen(err)) {
    	D(D_NOTICE, "Problems parsing XML request %d: %s\n", request_no, err);
    }
    
    /* check whether we have a soap envelope */
    if (strncasecmp_namespace(request->parsed_xml, request->parsed_xml->name, namespace_soap_env, "envelope", 8)) {
    	D(D_ERROR, "Envelope not found.\n");
    	RETURN_ERROR(FAULT_INVALID_MESSAGE, 0, "Envelope not found!");
    }
    
    /* search the envelope header */
    header = ezxml_child_namespace(request->parsed_xml, namespace_soap_env, "header");
    if (!header) {
    	D(D_ERROR, "Header not found.\n");
    	RETURN_ERROR(FAULT_INVALID_MESSAGE, 0, "Header not found!");
    }
    
    /* search the body and store it in the return structure */
    request->body = ezxml_child_namespace(request->parsed_xml, namespace_soap_env, "body");
    
    /* now parse the header */
    if ((ret = parse_header(request, header, fault_code, fault_detail, fault_detail_string))) {
    	goto bail;
    }
    
    /* check the interesting parts of the body */
    if (request->body && (ret = parse_body(request, request->body, fault_code, fault_detail, fault_detail_string))) {
    	goto bail;
    }

    /* check the header (all necessary elements included?) */
    /* we can skip this if we are in an identify request where the header elements are not relevant */
    if (!request->is_identify_request && (ret = check_header(request, fault_code, fault_detail, fault_detail_string))) {
    	goto bail;
    }
    
 bail:
    request_no++;
    return ret;
}

xml_parser_t xml_get_first_child(xml_parser_t parent) {
    return parent ? parent->child : NULL;
}

xml_parser_t xml_get_child(xml_parser_t parent, const char *namespace, const char *name) {
    return parent ? ezxml_child_namespace(parent, namespace, name) : NULL;
}

xml_parser_t xml_get_next_same_name(xml_parser_t current) {
    return current ? current->next : NULL;
}

xml_parser_t xml_get_next_different_name(xml_parser_t current) {
    return current ? current->sibling : NULL;
}

char *xml_get_name(xml_parser_t xml) {
    return get_tag_name_wo_namespace((char *)xml->name);
}

char *xml_get_text(xml_parser_t xml) {
    return remove_spaces(xml->txt);
}

int get_name_value(xml_parser_t xml, name_value_t *value) {
    value->name = xml_get_name(xml);
    value->value = xml_get_text(xml);
    
    return 0;
}

char *xml_find_namespace_shortcut(xml_parser_t xml, const char *full_ns) {
    return ezxml_find_namespace_shortcut(xml, full_ns);
}

char *xml_find_namespace_fullname(xml_parser_t xml, const char *short_ns) {
    return ezxml_find_namespace_fullname(xml, short_ns);
}

/******************************************************************************
* local helper functions                                                      *
******************************************************************************/

/* remove spaces from a string (leading and trailing) */
static char* remove_spaces(char *string) {
    int len;
    
    if (!string) {
    	return NULL;
    }
    
    /* remove leading spaces */
    while (*string == ' ' ||
           *string == '\t' ||
           *string == '\r' ||
           *string == '\n') {
    	string++;
    }
    
    /* remove trailing spaces */
    while (((len = strlen(string)) > 1) &&
            (string[len - 1] == ' ' ||
             string[len - 1] == '\t' ||
             string[len - 1] == '\r' ||
             string[len - 1] == '\n'
            )) {
    	string[len - 1] = '\0';
    }
    
    return string;
}

static char *get_tag_name_wo_namespace(char *tagname) {
    char *name;
    
    if (!tagname) {
    	return NULL;
    }
    
    name = strchr(tagname, ':');
    if (name) {
    	return name + 1;
    } else {
    	return tagname;
    }
}

static ezxml_t ezxml_child_namespace(ezxml_t xml, const char *namespace_uri, const char *name) {
    char temp[512];
    ezxml_t current, child;
    
    if (!namespace_uri || namespace_uri[0] == '\0') {
    	return ezxml_child(xml, name);
    }
    
    for (current = xml->child; current; current = current->ordered) {
    	char *namespace;
    	if (!strcasestr(current->name, name)) continue;
    	
    	namespace = ezxml_find_namespace_shortcut(current, namespace_uri);
    	if (!namespace) return NULL;
    	memset(temp, 0, sizeof(temp));
    	if (namespace[0] != '\0') {
    	    snprintf(temp, sizeof(temp), "%s:%s", namespace, name);
    	} else {
    	    snprintf(temp, sizeof(temp), "%s", name);
    	}
    	
    	if ((child = ezxml_child(xml, temp))) {
    	    return child;
    	}
    }
    
    return NULL;
}

static int strcasecmp_namespace(ezxml_t xml, const char *s1, const char *namespace_full, const char *s2) {
    char *namespace = ezxml_find_namespace_shortcut(xml, namespace_full);
    char temp[512];
    memset(temp, 0, sizeof(temp));
    if (!namespace) {
    	return 1;
    } else if (namespace[0] != '\0') {
    	snprintf(temp, sizeof(temp), "%s:%s", namespace, s2);
    } else {
    	snprintf(temp, sizeof(temp), "%s", s2);
    }
    
    return strcasecmp(s1, temp);
}

static int strncasecmp_namespace(ezxml_t xml, const char *s1, const char *namespace_full, const char *s2, size_t len) {
    char *namespace = ezxml_find_namespace_shortcut(xml, namespace_full);
    char temp[512];
    memset(temp, 0, sizeof(temp));
    if (!namespace) {
    	return 1;
    } else if (namespace[0] != '\0') {
    	snprintf(temp, sizeof(temp), "%s:%s", namespace, s2);
    	len += strlen(namespace) + 1;
    } else {
    	snprintf(temp, sizeof(temp), "%s", s2);
    }
    
    return strncasecmp(s1, temp, len);
}

/* is the MustUnderstand flag set? */
static int must_understand(ezxml_t elem) {
    const char *understand = ezxml_attr(elem, "mustUnderstand");
    return (understand && (!strcasecmp(understand, "true") || !strcasecmp(understand, "1")));
}

/* parse a WS-Management SOAP body */
static int parse_body(wsman_request_t *request, ezxml_t body,
		      int *fault_code, int *fault_detail,
		      char **fault_detail_string) {
    int ret = 0;
    ezxml_t elem;
    
    /* go through elements */
    for (elem = body ? body->child : NULL; elem; elem = elem->sibling) {
    	if ((ret = parse_body_element(request, elem,
    				      fault_code, fault_detail,
    				      fault_detail_string))) {
    	    break;
    	}
    }

    return ret;
}

static void parse_enum_context(wsman_request_t *request, ezxml_t elem) {
    ezxml_t child;
    if ((child = ezxml_child_namespace(elem, namespace_enumeration, "EnumerationContext"))) {
    	request->enumeration_context = remove_spaces(child->txt);
    }
}

static void parse_enum_max_elements(wsman_request_t *request, ezxml_t elem) {
    ezxml_t child;
    if ((child = ezxml_child_namespace(elem, namespace_enumeration, "MaxElements"))) {
    	request->enumeration_max_elements = atoi(remove_spaces(child->txt));
    }
}

static void parse_pull(wsman_request_t *request, ezxml_t elem) {
    parse_enum_context(request, elem);
    parse_enum_max_elements(request, elem);
    /* The MaxElements value is ignored here because we support only one element
       for enumeration (ignoring this value is allowed according to WS Management
       spec R5.4-11) */
}

static void parse_release(wsman_request_t *request, ezxml_t elem) {
    parse_enum_context(request, elem);
}

static void parse_enumerate(wsman_request_t *request, ezxml_t elem) {
    ezxml_t child;
    if ((child = ezxml_child_namespace(elem, namespace_management, "EnumerationMode"))) {
    	char epr[100], object[100], epr_object[100];
    	char *mode = remove_spaces(child->txt);
    	char *namespace = ezxml_find_namespace_shortcut(child, namespace_management);
    	
    	D(D_BLABLA, "EnumerationMode: %s\n", mode);

    	if (namespace && namespace[0] != '\0') {
    	    snprintf(epr, sizeof(epr), "%s:%s", namespace, "EnumerateEPR");
    	    snprintf(object, sizeof(epr), "%s:%s", namespace, "EnumerateObject");
    	    snprintf(epr_object, sizeof(epr), "%s:%s", namespace, "EnumerateObjectAndEPR");
    	} else {
    	    snprintf(epr, sizeof(epr), "%s", "EnumerateEPR");
    	    snprintf(object, sizeof(epr), "%s", "EnumerateObject");
    	    snprintf(epr_object, sizeof(epr), "%s", "EnumerateObjectAndEPR");
    	}
    	
    	if (!strcasecmp(mode, epr)) {
    	    request->enum_type = ENUMERATION_TYPE_EPR;
    	} else if (!strcasecmp(mode, object)) {
    	    request->enum_type = ENUMERATION_TYPE_OBJECT;
    	} else if (!strcasecmp(mode, epr_object)) {
    	    request->enum_type = ENUMERATION_TYPE_OBJECT_AND_EPR;
    	}
    	
    	D(D_BLABLA, "EnumerationMode: %d\n", request->enum_type);
    }
}

static int parse_body_element(wsman_request_t *request, ezxml_t elem,
			      int *fault_code UNUSED, int *fault_detail UNUSED,
			      char **fault_detail_string UNUSED) {
    
    D(D_BLABLA, "Parsing body element %s\n", elem->name);
    
    /* we are only interested in some parts, the rest is done later */
    if (!strcasecmp_namespace(elem, elem->name, namespace_enumeration, "Pull")) {
    	parse_pull(request, elem);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_enumeration, "Release")) {
    	parse_release(request, elem);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_enumeration, "Enumerate") && elem->txt) {
    	parse_enumerate(request, elem);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_identity, "Identify")) {
    	request->is_identify_request = 1;
    }
    
    return 0;
}

/* parse a WS-Management SOAP header */
static int parse_header(wsman_request_t *request, ezxml_t header,
			int *fault_code, int *fault_detail,
			char **fault_detail_string) {
    int ret = 0;
    ezxml_t elem;
    
    /* go throug the header elements and parse them */
    D(D_BLABLA, "Parsing WS-Management header.\n");
    for (elem = header ? header->child : NULL; elem; elem = elem->sibling) {
    	if ((ret = parse_header_element(request, elem,
    					fault_code, fault_detail,
    					fault_detail_string))) {
    	    break;
    	}
    }
    D(D_BLABLA, "Parsing WS-Management header done.\n");
    
    return ret;
}

/*--- header entry parsing ---*/
static void parse_address(char **address, char **resource_uri, ezxml_t elem) {
    char *ws_addr = remove_spaces(elem->txt);
    
    if (ws_addr) {
    	char *bracket, *q_mark;
    	
    	*address = ws_addr;

    	/* search for the beginning of the URI */
    	if ((q_mark = strchr(ws_addr, '?'))) {
    	    ws_addr = remove_spaces(q_mark + 1);
    	    *q_mark = '\0';
    	}
    	
    	/* is this really a ResourceURI? */
    	if (strncasecmp(ws_addr, "ResourceURI", 11)) {
    	    D(D_VERBOSE, "No Resource URI found in wsa:Address\n");
    	    goto bail;
    	}
    	
    	/* remove the leading and trailing '(' and ')' */
    	bracket = strchr(ws_addr, '(');
    	if (bracket) {
    	    ws_addr = remove_spaces(bracket + 1);
    	}
    	
    	bracket = strrchr(ws_addr, ')');
    	if (bracket) {
    	    *bracket = '\0';
    	}
    	
    	*resource_uri = ws_addr;
    }
    
 bail:
    D(D_BLABLA, "To address: %s, ResourceURI: %s\n", *address, *resource_uri);
}

static void parse_wsa_to(wsman_request_t *request, ezxml_t elem) {
    parse_address(&request->wsa_address, &request->resource_uri, elem);
    if (request->resource_uri) {
    	request->resource_uri_xml = elem;
    }
}

static void parse_wsman_resource_uri(wsman_request_t *request, ezxml_t elem) {
    request->resource_uri = remove_spaces(elem->txt);
    if (request->resource_uri) {
    	request->resource_uri_xml = elem;
    }
}

static void parse_wsa_address(wsman_epr_t *epr, ezxml_t elem) {
    ezxml_t address;
    xml_parser_t uri;
    
    /* find the wsa:Address element */
    if ((address = ezxml_child_namespace(elem, namespace_addressing, "Address"))) {
    	parse_address(&epr->address, &epr->resource_uri, address);
    	D(D_BLABLA, "Found address %s,\n", epr->address);
    	if (!strcasecmp(epr->address, NAMESPACE_ADDRESSING "/role/anonymous")) {
    	    epr->anonymous = 1;
    	}
    }
    
    epr->ref_params = ezxml_child_namespace(elem, namespace_addressing, "ReferenceParameters");
    if (epr->ref_params && (uri = ezxml_child_namespace(epr->ref_params, namespace_management, "ResourceURI"))) {
    	epr->resource_uri = remove_spaces(uri->txt);
    }
    epr->ref_props = ezxml_child_namespace(elem, namespace_addressing, "ReferenceProperties");
}

static void parse_wsa_message_id(wsman_request_t *request, ezxml_t elem) {
    request->message_id = remove_spaces(elem->txt);
}

static void parse_wsa_action(wsman_request_t *request, ezxml_t elem) {
    request->action_uri = remove_spaces(elem->txt);
}

static void parse_wsman_max_envelope_size(wsman_request_t *request, ezxml_t elem) {
    if (elem->txt) {
    	request->max_envelope_size = atoi(elem->txt);
    	if (request->max_envelope_size == 0) {
    	    request->max_envelope_size = -1;
    	}
    }
}

static void parse_wsman_selector_set(wsman_request_t *request, ezxml_t elem) {
    ezxml_t sel;
    
    /* find the first selector */
    D(D_BLABLA, "Parsing selectors\n");
    sel = ezxml_child_namespace(elem, namespace_management, "Selector");
    
    /* go through all selectors and process them */
    for (; sel; sel = sel->next) {
    	char *name;
    	
    	if ((name = (char *)ezxml_attr(sel, "Name"))) {
    	    D(D_BLABLA, "Found selector %s\n", name);
    	    wsman_req_selector_t *s;
    	    xml_parser_t child;
    	    
    	    /* add this resource */
    	    if (!request->selectors) {
    	    	request->selectors = malloc(sizeof(wsman_req_selector_t));
    	    } else {
    	    	request->selectors = realloc(request->selectors,
    	    		(request->no_selectors + 1) * sizeof(wsman_req_selector_t));
    	    }
    	    
    	    s = &request->selectors[request->no_selectors];
    	    
    	    memset(s, 0, sizeof(wsman_req_selector_t));
    	    
    	    s->name = name;
    	    s->xml = sel;
    	    
    	    if ((child = ezxml_child_namespace(sel, namespace_addressing, "EndpointReference"))) {
    	    	D(D_BLABLA, "Endpoint Reference selector\n");
    	    	s->type = SELECTOR_TYPE_EPR;
    	    	parse_wsa_address(&s->value.epr, child);
    	    } else if ((child = sel->child)) {
    	    	D(D_BLABLA, "Complex selector: %s\n", child->name);
    	    	s->type = SELECTOR_TYPE_COMPLEX;
    	    	s->value.complex = child;
    	    } else if (sel->txt) {
    	    	D(D_BLABLA, "Simple selector: %s\n", sel->txt);
    	    	s->type = SELECTOR_TYPE_SIMPLE;
    	    	s->value.simple = remove_spaces(sel->txt);
    	    }
    	    
    	    request->no_selectors++;
    	}
    }
}

static void frag_delelem(void *elem) {
    free(elem);
}

static int parse_fragment_transfer(wsman_request_t *request, ezxml_t elem,
				    int *fault_code, int *fault_detail,
				    char **fault_detail_string) {
    int ret = 0;
    const char *dialect;
    char *frag;
    tokenizer_t _tokenizer, *tokenizer;
    
    if (!elem->txt) return 0;
    if (request->fragment.fragment_full) {
    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT, 0);
    }
    
    frag = remove_spaces(elem->txt);

    if ((dialect = ezxml_attr(elem, "Dialect")) && strcasecmp(dialect, namespace_xpath)) {
    	D(D_NOTICE, "Fragment-level dialect %s not supported.\n", dialect);
    	RETURN_ERROR(FAULT_UNSUPPORTED_FEATURE, FAULT_DETAIL_UNSUPPORTED_FRAGMENT_LEVEL_ACCESS, NULL);
    }
    
    request->fragment.fragment_full = strdup(frag);
    request->fragment.fragments = vector_new(NULL, 0, frag_delelem);
    request->fragment.text_only = 0;
    
    tokenizer = tokenizer_new(&_tokenizer, frag, "/", TOKENIZER_DELI_NORET);
    while(tokenizer_has_next(tokenizer)) {
    	char *next = tokenizer_get_next(tokenizer);
    	wsman_fragment_t *f;
    	char *b;
    	
    	if (!next) continue;
    	
    	if (!strcasecmp(next, "Text()")) {
    	    D(D_BLABLA, "found /Text() in XPath\n");
    	    request->fragment.text_only = 1;
    	    if (tokenizer_has_next(tokenizer)) {
    	    	D(D_NOTICE, "Found /Text(), but this is not the last element in XPath!\n");
    	    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT, 0);
    	    }
    	    continue;	/* don't add the Text() element */
    	}
    	
    	f = malloc(sizeof(wsman_fragment_t));
    	memset(f, 0, sizeof(wsman_fragment_t));
    	
    	f->fragment = next;
    	if ((b = strrchr(next, '[')) && next[strlen(next) - 1] == ']') {
    	    unsigned int no;
    	    if (sscanf(b, "[%u]", &no) != 1) {
    	    	D(D_NOTICE, "Found array brackets, but could not parse index.\n");
    	    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT, 0);
    	    }
    	    f->array_index = no;
    	    *b = '\0';
    	}
    	
    	D(D_BLABLA, "Found XPath fragment %s, array index %d\n",
    		f->fragment, f->array_index);
    	vector_add(request->fragment.fragments, f);
    }
    
 bail:
    return ret;
}

/* parse a SOAP header entry */
static int parse_header_element(wsman_request_t *request, ezxml_t elem,
				int *fault_code, int *fault_detail,
				char **fault_detail_string) {
    int ret = 0;
    
    if (!strcasecmp_namespace(elem, elem->name, namespace_addressing, "To")) {
    	parse_wsa_to(request, elem);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_management, "ResourceURI")) {
    	parse_wsman_resource_uri(request, elem);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_addressing, "ReplyTo")) {
    	parse_wsa_address(&request->reply_to, elem);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_addressing, "FaultTo")) {
    	parse_wsa_address(&request->fault_to, elem);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_addressing, "MessageID")) {
    	parse_wsa_message_id(request, elem);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_addressing, "Action")) {
    	parse_wsa_action(request, elem);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_addressing, "From")) {
    	/* nothing to do here */
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_management, "OperationTimeout") &&
    		must_understand(elem)) {
    	RETURN_ERROR(FAULT_UNSUPPORTED_FEATURE,
    		     FAULT_DETAIL_UNSUPPORTED_OPERATION_TIMEOUT, NULL);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_management, "Locale") &&
    		must_understand(elem)) {
    	RETURN_ERROR(FAULT_UNSUPPORTED_FEATURE,
    		     FAULT_DETAIL_UNSUPPORTED_LOCALE, NULL);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_management, "MaxEnvelopeSize")) {
    	parse_wsman_max_envelope_size(request, elem);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_management, "SelectorSet")) {
    	parse_wsman_selector_set(request, elem);
    } else if (!strcasecmp_namespace(elem, elem->name, namespace_management, "FragmentTransfer")) {
    	ret = parse_fragment_transfer(request, elem, fault_code, fault_detail, fault_detail_string);
    } else if (must_understand(elem)) {
    	RETURN_ERROR(FAULT_NOT_UNDERSTOOD, 0, NULL);
    }
    
 bail:
    return ret;
}

/* check whether the message id is correct */
static int check_message_id(wsman_request_t *request) {
    /* message ID syntax is not defined as MUST any longer in 2005/06 spec */
    return request->message_id && request->message_id[0] ? 0 : 1;
}

#define MAX_ENVELOPE_SIZE_MINIMUM	8192

/* check a soap header for locigal errors */
static int check_header(wsman_request_t *request,
			int *fault_code, int *fault_detail,
			char **fault_detail_string) {
    int ret = 0;
    
    /* ResourceURI is a mandatory element */
    if (!request->resource_uri || !request->resource_uri[0]) {
    	D(D_ERROR, "ResourceURI not available!\n");
    	RETURN_ERROR(FAULT_DESTINATION_UNREACHABLE,
    		     FAULT_DETAIL_INVALID_RESOURCE_URI, NULL);
    }
    
    /* both the ReplyTo and FaultTo addresses have to be the
       anonymous address; if they are wrong, we adjust it here
       (error is reported later) */
    if (request->reply_to.address && !request->reply_to.anonymous) {
    	request->reply_to.address = NAMESPACE_ADDRESSING "/role/anonymous";
    }
    if (request->fault_to.address && !request->fault_to.anonymous) {
    	request->fault_to.address = NAMESPACE_ADDRESSING "/role/anonymous";
    }
    
    /* ReplyTo is in our cases necessary
       NOTE: this is not necessary in all cases, but for all our
       supported messages! */
    if (!request->reply_to.address || !request->reply_to.address[0]) {
    	D(D_ERROR, "ReplyTo not available!\n");
    	RETURN_ERROR(FAULT_MESSAGE_INFORMATION_HEADER_REQUIRED,
    		     0, "wsa:ReplyTo");
    }
    if (!request->reply_to.anonymous) {
    	D(D_ERROR, "ReplyTo not anonymous address!\n");
    	RETURN_ERROR(FAULT_UNSUPPORTED_FEATURE,
    		     FAULT_DETAIL_UNSUPPORTED_ADDRESSING_MODE, NULL);
    }
    
    /* FaultTo must be anonymous address */
    if (request->fault_to.address && !request->fault_to.anonymous) {
    	D(D_ERROR, "FaultTo not anonymous address!\n");
    	request->fault_to.address = NULL;	/* deliver to ReplyTo */
    	RETURN_ERROR(FAULT_UNSUPPORTED_FEATURE,
    		     FAULT_DETAIL_UNSUPPORTED_ADDRESSING_MODE, NULL);
    }
    
    /* MessageID is mandatory */
    if (check_message_id(request)) {
    	D(D_ERROR, "MessageID wrong or not available!\n");
    	RETURN_ERROR(FAULT_INVALID_MESSAGE_INFORMATION_HEADER,
    		     0, "<wsa:MessageID/>");
    }
    
    /* Action must be there */
    if (!request->action_uri || !request->action_uri[0]) {
    	D(D_ERROR, "Action not available!\n");
    	RETURN_ERROR(FAULT_ACTION_NOT_SUPPORTED,
    		     0, "");
    }
    
    /* MaxEnvelopeSize must not be <MAX_ENVELOPE_SIZE_MINIMUM */
    if (request->max_envelope_size > 0 && request->max_envelope_size < MAX_ENVELOPE_SIZE_MINIMUM) {
    	D(D_ERROR, "MaxEnvelopeSize out of range (%d)!\n", request->max_envelope_size);
    	RETURN_ERROR(FAULT_ENCODING_LIMIT,
    		     FAULT_DETAIL_MINIMUM_ENVELOPE_LIMIT, NULL);
    } else if (request->max_envelope_size < 0) {
    	D(D_ERROR, "MaxEnvelopeSize not parsed correctly (%d)!\n", request->max_envelope_size);
    	RETURN_ERROR(FAULT_INVALID_MESSAGE_INFORMATION_HEADER,
    		     0, "<wsman:MaxEnvelopeSize/>");
    }
    
 bail:
    return ret;
}
