#ifndef __XML_PARSER_H
#define __XML_PARSER_H

#include <pp/vector.h>
#include "ezxml.h"

/******************************************************************************
* data types                                                                  *
******************************************************************************/

typedef ezxml_t xml_parser_t;

/* name + value pair */
typedef struct {
    char *name;
    char *value;
} name_value_t;

/* enumeration mode */
typedef enum {
    ENUMERATION_TYPE_OBJECT = 0,
    ENUMERATION_TYPE_EPR,
    ENUMERATION_TYPE_OBJECT_AND_EPR,
} enum_type_t;

/* endpoint reference */
typedef struct {
    int anonymous;		/* anonymous address? */
    char *address;		/* address URI */
    char *resource_uri;		/* ResourceURI, if available */
    xml_parser_t ref_params;	/* reference parameters */
    xml_parser_t ref_props;	/* reference properties */
} wsman_epr_t;

/* a selector in a request */
typedef struct {
    char *name;
    enum {
    	SELECTOR_TYPE_NONE = 0,
    	SELECTOR_TYPE_SIMPLE,
    	SELECTOR_TYPE_EPR,
    	SELECTOR_TYPE_COMPLEX,
    } type;
    union {
    	char *simple;
    	wsman_epr_t epr;
    	xml_parser_t complex;
    } value;
    xml_parser_t xml;
} wsman_req_selector_t;

/* fragment level access */
typedef struct {
    char *fragment;
    int array_index;
} wsman_fragment_t;

typedef struct {
    char *fragment_full;	/* must be freed */
    vector_t *fragments;	/* vector of wsman_fragment_t */
    int text_only;
} wsman_fragment_access_t;

/* a parsed WS-Management request */
typedef struct {
    /* the structure processed by ezxml */
    xml_parser_t parsed_xml;
    
    /* required header fields */
    char *wsa_address;			/* wsa:To address */
    char *resource_uri;			/* name of the addressed resource */
    xml_parser_t resource_uri_xml;	/* xml node containing the resource URI */
    wsman_epr_t reply_to;		/* EPR the reply is sent to */
    char *action_uri;			/* the URI of the requested action */
    char *message_id;			/* the UUID of the request */
    
    /* optional header fields (only some common) */
    wsman_epr_t fault_to;		/* EPR the fault is sent to */
    int max_envelope_size;		/* maximum envelope size */
    wsman_fragment_access_t fragment;	/* fragment level access */
    
    /* selectors */
    int no_selectors;
    wsman_req_selector_t *selectors;
    
    // TODO: add option set here
    
    /* the body (includes data for Put operations), saved as ezxml structure */
    xml_parser_t body;
    
    /* commonly used parts of the body */
    char *enumeration_context;
    int enumeration_max_elements;
    enum_type_t enum_type;
    
    int is_identify_request;
} wsman_request_t;

/******************************************************************************
* functions                                                                   *
******************************************************************************/

/* initialize the request structure */
void init_wsman_request(wsman_request_t *request);

/* parse an XML request
   returns 0 if okay or != 0 on error */
int parse_wsman_request(wsman_request_t *request,
			char *string, int len,
			int *encoding,
			int *fault_code, int *fault_detail,
			char **fault_detail_string);

/* free allocated memory for a parsed request */
void free_wsman_request(wsman_request_t *request);

/* some helper functions for the xml_parser_t */
xml_parser_t xml_get_first_child(xml_parser_t parent);
xml_parser_t xml_get_child(xml_parser_t parent, const char *namespace, const char *name);
xml_parser_t xml_get_next_same_name(xml_parser_t current);
xml_parser_t xml_get_next_different_name(xml_parser_t current);
char *xml_get_name(xml_parser_t xml);
char *xml_get_text(xml_parser_t xml);

/* parse the given xml_parser_t and build name-value pairs */
int get_name_value(xml_parser_t xml, name_value_t *value);

/* namespace handling */
char *xml_find_namespace_shortcut(xml_parser_t xml, const char *full_ns);
char *xml_find_namespace_fullname(xml_parser_t xml, const char *short_ns);

#endif /* __XML_PARSER_H */
