/*
 *  linux/arch/arm/mach-faraday/intc.c
 *
 *  Faraday FTINTC010 Master Interrupt Controller Device Driver Implementation
 *
 *  Copyright (C) 2005 Faraday Corp. (http://www.faraday-tech.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Note
 * 
 *  This program implements only the master INTC of the platform. Slave INTCs must
 *  be initialized by themselves.
 * 
 * ChangeLog
 * 
 *  Luke Lee  09/14/2005  Created.
 */

#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <asm/io.h>

#include <asm/mach/irq.h>
#include <asm/arch/intc.h>
#include <asm/arch/spec.h>

#define IPMODULE INTC
#define IPNAME   FTINTC010

static void intc_ftintc010_edge_ack_irq(unsigned int irq);
static void intc_ftintc010_edge_mask_irq(unsigned irq);
static void intc_ftintc010_edge_unmask_irq(unsigned int irq);
static void intc_ftintc010_level_ack_mask_irq(unsigned int irq);
static void intc_ftintc010_level_mask_irq(unsigned int irq);
static void intc_ftintc010_level_unmask_irq(unsigned int irq);
static int intc_ftintc010_set_irq_type(unsigned int irq, unsigned int type);

/*
 * Common methods
 */
static struct irqchip intc_ftintc010_edge_chip = {
	.ack	= intc_ftintc010_edge_ack_irq,
	.mask	= intc_ftintc010_edge_mask_irq,
	.unmask = intc_ftintc010_edge_unmask_irq,
	.type   = intc_ftintc010_set_irq_type,
};

static struct irqchip intc_ftintc010_level_chip = {
	.ack	= intc_ftintc010_level_ack_mask_irq,
	.mask	= intc_ftintc010_level_mask_irq,
	.unmask = intc_ftintc010_level_unmask_irq,
	.type   = intc_ftintc010_set_irq_type,
};

/*
 * Edge trigger IRQ chip methods
 */
static void intc_ftintc010_edge_ack_irq(unsigned int irq)
{
        // ack
        *(volatile unsigned *)(IP_VA_BASE(0)+IRQ_CLEAR_REG) = 1<<irq;
}

static void intc_ftintc010_edge_mask_irq(unsigned irq)
{
        // disable
        *(volatile unsigned *)(IP_VA_BASE(0)+IRQ_MASK_REG) &= ~(1<<irq);
        
}

static void intc_ftintc010_edge_unmask_irq(unsigned int irq)
{
        // enable 
        *(volatile unsigned *)(IP_VA_BASE(0)+IRQ_MASK_REG) |= 1<<irq;
}

/*
 * Level trigger IRQ chip methods
 */
static void intc_ftintc010_level_ack_mask_irq(unsigned int irq)
{
        // ack and disable
        *(volatile unsigned *)(IP_VA_BASE(0)+IRQ_CLEAR_REG) = 1<<irq;
        *(volatile unsigned *)(IP_VA_BASE(0)+IRQ_MASK_REG) &= ~(1<<irq);
}

static void intc_ftintc010_level_mask_irq(unsigned int irq)
{
        // disable
        *(volatile unsigned *)(IP_VA_BASE(0)+IRQ_MASK_REG) &= ~(1<<irq);
}

static void intc_ftintc010_level_unmask_irq(unsigned int irq)
{
        // enable 
        *(volatile unsigned *)(IP_VA_BASE(0)+IRQ_MASK_REG) |= 1<<irq;
}


static int intc_ftintc010_set_irq_type(unsigned int irq, unsigned int type)
{
    if (type != IRQT_RISING && type != IRQT_FALLING &&
	type != IRQT_HIGH && type != IRQT_LOW) {

	printk("Unsupported IRQ type %d for IRQ %d\n",
	       type, irq);

	return -1;
    }
    
    if (type == IRQT_RISING || type == IRQT_FALLING) {
	/* edge triggered irq */
	set_irq_chip(irq, &intc_ftintc010_edge_chip);
	set_irq_handler(irq, do_edge_IRQ);

	*(volatile unsigned *)(IP_VA_BASE(0)+IRQ_MODE_REG) |= (1<<irq);
	if (type == IRQT_RISING) {
	    *(volatile unsigned *)(IP_VA_BASE(0)+IRQ_LEVEL_REG) &= ~(1<<irq);
	} else {
	    *(volatile unsigned *)(IP_VA_BASE(0)+IRQ_LEVEL_REG) |= (1<<irq);
	}
    } else {
	/* level triggered irq */
	set_irq_chip(irq, &intc_ftintc010_level_chip);
	set_irq_handler(irq, do_level_IRQ);

	*(volatile unsigned *)(IP_VA_BASE(0)+IRQ_MODE_REG) &= ~(1<<irq);
	if (type == IRQT_HIGH) {
	    *(volatile unsigned *)(IP_VA_BASE(0)+IRQ_LEVEL_REG) &= ~(1<<irq);
	} else {
	    *(volatile unsigned *)(IP_VA_BASE(0)+IRQ_LEVEL_REG) |= (1<<irq);
	}
    }

    return 0;
}

static struct resource intc_resource = {
        .name  = "Main interrupt controller",
        .start = IP_VA_BASE(0),
        .end   = IP_VA_BASE(0) + IP_VA_SIZE(0),
};

/*
 * Initialization of master interrupt controller, after this INTC is
 * enabled, the rest of Linux initialization codes can then be completed.
 * For example, timer interrupts and UART interrupts must be enabled during
 * the boot process.
 */
void __init intc_ftintc010_init_irq(void)
{
        int i,edge;

        /* Initialize the INTC */
	outl( 0x00000000, IP_VA_BASE(0) + IRQ_MASK_REG );
	outl( 0x00000000, IP_VA_BASE(0) + FIQ_MASK_REG );
	outl( 0xffffffff, IP_VA_BASE(0) + IRQ_CLEAR_REG );
	outl( 0xffffffff, IP_VA_BASE(0) + FIQ_CLEAR_REG );
        outl( PLATFORM_IRQ_TRIGGER_MODE, IP_VA_BASE(0) + IRQ_MODE_REG );
        outl( PLATFORM_FIQ_TRIGGER_MODE, IP_VA_BASE(0) + FIQ_MODE_REG );
        /* FTINTC010: bit 0=active high or rising edge, 1=active low or falling edge. */
        outl( ~PLATFORM_IRQ_TRIGGER_LEVEL, IP_VA_BASE(0) + IRQ_LEVEL_REG );
        outl( ~PLATFORM_FIQ_TRIGGER_LEVEL, IP_VA_BASE(0) + FIQ_LEVEL_REG );

        /* Register all IRQ */
        for (i = PLATFORM_IRQ_BASE, edge=1;
             i < PLATFORM_IRQ_BASE + PLATFORM_IRQ_TOTALCOUNT;
             i++, edge<<=1)
        {
                if (PLATFORM_IRQ_TRIGGER_MODE & edge) {
                        // edge trigger
                        set_irq_chip(i, &intc_ftintc010_edge_chip);
                        set_irq_handler(i, do_edge_IRQ);
                } else {
                        // level trigger
                        set_irq_chip(i, &intc_ftintc010_level_chip);
                        set_irq_handler(i, do_level_IRQ);
                }
                set_irq_flags(i, IRQF_VALID | IRQF_PROBE);
        }
        /* Register all FIQ :
         * Notice that FIQ is not supported by the system yet
         */
        for (i = PLATFORM_FIQ_BASE, edge=1;
             i < PLATFORM_FIQ_BASE + PLATFORM_FIQ_TOTALCOUNT;
             i++, edge<<=1)
        {
                if (PLATFORM_FIQ_TRIGGER_MODE & edge) {
                        // edge trigger
                        set_irq_chip(i, &intc_ftintc010_edge_chip);
                        set_irq_handler(i, do_edge_IRQ);
                } else {
                        // level trigger
                        set_irq_chip(i, &intc_ftintc010_level_chip);
                        set_irq_handler(i, do_level_IRQ);
                }
                set_irq_flags(i, IRQF_VALID | IRQF_PROBE);
        }

        /* Register I/O address range of this INTC */
        request_resource(&ioport_resource, &intc_resource);

        /*
         * Initialize IRQ of all daughter platforms if necessary.
         * 
         * 1. If the initialization *MUST* take place *IMMEDIATELY* right here,
         *    daughter_platform_init_irq() must be defined in platform dependent
         *    codes.
         *    Example of this case: if the timer in the whole system existed only
         *    in the daughter platform instead of the mother platform. Linux kernel
         *    require *AT LEAST ONE* timer in the whole system during the kernel
         *    initialization.
         *    
         * 2. If the initialization can take place later, use core_initcall(fn),
         *    postcore_initcall(fn), arch_initcall(fn) ... etc. to register the
         *    initialization code. They will be called later after the Linux
         *    kernel initialization is completed. Notice that *DO NOT* use the
         *    low priority "device_initcall(fn)" to register your slave INTCs
         *    initialization codes, otherwise some device on the daughter platform
         *    might got initialized *BEFORE* the INTC got initialized.
         */
        daughter_platform_init_irq();
}
