/*
    i2c-algo-asics.c i2c driver algorithms for Asics I2C controllers

    Copyright (C) 2004 Microtronix Datacom Ltd
    Written by Wentao Xu <wentao@microtronix.com>
    
    This file was highly leveraged from i2c-algo-pcf.c, which was  

	--------------------------------------------------------------------
    Copyright (C) 1995-1997 Simon G. Vogl
                   1998-2000 Hans Berglund

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   With some changes from Ky?ti M?kki <kmalkki@cc.hut.fi> and 
   Frodo Looijaard <frodol@dds.nl> ,and also from Martin Bailey
   <mbailey@littlefeet-inc.com> 

   Partially rewriten by Oleg I. Vdovikin <vdovikin@jscc.ru> to handle multiple
   messages, proper stop/repstart signaling during receive,
   added detect code
   --------------------------------------------------------------------
*/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/i2c.h>
#include <linux/i2c-algo-asics.h>
#include "../busses/i2c-asics.h"

MODULE_AUTHOR("Peppercon AG");
MODULE_DESCRIPTION("I2C-Bus Asics algorithm");
MODULE_LICENSE("GPL");

/* debug level: - 0 off; 1 normal; 2,3 more verbose; 9 asics-protocol */
#define DEBUG_LEVEL	0
        
/* ----- global defines ----------------------------------------------- */
#if DEBUG_LEVEL >= 1
#define DEB(x)	{ x }
#else
#define DEB(x)	{   }
#endif
#if DEBUG_LEVEL >= 2
#define DEB2(x)	{ x }
#else
#define DEB2(x)	{   }
#endif
#if DEBUG_LEVEL >= 3
#define DEB3(x)	{ x }
#else
#define DEB3(x)	{   }
#endif
#if DEBUG_LEVEL >= 4
#define DEB4(x)	{ x }
#else
#define DEB4(x)	{   }
#endif
#if DEBUG_LEVEL >= 9
#define DEBPROTO(x)	{ x }
#else
#define DEBPROTO(x)	{   }
#endif

#define DBG(fmt, x...)		i2c_asics_print(KERN_DEBUG "i2c-algo-asics: " fmt, ##x)
#define INFO(fmt, x...)		i2c_asics_print(KERN_INFO  "i2c-algo-asics: " fmt, ##x)
#define WARN(fmt, x...)		i2c_asics_print(KERN_WARNING  "i2c-algo-asics: " fmt, ##x)
#define CRIT(fmt, x...)		i2c_asics_print(KERN_CRIT  "i2c-algo-asics: " fmt, ##x)
#define ERR(fmt, x...)		i2c_asics_print(KERN_ERR   "i2c-algo-asics: " fmt, ##x)

#define BUS_WAIT_TIME		100
#define BUS_WAIT_RETRIES	100
#define ADDRESS_RETRIES		10

/* --- setting states on the bus with the right timing: ---------------	*/
#define get_ctl_cmd(adap) adap->getreg(adap->hw, REG_ASICSI2C_CCR)
#define set_ctl_cmd_force(adap, val) adap->setreg(adap->hw, REG_ASICSI2C_CCR, val)
#define set_ctl_cmd_control(adap, val) \
	adap->setreg(adap->hw, REG_ASICSI2C_CCR, val | (get_ctl_cmd(adap) & (BIT_ASICSI2C_CC_ENA | BIT_ASICSI2C_CC_IEN)))
#define set_ctl_cmd(adap, val) \
	adap->setreg(adap->hw, REG_ASICSI2C_CCR, val | (get_ctl_cmd(adap) & (BIT_ASICSI2C_CC_MS | BIT_ASICSI2C_CC_ENA | BIT_ASICSI2C_CC_IEN)))
	/* don't touch the MS bit here! */

#define get_sts(adap) adap->getreg(adap->hw, REG_ASICSI2C_SR)
#define set_sts(adap, val) adap->setreg(adap->hw, REG_ASICSI2C_SR, val)

#define get_data(adap) adap->getreg(adap->hw, REG_ASICSI2C_DR)
#define set_data(adap, val) adap->setreg(adap->hw, REG_ASICSI2C_DR, val)

#define asics_set_bit(adap, register, bit) \
	adap->setreg(adap->hw, register, adap->getreg(adap->hw, register) | bit)
#define asics_clear_bit(adap, register, bit) \
	adap->setreg(adap->hw, register, adap->getreg(adap->hw, register) & ~bit)

#define wait_for_tdone(adap) adap->wait_for_tdone(adap->hw)

/******************************************************************************
 * debugging into a buffer                                                    *
 ******************************************************************************/

#if DEBUG_TO_BUFFER

#define DBG_BUFFER_ENTRY_SIZE	160
#define DBG_BUFFER_SIZE		10000

static char dbg_buffer[DBG_BUFFER_SIZE][DBG_BUFFER_ENTRY_SIZE];
static spinlock_t dbg_lock = SPIN_LOCK_UNLOCKED;
static int dbg_buffer_pos = 0;

void i2c_asics_print(const char *fmt, ...) {
	unsigned long flags;
	
	spin_lock_irqsave(&dbg_lock, flags);
	if (dbg_buffer_pos < DBG_BUFFER_SIZE) {
		va_list args;
		va_start(args, fmt);
		vsnprintf(dbg_buffer[dbg_buffer_pos++], DBG_BUFFER_ENTRY_SIZE, fmt, args);
		va_end(args);
	}
	
	spin_unlock_irqrestore(&dbg_lock, flags);
}

static void i2c_asics_print_dbg(void) {
	unsigned long flags;
	int i;
	
	spin_lock_irqsave(&dbg_lock, flags);

	printk("**************** Collected %d debug messages:\n", dbg_buffer_pos);
	for (i = 0; i < dbg_buffer_pos; i++) {
		printk(dbg_buffer[i]);
	}
	printk("**************** debug messages finished\n");
	
	dbg_buffer_pos = 0;

	spin_unlock_irqrestore(&dbg_lock, flags);
}

EXPORT_SYMBOL(i2c_asics_print);

#endif /* DEBUG_TO_BUFFER */

/* --- other auxiliary functions --------------------------------------	*/

static void i2c_start(struct i2c_algo_asics_data *adap) 
{
	DEB4(DBG("Sending start condition.\n"););
	/* don't use set_ctl_cmd() here because it does not set the MS bit */
	adap->current_state = I2C_ASICS_MODE_MASTER;
	set_ctl_cmd_control(adap, BIT_ASICSI2C_CC_MS | BIT_ASICSI2C_CC_WR) ;
}

static void i2c_repstart(struct i2c_algo_asics_data *adap) 
{
	DEB4(DBG("Sending repeated start condition.\n"););
	set_ctl_cmd(adap, BIT_ASICSI2C_CC_RSTA | BIT_ASICSI2C_CC_WR); 
}

static void i2c_stop(struct i2c_algo_asics_data *adap) 
{
	DEB4(DBG("Sending stop condition.\n"););
	/* don't use set_ctl_cmd() here because it does not clear the MS bit */
	set_ctl_cmd_control(adap, 0);
}

static void i2c_reset(struct i2c_algo_asics_data *adap)
{
	DEB(DBG("Resetting core.\n"););
	
	adap->setreg(adap->hw, REG_ASICSI2C_SEC_CMD, BIT_ASICSI2C_SCR_RESET);
	udelay(10);
	adap->configure(adap->i2c_adap, 0);
}

static int wait_for_bb(struct i2c_algo_asics_data *adap) {

	int timeout = BUS_WAIT_RETRIES;
	int status;

	DEB4(DBG("waiting for bus to become free.\n");)

	status = get_sts(adap);
#ifndef STUB_I2C
	if (status & BIT_ASICSI2C_SR_BSY) {
		DEB(DBG("Bus busy.\n"););
		do {
			udelay(BUS_WAIT_TIME); /* wait for 100 us */
			status = get_sts(adap);
			timeout--;
		} while (timeout && (status & BIT_ASICSI2C_SR_BSY));
		
		if (status & BIT_ASICSI2C_SR_BSY) {
			WARN("Bus still not free, resetting.\n");
			i2c_reset(adap);
			status = get_sts(adap);
		}
	}
#endif
	if (status & BIT_ASICSI2C_SR_BSY) {
		ERR("Timeout waiting for Bus to become free.\n");
		return -1;
	}
	
	return 0;
}

/* ----- Utility functions
 */

static int wait_for_tdone_or_arbitration_lost(struct i2c_algo_asics_data *adap, char* function)
{
	int timeout = wait_for_tdone(adap);
	struct iic_asics_hw *hw = adap->hw;
	
	if (timeout) {
		i2c_stop(adap);
		ERR("%s: time out waiting for transfer done.\n", function);
		return (-1);
	}

#ifndef STUB_I2C
	if (hw->arbitration_lost) {
		DBG("%s: arbitration lost.\n", function);
		return (-1);
	}
#endif
	return 0;
}

static inline int try_address(struct i2c_algo_asics_data *adap,
		       unsigned char addr, int retries, int rep_start)
{
	int i, status, ret = -1;
	for (i=0;i<retries;i++) {
		set_data(adap, addr);
		if (rep_start) i2c_repstart(adap);
		else i2c_start(adap);
		rep_start = 1;
		if (wait_for_tdone(adap) >= 0) {
			status=get_sts(adap);
			if (!(status & BIT_ASICSI2C_SR_R_NO_ACK)) {
				ret=1;
				break;	/* success! */
			}
		}
		i2c_stop(adap);
		udelay(adap->udelay);
	}
	DEB2(if (i) DBG("needed %d retries for %d\n",i, addr);)
	return ret;
}

static int asics_sendbytes(struct i2c_adapter *i2c_adap, const char *buf, int count)
{
	struct i2c_algo_asics_data *adap = i2c_adap->algo_data;
	int wrcount, status, timeout;
    
	for (wrcount=0; wrcount<count; ++wrcount) {
		DEB2(DBG("asics_sendbytes: writing %2.2X\n", buf[wrcount]&0xff);)
		/* send a byte*/
		set_data(adap, buf[wrcount]);
		set_ctl_cmd(adap, BIT_ASICSI2C_CC_WR);

		timeout = wait_for_tdone_or_arbitration_lost(adap, "asics_sendbytes");
		status = get_sts(adap);
		if (timeout) {
			return -EREMOTEIO; /* got a better one ?? */
		}
#ifndef STUB_I2C
		if (status & BIT_ASICSI2C_SR_R_NO_ACK) {
			i2c_stop(adap);
			ERR("asics_sendbytes: error - no ack.\n");
			return -EREMOTEIO; /* got a better one ?? */
		}
#endif
	}

	return (wrcount);
}



static int asics_readbytes(struct i2c_adapter *i2c_adap, char *buf, int count)
{
	int i;
	struct i2c_algo_asics_data *adap = i2c_adap->algo_data;

	for (i = 0; i < count; i++) {
		if (i == count - 1) {
			set_ctl_cmd(adap, BIT_ASICSI2C_CC_RD | BIT_ASICSI2C_CC_T_NO_ACK);
		}
		else {
			set_ctl_cmd(adap, BIT_ASICSI2C_CC_RD);
		}

		if (wait_for_tdone_or_arbitration_lost(adap, "asics_readbytes")) {
			return -EREMOTEIO;
		}

		buf[i] = get_data(adap);
		
	}

	return (i);
}

static inline int asics_doAddress(struct i2c_algo_asics_data *adap,
                                struct i2c_msg *msg, int retries, int rep_start) 
{
	unsigned short flags = msg->flags;
	struct iic_asics_hw *hw = adap->hw;
	unsigned char addr;
	int ret;
	
	if ( (flags & I2C_M_TEN)  ) { 
		/* a ten bit address */
		addr = 0xf0 | (( msg->addr >> 7) & 0x03);
		DEB2(DBG("Addressing i2c slave at address 0x%03x\n", msg->addr);)
		/* try extended address code...*/
		ret = try_address(adap, addr, retries, rep_start);
		rep_start = 1;
		if (ret!=1) {
			ERR("died at extended address code.\n");
			return -EREMOTEIO;
		}
		/* the remaining 8 bit address */
		set_data(adap,msg->addr & 0x7f);
		set_ctl_cmd(adap, BIT_ASICSI2C_CC_WR);
		if (wait_for_tdone(adap)) {
			i2c_stop(adap);
			ERR("died at 2nd address code.\n");
			return -EREMOTEIO;
		}
		if ( flags & I2C_M_RD ) {
			i2c_repstart(adap);
			/* okay, now switch into reading mode */
			addr |= 0x01;
			ret = try_address(adap, addr, retries, rep_start);
			if (ret!=1) {
				ERR("died at extended address code.\n");
				return -EREMOTEIO;
			}
		}
	} else {		/* normal 7bit address	*/
		int status;
		int i=retries;
		int resetted = 0;

		while (i--) {
			addr = ( msg->addr << 1 );
			DEB2(DBG("Addressing i2c slave at address 0x%02x\n", msg->addr);)
			if (flags & I2C_M_RD )
				addr |= 1;
			if (flags & I2C_M_REV_DIR_ADDR )
				addr ^= 1;
			set_data(adap, addr);
			if (rep_start) i2c_repstart(adap);
			else i2c_start(adap);
			rep_start = 1;
			
			/* waiting for TDONE */
			if (wait_for_tdone(adap)) {
				i2c_stop(adap);
				if (!resetted) {
					resetted = 1;
					WARN("%s: time out waiting for transfer done, resetting.\n", "asics_doAddress");
					i2c_reset(adap);
					// another try
					i++;
					continue;
				} else {
					ERR("%s: time out waiting for transfer done.\n", "asics_doAddress");
					return -EREMOTEIO;
				}
			}

			/* check ACK */
			status=get_sts(adap);
			if (!(status & BIT_ASICSI2C_SR_R_NO_ACK))
				return 0; /* success */
			
			/* wait a while and try it again if we lost arbitration; give up otherwise */
			i2c_stop(adap);
			rep_start = 0;
			if (hw->arbitration_lost && i>0) mdelay(adap->mdelay);
			else break;
	
		}
		/* we tried enough times, failed to write address */		
		DBG("failed to address slave at %x\n", msg->addr);
		return -EREMOTEIO;

	}
	return 0;
}

static int asics_xfer(struct i2c_adapter *i2c_adap,
		    struct i2c_msg msgs[], 
		    int num)
{
	struct i2c_algo_asics_data *adap = i2c_adap->algo_data;
	struct i2c_msg *pmsg;
	int i;
	int ret=0, timeout, status;
	int sent = 0;
    
	DEB3(DBG("asics_xfer: adapter %s, num=%d\n", i2c_adap->name, num);)
	
	/* Check for bus busy */
	timeout = wait_for_bb(adap);
	if (timeout) {
		DEB2(ERR("Timeout waiting for BB in asics_xfer\n");)
		return -EIO;
	}
	
	for (i = 0;ret >= 0 && i < num; i++) {
		pmsg = &msgs[i];

		DEB2(DBG("%s %d bytes %s 0x%02x - %d of %d messages\n",
		     pmsg->flags & I2C_M_RD ? "Reading" : "Writing",
                     pmsg->len,
                     pmsg->flags & I2C_M_RD ? "from" : "to",
                     pmsg->addr, i + 1, num);)
    
		ret = asics_doAddress(adap, pmsg, i2c_adap->retries, i != 0);
		if (ret < 0) {
			DEB2(ERR("Could not send slave address\n");)
			continue;
		}


		/* Wait for TIP to negate */
		timeout = wait_for_tdone(adap);
		if (timeout) {
			DEB2(ERR("Timeout waiting for TIP to negate in asics_xfer\n");)
			i2c_stop(adap);
			return (-EREMOTEIO);
		}
    
		/* Check RxACK (last rcvd bit - slave ack) */
		status=get_sts(adap);
		if (status & BIT_ASICSI2C_SR_R_NO_ACK) {
			i2c_stop(adap);
			DEB2(DBG("No RxACK(1) in asics_xfer\n");)
			return (-EREMOTEIO);
		}
    
		DEB3(DBG("Msg %d, addr=0x%x, flags=0x%x, len=%d\n", i, msgs[i].addr, msgs[i].flags, msgs[i].len);)
    
		/* Read */
		if (pmsg->flags & I2C_M_RD) {
			/* read bytes into buffer*/
			ret = asics_readbytes(i2c_adap, pmsg->buf, pmsg->len);
        
			if (ret != pmsg->len) {
				DEB2(DBG("fail: only read %d bytes.\n",ret);)
			} else {
				DEB2(DBG("read %d bytes.\n",ret);)
			}
		} else { /* Write */
			ret = asics_sendbytes(i2c_adap, pmsg->buf, pmsg->len);
        
			if (ret != pmsg->len) {
				DEB2(DBG("fail: only wrote %d bytes.\n",ret);)
			} else {
				DEB2(DBG("wrote %d bytes.\n",ret);)
			}
		}
		
		++sent;
	}
	
	i2c_stop(adap);

        return ret < 0 ? ret : sent;
}

/* set own slave address */
int i2c_asics_slave_set_address(struct i2c_adapter *i2c_adap, unsigned int index, unsigned int address)
{
	struct i2c_algo_asics_data *adap = i2c_adap->algo_data;

	if (index >= NO_SLAVE_ADDRESSES) {
		return -EINVAL;
	}

	/* disable the core */
	adap->setreg(adap->hw, REG_ASICSI2C_CCR, 0);
	
	adap->setreg(adap->hw, REG_ASICSI2C_ESAR, 0);
	adap->setreg(adap->hw, REG_ASICSI2C_SAR0 + index, (address & 0x7f) << 1);
	DEB(DBG("set i2c slave address %u to 0x%x\n", index, address);)
	
	/* re-enable the core */
	adap->setreg(adap->hw, REG_ASICSI2C_CCR, BIT_ASICSI2C_CC_ENA | BIT_ASICSI2C_CC_IEN);
	
	return 0;
}

static u32 asics_func(struct i2c_adapter *adap)
{
	return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL | I2C_FUNC_10BIT_ADDR | 
	       I2C_FUNC_PROTOCOL_MANGLING; 
}

static int asics_algo_control(struct i2c_adapter *adap, 
			      unsigned int cmd, unsigned long arg)
{
	switch (cmd) {
	    case I2C_SLAVE_SET_ADDRESS:
	    	/* HACK: we use the higher word as index,
	    	         the lower word as slave address */
		i2c_asics_slave_set_address(adap, (unsigned int) (arg >> 16), (unsigned int) (arg & 0x7f));
#if DEBUG_TO_BUFFER
		i2c_asics_print_dbg();
#endif /* DEBUG_TO_BUFFER */
		break;
	    default:
		return -1;
	}
	return 0;
}

/* -----slave mode functions: ----------------------------------------	*/

static void block_next_slave_transfer(struct i2c_algo_asics_slave_data *slave_data)
{
	DEB(DBG("cannot put data to i2c-slave layer, next operation will be blocked.\n");)
	slave_data->block_next_slave_operation = 1;
}

static int i2c_asics_slave_write_complete(struct i2c_adapter *i2c_adap)
{
	struct i2c_algo_asics_data *adap = i2c_adap->algo_data;
	struct i2c_algo_asics_slave_data *slave_data = adap->slave_data;
	
	int ret = 0;
	int written_bytes;

	DEB3(DBG("Slave write operation complete.\n");)

	slave_data->last_slave_op = I2C_ASICS_SLAVE_OP_NONE;

	if (slave_data->slave_write_count <= 0) {
		return slave_data->slave_write_count;
	}
	
	/* copy the written data to the upper I2C slave layer */
	written_bytes = iic_slave_write(i2c_adap, slave_data->last_addressed_slave,
		slave_data->slave_write_buffer, slave_data->slave_write_count);
	if (written_bytes < 0) {
		DEB(WARN("i2c_asics_slave_write_complete: error exporting data to i2c-slave layer: %d!\n",
			written_bytes);)
		ret = -ENOMEM;
	}
	else if (written_bytes == 0) {
		DEB2(DBG("i2c_asics_slave_write_complete: 0 bytes to i2c-slave layer exported!\n");)
	}
	else if (written_bytes >= slave_data->slave_write_count) {
		DEB2(DBG("i2c_asics_slave_write_complete: moved %d bytes to i2c-slave layer.\n",
			written_bytes);)
		slave_data->slave_write_ptr = slave_data->slave_write_buffer;
		slave_data->slave_write_count = 0;
	}
	else {
		/* only a partial copy */
		DEB2(DBG("i2c_asics_slave_write_complete: moved %d bytes to i2c-slave layer, %d bytes requested.\n",
			written_bytes, slave_data->slave_write_count);)
		slave_data->slave_write_ptr -= written_bytes;
		slave_data->slave_write_count -= written_bytes;
	}
	
	return ret;
}

static int i2c_asics_slave_read_complete(struct i2c_adapter *i2c_adap)
{
	struct i2c_algo_asics_data *adap = i2c_adap->algo_data;
	struct i2c_algo_asics_slave_data *slave_data = adap->slave_data;

	DEB3(DBG("Slave read operation complete.\n");)

	slave_data->last_slave_op = I2C_ASICS_SLAVE_OP_NONE;

	if (slave_data->slave_read_ptr < slave_data->slave_read_buffer + slave_data->slave_read_count) {
		DEB2(DBG("i2c_asics_slave_read_complete: not all data was read! amount: %d, read: %d\n",
			slave_data->slave_read_count, slave_data->slave_read_ptr - slave_data->slave_read_buffer);)
	}
        else {
		DEB2(DBG("i2c_asics_slave_read_complete: read %d bytes\n",
			slave_data->slave_read_ptr - slave_data->slave_read_buffer);)
	}
	
	slave_data->slave_read_ptr = slave_data->slave_read_buffer;
	slave_data->slave_read_count = 0;
	
	return 0;
}

int i2c_asics_slave_restart_xfer(struct i2c_adapter *i2c_adap)
{
	struct i2c_algo_asics_data *adap = i2c_adap->algo_data;
	struct i2c_algo_asics_slave_data *slave_data = adap->slave_data;
	
	int ret = 0;

	unsigned long flags;
	
        spin_lock_irqsave(&slave_data->slave_lock, flags);
        
        /* the next message shall be blocked, but no blocked message read yet
           --> just deliver the last message */
        if (slave_data->block_next_slave_operation) {
        	slave_data->block_next_slave_operation = 0;
		if (i2c_asics_slave_write_complete(i2c_adap) < 0) {
			block_next_slave_transfer(slave_data);
			goto bail;
		}
        }
        
 bail:
        spin_unlock_irqrestore(&slave_data->slave_lock, flags);
	return ret;
}

static uint8_t finish_last_transfer(struct i2c_adapter *i2c_adap, slave_op_t last_slave_op)
{
	struct i2c_algo_asics_data *adap = i2c_adap->algo_data;
	struct i2c_algo_asics_slave_data *slave_data = adap->slave_data;
	int ret = 0;
	
	if (last_slave_op == I2C_ASICS_SLAVE_OP_WRITE) {
		DEB2(DBG("finish_last_transfer: finishing write operation.\n");)
		if (slave_data->slave_write_count > 0) {
			if (i2c_asics_slave_write_complete(i2c_adap) != 0) {
				block_next_slave_transfer(slave_data);
				ret = BIT_ASICSI2C_CC_T_NO_ACK;
				goto bail;
			}
		}
		else {
			DEB2(DBG("finish_last_transfer: no bytes received in write operation (%d).\n",
				slave_data->slave_write_count);)
		}
	}
	else if (last_slave_op == I2C_ASICS_SLAVE_OP_READ) {
		DEB2(DBG("finish_last_transfer: finishing read operation.\n");)
		if (i2c_asics_slave_read_complete(i2c_adap) != 0) {
			block_next_slave_transfer(slave_data);
			ret = BIT_ASICSI2C_CC_T_NO_ACK;
			goto bail;
		}
	}
	else {
		DEB4(DBG("finish_last_transfer: no operation to finish (%d)!\n", last_slave_op);)
	}
 bail:
	return ret;
}

static int i2c_asics_handle_slave_read(struct i2c_adapter *i2c_adap, uint8_t sr)
{
	int ret = -1;
	uint8_t cr = 0, res;

	struct i2c_algo_asics_data *adap = i2c_adap->algo_data;
	struct i2c_algo_asics_slave_data *slave_data = adap->slave_data;

	int is_first_slave_byte = sr & BIT_ASICSI2C_SR_CAS;
	slave_op_t last_slave_op = slave_data->last_slave_op;
	
	uint8_t data;
	
	slave_data->last_slave_op = I2C_ASICS_SLAVE_OP_READ;

	/* we cannot detect the end of a message when a repeated start
	   is driven, so assume that the last message is finished when
	   a new message starts */
	if (is_first_slave_byte && (res = finish_last_transfer(i2c_adap, last_slave_op)) != 0) {
		cr |= res;
		ret = cr;
		goto bail;
	}
	
	/* fetch data from slave layer */
	if (is_first_slave_byte) {
		slave_data->last_addressed_slave = (get_data(adap) >> 1) & 0x7f;
		
		DEB2(DBG("iic_handle_slave_first_read: Fetching slave read data from i2c-slave layer.\n");)
		
	        slave_data->slave_read_ptr = slave_data->slave_read_buffer;
	        slave_data->slave_read_count =
	        	iic_slave_read(i2c_adap, slave_data->last_addressed_slave, slave_data->slave_read_buffer, SLAVE_BUF_SIZE, -1);
	        
	        if (slave_data->slave_read_count > 0) {
	        	DEB2(DBG("iic_handle_slave_first_read: Got %d bytes.\n", slave_data->slave_read_count);)
	        }
	        else {
	        	DEB2(DBG("iic_handle_slave_first_read: cannot read bytes.\n");)
	        }
	}
	
	/* write a byte */
	if (slave_data->slave_read_count > 0) {
		/* avoid buffer overrun here */
		if (slave_data->slave_read_ptr < slave_data->slave_read_buffer + slave_data->slave_read_count) {
			data = *(slave_data->slave_read_ptr);
			(slave_data->slave_read_ptr)++;
		}
		else {
			/* better error handline here? */
			data = 0;
			DEB(DBG("i2c_asics_handle_slave_read: no data!\n");)
		}
	}
	else {
		/* FIXME: signal that we are not ready by not sending data */
		data = 0;
		DEB(DBG("i2c_asics_handle_slave_read: not ready, sending no byte\n");)
	}
	
	DEB2(DBG("i2c_asics_handle_slave_read: sending byte 0x%02x\n", data);)

	set_data(adap, data);
	ret = BIT_ASICSI2C_CC_WR;
	
 bail:
	return ret;
}

static int i2c_asics_handle_slave_write(struct i2c_adapter *i2c_adap, uint8_t sr)
{
	int ret = -1;
	uint8_t cr = 0, res;
	
	struct i2c_algo_asics_data *adap = i2c_adap->algo_data;
	struct i2c_algo_asics_slave_data *slave_data = adap->slave_data;

	int is_first_slave_byte = sr & BIT_ASICSI2C_SR_CAS;
	slave_op_t last_slave_op = slave_data->last_slave_op;

	slave_data->last_slave_op = I2C_ASICS_SLAVE_OP_WRITE;

	/* we cannot detect the end of a message when a repeated start
	   is driven, so assume that the last message is finished when
	   a new message starts */
	if (is_first_slave_byte && (res = finish_last_transfer(i2c_adap, last_slave_op)) != 0) {
		cr |= res;
		goto bail;
	}
	
	/* read the byte */
	if (is_first_slave_byte) {
		slave_data->last_addressed_slave = (get_data(adap) >> 1) & 0x7f;
	} else {
		slave_data->slave_write_count++;

		*(slave_data->slave_write_ptr) = get_data(adap);

		DEB2(DBG("i2c_asics_handle_slave_write: putting byte 0x%02x\n",
			*slave_data->slave_write_ptr);)

		/* avoid buffer overrun here */
		if (slave_data->slave_write_ptr < slave_data->slave_write_buffer + SLAVE_BUF_SIZE) {
			(slave_data->slave_write_ptr)++;
		}
	}
	
	ret = BIT_ASICSI2C_CC_RD;
	
 bail:
	DEB2(DBG("i2c_asics_handle_slave_write: ret=0x%x.\n", ret);)
	return ret;
}

int i2c_asics_handle_slave_request(struct i2c_adapter *i2c_adap)
{
	struct i2c_algo_asics_data *adap = i2c_adap->algo_data;
	struct i2c_algo_asics_slave_data *slave_data = adap->slave_data;
	
	uint8_t sr = get_sts(adap);
	uint8_t cr = BIT_ASICSI2C_CC_ENA | BIT_ASICSI2C_CC_IEN, res;
	int op_ret = 0;
	
	int is_slave_write_request = 0;
	int is_slave_read_request = 0;
	int is_slave_operation_complete = 0;
		
	unsigned long flags;
	
        spin_lock_irqsave(&slave_data->slave_lock, flags);
        DEB3(DBG("handling slave request.\n");)
        
        if (adap->current_state != I2C_ASICS_MODE_SLAVE) {
        	DEB3(DBG("We are not in slave mode, probably we detected another transmission.\n");)
        	goto bail;
        }
        
	/* parse the request */
	is_slave_write_request = !(sr & BIT_ASICSI2C_SR_SRW);
	is_slave_read_request = !is_slave_write_request;
	is_slave_operation_complete = (sr & BIT_ASICSI2C_SR_DSP) || (is_slave_read_request && (sr & BIT_ASICSI2C_SR_R_NO_ACK));
	
	/* if a request is finished (stop condition), finish the operation */
	if (is_slave_operation_complete) {
		DEB3(DBG("Slave operation complete.\n");)
		if ((res = finish_last_transfer(i2c_adap, slave_data->last_slave_op)) != 0) {
			cr |= res;
			block_next_slave_transfer(slave_data);
		}
		goto bail;
	}	

	/* if operation should be blocked, don't accept it */
	if (slave_data->block_next_slave_operation) {
		DEB(INFO("This operation is being blocked.\n");)
		cr |= BIT_ASICSI2C_CC_T_NO_ACK;
		goto bail;
	}
	
	/* process the read or write operation */
	if (is_slave_write_request) {
		DEB3(DBG("Slave write request.\n");)
		op_ret = i2c_asics_handle_slave_write(i2c_adap, sr);
	}
	else if (is_slave_read_request) {
		DEB3(DBG("Slave read request.\n");)
		op_ret = i2c_asics_handle_slave_read(i2c_adap, sr);
	}

	if (op_ret > 0) {
		cr |= (op_ret & 0xff);
	}
	
 bail:
        DEB3(DBG("slave request handling done.\n");)
        spin_unlock_irqrestore(&slave_data->slave_lock, flags);
        
        return cr;
}

/* -----exported algorithm data: -------------------------------------	*/

static struct i2c_algorithm asics_algo = {
	.name		= "Asics algorithm",
	.id		= I2C_ALGO_ASICS,
	.master_xfer	= asics_xfer,
	.algo_control	= asics_algo_control,
	.functionality	= asics_func,
};

/* 
 * registering functions to load algorithms at runtime 
 */

int i2c_asics_add_bus(struct i2c_adapter *adap)
{
	DEB2(DBG("hw routines of %s registered.\n", adap->name);)

	/* register new adapter to i2c module... */

	adap->id |= asics_algo.id;
	adap->algo = &asics_algo;

	adap->timeout = BUS_WAIT_TIME;
	adap->retries = ADDRESS_RETRIES;

	i2c_add_adapter(adap);
	return 0;
}


int i2c_asics_del_bus(struct i2c_adapter *adap)
{
	return i2c_del_adapter(adap);
}

int __init i2c_algo_asics_init (void)
{
	INFO("Asics i2c algorithm module\n");
	return 0;
}


void i2c_algo_asics_exit(void)
{
	return;
}

EXPORT_SYMBOL(i2c_asics_add_bus);
EXPORT_SYMBOL(i2c_asics_del_bus);
EXPORT_SYMBOL(i2c_asics_slave_restart_xfer);
EXPORT_SYMBOL(i2c_asics_handle_slave_request);
EXPORT_SYMBOL(i2c_asics_slave_set_address);

module_init(i2c_algo_asics_init);
module_exit(i2c_algo_asics_exit);
