/*  /drivers/i2c/busses/i2c-fia320.c
	Support of the Faraday i2c bus
	
	Copyright (C) 2004 Microtronix Datacom Ltd.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
    Written by Wentao Xu <wentao@microtronix.com>
*/

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/i2c.h>
#include <linux/i2c-algo-fia320.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/arch/pp_bd.h>
#include <asm/arch/cpe_clk.h>

#include <linux/i2c-fia320.h>

MODULE_AUTHOR("Peppercon AG");
MODULE_DESCRIPTION("I2C-Bus adapter routines for Faraday FIA320 i2c controller");
MODULE_LICENSE("GPL");

MODULE_PARM(i2c_debug,"i");
MODULE_PARM_DESC(i2c_debug, "debug level - 0 off; 1 normal; 2,3 more verbose; 9 fia-protocol");

#define FIA_I2C_IO_SIZE 		28

/******************************************************************************
* global defines                                                              *
******************************************************************************/

int i2c_debug = 0;

#define NDEBUG
// #undef NDEBUG

#ifndef NDEBUG
#define DEB(x) if (i2c_debug>=1) x
#define DEB2(x) if (i2c_debug>=2) x
#define DEB3(x) if (i2c_debug>=3) x
#define DEBPROTO(x) if (i2c_debug>=9) x
#else
#define DEB(x) do {} while (0);
#define DEB2(x) do {} while (0);
#define DEB3(x) do {} while (0);
#define DEBPROTO(x) do {} while (0);
#endif

#define FIAI2C_SCL_DEFAULT	400000

#if DEBUG_TO_BUFFER
#define dbg	i2c_fia_print
#else /* DEBUG_TO_BUFFER */
#define dbg(fmt, x...)	printk(KERN_DEBUG fmt, ##x)
#endif /* DEBUG_TO_BUFFER */

/******************************************************************************
* function prototypes                                                         *
******************************************************************************/

static void fia_hw_exit(struct iic_fia320_hw* hw);

/******************************************************************************
* board configuration                                                         *
******************************************************************************/

static struct i2c_fia_interface_configuration i2c_fia_interfaces[] = {

    {
    	base:		CPE_I2C1_VA_BASE,
    	irq:		IRQ_I2C1,
    	scl:		400000,
    	clock:		0,
    	slave_address:	0x55,
    	name:		"FIA320 I2C 1"
    },
    {
    	base:		CPE_I2C2_VA_BASE,
    	irq:		IRQ_I2C2,
    	scl:		400000,
    	clock:		0,
    	slave_address:	0x55,
    	name:		"FIA320 I2C 2"
    },
    {
    	base:		CPE_I2C3_VA_BASE,
    	irq:		IRQ_I2C3,
    	scl:		400000,
    	clock:		0,
    	slave_address:	0x55,
    	name:		"FIA320 I2C 3"
    },
    {
    	base:		CPE_I2C4_VA_BASE,
    	irq:		IRQ_I2C4,
    	scl:		400000,
    	clock:		0,
    	slave_address:	0x55,
    	name:		"FIA320 I2C 4"
    },
    {
    	base:		CPE_I2C5_VA_BASE,
    	irq:		IRQ_I2C5,
    	scl:		400000,
    	clock:		0,
    	slave_address:	0x55,
    	name:		"FIA320 I2C 5"
    },
#if defined (PP_FEAT_KIRA_USE_I2C6)
    {
    	base:		CPE_I2C6_VA_BASE,
    	irq:		IRQ_I2C6,
    	scl:		400000,
    	clock:		0,
    	slave_address:	0x55,
    	name:		"FIA320 I2C 6"
    },
#endif
};

#define FIA_NUM_INTERFACES	(sizeof(i2c_fia_interfaces) / sizeof(struct i2c_fia_interface_configuration))

/******************************************************************************
* helper functions                                                            *
******************************************************************************/

static void fia_i2c_setreg (struct iic_fia320_hw *hw, int regno, int val)
{
	int address = hw->base + regno;
	
	DEB2(dbg("fia_i2c_setreg: address %08x, value %08x\n",address,val));
	outl(val, address);
}

static int fia_i2c_getreg(struct iic_fia320_hw *hw, int regno)
{
	int address = hw->base + regno;
	
	return inl(address);
}

static int fia_i2c_reset(struct i2c_adapter *i2c_adap) {
	struct i2c_algo_fia320_data *adap = (struct i2c_algo_fia320_data *) i2c_adap->algo_data;
	struct iic_fia320_hw *hw = adap->hw;
	struct i2c_fia_interface_configuration *config = adap->config;

	int cdr, scl, gsr, tsr;

	int timeout = 10;

	/* reset i2c chip  */
	fia_i2c_setreg(hw, REG_FA320_CR, FA320_CR_I2CRST);

	/* 2 pclk clock cycles */
	while (timeout > 0) {
		wait_event_interruptible_timeout(hw->wait, ((! (fia_i2c_getreg(hw, REG_FA320_CR) & FA320_CR_I2CRST) )), HZ);
		if (!(fia_i2c_getreg(hw, REG_FA320_CR) & FA320_CR_I2CRST)) {
			break;
		}
		timeout--;
	}
	hw->isr = 0;

	/* stop a possible running transfer here by clocking the scl line until the sda line gets released */
        if (!(fia_i2c_getreg(hw, REG_FA320_BMR) & FA320_BMR_SDA)) {
            DEB(dbg("fia_i2c_reset: found SDA line low, trying to release it.\n"));	   
	    if (i2c_fia320_clock_bus_free(i2c_adap) < 0) {
                printk(KERN_ERR "could not release i2c line\n");
            } else {
	        /* send a stop condition by bit banging */
		i2c_fia320_send_stop(i2c_adap);
	    }
        }

	/* compute prescale */
	scl = (hw->scl) ? hw->scl : FIAI2C_SCL_DEFAULT; /* maximum speed as default */

	gsr = 0x2;
	tsr = 0xf;
	cdr = (hw->clock / scl + gsr - 1) / 2;

	if (cdr < 0 || cdr > 0x3ff ||
		!(cdr > 3 + gsr + tsr)
	) {
		printk(KERN_ERR "prescale error clk_i %d scl %d cdr: %d, gsr: %d, tsr: %d\n", hw->clock, scl, cdr, gsr, tsr);
		return -1;
	}

	/* set prescale */
	fia_i2c_setreg(hw, REG_FA320_CDR, cdr & 0x000003FF);
	fia_i2c_setreg(hw, REG_FA320_TGSR, ((gsr & 0x7) << 10) | (tsr & 0x3ff));
	fia_i2c_setreg(hw, REG_FA320_CR, FA320_CR_ENABLE);

	i2c_fia320_slave_set_address(i2c_adap, config->slave_address);

	return 0;
}

/******************************************************************************
* interrupt handler                                                           *
******************************************************************************/

static spinlock_t      irq_lock = SPIN_LOCK_UNLOCKED;

static irqreturn_t fia_handler(int this_irq, void *dev_id, struct pt_regs *regs) 
{
	struct iic_fia320_hw *hw= (struct iic_fia320_hw *)dev_id;
	unsigned long flags;

	spin_lock_irqsave(&irq_lock, flags);
	i2c_fia320_isr(hw->adap, 1);	
        spin_unlock_irqrestore(&irq_lock, flags);
	return IRQ_HANDLED;
}

/******************************************************************************
* module data                                                                 *
******************************************************************************/

static struct iic_fia320_hw fia_hw[FIA_NUM_INTERFACES];

static struct i2c_algo_fia320_data fia_data[FIA_NUM_INTERFACES];

static struct i2c_adapter fia_ops[FIA_NUM_INTERFACES];

static struct i2c_algo_fia320_slave_data fia_slave[FIA_NUM_INTERFACES];

static struct i2c_slave_adapter fia_slave_adap[FIA_NUM_INTERFACES];

/******************************************************************************
* initialization functions                                                    *
******************************************************************************/

static void i2c_fia_init_adapters(int clock) {
	int i;
	
	memset(fia_ops, 0, sizeof(fia_ops));
	memset(fia_data, 0, sizeof(fia_data));
	memset(fia_hw, 0, sizeof(fia_hw));
	memset(fia_slave, 0, sizeof(fia_slave));
	memset(fia_slave_adap, 0, sizeof(fia_slave_adap));

	for (i = 0; i < FIA_NUM_INTERFACES; i++) {
		/* fill i2c_adapter struct */
		fia_ops[i].owner		= THIS_MODULE;
		fia_ops[i].id			= I2C_HW_FIA320_UKIT;
		fia_ops[i].algo_data		= &fia_data[i];
		strncpy(fia_ops[i].name, i2c_fia_interfaces[i].name, sizeof(fia_ops[i].name));
		
		/* fill i2c_algo_fia_data struct */
		fia_data[i].hw			= &fia_hw[i];
		fia_data[i].slave_data		= &fia_slave[i];
		fia_data[i].slave_adapter	= &fia_slave_adap[i];
		fia_data[i].config		= &i2c_fia_interfaces[i];
		fia_data[i].setreg		= fia_i2c_setreg;
		fia_data[i].getreg		= fia_i2c_getreg;
		fia_data[i].reset		= fia_i2c_reset;
		fia_data[i].udelay		= 10,
		fia_data[i].mdelay		= 10,
		fia_data[i].timeout		= 100,
		
		/* fill iic_fia_hw struct */
		fia_hw[i].base			= i2c_fia_interfaces[i].base ? i2c_fia_interfaces[i].base : 0;
		fia_hw[i].irq			= i2c_fia_interfaces[i].irq ? i2c_fia_interfaces[i].irq : 0;
		fia_hw[i].clock			= i2c_fia_interfaces[i].clock ? i2c_fia_interfaces[i].clock : clock;
		fia_hw[i].scl			= i2c_fia_interfaces[i].scl ? i2c_fia_interfaces[i].scl : FIAI2C_SCL_DEFAULT;
		fia_hw[i].idx			= i;
	}
}

static int fia_hw_init(struct i2c_adapter *i2c_adap, struct iic_fia320_hw *hw)
{
	hw->isr = 0;
	hw->initialized = 0;

	if (!request_mem_region(hw->base, FIA_I2C_IO_SIZE, "i2c (bus adapter)")) {
		printk(KERN_ERR
		       "i2c-fia320: requested MMIO region (0x%X:2) "
		       "is in use.\n", hw->base);
		return -ENODEV;
	}
	
	hw->adap = i2c_adap;
	if (hw->irq > 0) {
		if (request_irq(hw->irq, &fia_handler, 0, "Faraday FIA320 I2C", hw) < 0) {
			printk(KERN_ERR "i2c-fia320: Request irq%d failed\n", hw->irq);
			hw->irq = 0;
			goto fail;
		} else {
			set_irq_type(hw->irq, IRQT_HIGH);
		}
	}

	/* reset i2c chip  */
	if (fia_i2c_reset(i2c_adap)) {
		goto fail;
	}

	hw->initialized = 1;

	DEB2(dbg("fia_hw_init successful\n"));

	return 0;
fail:
	release_mem_region(hw->base, FIA_I2C_IO_SIZE);

	return -1;
}

static void fia_slave_init(struct i2c_algo_fia320_slave_data * slave_data)
{
	spin_lock_init(slave_data->slave_lock);
	
	slave_data->block_next_slave_operation = 0;
	slave_data->slave_write_in_progress = 0;
	slave_data->slave_read_in_progress = 0;

	slave_data->slave_write_ptr = slave_data->slave_write_buffer;
	slave_data->slave_write_count = 0;
	slave_data->slave_read_ptr = slave_data->slave_read_buffer;
	slave_data->slave_read_count = 0;
}

/* register adapter in slave layer */
static int fia_slave_init_adapter(struct i2c_slave_adapter *slave, struct i2c_adapter *adap)
{
	int ret = 0;

	slave->adapter = adap;
	slave->reactivate = i2c_fia320_slave_restart_xfer;
	if ((ret = i2c_slave_add_adapter(slave)) != 0){
		printk("failed to register i2c slave adapter\n");
		return ret;
	}
	
	return 0;
}

static int i2c_fia_adapter_init(struct i2c_adapter *i2c_adap)
{
	struct i2c_algo_fia320_data *adap = (struct i2c_algo_fia320_data *) i2c_adap->algo_data;
	struct iic_fia320_hw *hw = adap->hw;
	struct i2c_algo_fia320_slave_data *slave_data = adap->slave_data;
	struct i2c_slave_adapter *slave_adapter = adap->slave_adapter;
	
	DEB3(dbg("Initializing hardware for %s.\n", i2c_adap->name));
	
	init_waitqueue_head(&hw->wait);
	if (fia_hw_init(i2c_adap, hw) == 0) {
		if (i2c_fia320_add_bus(i2c_adap) < 0) {
			fia_hw_exit(hw);
			return -1;
		}
		if (fia_slave_init_adapter(slave_adapter, i2c_adap) < 0) {
			fia_hw_exit(hw);
			return -1;
		}

		fia_slave_init(slave_data);
	}
	
	return 0;
}

/******************************************************************************
* cleanup functions                                                           *
******************************************************************************/

/* deregister */
static int fia_slave_cleanup_adapter(struct i2c_slave_adapter *slave)
{
	int ret;
	
	if ((ret = i2c_slave_remove_adapter(slave)) != 0) {
		printk("failed to deregister i2c slave adapter\n");
		return ret;
	}
	
	return 0;
}

static void fia_hw_exit(struct iic_fia320_hw* hw)
{
	/* free irq */
	if (hw->irq > 0) {
		disable_irq(hw->irq);
		free_irq(hw->irq, hw);
	}	
	/* free mem region */
	release_mem_region(hw->base, FIA_I2C_IO_SIZE);
}

static void fia_adapter_cleanup(struct i2c_adapter *i2c_adap)
{
	struct i2c_algo_fia320_data *data = (struct i2c_algo_fia320_data *) i2c_adap->algo_data;
	struct iic_fia320_hw *hw = data->hw;
	struct i2c_slave_adapter *slave_adapter = data->slave_adapter;

	if (hw->initialized == 0) {
	    DEB(dbg("i2c_asics_adapter_cleanup: adapter '%s' not used, skipping\n", i2c_adap->name));
	    return;
	}
	
	i2c_fia320_del_bus(i2c_adap);
	fia_hw_exit(hw);
	fia_slave_cleanup_adapter(slave_adapter);
}


/******************************************************************************
* module stuff                                                                *
******************************************************************************/

static int __init i2c_fia320_init(void) 
{
        bd_arm_t* bd = (bd_arm_t*)__res;
        int i;

	printk(KERN_INFO "i2c-fia320: i2c Faraday FIA320 adapter driver\n");

	/* parameters */
	i2c_fia_init_adapters(BD_APB_CLK(bd));

	for (i = 0; i < FIA_NUM_INTERFACES; i++) {
		i2c_fia_adapter_init(&fia_ops[i]);
	}
	
	return 0;
}


static void i2c_fia320_exit(void)
{
        int i;

	for (i = 0; i < FIA_NUM_INTERFACES; i++) {
		fia_adapter_cleanup(&fia_ops[i]);
	}
}


module_init(i2c_fia320_init);
module_exit(i2c_fia320_exit);
