// downsampling engine

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <netinet/in.h>
#include <byteswap.h>

#include "endian.h"
#include "bmp.h"
#include "rle_util.h"
#include "ccr.h"

// Globals
RLEINFO rleInfo;

unsigned char cellMap[MAXWIDTH][MAXWIDTH][3];

int mode = 0;
FILE *inFP;

// Functions
void usage();
void encodeCell(), encodeFrame();

// Main program
int main(int argc, char *argv[])
{
    char *inFile = 0;
    char use_ccr = 0;
    unsigned long ccr;
    int c;
    
    memset(&rleInfo, 0, sizeof(rleInfo));
    
    //
    // Process args
    //
    while ((c = getopt(argc, argv, "c:")) != -1) {
	switch (c) {
	  case 'c':
	      if (optarg != NULL) {
		  ccr = strtoul(optarg, NULL, 16);
		  use_ccr = 1;
	      } else {
		  usage();
	      }
	      break;
	}
    }

    if (optind < argc) {
	inFile = argv[optind];
    }

    // override options according to the CCR register
    if (use_ccr) {
	mode = (ccr & CCR_DOWN_MODE_MASK) >> CCR_DOWN_MODE_SHIFT;
    }

//
// Read Input File
//

// Open it
    if (inFile) {
	if (!(inFP = (FILE *) fopen(inFile, "r"))) {
	    fprintf(stderr, "ERROR: can't open file '%s'\n", inFile);
	    exit(-1);
	}
    } else
	inFP = stdin;

//
// Read file header
//
    if (getRLEHdr(inFP, &rleInfo) < 0)
	exit(-1);

//
// Report parameters
//
    fprintf(stderr, ">>DOWNSAMPLE:\n");
    if (inFile) {
	fprintf(stderr, "\tInput file: %s\n", inFile);
	fprintf(stderr, "\tOriginal Input file: %s\n", rleInfo.name);
	fprintf(stderr, "\tFrame size: %dx%d; cell size: %dx%d\n",
		rleInfo.fWidth, rleInfo.fHeight, rleInfo.cWidth,
		rleInfo.cHeight);
    }
    fprintf(stderr, "\tDownSample parameters: mode=%d\n", mode);

    //putRLEHdr(rleInfo); - single line header
//    fprintf(stdout, "%-4d %-4d %lx\n", rleInfo.fWidth/(1<<mode), 
//	    rleInfo.fHeight/(1<<mode), ccr);, we blow the image up
    fprintf(stdout, "%-4d %-4d %lx\n", rleInfo.fWidth, 
	    rleInfo.fHeight, ccr);

//
// Generate RLE data
//
    encodeFrame(inFP);
    return 0;
}

/*
 *===================
 * Encoding Functions
 *===================
 */

//
// Encode a frame by dividing it into cells and encoding each cell.
// Generate RLE code in rleCode[] array.
void encodeFrame(FILE * inFP)
{
    int rv;
    int x, y;
    int cw, ch;
    char s[5 * MAXWIDTH], *sp, hd;
    unsigned long pixel;

    while ((rv = getBlockHdr(inFP, sizeof(s), s, "CELLDATA"))) {
	// parse cell header
	if (rv < 0) {
	    fprintf(stderr,
		    "DOWN_ENGINE ERROR: unexpected block type '%s'\n", s);
	    exit(-1);
	}
	if (sscanf(s, " %d %d", &cw, &ch) != 2) {
	    fprintf(stderr,
		    "DOWN_ENGINE ERROR: bad celldata header '%s'\n", s);
	    exit(-1);
	}
	// load pixel data into cellMap
	for (y = 0; y < ch; y++) {
	    // get a scanline
	    if (!(rv = getBlockData(inFP, sizeof(s), s))) {
		fprintf(stderr,
			"DOWN_ENGINE ERROR: unexpected EOF (saw only %d out of %d lines in cell\n",
			y, ch);
		exit(-1);
	    }
	    if (rv < 0) {
		fprintf(stderr,
			"DOWN_ENGINE ERROR: unexpected END (saw only %d out of %d lines in cell\n",
			y, ch);
		exit(-1);
	    }
	    // extract pixel values from scanline
	    for (x = 0, sp = s; x < cw; x++) {
		while (*sp == ' ')	// skip whitespace
		    sp++;
		if (!*sp) {	// unexpected EOL?
		    fprintf(stderr,
			    "DOWN_ENGINE ERROR: unexpected EOL (saw only %d out of %d pixels in line\n",
			    x, cw);
		    exit(-1);
		}
		// cheesy home-brew input routine because strtok() and sscanf()
		// were too slow
		pixel = 0;
		while (1) {
		    hd = toupper(*sp++);
		    if (hd >= '0' && hd <= '9')
			pixel = (pixel << 4) + hd - '0';
		    else if (hd >= 'A' && hd <= 'F')
			pixel = (pixel << 4) + 10 + hd - 'A';
		    else if (hd == 0 || hd == ' ') {
			sp--;
			break;
		    } else {
			fprintf(stderr,
				"DOWN_ENGINE ERROR: bad pixel value\n");
			exit(-1);
		    }

		    if (pixel >> 16) {
			fprintf(stderr,
				"DOWN_ENGINE ERROR: pixel value too large\n");
			exit(-1);
		    }
		}
		cellMap[y][x][BLUE] = pixel & 0x1F;
		pixel >>= 5;
		cellMap[y][x][GREEN] = pixel & 0x3F;
		pixel >>= 6;
		cellMap[y][x][RED] = pixel & 0x1F;
	    }
	}
	if (getBlockData(inFP, sizeof(s), s) >= 0 || strcmp(s, "CELLDATA")) {
	    fprintf(stderr,
		    "DOWN_ENGINE ERROR: bad or missing END to CELLDATA\n");
	    exit(-1);
	}
	// Encode the cell
	encodeCell(cw, ch);
    }
}

//
// Create the downsampled code for a single cell.  Pixel data is found in 
// cellMap[0..ch-1][0..cw-1]
//
void encodeCell(int lastX, int lastY)
{
    int x, y;
    unsigned long word;
    int r[8], g[8], b[8];
    int rpix, gpix, bpix;
    for(y=0; y<8; y++) {
	r[y]=0;	g[y]=0;	b[y]=0;
    }
    
    
// Loop through cell pixel data
    for (y = 0; y < lastY; y++) {
	for (x = 0; x < lastX; x=x+2) {
	    rpix = (cellMap[y][x][RED] + cellMap[y][x+1][RED]);
	    gpix = (cellMap[y][x][GREEN] + cellMap[y][x+1][GREEN]);
	    bpix = (cellMap[y][x][BLUE] + cellMap[y][x+1][BLUE]);
	    switch(mode) {
	    case DOWN_MODE_1X1:
		rpix=rpix/2;
		gpix=gpix/2;
		bpix=bpix/2;
		word=rpix<<27 | gpix<<21 | bpix <<16 |
		     rpix<<11 | gpix<<5  | bpix <<0;
		word = cpu_to_be16(word);
		fwrite(&word, sizeof(word), 1, stdout);
		break;
	    case DOWN_MODE_2X2:
		r[x/2]=r[x/2]+rpix;
		g[x/2]=g[x/2]+gpix;
		b[x/2]=b[x/2]+bpix;
		if((y & 1) == 1) {
		    if(x==2 || x==6 ||  x==10 || x==14) {
			r[x/2]=r[x/2]/4;
			g[x/2]=g[x/2]/4;
			b[x/2]=b[x/2]/4;
			r[x/2-1]=r[x/2-1]/4;
			g[x/2-1]=g[x/2-1]/4;
			b[x/2-1]=b[x/2-1]/4;
			word=r[x/2]<<27   | g[x/2]<<21  | b[x/2]<<16 |
			     r[x/2-1]<<11 | g[x/2-1]<<5 | b[x/2-1]<<0;
			word = cpu_to_be16(word);
			fwrite(&word, sizeof(word), 1, stdout);
			r[x/2]=0; g[x/2]=0; b[x/2]=0;
			r[x/2-1]=0; g[x/2-1]=0; b[x/2-1]=0;
		    }
		}
		break;
	    case DOWN_MODE_4X4:
		r[x/4]=r[x/4]+rpix;
		g[x/4]=g[x/4]+gpix;
		b[x/4]=b[x/4]+bpix;
		if((y & 3) == 3) {
		    if(x==6 || x==14) {
			r[x/4]=r[x/4]/16;
			g[x/4]=g[x/4]/16;
			b[x/4]=b[x/4]/16;
			r[x/4-1]=r[x/4-1]/16;
			g[x/4-1]=g[x/4-1]/16;
			b[x/4-1]=b[x/4-1]/16;
			word=r[x/4]<<27   | g[x/4]<<21  | b[x/4]<<16 |
			     r[x/4-1]<<11 | g[x/4-1]<<5 | b[x/4-1]<<0;
			word = cpu_to_be16(word);
			fwrite(&word, sizeof(word), 1, stdout);
			r[x/4]=0; g[x/4]=0; b[x/4]=0;
			r[x/4-1]=0; g[x/4-1]=0; b[x/4-1]=0;
		    }
		}
		break;
	    case DOWN_MODE_8X8:
		r[x/8]=r[x/8]+rpix;
		g[x/8]=g[x/8]+gpix;
		b[x/8]=b[x/8]+bpix;
		if((y & 7) == 7) {
		    if(x == 14) {
			r[x/8]=r[x/8]/64;
			g[x/8]=g[x/8]/64;
			b[x/8]=b[x/8]/64;
			r[x/8-1]=r[x/8-1]/64;
			g[x/8-1]=g[x/8-1]/64;
			b[x/8-1]=b[x/8-1]/64;
			word=r[x/8]<<27   | g[x/8]<<21   | b[x/8]<<16 |
			     r[x/8-1]<<11 | g[x/8-1]<<5  | b[x/8-1]<<0;
			word = cpu_to_be16(word);
			fwrite(&word, sizeof(word), 1, stdout);
			r[x/8]=0; g[x/8]=0; b[x/8]=0;
			r[x/8-1]=0; g[x/8-1]=0; b[x/8-1]=0;
		    }
		}
		break;
	    }
	}
    }
    return;
}

void usage()
{
    fprintf(stderr,
	    "usage: downsample [<infile>] [-c <CCR>]\n"
	    "\t<infile> must be cell data as generated by rle_cellsplit\n"
	    "\tIf <infile> is not specified, standard input will be used.\n"
	    "\t<CCR> = 32bit hex value for the CCR register, overrides all other settings\n"
	    "\tStandard output is the raw binary pixel file in tile order\n");

    exit(-1);
}
