// rle_format
//              This takes RLE data and formats into the final binary stream.  Output
//              file may be decoded with rle_decode

// Includes
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>

#include "bmp.h"
#include "rle_util.h"
#include "ccr.h"

// Globals
RLEINFO rleInfo;
FILE *inFP;

// Settable parameters
char format = 'c';
int depth = 15;

// Functions
void usage(), rleFormat();

// Main program
int main(int argc, char *argv[])
{
    char *inFile = 0;
    char use_ccr = 0;
    int c;
    unsigned long ccr;
    
    //
    // Process args
    //
    while ((c = getopt(argc, argv, "G:C:c:")) != -1) {
	switch (c) {
	  case 'G':
	      if (optarg != NULL) {
		  format = 'g';
		  depth = strtol(optarg, NULL, 10);
		  if (depth < 1 || depth > 6) {
		      fprintf(stderr,
			      "RLE_FORMAT ERROR: bad greyscale depth value '%s'\n",
			      optarg);
		      usage();
		  }
	      } else {
		  usage();
	      }
	      break;
	  case 'C':
	      if (optarg != NULL) {
		  format = 'c';
		  depth = strtol(optarg, NULL, 10);
		  if (depth != 4 && depth != 7 && depth != 15) {
		      fprintf(stderr,
			      "RLE_FORMAT ERROR: bad color depth value '%s'\n",
			      optarg);
		      usage();
		  }
	      } else {
		  usage();
	      }
	      break;
	  case 'c':
	      if (optarg != NULL) {
		  ccr = strtoul(optarg, NULL, 16);
		  use_ccr = 1;
	      } else {
		  usage();
	      }
	      break;
	}
    }

    if (optind < argc) {
	inFile = argv[optind];
    }
    
    if (use_ccr) {
	unsigned char mode = (ccr & CCR_LRLE_COLOR_MASK) >> CCR_LRLE_COLOR_SHIFT;
	if (mode <= LRLE_COLOR_8BIT_GREY) { 	    
	    format = 'g';
	    depth = mode + 1;
	} else {
	    format = 'c';
	    switch (mode) {
	      case LRLE_COLOR_15BIT_DIRECT:
		  depth = 15;
		  break;
	      case LRLE_COLOR_7BIT_DIRECT:
		  depth = 7;
		  break;
	      case LRLE_COLOR_4BIT_PALETTE:
		  depth = 4;		  
		  break;
	      default:
		  fprintf(stderr,
			  "RLE_FORMAT ERROR: unknown color mode '%d'\n", mode);
		  usage();
	    }
	}
    }
	
    if (depth <= 4)
	format = toupper(format);
//
// Open input file
//
    if (inFile) {
	if (!(inFP = (FILE *) fopen(inFile, "r"))) {
	    fprintf(stderr, "RLE_FORMAT ERROR: can't open file '%s'\n",
		    inFile);
	    exit(-1);
	}
    } else
	inFP = stdin;

//
// Read file header
//
    if (getRLEHdr(inFP, &rleInfo) < 0)
	exit(-1);

//
// Report parameters
//
    fprintf(stderr, ">>RLE_FORMAT:\n");
    if (inFile) {
	fprintf(stderr, "\tInput file: %s\n", inFile);
	fprintf(stderr, "\tFrame size: %dx%d; cell size: %dx%d\n",
		rleInfo.fWidth, rleInfo.fHeight, rleInfo.cWidth,
		rleInfo.cHeight);
    }
    fprintf(stderr, "\tFormat: %d-bit %s\n", depth,
	    (depth > 4 ? "RLE" : (depth >= 3 ? "compact RLE" : "raw")));
/*
	putRLEHdr(rleInfo);
*/
    printf("%-4d %-4d %c%02d\n", rleInfo.fWidth, rleInfo.fHeight, format,
	   depth);

    rleFormat(inFP);
    return 0;
}

void rleFormat(FILE * inFP)
{
    int r, g, b, run, isGrey;
    int rv, isCopy, isRun, isEnd, isNew, isStart;
    char s[9999];
    unsigned char cellCode[MAXCWIDTH * MAXCHEIGHT * 2 + 2];
    int cellSize, outSize, lastPixdata;

    unsigned char outByte;	// used for raw mode decoding
    int pixCount;

    outSize = 0;

    isEnd = 1;
    while (1) {
	// fetch new line of data
	if (isEnd) {		// look for start of CELLRLE block
	    if (!(rv = getBlockHdr(inFP, sizeof(s), s, "CELLRLE")))	// Done
		break;
	    else if (rv < 0) {	// Bad
		fprintf(stderr,
			"RLE_FORMAT ERROR: unexpected block type '%s'\n",
			s);
		exit(-1);
	    } else		// don't care about cw or ch
		;
	    isStart = 1;
	    isRun = isCopy = isEnd = 0;
	    cellSize = 0;
	} else {		// look for data or END
	    if (!(rv = getBlockData(inFP, sizeof(s), s))) {
		fprintf(stderr,
			"RLE_FORMAT ERROR: unexpected EOF in cell\n");
		exit(-1);
	    } else if (rv < 0) {
		isEnd = 1;
		isRun = isCopy = isStart = 0;
	    } else {
		isStart = isEnd = 0;
		if (!strncmp(s, "RUN ", 4)) {	// parse a run
		    if (!parseRunData
			(s, &isGrey, &isNew, &run, &r, &g, &b)) {
			fprintf(stderr,
				"RLE_FORMAT ERROR: bad run format '%s'\n",
				s);
			exit(-1);
		    }
		    isRun = 1;
		    isCopy = 0;
		} else if (!strncmp(s, "LC ", 3)) {	// parse a line copy
		    if (!parseInt(s + 3, &run, 0)) {
			fprintf(stderr,
				"RLE_FORMAT ERROR: bad LC format '%s'\n",
				s);
			exit(-1);
		    }
		    isCopy = 1;
		    isRun = 0;
		}
	    }
	}

	// format according to selected mode
	switch (depth) {
	    //
	    // formatter for 1/2 bit raw modes
	    //
	case 1:
	case 2:
	    if (isStart) {	// init for new cell
		cellSize = 0;
		pixCount = 0;
		outByte = 0;
	    } else if (isCopy) {	// line copy; disallowed here
		fprintf(stderr,
			"RLE_FORMAT ERROR: line copies unsupported in 1/2 bit raw modes\n");
		exit(-1);
	    } else if (isRun) {	// normal run
		for (; run >= 0; run--) {
		    outByte = (outByte << depth) + g;
		    if ((pixCount += depth) == 8) {
			cellCode[cellSize++] = outByte;
			outByte = 0;
			pixCount = 0;
		    }
		}
	    } else		// end of cell
		outSize += cellSize;
	    break;
	    //
	    // formatter for 3/4 bit compact RLE modes
	    //
	case 3:
	case 4:
	    if (isStart) {	// init for new cell
		cellSize = 0;
		lastPixdata = 0;
	    } else if (isCopy) {	// line copy
		if (run >= 31) {
		    cellCode[cellSize++] = 0xFF;
		    cellCode[cellSize++] = run;
		} else
		    cellCode[cellSize++] = 0xE0 + run;
	    } else if (isRun) {	// normal run
		cellCode[cellSize++] = g + (run << depth);
		lastPixdata = cellSize;
	    }
/*
				else {					// end of cell
					cellSize = lastPixdata;
					lastPixdata = 0;
					cellCode[cellSize++] = 0xFF;
				  }
*/
	    break;
	    //
	    // formatter for 7/15 bit RLE modes
	    //
	case 7:
	case 15:
	    if (isStart) {	// init for new cell
		cellSize = 0;
		lastPixdata = 0;
	    } else if (isCopy) {	// line copy
		if (run >= 31) {
		    cellCode[cellSize++] = 0xFF;
		    cellCode[cellSize++] = run;
		} else
		    cellCode[cellSize++] = 0xE0 + run;
	    } else if (isRun) {	// normal run
		if (isNew) {
		    if (isGrey)
			cellCode[cellSize++] = 0x80 + g;
		    else if (depth == 7)
			cellCode[cellSize++] = (r << 5) + (g << 2) + b;
		    else {
			cellCode[cellSize++] = (r << 2) + (g >> 3);
			cellCode[cellSize++] = ((g & 0x07) << 5) + b;
		    }
		    run--;
		}
		if (run >= 0)
		    cellCode[cellSize++] = 0xC0 + run;
		lastPixdata = cellSize;
	    }
/*
				else {					// end of cell
					cellSize = lastPixdata;
					cellCode[cellSize++] = 0xFF;
					cellCode[cellSize++] = 0xDF;
				  }
*/
	    break;
	default:
	    break;
	}
	if (isEnd) {		// write cell data to output
	    fwrite(cellCode, 1, cellSize, stdout);
	    outSize += cellSize;
	}
    }
    
    if (outSize % 4) {
	    int i;
	    for (i=0; i < 4-(outSize % 4); i++) {
		    fputc('\0', stdout);
	    }
    }
}

void usage()
{
    fprintf(stderr,
	    "usage: rle_format [<infile>] [-c <CCR>] [-c <depth> | -g <depth>]\n"
	    "\t<infile> must be RLE data as generated by lrle_engine\n"
	    "\tIf <infile> is not specified, standard input will be used.\n"
	    "\t<CCR> = 32bit hex value for the CCR register, overrides all other settings\n");
    exit(-1);
}
