// rle_reduce
//		This takes RLE data and reduces the color space to the selected level.

// Includes
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

#include "bmp.h"
#include "rle_util.h"
#include "ccr.h"

// Globals
RLEINFO	rleInfo;
FILE *inFP;

int roundRB = 7;
int roundG = 6;

unsigned char c5Table[8] = { 0,0,0,1,1,2,2,2 };
unsigned char c7Table[8] = { 0,1,1,2,2,3,3,4 };

int c4table[512] = {
	0,0,4,4,4,4,12,12,
	0,0,4,4,4,4,12,12,
	2,2,6,6,6,6,12,12,
	2,2,6,6,6,6,6,12,
	2,2,6,6,6,6,6,14,
	2,2,6,6,6,6,14,14,
	10,10,10,6,6,14,14,14,
	10,10,10,10,14,14,14,14,
	0,0,4,4,4,4,12,12,
	0,0,4,4,4,4,12,12,
	2,2,6,6,6,6,12,12,
	2,2,6,6,6,6,6,12,
	2,2,6,6,6,6,6,14,
	2,2,6,6,6,6,14,14,
	10,10,10,6,6,14,14,14,
	10,10,10,10,14,14,14,14,
	1,1,5,5,5,5,12,12,
	1,1,5,5,5,5,12,12,
	3,3,7,7,7,7,7,12,
	3,3,7,7,7,7,7,7,
	3,3,7,7,7,7,7,7,
	3,3,7,7,7,7,7,14,
	10,10,7,7,7,7,14,14,
	10,10,10,7,7,14,14,14,
	1,1,5,5,5,5,5,12,
	1,1,5,5,5,5,5,12,
	3,3,7,7,7,7,7,7,
	3,3,7,7,7,7,7,7,
	3,3,7,7,7,7,7,8,
	3,3,7,7,7,7,8,8,
	3,3,7,7,7,8,8,8,
	10,10,7,7,8,8,8,8,
	1,1,5,5,5,5,5,13,
	1,1,5,5,5,5,5,13,
	3,3,7,7,7,7,7,7,
	3,3,7,7,7,7,7,8,
	3,3,7,7,7,7,8,8,
	3,3,7,7,7,8,8,8,
	3,3,7,7,8,8,8,8,
	11,11,7,8,8,8,8,8,
	1,1,5,5,5,5,13,13,
	1,1,5,5,5,5,13,13,
	3,3,7,7,7,7,7,13,
	3,3,7,7,7,7,8,8,
	3,3,7,7,7,8,8,8,
	3,3,7,7,8,8,8,8,
	11,11,7,8,8,8,8,8,
	11,11,11,8,8,8,8,8,
	9,9,9,5,5,13,13,13,
	9,9,9,5,5,13,13,13,
	9,9,7,7,7,7,13,13,
	3,3,7,7,7,8,8,8,
	3,3,7,7,8,8,8,8,
	11,11,7,8,8,8,8,8,
	11,11,11,8,8,8,8,8,
	11,11,11,8,8,8,8,15,
	9,9,9,9,13,13,13,13,
	9,9,9,9,13,13,13,13,
	9,9,9,7,7,13,13,13,
	9,9,7,7,8,8,8,8,
	11,11,7,8,8,8,8,8,
	11,11,11,8,8,8,8,8,
	11,11,11,8,8,8,8,15,
	11,11,11,8,8,8,15,15,
  };

// Settable parameters
int greyscale = 0;		// whether this is a greyscale mode
int depth = 15;			// total color depth

// Functions
void	usage(), rleReduce();

// Main program
int
main(int argc, char *argv[]) {
    char *inFile = 0;
    char use_ccr = 0;
    int c;
    unsigned long ccr;
    
    //
    // Process args
    //
    while ((c = getopt(argc, argv, "G:C:c:")) != -1) {
	switch (c) {
	  case 'G':
	      if (optarg != NULL) {
		  greyscale = 1;
		  depth = strtol(optarg, NULL, 10);
		  if (depth < 1 || depth > 6) {
		      fprintf(stderr,
			      "RLE_REDUCE ERROR: bad grey depth value '%s'\n",
			      optarg);
		      usage();
		  }
	      } else {
		  usage();
	      }
	      break;
	  case 'C':
	      if (optarg != NULL) {
		  greyscale = 0;
		  depth = strtol(optarg, NULL, 10);
		  if (depth != 4 && depth != 5 && depth != 7 && depth != 15) {
		      fprintf(stderr,
			      "RLE_REDUCE ERROR: bad color depth value '%s'\n",
			      optarg);
		      usage();
		  }
	      } else {
		  usage();
	      }
	      break;
	  case 'c':
	      if (optarg != NULL) {
		  ccr = strtoul(optarg, NULL, 16);
		  use_ccr = 1;
	      } else {
		  usage();
	      }
	      break;
	}
    }

    if (optind < argc) {
	inFile = argv[optind];
    }

    if (use_ccr) {
	unsigned char mode = (ccr & CCR_LRLE_COLOR_MASK) >> CCR_LRLE_COLOR_SHIFT;
	if (mode <= LRLE_COLOR_8BIT_GREY) { 	    
	    greyscale = 1;
	    depth = mode + 1;
	} else {
	    greyscale = 0;
	    switch (mode) {
	      case LRLE_COLOR_15BIT_DIRECT:
		  depth = 15;
		  break;
	      case LRLE_COLOR_7BIT_DIRECT:
		  depth = 7;
		  break;
	      case LRLE_COLOR_4BIT_PALETTE:
		  depth = 4;		  
		  break;
	      default:
		  fprintf(stderr,
			  "RLE_REDUCE ERROR: unknown color mode '%d'\n", mode);
		  usage();
	    }
	}
    }
    
//
// Open input file
//
	if (inFile) {
		if (!(inFP = (FILE *) fopen(inFile, "r"))) {
			fprintf(stderr, "RLE_REDUCE ERROR: can't open file '%s'\n", inFile);
			exit(-1);
		  }
	  }
	else
		inFP = stdin;

//
// Read file header
//
	if (getRLEHdr(inFP, &rleInfo) < 0)
		exit(-1);

//
// Report parameters
//
	fprintf(stderr, ">>RLE_REDUCE:\n");
	if (inFile) {
		fprintf(stderr, "\tInput file: %s\n", inFile);
		fprintf(stderr, "\tFrame size: %dx%d; cell size: %dx%d\n",
			rleInfo.fWidth,rleInfo.fHeight, rleInfo.cWidth,rleInfo.cHeight);
	  }
	fprintf(stderr, "\tMode: %d-bit %s\n", depth,
		(greyscale ? "greyscale" : "color"));

	putRLEHdr(rleInfo);

	rleReduce(inFP);

	return(0);
  }

// Do it
void
rleReduce(FILE *inFP) {
	char s[160];
	int	r, g, b, run, isGrey, isNew;
	int	c;
//
// Process input lines; convert color when appropriate
//
	while (fgets(s, sizeof(s), inFP)) {
		if (!strncmp(s, "RUN ", 4)) {
			if (!parseRunData(s, &isGrey, &isNew, &run, &r, &g, &b)) {
				fprintf(stderr, "RLE_REDUCE ERROR: bad run format '%s'\n",s);
				exit(-1);
			  }
		// Greyscale modes: most of the work already done
			if (greyscale)
				putRun(1, isNew, run, 0, (g >> (6 - depth)), 0);
		// Color modes: all different
			else switch(depth) {
				case 4:
					if (isGrey) {
						if (g < 4)			g = 0;
						else if (g < 10)	g = 2;
						else if (g < 14)	g = 3;
						else				g = 1;
					  }
					else
						g = c4table[(r/4)*64 + (g/8)*8 + (b/4)];
					putRun(0, isNew, run, r, g, b);
					break;
				case 5:
					if (isGrey)
						g = c7Table[g>>3] + 27;
					else 
						g = 9*c5Table[r/4] + 3*c5Table[g/8] + c5Table[b/4];
					putRun(0, isNew, run, r, g, b);
					break;
				case 7:
					if (isGrey) 	// chop off two bits of grey
						g >>= 2;
					else {			// reduce R, G, and B
/* OLD 7-BIT MODE, REPLACED BY THE NEWER ONE
						r = (r >> 3) + (!(r>>4) && ((r & roundRB) == roundRB));
						g = (g >> 3) + (!(g>>5) && ((g & roundG ) == roundG ));
						b = (b >> 3) + (!(b>>4) && ((b & roundRB) == roundRB));
*/
					// New, much superior 7-bit mode
						c = 25*c7Table[r/4] + 5*c7Table[g/8] + c7Table[b/4];
						r = c >> 5;
						g = (c >> 2) & 7;
						b = c & 3;
					  }
					putRun(isGrey, isNew, run, r, g, b);
					break;
				case 15:
					if (isGrey)		// don't touch greys
						putRun(1, isNew, run, r, g, b);
					else { 			// for color, just chop off a bit of green
						g = (g>>1) ;//+ ((g & 1) && g < 0x3E);
						putRun(0, isNew, run, r, g, b);
					  }
					break;
			  }
		  }
		else 
			fputs(s, stdout);
	  }
  }

void
usage() {
    fprintf(stderr,
	    "usage: rle_reduce [<infile>] [-c <CCR>] [-G <depth> | -C <depth>]\n"
	    "\t<infile> must be RLE data as generated by lrle_engine\n"
	    "\tIf <infile> is not specified, standard input will be used.\n"
	    "\tFor greyscale modes (-G), <depth> is in the range [1,6]\n"
	    "\tFor color modes (-C), <depth> may be 4, 7, or 15.\n"
	    "\t<CCR> = 32bit hex value for the CCR register, overrides all other settings\n");
    exit(-1);
}
