###############################################################################
#
# eRIC_Firmware - firmware manipulation routines
#

package eRIC_Firmware;

use strict;

use vars qw(
            @ISA
            @EXPORT
	    $fw
	    @part_types
           );

#
###############################################################################
# imports
#

use FindBin;
use lib "$FindBin::Bin";
use lib "$FindBin::Bin/cpan";

use MIME::Base64;
use Digest::MD5;
use File::Path;
use File::Find;
use File::Copy;
use Cwd;
use pp_hw_data;

#
###############################################################################
# global variables
#

$fw            = {};
my $fw_magic;
my $fw_part_magic;
my $FW_MAGIC_OFFSET = 16;
my $FW_LARA_MAGIC = "LARA firmware ";

# magic before firmware version string in libpp_firmware.so
my $pp_fw_ver_hdr_magic = "pp_fw_ver_hdr251105v10_fzsaiudl\0";

my $debug         = 0;

## definition of supported firmware types and subpart types

# def of symbolic names of flash mappings
my $flash_map_names = {
    0 => "map_ramfs",  # classic FW with initrd
    1 => "map_jffs"    # JFFS FW
};

# defs of valid "Flash" subparts
#     partition nums of subparts depend on type of flash mapping
#     -> define different sets and select the appropriate one
my $flash_subparts = {
    "map_ramfs" =>	{ 0 => { "name"	=> "Kernel",
				 "max_size"	=> (896 * 1024),
				 "file"	=> "zImage.bin",
				 "dest"	=> "fw"
			       },
			  1 => { "name"	=> "Initrd",
				 "max_size"	=> (3072 * 1024),
				 "file"	=> "initrd.bin",
				 "dest"	=> "fw"
			       },
			  3 => { "name"	=> "U-Boot",
				 "max_size"	=> (128 * 1024),
				 "file"	=> "u-boot.bin",
				 "dest"	=> "fw"
			       }
			},
    "map_jffs" =>	{ 1 => { "name"	=> "U-Boot",
				 "max_size"	=> (256 * 1024),
				 "file"	=> "u-boot.bin",
				 "dest"	=> "fw"
			       }
			}
};

# global part type definition table
my $part_types = { 0 => { "name" => "Flash",
			  # hack: gets initialized with one of $flash_subparts,
			  #     depending on flash mapping of FW file;
                          #     see _set_part_types_flash_subparts()
			  "part_subtype"=> undef 
			},
		   1 => { "name" => "KME",
			  "part_subtype"=> { 4 => { "name"	=> "KME (AVR 8515)",
					     "max_size"	=> (8 * 1024),
					     "file"	=> "kme_8515.bin",
					     "dest"	=> "fw"
					   }
				    }
			},
		   2 => { "name" => "KME2",
			  "part_subtype"=> { 4 => { "name"	=> "KME (ATMEGA 16L)",
					     "max_size"	=> (16 * 1024),
					     "file"	=> "kme_m16l.bin",
					     "dest"	=> "fw"
					   },
				      5 => { "name"	=> "KME (ATMEGA 16L)",
					     "max_size"	=> (16 * 1024),
					     "file"	=> "kme_m16l.bin",
					     "dest"	=> "fw"
					   }
				    }
			},
		   # these subpart_types must be in sync with libpp_minicom/src/proto.h
		   3 => { "name" => "SmartIPC Firmware",
			  "part_subtype"=> { 0 => { "name"	=> "MASTER" },
				      1 => { "name"	=> "OSD" },
				      2 => { "name"	=> "RICC_USB" },
				      3 => { "name"	=> "RICC_PS2" },
				      4 => { "name"	=> "RICC_SUN" }
				    }
			},
		   # these subpart_types must be in sync with 'supported_flash_dirs'
		   4 => { "name" => "Flashdisk Directory",
			  "part_subtype"=> { 0 => { "name"		=> "OEM directory",
					     "max_plain_size"	=> (3 * 1024 * 1024),
					     "file"		=> "oem.tgz",
					     "dest"		=> "fd"
					   },
				      1 => { "name"		=> "Root-FS",
					     "max_plain_size"	=> (12 * 1024 * 1024),
					     "file"		=> "rootfs.tgz",
					     "dest"		=> "fd"
					   },
				      2 => { "name"		=> "Internationalization directory",
					     "max_plain_size"	=> (2 * 1024 * 1024),
					     "file"		=> "i18n.tgz",
					     "dest"		=> "fd"
					   }
				    }
			},

		   5 => { "name" => "ADC Atmel",
			  "part_subtype"=> { 4 => { "name"		=> "ADC (ATMEGA 16L)",
					     "max_plain_size"	=> (16 * 1024),
					     "file"		=> "adc_m16l.bin",
					     "dest"		=> "fw"
					   }
				    }
			},

		   # this is the MasterConsole hex file
		   6 => { "name" => "MasterConsole",
			  "part_subtype"=> { 0 => { "name"		=> "MasterConsole Hex",
			      		     "max_plain_size"	=> (256 * 1024),
					     "file"		=> "mcc.bin",
					     "dest"		=> "fw"
					   }
				    }
			},

		   # this is the MAX II file for Supermicro KIMs
		   7 => { "name" => "MAX II JAM",
			  "part_subtype"=> { 0 => { "name"		=> "MAX II Jam",
			      		     "max_plain_size"	=> (64 * 1024),
					     "file"		=> "dvo.jbc",
					     "dest"		=> "fw"
					   }
				    }
			},
                   8 => { "name" => "RPC Branch Controller",
			  "part_subtype"=> { 0 => { "name"      => "cypress cy8c27 hex",
                                             "max_plain_size"   => (8 * 1024),
					     "file"             => "rbc_cy8c27.bin",
                                             "dest"             => "fw"
                                           },
			              1 => { "name"      => "cypress cy8c26 hex",
                                             "max_plain_size"   => (8 * 1024),
					     "file"             => "rbc_cy8c26.bin",
                                             "dest"             => "fw"
                                           }
                                    }
                        } 
		 };

#### raw FW parts
#
# - defs here correspond with defs for "insmod cmdlinepart mtdparts=..."
# - mind the order of parts below - will be the same in flash image
#

my $raw_fw_info =  { "lara8_ramfs" => [
			{ "physpart_type" => "kernel",      "size" => 0x000e0000 },
			{ "physpart_type" => "initrd",      "size" => 0x00300000 },
			{ "physpart_type" => "flashdisk",   "size" => 0x00380000 },
			{ "physpart_type" => "config",      "size" => 0x00040000 },
			{ "physpart_type" => "uboot",       "size" => 0x00020000 },
			{ "physpart_type" => "config",      "size" => 0x00040000 } ],

                     "kira8_ramfs" => [
			{ "physpart_type" => "uboot",       "size" => 0x00020000 },
			{ "physpart_type" => "kernel",      "size" => 0x000e0000 },
			{ "physpart_type" => "initrd",      "size" => 0x00300000 },
			{ "physpart_type" => "flashdisk",   "size" => 0x00380000 },
			{ "physpart_type" => "config",      "size" => 0x00040000 },
			{ "physpart_type" => "config",      "size" => 0x00040000 } ] };

###############################################################################
# the API implementation
#

sub make_firmware ( $$$$$$$$\@\@$$ ) {
    my $parts = shift;
    my $ver = shift;
    my $build_nr = shift;
    my $tag = shift;
    my $device = shift;
    my $oem = shift;
    my $product = shift;
    my $min_req_ver = shift;
    my $include_ids = shift;
    my $exclude_ids = shift;
    my $flash_map_name = shift;
    my $glbl_hdr_ver = shift;
    my $part_hdr_ver;

    if ($device eq "lara" || $device eq "kira") {
      $fw_magic      = $FW_LARA_MAGIC;
      $fw_part_magic = "LARA partition ";
    } else {
      print "Cannot make firmware for device '$device'\n";
      return -1;
    }

    # hdr_ver of FW
    if ($glbl_hdr_ver) {
	# force a global header version (mainly for transition FWs)
	$fw->{"hdr"}->{"hdr_ver"} = $glbl_hdr_ver;
    } else {
	if ($flash_map_name eq "map_ramfs") {
		# Avoid need for transition FW when just going from version 2 to 3.
		# Going from RAMFS to JFFS will require a transition FW anyway and
		# we can do the ver-2-3 transition for free.
		$fw->{"hdr"}->{"hdr_ver"}     = 2;
	} else {
		$fw->{"hdr"}->{"hdr_ver"}     = 3;
	}
    }
    # hdr_ver of parts
    $part_hdr_ver = 3;

    $fw->{"hdr"}->{"magic"}       = $fw_magic;
    $fw->{"hdr"}->{"ver"}         = $ver;
    $fw->{"hdr"}->{"min_req_ver"} = $min_req_ver;
    $fw->{"hdr"}->{"build_nr"}    = $build_nr;
    $fw->{"hdr"}->{"tag"}         = $tag;
    $fw->{"hdr"}->{"oem"}         = $oem;
    $fw->{"hdr"}->{"product"}     = $product;
    $fw->{"hdr"}->{"include_ids"} = $include_ids;
    $fw->{"hdr"}->{"exclude_ids"} = $exclude_ids;
    $fw->{"hdr"}->{"flash_map_sel"} = get_flash_map_sel($flash_map_name);

    _set_part_types_flash_subparts($fw->{"hdr"}->{"flash_map_sel"});

    return _make_firmware_common($parts, $part_hdr_ver);
}

sub update_parts ( $$ ) {
    my $parts = shift;
    my $part_hdr_ver = shift;
    $fw_part_magic = "LARA partition ";
    return _make_firmware_common($parts, $part_hdr_ver);
}

sub tag_firmware ( $ ) {
    my $tag = shift;
    $fw->{"hdr"}->{"tag"} = $tag;
}

sub set_version ( $ ) {
    my $ver = shift;

    $fw->{"hdr"}->{"ver"} = $ver;
}

sub min_req_ver_firmware( $ ) {
    my $min_req_ver = shift;
    $fw->{"hdr"}->{"min_req_ver"} = $min_req_ver;
}

sub load_firmware ( $ ) {
    my $file = shift;
    my $data;

    open FILE, $file or
      ( print "Error: Can't open '$file'. ($!)\n" and return 0 );

    while (<FILE>) {
	    #$data .= decode_base64($_);
	    $data .= $_;
    }

    close FILE;

    _parse_firmware($data);

    return 1;
}

sub save_firmware ( $ ) {
    my $file = shift;
    my $data;
    my $hdr;
    my $pad_len;
    my $count;

    if ($fw->{"type"} eq "raw") {
      foreach my $part (@{$fw->{"parts"}}) {
	$data .= $part->{"data"};
      }

      open FILE,">$file" or ( print "Error: Can't open '$file' for writing. ($!)\n" and return 0 );
      print FILE $data;
      close FILE;

      return 1;
    }

    $hdr  = $fw->{"hdr"}->{"magic"};
    $hdr .= pack "C", $fw->{"hdr"}->{"hdr_ver"};
    $hdr .= $fw->{"hdr"}->{"ver"};
    $hdr .= $fw->{"hdr"}->{"min_req_ver"};

    if ($fw->{"hdr"}->{"hdr_ver"} >= 2) {
	$hdr .= pack "N", $fw->{"hdr"}->{"build_nr"};
	$hdr .= $fw->{"hdr"}->{"tag"};
	$hdr .= chr(0) x (64 - length($fw->{"hdr"}->{"tag"})); # pad tag

	$hdr .= $fw->{"hdr"}->{"oem"};
	$hdr .= chr(0) x (16 - length($fw->{"hdr"}->{"oem"})); # pad oem

	$hdr .= $fw->{"hdr"}->{"product"};
        $hdr .= chr(0) x (16 - length($fw->{"hdr"}->{"product"})); # pad product

	# add in/excluded hardware ids as array of bytes, pad the rest
	$count = $#{@{$fw->{"hdr"}->{"include_ids"}}} + 1;
	for (my $i = 0; $i < $count; $i++) {
	  $hdr .= pack "C", $fw->{"hdr"}->{"include_ids"}->[$i];
	}
	$hdr .= chr(0) x (16 - $count);

	$count = $#{@{$fw->{"hdr"}->{"exclude_ids"}}} + 1;
	for (my $i = 0; $i < $count; $i++) {
	  $hdr .= pack "C", $fw->{"hdr"}->{"exclude_ids"}->[$i];
	}
	$hdr .= chr(0) x (16 - $count);
     }
    if ($fw->{"hdr"}->{"hdr_ver"} >= 3) {
	# "n" = uint16_t, big-endian
	$hdr .= pack "n", $fw->{"hdr"}->{"flash_map_sel"};
    }

    # determine length of padding - return 0 if negative padding
    $pad_len = $fw->{"hdr_size"} - length($hdr) - 16; # 16 -> length of the md5
    ($pad_len >= 0) or
      ( print "Error: firmware header to big.\n" and return 0 );

    # pad firmware header with \0
    $data = $hdr . chr(0) x $pad_len;

    foreach my $part (@{$fw->{"parts"}}) {
	my $part_num = $part->{"part_subtype"};
	# create firmware partition header
	$hdr  = $part->{"hdr"}->{"magic"};
	$hdr .= pack "C", $part->{"hdr"}->{"hdr_ver"};	
	$hdr .= pack "N", $part_num;
	$hdr .= pack "N", 16 + length($data) + $fw->{"part_hdr_size"} +
	  length($part->{"data"});
	if ($part->{"hdr"}->{"hdr_ver"} >=3) {
	  $hdr .= pack "N", $part->{"hdr"}->{"part_type"};
	}

	# determine length of padding - return 0 if negative padding
	$pad_len = $fw->{"part_hdr_size"} - length($hdr);
	($pad_len >= 0) or
	  ( print "Error: firmware partition header to big.\n" and return 0 );

	# pad firmware header with \0
	$data .= $hdr . chr(0) x $pad_len;

	# add file data to firmware
	$data .= $part->{"data"};
    }

    my $md5 = Digest::MD5->new;
    $md5->add($data);

    open FILE,">$file" or 
      ( print "Error: Can't open '$file' for writing. ($!)\n" and return 0 );
#    print FILE encode_base64(pack("H32", $md5->hexdigest) . "$data");
    print FILE pack("H32", $md5->hexdigest)."$data";
    close FILE;

    return 1;
}

sub save_partitions ( $ ) {
    my $prefix = shift;

    foreach my $part (@{$fw->{"parts"}}) {
	my $part_num = $part->{"part_subtype"};
	my $type = $part->{"hdr"}->{"part_type"};
	open FILE, ">${prefix}${part_num}_$type"
	    or (print "Error: Can't open '${prefix}${part_num}_$type' " .
		    "for writing. ($!)\n" and return 0);
	print FILE $part->{"data"};
	close FILE;
    }

    return 1;
}

sub _cp_file ($$) {
    my $from = shift;
    my $to = shift;
    ($debug) and print "Copying file \"$from\" to \"$to\".\n";
    return !system("cp \"$from\" \"$to\"");
}

sub _append_file ($$) {
    my $from = shift;
    my $to = shift;
    ($debug) and print "Appending file \"$from\" to \"$to\".\n";
    return !system("cat \"$from\" >> \"$to\"");
}

sub get_hal_wrapper() {
    my $hal_wrapper = "/usr/local/bin/make_initrd_image";
    return (-x "$hal_wrapper") ? "$hal_wrapper" : "";
}

sub mount_initrd($$) {
    my $filename = shift;
    my $flash_map_name = shift;

    my $cmd = "sudo @{[get_hal_wrapper()]} ./root_fw.sh -m '$filename' -F '$flash_map_name'";
    system($cmd) == 0 or die "Error: system('$cmd') failed: $?\n";
}

sub umount_initrd_save($$) {
    my $filename = shift;
    my $flash_map_name = shift;

    my $cmd = "sudo @{[get_hal_wrapper()]} ./root_fw.sh -u '$filename'  -F '$flash_map_name'";
    system($cmd) == 0 or die "Error: system('$cmd') failed: $?\n";
}

sub umount_initrd_nosave($) {
    my $flash_map_name = shift;

    my $cmd = "sudo @{[get_hal_wrapper()]} ./root_fw.sh -U -F '$flash_map_name'";
    system($cmd) == 0 or die "Error: system('$cmd') failed: $?\n";
}

# populates 'cf...' dir with config settings from initrd and (optional) oem part
sub _extract_config ($$) {
    my $dirs = shift;
    my $flash_map_name = shift;

    my $working_dir = $dirs->{"work"};
    my $firmware_dir = $dirs->{"fw"};
    my $flashdisk_dir = $dirs->{"fd"};
    my $config_dir = $dirs->{"cf"};
    my $oem_tgz = $flashdisk_dir . "/oem.tgz";
    my $tmp_oem_dir = $dirs->{"oem"};
    my $initrd_mnt_dir = $dirs->{"initrd"};
    my $cmd;
    my ($from, $to);
    my $new_config_fs;
    my $pp_fw_ver_patching;
    my $initrd_mounted = 0;

    # try block
    eval {
	if ($flash_map_name eq "map_jffs") {
	    # assumption: all JFFS FWs use new config FS, so we need no test
	    $new_config_fs = 1;
	    # assumption: all JFFS FWs use version patching
	    $pp_fw_ver_patching = 1;
	} elsif ($flash_map_name eq "map_ramfs") {
	    # ramfs => we have an initrd -> test whether old or new config FS

	    if (-e "$oem_tgz") {
		my $cwd = cwd;
		mkdir($tmp_oem_dir)
		    or die "ERROR: Can't create \"$tmp_oem_dir\" directory.";
		system("tar -xzf '$oem_tgz' -C '$tmp_oem_dir' >/dev/null 2>&1");
		    ($? >> 8 == 0) or die "ERROR: Could not untar \"$oem_tgz\".";
	    }
	    mount_initrd("$firmware_dir/initrd.bin", $flash_map_name);
	    $initrd_mounted = 1;

	    # determine whether old or new config FS
	    $new_config_fs = (-e "$initrd_mnt_dir/etc/e-RIC/default/!global") ? 0 : 1;

	    # determine whether version patching is used in the target FW
	    $pp_fw_ver_patching = (read_pp_fw_ver("target_root/lib/libpp_firmware.so.1.0") eq "") ? 0 : 1;
	}

	(-d "$config_dir")
	    or mkdir($config_dir)
		or die "ERROR: Could not create config dir '$config_dir'!";

	if ($new_config_fs) {
	    ## new config FS

	    # just create empty !config -> defaults from base.cdl (product.cdl)
	    #     will be used
	    $cmd = "touch \"$config_dir/!config\"";
	    system($cmd) == 0 or die "ERROR: $cmd failed!";
	} else {
	    ## old config FS

	    # /config/!global
	    $to = "$config_dir/!global";
	    $from = "$initrd_mnt_dir/etc/e-RIC/default/!global";
	    _cp_file($from, $to)
		or die "ERROR: Copying \"$from\" to \"$to\" failed!";

	    $from = "$initrd_mnt_dir/etc/e-RIC/default/!global_oem";
	    if (-e $from) {
		_append_file($from, $to)
		    or die "ERROR: Appending \"$from\" to \"$to\" failed!";
	    }

	    $from = "$tmp_oem_dir/config/!global_oem";
	    if (-e $from) {
		_append_file($from, $to)
		    or die "ERROR: Appending \"$from\" to \"$to\" failed!";
	    }

	    # /config/0
	    $to = "$config_dir/0";
	    $from = "$tmp_oem_dir/config/0_oem";
	    if (-e $from) {
		_cp_file($from, $to)
		    or die "ERROR: Copying \"$from\" to \"$to\" failed!";
	    } else {
		$from = "$initrd_mnt_dir/etc/e-RIC/default/0";
		if  (-e $from) {
		    _cp_file($from, $to)
			or die "ERROR: Copying \"$from\" to \"$to\" failed!";
		}
	    }

	    # /config/1
	    $to = "$config_dir/1";
	    $from = "$tmp_oem_dir/config/1_oem";
	    if (-e $from) {
		_cp_file($from, $to)
		    or die "ERROR: Copying \"$from\" to \"$to\" failed!";
	    } else {
		$from = "$initrd_mnt_dir/etc/e-RIC/default/1";
		if  (-e $from) {
		    _cp_file($from, $to)
			or die "ERROR: Copying \"$from\" to \"$to\" failed!";
		}
	    }

	    # /config/user
	    $to = "$config_dir/user";
	    $from = "$tmp_oem_dir/config/user_oem";
	    if (-e $from) {
		_cp_file($from, $to)
		    or die "ERROR: Copying \"$from\" to \"$to\" failed!";
	    } else {
		if  (-e $from) {
		    $from = "$initrd_mnt_dir/etc/e-RIC/default/user";
		    _cp_file($from, $to)
			or die "ERROR: Copying \"$from\" to \"$to\" failed!";
		}
	    }

	    # /config/administrator
	    $to = "$config_dir/administrator";
	    $from = "$tmp_oem_dir/config/administrator_oem";
	    if (-e $from) {
		_cp_file($from, $to)
		    or die "ERROR: Copying \"$from\" to \"$to\" failed!";
	    } else {
		if  (-e $from) {
		    $from = "$initrd_mnt_dir/etc/e-RIC/default/administrator";
		    _cp_file($from, $to)
			or die "ERROR: Copying \"$from\" to \"$to\" failed!";
		}
	    }
	}
    };
    my $exception = $@;
    # final
    {
	# clean up
	if ($initrd_mounted) {
	    $cmd = "rm -rf \"$tmp_oem_dir\" &>/dev/null";
	    system($cmd) == 0 or print "WARNING: \"$cmd\" failed!";

	    umount_initrd_nosave($flash_map_name);
	}
    }
    # catch
    if ($exception) {
	die "$exception";
    }
    return { "is_new_config_fs" => $new_config_fs, "pp_fw_ver_patching" => $pp_fw_ver_patching };
}

# Splits a FW .bin file suitable for Kprod.
# Based on this, it can also create a raw FW image (complete flash image).
sub fw_bin_to_production ( $$$ ) {
    my $light_oem   = shift;
    my $do_create_raw  = shift;  # whether to create a raw FW image
    my $raw_file_name  = shift;  # name of raw FW file to create
    my $oemsuffix   = $light_oem ? "_$light_oem" : "";
    my $board       = "";
    my $subboard    = "";
    my $new_config  = 0;
    my $cwd;
    my $flash_map_name;
    my $targ_file;
    my $dirs =  {
		    "work"	=> "work",
		    "initrd"	=> "target_root",
		    "oem"	=> "work/oem",
		    "fw"        => undef,
		    "fd"        => undef,
		    "cf"        => undef,
		};
    my $working_dir = $dirs->{"work"};
    my @boards = ("lara", "kira");

    ($fw->{"hdr"}->{"hdr_ver"} >= 2) or die "ERROR: header version not >= 2, cannot proceed!";
    ($fw->{"hdr"}->{"build_nr"} ne "") or die "ERROR: no 'build number' field available!";
    ($fw->{"hdr"}->{"tag"} ne "") or die "ERROR: no 'tag' field available!";
    ($fw->{"hdr"}->{"product"} ne "") or die "ERROR: no 'product' field available!";
    ($fw->{"hdr"}->{"oem"} ne "") or die "ERROR: no 'oem' field available!";
    ($fw->{"hdr"}->{"flash_map_sel"} ne "") or die "ERROR: no 'flash_map_sel' field available!";

    $flash_map_name = get_flash_map_name($fw->{"hdr"}->{"flash_map_sel"});

    # iterate over supported hardware IDs
    my $hw_id_map_erla = $hw_id_map->{"erla"};
    my @include_ids = @{$fw->{"hdr"}->{"include_ids"}};
    foreach my $brd (@boards) {
	foreach my $sub (keys %{$hw_id_map_erla->{$brd}}) {
	    foreach my $prod (keys %{$hw_id_map_erla->{$brd}->{$sub}}) {
		my $hw_id = $hw_id_map_erla->{$brd}->{$sub}->{$prod};
		# iterate over this FW's allowed IDs
		foreach my $incl_id (@include_ids) {
		    if ($hw_id eq sprintf("%02X", $incl_id)) {
			$board = $brd;
			$subboard = $sub;
			goto subboard_found;
		    }
		}
	    }
	}
    }
    die "ERROR: no supported subboard found for this FW (included HW IDs: @include_ids)!";
subboard_found:

    if (-d $working_dir) {
	my $cmd;

	# remove old $working_dir
	$cmd = "rm -rf \"$working_dir\" &>/dev/null";
	system($cmd) == 0 or die "ERROR: \"$cmd failed!";
    }
    mkdir($working_dir) or
	die "ERROR: Can't create \"$working_dir\" directory!";

    ## initialize dir names
    #
    my $append = 
	$fw->{hdr}->{ver} . "_" .
	$fw->{hdr}->{build_nr} . "_" .
	$fw->{hdr}->{product} . "_" .
	$board . "_" .
	$subboard . "_" .
	$fw->{hdr}->{oem};
    # firmware dir
    $dirs->{"fw"} = $working_dir . "/fw" . $append;
    # flashdisk dir
    $dirs->{"fd"} = $working_dir . "/fd" . $append . $oemsuffix;
    # config dir
    $dirs->{"cf"} = $working_dir . "/cf" . $append . $oemsuffix;

    ## dissect parts into files
    #
    print "Splitting FW to \"$working_dir/\"...\n";
    foreach my $part (@{$fw->{"parts"}}) {
	my $part_num = $part->{"part_subtype"};
	my $type = $part->{"hdr"}->{"part_type"};

	# determine file name
	my $dest_sel = get_subpart_types($type)->{$part_num}->{"dest"};
	($dest_sel ne "")
	    or die "ERROR: no 'dest' field for type=$type, part_num=$part_num!";
	my $dest_dir = $dirs->{$dest_sel};
	if (! -d $dest_dir) {
	    mkdir($dest_dir)
		or die "ERROR: Could not create destination dir '$dest_dir'!";
	}
	my $dest_file = get_subpart_types($type)->{$part_num}->{"file"};
	($dest_file ne "")
	    or die "ERROR: no 'file' field for type=$type, part_num=$part_num!";
	# write
	open FILE, ">${dest_dir}/${dest_file}"
	    or die "ERROR: Can't open ${dest_dir}/${dest_file} for writing. ($!)\n";
	print FILE $part->{"data"};
	close FILE;
    }

    ## post-processing
    #
    my $cfg_res = _extract_config($dirs, $flash_map_name);

    ## pass info to production system
    #
    # version/build/tag info
    if ($cfg_res->{"pp_fw_ver_patching"} eq "0") {
	$targ_file = $dirs->{"fw"} . "/fw.version";
	open FILE, ">$targ_file"
	    or die "Error: Can't open $targ_file for writing!";
	print FILE "firmware_version=$fw->{hdr}->{ver}\n";
	print FILE "firmware_build_nr=$fw->{hdr}->{build_nr}\n";
	print FILE "firmware_tag=$fw->{hdr}->{tag}\n";
	close FILE;
    } else {
	# with version patching, writing version info to config fs is obsolete
    }

    ## extra info (type of flash mapping, type of config FS, etc.)
    $targ_file = $dirs->{"fw"} . "/fw.info";
    open FILE, ">$targ_file"
	or die "Error: Can't open $targ_file for writing!";
    # add flash map selector
    if ($fw->{"hdr"}->{"hdr_ver"} >= 3) {
	print FILE "flash_map_name=$flash_map_name\n";
    }
    # add hardware IDs supported by this FW
    my $hw_ids = join(',', grep(($_ != 0), @{$fw->{"hdr"}->{"include_ids"}}));
    print FILE "hardware_ids=$hw_ids\n";
    # add config FS type
    if ($cfg_res->{"is_new_config_fs"} eq "1") {
	print FILE "config_fs_type=cfg_fs_new\n";
    } elsif ($cfg_res->{"is_new_config_fs"} eq "0") {
	print FILE "config_fs_type=cfg_fs_old\n";
    } else {
	die "ERROR: invalid value: is_new_config_fs='$cfg_res->{'is_new_config_fs'}'!";
    }
    # add platform (ppc | kira)
    if ($board eq "lara") {
	print FILE "platform=ppc\n";
    } elsif ($board eq "kira") {
	print FILE "platform=kira\n";
    } else {
	die "ERROR: board '$board' not supported, yet!";
    }
    # add tag info (for CFwFinder.java)
    if ($cfg_res->{"pp_fw_ver_patching"} eq "1") {
	# since fw.version is discontinued, add this here
	print FILE "firmware_tag=$fw->{hdr}->{tag}\n";
    }
    close FILE;

    if ($do_create_raw) {
	_fw_kprod_to_raw($dirs, $raw_file_name);
    }

    return 1;
}

sub delete_partitions ( $ ) {
    my $prefix = shift;

    foreach my $part (@{$fw->{"parts"}}) {
	my $part_num = $part->{"part_subtype"};
	my $type = $part->{"hdr"}->{"part_type"};
	unlink "${prefix}${part_num}_$type";
    }

    return 1;
}

sub get_and_save_parts ( $ ) {
    my $prefix = shift;
    my $parts = ();

    save_partitions($prefix) or exit 1;

    foreach my $part (@{$fw->{"parts"}}) {
	my $p;
	my $type = 0;
	if ($part->{"hdr"}->{"hdr_ver"} >= 3) {
	    $type = $part->{"hdr"}->{"part_type"};
	}
	my $part_num = $part->{"part_subtype"};
	$p->{"file"} = "${prefix}${part_num}_$type";
	$p->{"part_type"} = $type;
	$p->{"part_subtype"} = $part_num;
	push @{$parts}, $p;
    }
    return $parts;
}

sub get_subpart_types($) {
  my $part_type = shift;
  return $part_types->{$part_type}->{"part_subtype"};
}

# accepts index or name of a flash mapping and returns the index
#     (or undef, if invalid)
sub get_flash_map_sel($) {
    my $flash_map_spec = shift;

    if (!defined($flash_map_spec)) {
	die "ERROR: \$flash_map_spec is undef!";
    } elsif ($flash_map_spec =~ /\d+/) {
	# number -> check range
	if (defined($flash_map_names->{$flash_map_spec})) {
	    return $flash_map_spec;
	} else {
	    die "ERROR: \$flash_map_spec=$flash_map_spec is out of range!";
	}
    } else {
	# name -> find index
	my ($idx, $name);
	foreach $idx (keys(%{$flash_map_names})) {
	    if ($flash_map_names->{$idx} eq $flash_map_spec) {
		return $idx;
	    }
	}
	die "ERROR: \$flash_map_spec='$flash_map_spec' is unknown!";
    }
}

sub get_flash_map_name($) {
    my $flash_map_sel = shift;
	if (defined($flash_map_names->{$flash_map_sel})) {
	    return $flash_map_names->{$flash_map_sel};
	} else {
	    die "ERROR: unable to get_flash_map_name() for $flash_map_sel!";
	}
}

# read FW version string from $filename (libpp_firmware.so)
sub read_pp_fw_ver ( $ ) {
  my $filename = shift;
  my $filedata;
  my $version_str;
  
  # read the file
  open FILE, $filename or do {
	  print "Warning: can't open '$filename'. ($!)\n";
	  return "";
  };
  $filedata="";
  while (<FILE>) {
    $filedata .= $_;
  }
  close FILE;
  
  # return only version string, no header magic
  if (($version_str) = ($filedata =~ m/$pp_fw_ver_hdr_magic(.{85})/gs)) {
    return $version_str;
  } else {
    return "";
  }
}

# write internal FW ver string to $filename (data only, no header magic)
sub write_pp_fw_ver ( $$ ) {
  my $filename = shift;
  my $version_str = shift;
  my $filedata;
  my $version_str_full;
  my $count;
  
  $version_str_full = $pp_fw_ver_hdr_magic;
  $version_str_full .= _pad_terminate_str($version_str, 85);

  # read the file
  open FILE, $filename or ( die "Error: can't open '$filename'. ($!)\n");
  $filedata="";
  while (<FILE>) {
    $filedata .= $_;
  }
  close FILE;
  
  $count = ($filedata =~ s/$pp_fw_ver_hdr_magic.{85}/$version_str_full/gs);
  
  # write the patched file
  if ($count == 1) {
    open  FILE,">$filename" or ( die "Error: Can't open '$filename' for writing. ($!)\n" and return 0 );
    print FILE $filedata;
    close FILE;
    print "Version information successfully written\n\n";
  } else {
    die "Error: version information in '$filename' could not be patched\n";
  }
  
  return 0;
}

# updates the specified versionstring with new version data
sub version_str_set_version( $$ ) {
  my $version_str = shift;
  my $new_version = shift;
  
  substr($version_str, 0, 7, _pad_terminate_str($new_version, 7));
  
  return $version_str;
}

# updates the specified versionstring with a new build number
sub version_str_set_buildno( $$ ) {
  my $version_str = shift;
  my $buildno = shift;

  substr($version_str, 7, 6, _pad_terminate_str($buildno, 6));
  
  return $version_str;
}

# updates the specified versionstring with a new downgrade version number
sub version_str_set_downgrade_version( $$ ) {
  my $version_str = shift;
  my $downgrade_version = shift;
  
  substr($version_str, 13, 7, _pad_terminate_str($downgrade_version, 7));

  return $version_str;
}

# updates the specified versionstring with a new version tag
sub version_str_set_tag( $$ ) {
  my $version_str = shift;
  my $tag = shift;

  substr($version_str, 20, 65, _pad_terminate_str($tag, 65));
  
  return $version_str;
}


#
###############################################################################
# internal routines
#

# creates a raw FW image, based on K-Prod FW dirs/files
sub _fw_kprod_to_raw ($$) {
    my $dirs   = shift;
    my $raw_file_name = shift;
    my $fw_info_fname = $dirs->{"fw"} . "/fw.info";
    my $fw_info = {};
    my $physmap_name;

    if (-e "$fw_info_fname") {
	# get key value pairs
	open FW_INFO, $fw_info_fname;
	while (<FW_INFO>) {
	    chomp;
	    my ($key, $val) = split(/=/);
	    $fw_info->{$key} = $val;
	}
	close FW_INFO;
    } else {
	$fw_info->{"platform"} = "ppc";
    }

    my $board = $fw_info->{"platform"};
    if ($board eq "ppc") {
	$physmap_name = "lara8_ramfs";
    } elsif ($board eq "kira") {
	$physmap_name = "kira8_ramfs";
    } else {
	die "ERROR: could not determine physmap_name!";
    }

    # collect raw data
    my $physmap = $raw_fw_info->{$physmap_name};
    my $raw_data = "";
    foreach my $physpart (@{$physmap}) {
	if ($physpart->{"physpart_type"} eq "uboot") {
	    my $fname = $dirs->{"fw"} . "/u-boot.bin";
	    $raw_data .= _read_file_padded($fname, $physpart->{"size"});
	} elsif ($physpart->{"physpart_type"} eq "kernel") {
	    my $fname = $dirs->{"fw"} . "/zImage.bin";
	    $raw_data .= _read_file_padded($fname, $physpart->{"size"});
	} elsif ($physpart->{"physpart_type"} eq "initrd") {
	    my $fname = $dirs->{"fw"} . "/initrd.bin";
	    $raw_data .= _read_file_padded($fname, $physpart->{"size"});
	} elsif ($physpart->{"physpart_type"} eq "flashdisk") {
	    $raw_data .= _get_flashdisk_data($dirs, $physpart->{"size"}, $board);
	} elsif ($physpart->{"physpart_type"} eq "config") {
	    $raw_data .= _get_zero_data($physpart->{"size"});
	} else {
	    die "ERROR: unsupported physpart_type '$physpart->{'physpart_type'}'!";
	}
    }

    # write out raw FW
    open (RAW_FILE, '>', $raw_file_name) or
	die "ERROR: could not open '$raw_file_name' for writing raw FW!";
    print RAW_FILE $raw_data;
    close (RAW_FILE);

    print "Wrote raw FW for physmap '$physmap_name' to file '$raw_file_name'.\n";
}

sub _get_flashdisk_data ($$) {
    my $dirs      = shift;
    my $part_size = shift;
    my $board     = shift;
    # temp dir to build JFFS2 image from
    my $fd_tmp_dir = $dirs->{"work"} . "/flashdisk_tmp";
    # path of JFFS2 image to build
    my $fd_img_fname = $dirs->{"work"} . "/flashdisk.jffs2";
    # K-Prod dir with flashdisk 'tgz's
    my $fw_fd_dir = $dirs->{"fd"};
    my $cmd;

    mkdir($fd_tmp_dir)
	or die "ERROR: could not create tmp dir '$fd_tmp_dir'\n";

    foreach my $part (@{$fw->{"parts"}}) {
	my $type_i = $part->{"hdr"}->{"part_type"};
	my $subtype_i = $part->{"part_subtype"};
	my $subtype = get_subpart_types($type_i)->{$subtype_i};
	my $targ_dir = "";

	# skip non-flashdisk parts
	($subtype->{"dest"} eq "fd") or next;

	# determine target subdir under flashdisk
	my $fname = $subtype->{"file"};
	if ($fname eq "rootfs.tgz") {
	    # extract jffs rootfs to basedir
	    $targ_dir = ".";
	} else {
	    # name dir like fname, but without .tgz
	    ($targ_dir) = ($fname =~ /(.*)\.tgz/)
		or die "ERROR: could not determine targ_dir from fname=$fname!";
	}
	my $full_targ_dir = "$fd_tmp_dir/$targ_dir";

	# extract to tmp target dir
	$cmd = "rm -rf \"$full_targ_dir\" &>/dev/null";
	system($cmd) == 0 or die "ERROR: \"$cmd failed!";

	mkdir($full_targ_dir)
	    or die "ERROR: could not create full target dir '$full_targ_dir'\n";

	$cmd = "tar -xzf '$fw_fd_dir/$fname' -C '$full_targ_dir' >/dev/null 2>&1";
	system($cmd);
	    ($? >> 8 == 0) or die "ERROR: Could not untar \"$fw_fd_dir/$fname\".";
    }

    # set endian_opt
    my $endian_opt = "";
    if ($board eq "ppc") {
	$endian_opt = "--big-endian";
    } elsif ($board eq "kira") {
	$endian_opt = "--little-endian";
    }

    # create JFFS2 image
    #     NB: piping to stdout seems broken with mkfs.jffs2, so we use -o <file> and read back in
    $cmd="./mkfs.jffs2 -v --pad=$part_size $endian_opt --root='$fd_tmp_dir'" .
	    " --eraseblock=0x20000 --output='$fd_img_fname' &>/dev/null";
    system($cmd);
	($? >> 8 == 0) or die "ERROR: Could not create JFFS2 FS.";
    my $data = "";
    open(IMG, "<", $fd_img_fname);
    while (<IMG>) { $data .= $_; }
    close(IMG);

    # remove temp stuff
    $cmd = "rm -rf \"$fd_tmp_dir\" \"$fd_img_fname\" &>/dev/null";
    system($cmd);
	($? >> 8 == 0) or print "WARNING: \"$cmd\" failed!\n";

    (length($data) == $part_size) or die "ERROR: size of created JFFS2 image differs from target size!";

    return $data;
}

sub _get_zero_data ($) {
    my $size = shift;
    return pack("x[$size]");
}

# reads <file> and pads it with zeros to <length>
sub _read_file_padded($$) {
    my $file_name = shift;
    my $targ_size = shift;
    my $data = "";

    # read data
    open FILE, $file_name or die "ERROR: could not open '$file_name''!";
    while (<FILE>) {
	$data .= $_;
    }
    close FILE;
    my $file_size = length($data);

    # check size
    ($file_size <= $targ_size) or
	die "ERROR: file '$file_name' (size=$file_size) longer than target size (=$targ_size)!";

    # add padding
    my $pad_size = $targ_size - $file_size;
    $data .= pack("x[$pad_size]");

    return $data;
}

sub _get_dir_size ( $ ) {
  my $dir = shift;
  # do not add size of tmp dir!
  our $count = -(lstat($dir))[7];

  sub _count_file {
    $count += (lstat($_))[7];
  }

  find({ wanted => \&_count_file, no_chdir => 1 }, $dir);

  return $count;
}

sub _check_firmware_part ( $$$ ) {
  my $file = shift;
  my $type = shift;
  my $partnr = shift;
  my $part_info = $part_types->{$type}->{"part_subtype"}->{$partnr};
  my $fs_dir = "_firmware_tmpdir_" . time();
  my $ret = 0;

  # check if the file exists
  (-r $file) or return 0;

  # do a max size check against the filesize
  if ($part_info->{"max_size"}) {
    my $size = (lstat($file))[7];
    if ($size > $part_info->{"max_size"}) {
      print "Error: $file exceeds maximum size of " . $part_info->{"max_size"} . "bytes\n" and goto bail;
    }
  }

  # special checks for flashdisk parts
  if ($type == 4) {
    # untar in temp dir
    (-d "$fs_dir") and ( rmtree($fs_dir) );
    mkdir($fs_dir);
    system("tar -C ".$fs_dir." -x -z -f ".$file . ">/dev/null 2>&1");
    ($? >> 8 == 0) or ( print "Error: $file doesn't seem to be a valid tar.gz archive\n" and goto bail);

    my $usage = _get_dir_size($fs_dir);

    # maximum uncompressed filesize
    if ($part_info->{"max_plain_size"}) {
      if ($usage > $part_info->{"max_plain_size"}) {
	print "Error: Uncompressed size of all files in $file exceeds maximum size of " .
	  $part_info->{"max_plain_size"} . " bytes\n";
	goto bail;
      }
    }

    # check for emtpy files
    if ($usage == 0) {
      print "Error: File $file is empty and may not be added to firmware!\n";
      goto bail;
    }
  }

  $ret = 1;

 bail:
  (-d "$fs_dir") and rmtree($fs_dir);
  return $ret;
}

sub _make_firmware_common ( $$ ) {
    my $parts = shift;
    my $part_hdr_ver = shift;

    $fw->{"parts"} = ();
    $fw->{"type"} = "";
    foreach my $part (@{$parts}) {
        my $file = $part->{"file"};
	my $type = $part->{"part_type"};
	my $num = $part->{"part_subtype"};
        my $data = "";
        my $n;
	my $mypart;

        open FILE, $file or
	  ( print "Error: Can't open '$file'. ($!)\n" and return 0 );

        do {
	    my $buf;
	    $n = read FILE, $buf, 8192;
	    defined $n or
	      ( print "Error: Reading '$file' failed. ($!)\n" and return 0 );
	    $data .= $buf;
        } while ($n > 0);

	close FILE;

	_check_firmware_part($part->{"file"}, $part->{"part_type"}, $part->{"part_subtype"}) or return 0;
	
	$mypart->{"hdr"}->{"magic"}   = $fw_part_magic;
	$mypart->{"hdr"}->{"hdr_ver"} = $part_hdr_ver;
	$mypart->{"hdr"}->{"part_type"}    = $type;
	$mypart->{"data"}             = $data;
	$mypart->{"part_subtype"} = $num;
	push @{$fw->{"parts"}}, $mypart;
    }

    _set_header_sizes();

    return 1;
}


sub print_fw_info () {
    my $id;

    print "magic:       ", $fw->{"hdr"}->{"magic"},       "\n";
    print "hdr_ver:     ", $fw->{"hdr"}->{"hdr_ver"},     "\n";
    print "ver:         ", $fw->{"hdr"}->{"ver"},         "\n";
    print "min_req_ver: ", $fw->{"hdr"}->{"min_req_ver"}, "\n";
    if ($fw->{"hdr"}->{"hdr_ver"} >= 2) {
	print "build_nr:    ", $fw->{"hdr"}->{"build_nr"},    "\n";
	print "tag:         ", $fw->{"hdr"}->{"tag"},         "\n";
	print "oem:         ", $fw->{"hdr"}->{"oem"},         "\n";
	print "product:     ", $fw->{"hdr"}->{"product"},     "\n";
	print "include hw:  ";
	foreach $id (@{$fw->{"hdr"}->{"include_ids"}}) {
	    if ($id != 0) { printf "%02X ", $id; }
	}
	print "\n";
	print "exclude hw:  ";
	foreach $id (@{$fw->{"hdr"}->{"exclude_ids"}}) {
	    if ($id != 0) { printf "%02X ", $id; }
	}
	print "\n";
	
    }
    print "parts:\n";
    foreach my $part (@{$fw->{"parts"}}) {
	print "  magic:     ",
	  $part->{"hdr"}->{"magic"}, "\n";
	print "  hdr_ver:   ",
	  $part->{"hdr"}->{"hdr_ver"}, "\n";
	print "  size:      ",
	  $part->{"size"}, "\n";
	if ($part->{"hdr"}->{"hdr_ver"} >= 3) {
	    print "  type:      ",
	      $part->{"hdr"}->{"part_type"}, " ( ",
		$part_types->{$part->{"hdr"}->{"part_type"}}->{"name"}, " )\n";
	}
	print "  subtype:   ",
	  $part->{"part_subtype"}, " ",
	    "( " . $part_types->{$part->{"hdr"}->{"part_type"}}->{"part_subtype"}->{$part->{"part_subtype"}}->{"name"} . " )\n";
	print "\n";
    }
}

sub print_fw_info_scarce($) {
    my $show_subtype = shift;   # print part_subtype information only when $show_subtype matches
                                # use -1 to display subtype information for all partitions

    # global header
    print "ver:         ", $fw->{"hdr"}->{"ver"},         "\n";
    print "min_req_ver: ", $fw->{"hdr"}->{"min_req_ver"}, "\n";

    if ($fw->{"hdr"}->{"hdr_ver"} >= 2) {
	print "build_nr:    ", $fw->{"hdr"}->{"build_nr"},    "\n";
	print "tag:         ", $fw->{"hdr"}->{"tag"},         "\n";
	print "product:     ", $fw->{"hdr"}->{"product"},     "\n";
    }
    if ($fw->{"hdr"}->{"hdr_ver"} >= 3) {
	print "flash map:   ", $flash_map_names->{$fw->{"hdr"}->{"flash_map_sel"}}, "\n";
    }

    # parts
    print "parts:\n";
    foreach my $part (@{$fw->{"parts"}}) {
	print "  size:      ",
	      $part->{"size"}, "\n";
	if ($part->{"hdr"}->{"hdr_ver"} >= 3) {
	    print "  type:      ",
		  $part->{"hdr"}->{"part_type"}, " ( ",
		  $part_types->{$part->{"hdr"}->{"part_type"}}->{"name"}, " )\n";
	}
	print "  subtype:   ",
	      $part->{"part_subtype"}, " ",
	      ( (($part->{"hdr"}->{"part_type"} == $show_subtype) || ($show_subtype == -1)) ?
	      ("( " . get_subpart_types($part->{"hdr"}->{"part_type"})->{$part->{"part_subtype"}}->{"name"} . " )"):""),
	      "\n";
	print "\n";
    }
    
    return 1;
}

# decodes FW, if it's base64-encoded (uses $FW_LARA_MAGIC as heuristics)
sub _decode64_if_needed ( $ ) {
    my $data_ref = shift;

    my $magic = substr $$data_ref, $FW_MAGIC_OFFSET, 14;
    if ($magic ne $FW_LARA_MAGIC) {
	# no magic found -> try base64 decoding
	print "NOTICE: no FW magic found, will try again after decoding base64...\n";
	my $tmpbuf .= decode_base64($$data_ref);
	$magic = substr $tmpbuf, $FW_MAGIC_OFFSET, 14;
	if ($magic eq $FW_LARA_MAGIC) {
	    $$data_ref = $tmpbuf;
	} else {
	    # no magic found either -> FW seems invalid
	    die "ERROR: Could not find magic string '$FW_LARA_MAGIC' in data.  Invalid FW?\n";
	}
    }
}

sub _parse_firmware ( $ ) {
    my $data = shift;

    _decode64_if_needed(\$data);
    $fw->{"parts"} = [];
    $fw->{"hdr"}->{"md5_hash"} = substr $data, 0, 16;
    $fw->{"hdr"}->{"magic"} = substr $data, $FW_MAGIC_OFFSET, 14;
    $fw->{"hdr"}->{"hdr_ver"} = unpack "C", substr $data, 30, 1;
    $fw->{"hdr"}->{"ver"} = substr $data, 31, 6;
    $fw->{"hdr"}->{"min_req_ver"} = substr $data, 37, 6;

    if ($fw->{"hdr"}->{"hdr_ver"} >= 2) {
	$fw->{"hdr"}->{"build_nr"} = unpack "N", substr $data, 43, 4;
	$fw->{"hdr"}->{"tag"} = unpack "Z*", substr $data, 47, 64;
	$fw->{"hdr"}->{"oem"} = unpack "Z*", substr $data, 111, 16;
	$fw->{"hdr"}->{"product"} = unpack "Z*", substr $data, 127, 16;
	@{$fw->{"hdr"}->{"include_ids"}} = unpack "C16", substr $data, 143, 16;
	@{$fw->{"hdr"}->{"exclude_ids"}} = unpack "C16", substr $data, 159, 16;
    }

    if ($fw->{"hdr"}->{"hdr_ver"} >= 3) {
	# "n" = uint16_t, big-endian
	$fw->{"hdr"}->{"flash_map_sel"} = unpack "n", substr $data, 175, 2;
    } else {
	# legacy case - assumption is: hdr_ver < 3 implies "map_ramfs".
	#     (hdr_ver 3 was introduced in parallel with JFFS-FWs)
	$fw->{"hdr"}->{"flash_map_sel"} = "0";
    }

    _set_part_types_flash_subparts($fw->{"hdr"}->{"flash_map_sel"});
    _set_header_sizes();

    my $part_offs = $fw->{"hdr_size"};

    while ($part_offs < length $data) {
        my $part_num = unpack "N", substr $data, $part_offs + 16, 4;
	my $hdr_ver  = unpack "C", substr $data, $part_offs + 15, 1;
	my $next_part_offs = unpack "N", substr $data, $part_offs + 20, 4;
	my $size = $next_part_offs - $part_offs - $fw->{"part_hdr_size"};
	my $mypart;

	$mypart->{"size"} = $size;
	$mypart->{"part_subtype"} = $part_num;
	$mypart->{"hdr"}->{"magic"} =  substr $data, $part_offs, 15;
	$mypart->{"hdr"}->{"hdr_ver"} = $hdr_ver;
	if ($hdr_ver >= 3) {
	  $mypart->{"hdr"}->{"part_type"} = unpack "N", substr $data, $part_offs + 24, 4;
	}
	$mypart->{"data"} =
	  substr $data, $part_offs + $fw->{"part_hdr_size"},
	    $next_part_offs - $part_offs - $fw->{"part_hdr_size"};
	$part_offs = $next_part_offs;
	push @{$fw->{"parts"}}, $mypart;
    }

    if ($debug) {
	print_fw_info();
    }

    return 1;
}

sub _set_part_types_flash_subparts ($) {
    my $flash_map_sel = shift;
    my $map_name;

    if (!defined($flash_map_sel) || $flash_map_sel eq "") {
	# fallback to "ramfs" for legacy FWs without flash_map_sel in the header
	$map_name = "map_ramfs";
    } else {
	$map_name = get_flash_map_name($flash_map_sel);
    }
    # Initialize previously "undef"ed substructure with subparts appropriate for
    #     current FW's flash mapping
    ($part_types->{0}->{"name"} eq "Flash") or
	die "ERROR: part type 0 should be \"Flash\", but is not!\n";
    $part_types->{0}->{"part_subtype"} = $flash_subparts->{$map_name};
}

sub _set_header_sizes {
    if ($fw->{"hdr"}->{"hdr_ver"} >= 2) {
	$fw->{"hdr_size"} = 256;
	$fw->{"part_hdr_size"} = 64;
    } else {
	$fw->{"hdr_size"} = 43;
	$fw->{"part_hdr_size"} = 24;
    }
}

# convert (pad) input data to \0 padded string with req_len bytes size (including \0)
# (internal helper only)
sub _pad_terminate_str ( $$ ) {
  my $data = shift;
  my $req_len = shift;
  my $pad_size;

  my $l = length($data)-1;  
  if ($data =~ m/.{$l}\0/) { chop($data); } # prevent errors if string is already correct size and padded
  if (length($data) >= $req_len) { die("Error: '$data' can only be $req_len bytes long\n"); }
  
  $pad_size = $req_len - length($data);
  $data .= pack("x[$pad_size]");
  
  return $data;
}


#
###############################################################################
# exports
#

@ISA    = qw(Exporter);
@EXPORT = qw(
	     make_firmware
	     update_parts
	     tag_firmware
	     set_version
             version_firmware
             min_req_ver_firmware
	     load_firmware
	     save_firmware
	     save_partitions
	     fw_bin_to_production
	     delete_partitions
	     print_fw_info
	     print_fw_info_scarce
	     get_and_save_parts
	     get_subpart_types
	     $fw
	     @part_types
	     get_flash_map_sel
	     get_flash_map_name
	     mount_initrd
	     umount_initrd_nosave
	     umount_initrd_save
	     get_hal_wrapper
	     read_pp_fw_ver
	     write_pp_fw_ver
	     version_str_set_version
	     version_str_set_buildno
	     version_str_set_downgrade_version
	     version_str_set_tag
	    );

#
###############################################################################
