#!/usr/bin/perl -w
#############################################################################################
#
# mktag - create or read a customization tag file to put into the baseboards
#	  RTC eeprom, see KIM customization document for details
#		
# Tag format (version 1)
#
# Offset	Length (bytes)		Description
#   0		  1			Version (0x00,0xFF -> invalid/empty tag)
#   1		  1			Hardware-Info, Mask for single parts:
#					0x03	- Speed (PLL)
#						  (0=auto,1=slow,2=fast)
#					0xFC	- reserved
#   2		 14			reserved
#  16		 16			style (ascii, if <16 bytes terminated with \0)
#  32		 16			structure (ascii, if <16 bytes terminated with \0)
#  48		 16			conf values (ascii, if <16 bytes terminated with \0)
#  64		 16			reserved
#  80		 64			signature over tag (from offset 16, length 64)
#					(RSA encrypted SHA-1 hash)
# 144		368			reserved
#############################################################################################

package mktag;

use strict;
use Getopt::Std;

my $opts = {};
my $tag = {
	   "version"	=> 1,	# version for newly created tags
	   "style"	=> "*",
	   "structure"	=> "*",
	   "conf"	=> "*",
	   "hw_opts"	=> {}
};
my $usage =
  "Usage: \n\n" .
  "  mktag -f <filename> [-s <style>] [-S <structure>] [-c <conf>]\n" .
  "                      [-h <hardware opts> ]\n\n" .
  "    <filename>       - filename of the created tag\n" .
  "    <style>          - style name, defaults to * if left empty\n" .
  "    <structure>      - structure name, defaults to * if left empty\n" .
  "    <conf>           - config name, defaults to * if left empty\n" .
  "    <hardware opts>  - comma separated list of hardware options\n" .
  "                       as key=value pairs\n" .
  "\n";

#
###############################################################################
# prototypes
#
sub save_tag($$);
sub parse_tag($$);
sub output_tag($);

#
###############################################################################
# main program, parse options and call the appropriate subs
#

getopts('f:s:S:c:h:', $opts);

# output file, maybe set default
defined $opts->{f} or die "No tag filename (-f) specified\n$usage";

# tag part names, default to "*" (everything)
defined $opts->{"s"} and $tag->{"style"} = $opts->{"s"};
defined $opts->{"S"} and $tag->{"structure"} = $opts->{"S"};
defined $opts->{"c"} and $tag->{"conf"} = $opts->{"c"};

# parse hardware-opts
if (defined $opts->{h}) {
  my @pairs = split (/,/, $opts->{h});
  my $key;
  my $value;

  foreach my $pair (@pairs) {
    ($key, $value) = split (/=/, $pair);
    $tag->{"hw_opts"}->{$key} = $value;
  }

  # TODO: check hardware options against their allowed values
}

print "Creating tag...\n";
output_tag($tag);
save_tag($tag, $opts->{f});

#
###############################################################################
# subroutines for tag handling
#

sub save_tag($$) {
  my $tag = shift;
  my $filename = shift;
  my $data;
  my $tagdata;

  $data .= pack "C", $tag->{"version"};
  $data .= pack "C", create_hw_opts($tag);
  $data	.= chr(0) x 14;

  $tagdata .= $tag->{"style"};
  $tagdata .= chr(0) x (16 - length($tag->{"style"}));
  $tagdata .= $tag->{"structure"};
  $tagdata .= chr(0) x (16 - length($tag->{"structure"}));
  $tagdata .= $tag->{"conf"};
  $tagdata .= chr(0) x (16 - length($tag->{"conf"}));
  $tagdata .= chr(0) x 16;

  # TODO sign the stuff
  $data .= $tagdata;
  $data .= chr(0) x 64;

  open FILE, ">$filename" or die "Error during save_tag ($!)";
  print FILE $data;
  close FILE;
}
		
sub create_hw_opts($) {
  my $tag = shift;
  my $speed = 0;
  my $ret = 0;

  # TODO make this more generic if necessary
  defined $tag->{hw_opts}->{"speed"} and $speed = $tag->{"hw_opts"}->{"speed"};

  $ret = $speed & 0x03;

  return $ret;
}


sub parse_tag($$) {
  my $tag = shift;
  my $filename = shift;

  die "parse_tag not implemented yet!\n";
}

sub parse_hw_opts($) {

  die "parse_hw_opts not implemented yet!\n";
}


sub output_tag($) {
  my $tag = shift;

  print "Tag-Version: ".$tag->{"version"} . "\n";
  print "Style      : ".$tag->{"style"} . "\n";
  print "Structure  : ".$tag->{"structure"} . "\n";
  print "Config     : ".$tag->{"conf"} . "\n\n";
  print "Hardware-Options:\n";

  foreach my $key (keys %{$tag->{"hw_opts"}}) {
    print "  $key=" . $tag->{"hw_opts"}->{$key} . "\n";
  }
}
