#!/usr/bin/perl -w
###############################################################################
# create_build_cfg.pl
#
# Helper script for mksdk
# Parses a build_fw.*.cfg script and matches each component against
# an SDK config script.
# Writes a filtered build_fw.cfg script which only contains components we want 
# in this SDK, but in the correct order as it was originally in the build_fw.*.cfg

use strict;

use Getopt::Std;

#
###############################################################################
# subroutines

my $usage ="\n\n".
"Usage: build_fw -b <input build cfg> -s <sdk source cfg> [ -o <output build cfg> ]\n".
"\n\n".
"    -b <input build cfg>  - firmware build config file\n" .
"    -s <sdk source cfg>   - sdk source contents config file\n" .
"    -o <output build cfg> - output build config file, STDOUT if omitted\n";

my $build_in_file;
my $build_out_file;
my $sdk_cfg_file;
my $build_in;
my $sdk_cfg = {};
my $opts = {};
my $out_handle;

getopts('b:s:o:', $opts);

unless (defined($build_in_file = $opts->{b})) {
  die "Missing input build config file $usage";
}
if (defined($build_out_file = $opts->{o})) {
  open OUT, ">$build_out_file"
    or die "Cannot open $build_out_file $usage";
  $out_handle = \*OUT;
} else {
  $out_handle = \*STDOUT;
}
unless (defined($sdk_cfg_file = $opts->{s})) {
  die "Missing sdk source config file $usage";
}

# load the sdk config
#
open FILE, "$sdk_cfg_file"
  or die "Cannot open $sdk_cfg_file $usage";

while (<FILE>) {
  $_ =~ s/#.*//; # strip commented lines
  chomp($_); # remove newlines
  $_ eq "" and next; # ignore empty lines

  $sdk_cfg->{$_} = "1";
}

close FILE;

# go through the build config and add those entries to the
# output file, which do exist in the sdk config

open FILE, "$build_in_file"
  or die "Cannot open $build_in_file $usage";

while (<FILE>) {
  $_ =~ s/#.*//; # strip commented lines
  chomp($_); # remove newlines
  $_ eq "" and next; # ignore empty lines

  if ($_ =~ /^[^:]+:\s*(\S+?)\s*$/) {
    my $dir = $1;

    if (defined($sdk_cfg->{$dir})) {
      print $out_handle "$_\n";
    }
  }
}

close FILE;

if (defined($build_out_file = $opts->{o})) {
  close $out_handle;
}
