/*
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Marius Groeger <mgroeger@sysgo.de>
 *
 * SPI flash adaptions made by
 * Danny Baumann <danny.baumann@raritan.com>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <asm/u-boot.h>
#include <flash.h>
#include <kira/porting.h>
#include <kira/kira_ssp.h>

#define FLASH_BANK_SIZE 0x1000000
#define MAIN_SECT_SIZE  0x10000

flash_info_t    flash_info[CFG_MAX_FLASH_BANKS];

#define K								(1024)

ulong flash_init()
{
    int i, j;
    uchar inbuf[512], outbuf[512];
    uchar manuf_id;
    ushort device_id;
    ulong size = 0, sect_size = 0;
    
    ssp_init();
    
    memset(&(flash_info[0]), 0, sizeof(flash_info_t));
    
    /* read ID code from flash */
    outbuf[0] = 0x9f;
    outbuf[1] = 0x00;
    outbuf[2] = 0x00;
    outbuf[3] = 0x00;
    ssp_txrx(outbuf, inbuf, 4);
    manuf_id = inbuf[1];
    device_id = (inbuf[2] << 8) | inbuf[3];
    
    flash_info[0].flash_id = (manuf_id << 16) | device_id;
    switch (manuf_id) {
    case SPI_WINBOND_MANUFACT:
        switch (device_id) {
        case SPI_WINBOND_ID_W25P80:
            flash_info[0].size = SPI_WINBOND_W25P80_SIZE;
            flash_info[0].sector_count = SPI_WINBOND_W25P80_SECTS;
            sect_size = SPI_WINBOND_W25P80_SECTSIZE;
            break;
        case SPI_WINBOND_ID_W25P16:
            flash_info[0].size = SPI_WINBOND_W25P16_SIZE;
            flash_info[0].sector_count = SPI_WINBOND_W25P16_SECTS;
            sect_size = SPI_WINBOND_W25P16_SECTSIZE;
            break;
        case SPI_WINBOND_ID_W25P32:
            flash_info[0].size = SPI_WINBOND_W25P32_SIZE;
            flash_info[0].sector_count = SPI_WINBOND_W25P32_SECTS;
            sect_size = SPI_WINBOND_W25P32_SECTSIZE;
            break;
        }
        break;
    case SPI_MXIC_MANUFACT:
        switch (device_id) {
        case SPI_MXIC_ID_MX25L6405:
            flash_info[0].size = SPI_MXIC_MX25L6405_SIZE;
            flash_info[0].sector_count = SPI_MXIC_MX25L6405_SECTS;
            sect_size = SPI_MXIC_MX25L6405_SECTSIZE;
            break;
        }
        break;
    case SPI_SPANSION_MANUFACT:
        switch (device_id) {
        case SPI_SPANSION_ID_S25FL128P:
            flash_info[0].size = SPI_SPANSION_S25FL128P_SIZE;
            flash_info[0].sector_count = SPI_SPANSION_S25FL128P_SECTS;
            sect_size = SPI_SPANSION_S25FL128P_SECTSIZE;
            break;
        }
        break;
    case SPI_ST_MANUFACT:
        switch (device_id) {
        case SPI_ST_ID_M25P128:
            flash_info[0].size = SPI_ST_M25P128_SIZE;
            flash_info[0].sector_count = SPI_ST_M25P128_SECTS;
            sect_size = SPI_ST_M25P128_SECTSIZE;
            break;
        }
        break;
    }
    if (flash_info[0].size == 0) flash_info[0].size = PHYS_FLASH_SIZE;
    if (flash_info[0].sector_count == 0) flash_info[0].sector_count = CFG_MAX_FLASH_SECT;
    if (sect_size == 0) sect_size = MAIN_SECT_SIZE;
    memset(flash_info[0].protect, 0, flash_info[0].sector_count);
    
    for (i = 0; i < CFG_MAX_FLASH_BANKS; i++) {
        ulong flashbase = 0;

        if (i == 0)
            flashbase = PHYS_FLASH_1;
        else
            panic("configured to many flash banks!\n");
        for (j = 0; j < flash_info[i].sector_count; j++) {
            flash_info[i].start[j] = flashbase + j*sect_size;
        }
        size += flash_info[i].size;
    }
    
    /* Protect monitor and environment sectors
    */
    flash_protect(FLAG_PROTECT_SET,
                  CFG_FLASH_BASE,
                  CFG_FLASH_BASE + _armboot_end - _armboot_start,
                  &flash_info[0]);
    
    flash_protect(FLAG_PROTECT_SET,
                  CFG_ENV_ADDR,
                  CFG_ENV_ADDR + CFG_ENV_SIZE - 1,
                  &flash_info[0]);

    return size;
}

/*-----------------------------------------------------------------------
 */
void flash_print_info  (flash_info_t *info)
{
    int i;

    switch (info->flash_id) {
    case ((SPI_WINBOND_MANUFACT << 16) | SPI_WINBOND_ID_W25P80):
        printf("Winbond: W25P80 (8Mbit)\n");
        break;
    case ((SPI_WINBOND_MANUFACT << 16) | SPI_WINBOND_ID_W25P16):
        printf("Winbond: W25P16 (16Mbit)\n");
        break;
    case ((SPI_WINBOND_MANUFACT << 16) | SPI_WINBOND_ID_W25P32):
        printf("Winbond: W25P32 (32Mbit)\n");
        break;
    case ((SPI_MXIC_MANUFACT << 16) | SPI_MXIC_ID_MX25L6405):
        printf("Macronix: MX25L6405 (64MBit)\n");
        break;
    case ((SPI_SPANSION_MANUFACT << 16) | SPI_SPANSION_ID_S25FL128P):
        printf("Spansion: S25FL128P (128MBit)\n");
        break;
    case ((SPI_ST_MANUFACT << 16) | SPI_ST_ID_M25P128):
        printf("ST Micro: M25P128 (128MBit)\n");
        break;
    default:
        printf("Unknown Chip (%02x / %04x)", (info->flash_id >> 16), (info->flash_id & FLASH_TYPEMASK));
        goto Done;
        break;
    }
    
    printf("  Size: %ld MB in %d Sectors\n",
           info->size >> 20, info->sector_count);
    
    printf("  Sector Start Addresses:");
    for (i = 0; i < info->sector_count; i++) {
        if ((i % 5) == 0) 
            printf ("\n   ");
        printf (" %08lX%s", info->start[i], info->protect[i] ? " (RO)" : "     ");
    }
    printf ("\n");
    
Done:
        return;
}

/*-----------------------------------------------------------------------
 */

int flash_erase (flash_info_t *info, int s_first, int s_last)
{
    int flag, prot, sect, i;
    unsigned char val;
    int rc = ERR_OK;
    char outbuf[512], inbuf[512];
   
    if (info->flash_id == FLASH_UNKNOWN)
        return ERR_UNKNOWN_FLASH_TYPE;

    if ((s_first < 0) || (s_first > s_last)) {
        return ERR_INVAL;
    }

    prot = 0;
    for (sect=s_first; sect<=s_last; ++sect) {
        if (info->protect[sect]) {
            prot++;
        }
    }
    if (prot)
        return ERR_PROTECTED;

    /* 
     * Disable interrupts which might cause a timeout
     * here. Remember that our exception vectors are
     * at address 0 in the flash, and we don't want a
     * (ticker) exception to happen while the flash
     * chip is in programming mode.
    */
    flag = disable_interrupts();

    /* Start erase on unprotected sectors */
    for (sect = s_first; sect<=s_last && !ctrlc(); sect++) {

        printf("Erasing sector %2d ... ", sect);

        /* arm simple, non interrupt dependent timer */
        reset_timer_masked();
        
        if (info->protect[sect] == 0) { /* not protected */
            outbuf[0] = 0x06; //flash write enable
            ssp_txrx(outbuf, NULL, 1);
            
            outbuf[0] = 0xd8; //sector erase
            for (i = 2; i >= 0; i--) outbuf[3-i] = (info->start[sect] >> (8*i)) & 0xff;
            ssp_txrx(outbuf, NULL, 4);
            
            outbuf[0] = 0x05; //read status reg
            outbuf[1] = 0x00; //dummy
            val = 1;
            while (val & 0x1) {
                if (get_timer_masked() > CFG_FLASH_ERASE_TOUT) {
                    rc = ERR_TIMOUT;
                    goto outahere;
                }
                ssp_txrx(outbuf, inbuf, 2);
                val = inbuf[1];
            }
        }
        printf("ok.\n");
    }
    if (ctrlc())
      printf("User Interrupt!\n");

outahere:
   
    /* allow flash to settle - wait 10 ms */
    udelay_masked(10000);
   
    if (flag)
      enable_interrupts();
   
    return rc;
}

/*
  Read from Flash
*/

int read_buff(flash_info_t *info, uchar *dest, ulong addr, ulong cnt)
{
    int rc = ERR_OK;
    int page, pages, i;
    ulong addr_curr, count, transferred = 0;
    uchar inbuf[512], outbuf[512];
    
    outbuf[0] = 0x03;   //read command
    pages = cnt / 256;
    if (cnt % 256) pages++;
    for (page = 0; page < pages;page++) {
        addr_curr = addr + (page*256);
        count = ((cnt % 256) && (page == (pages-1))) ? (cnt % 256) : 256;
        for (i = 2; i >= 0; i--) outbuf[3-i] = (addr_curr >> (8*i)) & 0xff; //src address
        memset(outbuf+4, 0, 508);
        ssp_txrx(outbuf, inbuf, count+4);
        memcpy(dest+transferred, inbuf+4, count);
        transferred += count;
    }
    return rc;
}

/*-----------------------------------------------------------------------
 * Copy memory to flash
 */

static int write_data (flash_info_t *info, ulong dest, uchar *data, ulong cnt, ulong verify)
{
    ushort val;
    uchar inbuf[512], outbuf[512];
    int rc = ERR_OK;
    int flag, i;

    /* we only want to write up to CFG_FLASH_PAGE_SIZE bytes */
    if (cnt > CFG_FLASH_PAGE_SIZE)
        return ERR_INVAL;
    
    /* Check if Flash is (sufficiently) erased
    */
    if (verify) {
        if (read_buff(info, inbuf, dest, cnt) == ERR_OK) {
            for (i=0;i < cnt; i++) {
                if ((inbuf[i] & data[i]) != data[i]) return ERR_NOT_ERASED;
            }
        } else return ERR_INVAL;
    }
    
    /* 
     * Disable interrupts which might cause a timeout
     * here. Remember that our exception vectors are
     * at address 0 in the flash, and we don't want a
     * (ticker) exception to happen while the flash
     * chip is in programming mode.
    */
    flag = disable_interrupts();

    outbuf[0] = 0x06; //flash write enable
    ssp_txrx(outbuf, NULL, 1);
            
    outbuf[0] = 0x02; //page program
    for (i = 2; i >= 0; i--) outbuf[3-i] = (dest >> (8*i)) & 0xff;
    memcpy(outbuf+4, data, cnt);
    ssp_txrx(outbuf, NULL, cnt+4);
            
    /* arm simple, non interrupt dependent timer */
    reset_timer_masked();
    
    outbuf[0] = 0x05; //read status reg
    outbuf[1] = 0x00; //dummy
    val = 1;
    while (val & 0x1) {
        if (get_timer_masked() > CFG_FLASH_ERASE_TOUT) {
            rc = ERR_TIMOUT;
            goto outahere;
        }
        ssp_txrx(outbuf, inbuf, 2);
        val = inbuf[1];
    }
    
outahere:
    if (flag)
      enable_interrupts();

    return rc;
}

/*-----------------------------------------------------------------------
 * Copy memory to flash.
 */

int write_buff (flash_info_t *info, uchar *src, ulong addr, ulong cnt)
{
    ulong wp = addr;
    uchar tempdata[2];
    uchar *curr_data = src;
    int l, rc, i = 0;

    /* first handle unaligned start bytes */
    if (addr & 0x1) {
        tempdata[0] = 0xff;
        tempdata[1] = *curr_data;
        if ((rc = write_data(info, wp, tempdata, 2, 0)) != ERR_OK) return rc;
        wp += 1;
        curr_data += 1;
    }
    /* first write the bytes up to the next page boundary */
    l = (CFG_FLASH_PAGE_SIZE - (wp & CFG_FLASH_PAGE_SIZE));
    if (l > cnt) l = cnt;
    if ((rc = write_data(info, wp, curr_data, l, 1)) != ERR_OK) return rc;
    wp += l; curr_data += l;
    
    /* then write all full pages */
    l = (cnt - (wp-addr)) / CFG_FLASH_PAGE_SIZE;
    while (l > 0) {
        if ((rc = write_data(info, wp, curr_data, CFG_FLASH_PAGE_SIZE, 1)) != ERR_OK) return rc;
        wp += CFG_FLASH_PAGE_SIZE;
        curr_data += CFG_FLASH_PAGE_SIZE;
        // print one dot every 4K
        if (((wp - addr) >> 12) > i) {
            printf(".");
            i++;
        }
        l = (cnt - (wp-addr)) / CFG_FLASH_PAGE_SIZE;
    }
    
    /* and at last write the rest correctly handling byte alignment */
    l = (cnt - (wp-addr)) & (~1);
    if ((rc = write_data(info, wp, curr_data, l, 1)) != ERR_OK) return rc;
    wp += l;curr_data += l;
    
    l = (cnt - (wp-addr));
    if (l != 0) {   //we have a last byte
        tempdata[0] = *curr_data;
        tempdata[1] = 0xff;
        if ((rc = write_data(info, wp, tempdata, 2, 0)) != ERR_OK) return rc;
    }
    
    printf("\n");
    return ERR_OK;
}
