/*****************************************************************************
 *  Flash Utility Programs Menu
 *
 *  FILE: flash_menu.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <common.h>
#include <flash.h>
#include <check.h>
#include <flash_menu.h>
#include <mem.h>
#include <util.h>


/* info for FLASH chips */
extern flash_info_t flash_info[];

/* Flash Read size: 1 = 8-bit, 2 = 16-bit, 4 = 32-bit */
static char flash_r_width = 4;


/*-----------------------------------------------------------------------------
 * flash address validation
 *---------------------------------------------------------------------------*/
int check_flashaddr( ulong addr )
{
    /* info for FLASH chips */
    extern flash_info_t flash_info[];

    if (addr2info(addr) == NULL) {
       printf("\nERROR: address 0x%08lx does not reside in flash\n", addr);
       printf("Valid flash range is 0x%08lx - 0x%08lx.\n", 
              flash_info[0].start[0],
              flash_info[0].start[0] + flash_info[0].size - 1);
       return -1;
    }
    if (addr2info(addr + flash_r_width - 1) == NULL) {
       printf("\nERROR: address 0x%08lx with data size %d overbounds "
              "flash memory\n", addr, flash_r_width);
       printf("Valid flash range is 0x%08lx - 0x%08lx.\n", 
              flash_info[0].start[0],
              flash_info[0].start[0] + flash_info[0].size - 1);
       return -1;
    }
    return 0;
}

static void print_flash_menu( void )
{
    printf("\nFlash Utilities:");
    print_exit();
    printf("\n\t 1: Print Flash Info");
    printf("\n\t 2: Change Flash Read Option");
    printf("\n\t 3: Read Flash");
    printf("\n\t 4: Dump Flash");
    printf("\n\t 5: Erase Flash Chip");
    printf("\n\t 6: Erase Flash Sector");
    printf("\n\t 7: Set Flash Sector Protection");
    printf("\n\t 8: Clear Flash Sector Protection");
    printf("\n\t 9: Write Flash");
    printf("\n\n");

    return;
}

int flash_menu( void )
{
    char flash_flag = 1;
    int result = 1; /* result is used for reverting back to U-boot cmds */
    unsigned long addr, width, length;
    unsigned long start_sector, last_sector, bank;
    int num_chars;
    unsigned long diag_opt;
    int opt_res;
    int i;

    while (flash_flag) {
        print_flash_menu();

        opt_res = menu_get_user_input(&diag_opt);
        if( opt_res > 0 ) {
            return 0;
        }
        else if( opt_res < 0 ) {
            continue;
        }

        switch( diag_opt ) {
            case FLASH_MENU_EXIT:
                flash_flag = 0;
                break;

            case FLASH_MENU_RW_OPTIONS:
                printf("\nCurrent Flash Read width is ");
                switch (flash_r_width) {
                    case (sizeof(ulong)):
                        printf("32-bit\n");
                        break;
                    case (sizeof(ushort)):
                        printf("16-bit\n");
                        break;
                    case (sizeof(uchar)):
                        printf("8-bit\n");
                        break;
                    default:
                        printf("Invalid\n");
                        break;
                }
                num_chars = print_prompt("\nEnter new data size [(1) 8-bit, "
                                         "(2) 16-bit (4) 32-bit]: ", &width, 10);
                if (num_chars > 0 && check_byte_width(width) == 0) {
                    flash_r_width = width;
                }
                break;

            case FLASH_MENU_INFO:
                printf("\n");
                for( i = 0; i < CFG_MAX_FLASH_BANKS; i++ ) {
                    flash_print_info(&flash_info[i]);
                    print_pause();
                }
                break;

            case FLASH_MENU_READ:
                num_chars = print_prompt("\nEnter address: 0x", &addr, 16);
                if (num_chars > 0 && check_flashaddr(addr) == 0) {
                    printf("\n");
                    mem_dump(addr, 1, flash_r_width); 
                }
                break;

            case FLASH_MENU_DUMP:
                num_chars = print_prompt("\nEnter address: 0x", &addr, 16);
                if (num_chars <= 0 || check_flashaddr(addr) != 0) {
                    break;
                }
                num_chars = print_prompt("Enter length : 0x", &length, 16);
                if (num_chars > 0 && 
                    check_flashaddr(addr + length - flash_r_width) == 0) {
                    printf("\n");
                    mem_dump(addr, length, flash_r_width);
                }
                break;

            case FLASH_MENU_ERASE_CHIP:
                for( i = 0;  i < CFG_MAX_FLASH_BANKS; i++ ) {
                    if (flash_erase(&flash_info[i], 0, flash_info[i].sector_count-1) != 0 ) {
                        printf("ERROR: Failed to erase the flash\n");
                        break;
                    }
                }
                break;

            case FLASH_MENU_ERASE_SECTOR:
                num_chars = print_prompt("\nFlash Bank        : ",
                                         &bank, 10);
                if (num_chars <= 0 || check_flashbank(bank) != 0) {
                    break;
                }
                num_chars = print_prompt("First erase sector: ",
                                         &start_sector, 10);
                if (num_chars <= 0 || check_flashsect(start_sector) != 0) {
                    break;
                }
                num_chars = print_prompt("Last erase sector : ",
                                         &last_sector, 10);
                if (num_chars > 0 && check_flashsect(last_sector) == 0) {
                    if (flash_erase(&flash_info[bank], start_sector, last_sector)
                        != 0) {
                        printf("\nERROR: Failed to erase a flash sector\n");
                    }
                }
                break;

            case FLASH_MENU_PROTECT_SET:
                num_chars = print_prompt("\nFlash Bank          : ",
                                         &bank, 10);
                if (num_chars <= 0 || check_flashbank(bank) != 0) {
                    break;
                }
                num_chars = print_prompt("First protect sector: ",
                                         &start_sector, 10);
                if (num_chars <= 0 || check_flashsect(start_sector) != 0) {
                    break;
                }
                num_chars = print_prompt("Last protect sector : ",
                                         &last_sector, 10);
                if (num_chars > 0 && check_flashsect(last_sector) == 0) {
                    flash_protect_on(bank, start_sector, last_sector);
                }
                break;

            case FLASH_MENU_PROTECT_CLEAR:
                num_chars = print_prompt("\nFlash Bank        : ",
                                         &bank, 10);
                if (num_chars <= 0 || check_flashbank(bank) != 0) {
                    break;
                }
                num_chars = print_prompt("First clear sector: ",
                                         &start_sector, 10);
                if (num_chars <= 0 || check_flashsect(start_sector) != 0) {
                    break;
                }
                num_chars = print_prompt("Last clear sector : ",
                                         &last_sector, 10);
                if (num_chars > 0 && check_flashsect(last_sector) == 0) {
                    flash_protect_off(bank, start_sector, last_sector);
                }
                break;

           case FLASH_MENU_WRITE:
               num_chars = print_prompt("\nWrite flash address: 0x", &addr, 16);
               if (num_chars > 0 && check_flashaddr(addr) == 0) {
                   num_chars = readline("Fill write buffer [max 256 chars]: ");
                   if (num_chars > 0) {
                       result = flash_write(console_buffer, addr, num_chars);
                       flash_perror( result );
                       switch( result ) {
                           case ERR_OK:
                               result = 1;
                               break;
                           default:
                               result = 0;
                               break;
                       }
                   }
               }
               break;

            default:
                print_invalid();
                break;
        }
    }

    return result;
}
