/*****************************************************************************
 *  Memory Test and Utility Menu
 *
 *  FILE: memory_menu.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <common.h>
#include <command.h>
#include <check.h>
#include <mem.h>
#include <memory_menu.h>
#include <util.h>


#define MEMTEST_FILL_5    0x55555555
#define MEMTEST_FILL_A    0xaaaaaaaa

/* SDRAM Read/Write size: 1 = 8-bit, 2 = 16-bit, 4 = 32-bit */
static char mem_rw_width = 4;

/* SDRAM Test Start/End address */
static uint memtest_start = CFG_MEMTEST_START;
static uint memtest_end = CFG_MEMTEST_END;

static uint cfg_sdram_size = 0;


static int check_test_memaddr( uint addr, char start )
{
    /* have to be word-aligned */
    if (addr & 0x3) {
       return -1;
    }

    /* have to be in range of valid values */
    if (addr < CFG_MEMTEST_START || addr > CFG_MEMTEST_END) {
       return -1;
    }

    if (start) {
       if (addr == CFG_MEMTEST_END) {
          return -1;
       }
    }
    else {
       if (addr <= memtest_start) {
          return -1;
       }
    }

    return 0;
}

static int check_memaddr( ulong addr )
{
    /* have to be in range of valid values */
    if (addr >= CFG_SDRAM_BASE + cfg_sdram_size || addr < CFG_SDRAM_BASE) {
       printf("\nERROR: address 0x%08lx does not reside in SDRAM\n", addr);
       printf("Valid SDRAM range is 0x%08x - 0x%08x.\n", CFG_SDRAM_BASE,
              CFG_SDRAM_BASE + cfg_sdram_size - 1);
       return -1;
    }
    else if ((addr + mem_rw_width - 1) >= CFG_SDRAM_BASE + cfg_sdram_size) {
       printf("\nERROR: address 0x%08lx with data size %d overbounds SDRAM\n",
              addr, mem_rw_width);
       printf("Valid SDRAM range is 0x%08x - 0x%08x.\n", CFG_SDRAM_BASE,
              CFG_SDRAM_BASE + cfg_sdram_size - 1);
       return -1;
    }
    return 0;
}

static int check_write_memaddr( ulong addr )
{
    /* have to be in range of valid values */
    if (addr < CFG_MEMTEST_START || addr > CFG_MEMTEST_END) {
       printf("\nERROR: address 0x%08lx is in reserved SDRAM space\n", addr);
       printf("Valid SDRAM range is 0x%08x - 0x%08x.\n", CFG_MEMTEST_START,
              CFG_MEMTEST_END - 1);
       return -1;
    }
    else if ((addr + mem_rw_width - 1) >= CFG_MEMTEST_END) {
       printf("\nERROR: address 0x%08lx with data size %d overbounds "
              "writable SDRAM\n", addr, mem_rw_width);
       printf("Valid SDRAM range is 0x%08x - 0x%08x.\n", CFG_SDRAM_BASE,
              CFG_SDRAM_BASE + cfg_sdram_size - 1);
       return -1;
    }
    return 0;
}

static void print_memory_util_menu( void )
{
    printf("\nMemory Utilities:");
    print_exit();
    printf("\n\t 1: Change Read/Write Options");
    printf("\n\t 2: Read SDRAM");
    printf("\n\t 3: Write SDRAM");
    printf("\n\t 4: Dump SDRAM");
    printf("\n\t 5: Fill SDRAM");
    printf("\n\t 6: Search Pattern");
    printf("\n\n");

    return;
}

int memory_util_menu( void )
{
    char memutil_flag = 1;
    int result = 1; /* result is used for reverting back to U-boot cmds */
    unsigned long addr, width, length;
    unsigned long rw_val;
    int num_chars;
    unsigned long diag_opt;
    int opt_res;
    unsigned long pattern, data;
    char ignore;
    static unsigned long search_start = 0;
    static unsigned long search_end = 0;

    if (cfg_sdram_size == 0) {
        cfg_sdram_size = initdram(0);
        search_end = cfg_sdram_size - sizeof(int);
    }

    while (memutil_flag) {
       print_memory_util_menu();

       opt_res = menu_get_user_input(&diag_opt);
       if( opt_res > 0 ) {
           return 0;
       }
       else if( opt_res < 0 ) {
           continue;
       }

       switch( diag_opt ) {
           case MEM_UTIL_MENU_EXIT:
              memutil_flag = 0;
              break;

           case MEM_UTIL_MENU_RW_OPTIONS:
              printf("\nCurrent Read/Write width is ");
              switch (mem_rw_width) {
                 case (sizeof(ulong)):
                    printf("32-bit\n");
                    break;
                 case (sizeof(ushort)):
                    printf("16-bit\n");
                    break;
                 case (sizeof(uchar)):
                    printf("8-bit\n");
                    break;
                 default:
                    printf("Invalid\n");
                    break;
              }
              num_chars = print_prompt("\nEnter new data size [(1) 8-bit, "
                                       "(2) 16-bit (4) 32-bit]: ", &width, 10);
              if (num_chars > 0 && check_byte_width(width) == 0) {
                 mem_rw_width = width;
              }
              break;

           case MEM_UTIL_MENU_READ:
              num_chars = print_prompt("\nEnter address: 0x", &addr, 16);
              if (num_chars > 0 && check_memaddr(addr) == 0) {
                 printf("\n");
                 mem_dump(addr, 1, mem_rw_width); 
              }
              break;

           case MEM_UTIL_MENU_WRITE:
              num_chars = print_prompt("\nEnter SDRAM address: 0x", &addr, 16);
              if (num_chars > 0 && check_write_memaddr(addr) == 0) {
                 num_chars = print_prompt("Enter value to write: 0x",
                                          &rw_val, 16);
                 if (num_chars > 0) {
                    printf("\n");
                    mem_fill(addr, 1, mem_rw_width, rw_val); 
                 }
              }
              break;

           case MEM_UTIL_MENU_DUMP:
              num_chars = print_prompt("\nEnter address: 0x", &addr, 16);
              num_chars = print_prompt("Enter length : 0x", &length, 16);
              if (check_memaddr(addr) == 0
                  && check_memaddr(addr + length - mem_rw_width) == 0) {
                 printf("\n");
                 mem_dump(addr, length, mem_rw_width);
              }
              break;

           case MEM_UTIL_MENU_FILL:
              num_chars = print_prompt("\nEnter SDRAM address: 0x", &addr, 16);
              addr &= ~((unsigned long)(mem_rw_width-1));
              if (num_chars <= 0 || check_write_memaddr(addr) != 0) {
                 break;
              }
              num_chars = print_prompt("Enter length: 0x", &length, 16);
              if (num_chars > 0 && check_write_memaddr(addr+length-mem_rw_width) == 0) {
                 num_chars = print_prompt("Enter value to write: 0x",
                                          &rw_val, 16);
                 if (num_chars > 0) {
                    printf("\n");
                    mem_fill(addr, (length+mem_rw_width-1)/mem_rw_width, mem_rw_width, rw_val); 
                 }
              }
              break;

           case MEM_UTIL_MENU_SEARCH_PATTERN:
              printf("\nCurrent search start addr is 0x%08lx", search_start);
              printf("\nCurrent search end addr is 0x%08lx\n\n", search_end);

              ignore = 0;
              do {
                 num_chars = readline("Enter new search start address: 0x");
                 if (num_chars > 0) {
                    addr = simple_strtoul(console_buffer, NULL, 16);
                 }
                 else {
                    /* keep the start address */
                    addr = search_start;
                    ignore = 1;
                 }
              } while (!ignore && check_memaddr(addr) < 0);

              search_start = addr;

              ignore = 0;
              do {
                 num_chars = readline("Enter new search end address: 0x");
                 if (num_chars > 0) {
                    addr = simple_strtoul(console_buffer, NULL, 16);
                 }
                 else {
                    addr = search_end;
                    /* keep the end address only if it is still valid */
                    if (search_end > search_start) {
                       ignore = 1;
                    }
                 }
              } while (!ignore && (check_memaddr(addr) < 0) &&
                       addr < search_start);

              search_end = addr;

              num_chars = readline("Enter 32-bit pattern to search: 0x");
              if (num_chars > 0) {
                  pattern = simple_strtoul(console_buffer, NULL, 16);
                  addr = search_start;
                  do {
                      data = *(unsigned long *)addr;
                      if (data == pattern) {
                          printf("address 0x%08lx has pattern\n", addr);
                      }
                      addr += sizeof(int);
                  }while (addr < search_end);
              }
              break;

           default:
              print_invalid();
              break;
       }
    }

    return result;
}

static void print_memory_test_menu( void )
{
    printf("\nMemory Test:");
    print_exit();
    printf("\n\t 1: Change Memory Test Start/End Address");
    printf("\n\t 2: Single-Pass Test");
    printf("\n\t 3: Continuous Test");
    printf("\n\t 4: Walking Ones Test");
    printf("\n\t 5: Address in Address Test");
    printf("\n\t 6: Fill 0x55555555 Test");
    printf("\n\t 7: Fill 0xaaaaaaaa Test");
    printf("\n\t 8: Fill Modulo Test");
    printf("\n\n");

    return;
}

int memory_test_menu( void )
{
    char memutil_flag = 1;
    int result = 1; /* result is used for reverting back to U-boot cmds */
    int test_result;
    int num_chars;
    char ignore;
    uint size, addr;
    unsigned long diag_opt;
    int opt_res;

    if (cfg_sdram_size == 0) {
        cfg_sdram_size = initdram(0);
    }

    while (memutil_flag) {
       print_memory_test_menu();

       opt_res = menu_get_user_input(&diag_opt);
       if( opt_res > 0 ) {
           return 0;
       }
       else if( opt_res < 0 ) {
           continue;
       }

       switch( diag_opt ) {
           case MEM_TEST_MENU_EXIT:
              memutil_flag = 0;
              break;

           case MEM_TEST_MENU_SEADDR_OPT:
              printf("\nCurrent Test start address is 0x%08x", memtest_start);
              printf("\nCurrent Test end address is 0x%08x\n\n", memtest_end);

              ignore = 0;
              do {
                 num_chars = readline("Enter new test start address: 0x");
                 if (num_chars > 0) {
                    addr = simple_strtoul(console_buffer, NULL, 16);
                 }
                 else {
                    /* keep the start address */
                    addr = memtest_start;
                    ignore = 1;
                 }
              } while (!ignore && check_test_memaddr(addr, 1) < 0);

              memtest_start = addr;

              ignore = 0;
              do {
                 num_chars = readline("Enter new test end address: 0x");
                 if (num_chars > 0) {
                    addr = simple_strtoul(console_buffer, NULL, 16);
                 }
                 else {
                    addr = memtest_end;
                    /* keep the end address only if it is still valid */
                    if (check_test_memaddr(memtest_end, 0) == 0) {
                       ignore = 1;
                    }
                 }
              } while (!ignore && check_test_memaddr(addr, 0) < 0);

              memtest_end = addr;
              printf("\nNew Test start address is 0x%08x", memtest_start);
              printf("\nNew Test end address is 0x%08x\n", memtest_end);
              break;

           case MEM_TEST_MENU_SINGLE:
              test_result = mem_test(memtest_start, memtest_end, 0);
              if (test_result == 0) {
                 printf("\bpassed.\n");
              }
              else if (test_result == 1) {
                 printf("\bhalted\n");
              }
              break;

           case MEM_TEST_MENU_CONTINUOUS:
              test_result = mem_test(memtest_start, memtest_end, 1);
              printf("\n");
              break;

           case MEM_TEST_MENU_WALKING_ONES:
              size = memtest_end - memtest_start;
              test_result = memtest_walking_ones(memtest_start, size);
              if (test_result == 0) {
                 printf("\bpassed.\n");
              }
              else if (test_result == 1) {
                 printf("\bhalted\n");
              }
              break;

           case MEM_TEST_MENU_ADDR:
              size = memtest_end - memtest_start;
              test_result = memtest_address(memtest_start, size);
              if (test_result == 0) {
                 printf("\bpassed.\n");
              }
              else if (test_result == 1) {
                 printf("\bhalted\n");
              }
              break;

           case MEM_TEST_MENU_FILL_5:
              size = memtest_end - memtest_start;
              test_result = memtest_fill(memtest_start, size, MEMTEST_FILL_5);
              if (test_result == 0) {
                 printf("\bpassed.\n");
              }
              else if (test_result == 1) {
                 printf("\bhalted\n");
              }
              break;

           case MEM_TEST_MENU_FILL_A:
              size = memtest_end - memtest_start;
              test_result = memtest_fill(memtest_start, size, MEMTEST_FILL_A);
              if (test_result == 0) {
                 printf("\bpassed.\n");
              }
              else if (test_result == 1) {
                 printf("\bhalted\n");
              }
              break;

           case MEM_TEST_MENU_FILL_MOD:
              size = memtest_end - memtest_start;
              test_result = memtest_modfill(memtest_start, size);
              if (test_result == 0) {
                 printf("\bpassed.\n");
              }
              else if (test_result == 1) {
                 printf("\bhalted\n");
              }
              break;

           default:
              print_invalid();
              break;
       }
    }

    return result;
}
