/*****************************************************************************
 *  Intel LXT972A MII PHY Driver
 *
 *  FILE: mii_phy_10_100.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <common.h>
#include <mii_phy.h>
#include <miiphy.h>
#include <mii_phy_10_100.h>
#include <lxt972.h>
#include <dm9161.h>

#define MII_PHY_CNT    1

static int debug = 0;

#define PPDEBUG(fmt, args...)	 {					\
	if(debug != 0) {						\
		printf("[%s %d %s]: ",__FILE__,__LINE__,__FUNCTION__);	\
		printf(fmt, ##args);					\
	}								\
}

typedef struct mii_phy_private {
    uint   phy_address;
    uint   phy_type;
    uint   phy_ident;
    ushort phy_duplex;
    ushort phy_speed;
    int	   auto_neg;
    int	   link_options;
} mii_phy_private_t;

static mii_phy_private_t mii_priv[MII_PHY_CNT];
static int  phyaddr = -1;
static int  phy_id  = 0;
static uint phytype;
static int  mii_phy_init = 0;

/* global variable */
int  found_phy = -1;


unsigned short
mii_phy_read(unsigned short reg)
{
    unsigned short val;
    miiphy_read(phyaddr, reg, &val);
    return val;
}

void
mii_phy_write(unsigned short reg, unsigned short val)
{
    miiphy_write(phyaddr, reg, val);
    return;
}

static int
mii_discover_phy_poll(struct mii_phy_private *cep)
{
    /* limit one phy discovery for each port */
    ushort rv;
    int	i, go_on = 1;
    int auton, linkopt;
    static int start = 0;

    found_phy = 0;
    cep->phy_type = 0;
    cep->phy_ident = 0;
    for (i = start; i < 32; i++) {
        if (miiphy_read(i, PHY_PHYIDR1, &rv) < 0)
            continue;

        if (((phytype = (rv & 0xffff)) != 0xffff) &&
            ((rv & 0xffff) != 0x0000)) {
            if (miiphy_read(i, PHY_PHYIDR2, &rv) < 0)
                continue;
            ++found_phy;
            phyaddr = i;
            phytype <<= 16;
            phytype |= rv;
            PPDEBUG("\nfec: Phy @ 0x%x, type 0x%08x\n", phyaddr, phytype);
            cep->phy_type = phytype;
            cep->phy_address = phyaddr;
            start = i + 1;
            go_on = 0;
        }
        if (!go_on) break;
    }

    if (cep->phy_type) {
        /*
         * It would be really nice if this information was in a
         * standard register........
         * If we don't know the type of phy, just report 10 Mbits, half duplex.
         */
        cep->phy_duplex = 0;
        cep->phy_speed = 10;

        if ((cep->phy_type & 0xfffffff0) == PHY_ID_LXT972) {
            /* Intel LXT972A */
            printf("Found LXT972A PHY @ 0x%x\n", mii_priv[phy_id].phy_address);
            cep->phy_ident = PHY_ID_LXT972;

            miiphy_read(phyaddr, LXT972_SR2, &rv);
            PPDEBUG("Status Register 2 (17): %04x\n", rv);
            if (rv & LXT972_SR2_SPEED)
                cep->phy_speed = 100;
            else
                cep->phy_speed = 10;

            if (rv & LXT972_SR2_DM)
                cep->phy_duplex = 1;
        }
        else if ((cep->phy_type & 0xfffffc00) == PHY_ID_DM9161) {
            /* Davicom DM9161 */
            printf("Found DM9161 PHY @ 0x%x\n", mii_priv[phy_id].phy_address);
            cep->phy_ident = PHY_ID_DM9161;
        }
        PPDEBUG("phy_address = 0x%x phy_type = 0x%x phy_duplex = %d "
              "phy_speed = %d\n", cep->phy_address, cep->phy_type,
              cep->phy_duplex, cep->phy_speed);

        if (mii_phy_get_configuration(&auton, &linkopt) == 0) {
            mii_phy_print_configuration(auton, linkopt);
            mii_priv[phy_id].auto_neg = auton;
            mii_priv[phy_id].link_options = linkopt;
        }
        else {
            /* guess for now */
            mii_priv[phy_id].auto_neg = auton;
            mii_priv[phy_id].link_options = LINK_ALL;
        }

        if( cep->phy_ident == PHY_ID_DM9161 ) {
            /* continue getting status */
            if( mii_priv[phy_id].auto_neg ) {
                /* get status from DSCSR */
                miiphy_read(phyaddr, DM9161_DSCSR, &rv);
                PPDEBUG("DSCSR: %04x\n", rv);
                if (rv & DM9161_100FDX || rv & DM9161_100HDX)
                    cep->phy_speed = 100;
                else
                    cep->phy_speed = 10;

                if (rv & DM9161_100FDX || rv & DM9161_10FDX)
                    cep->phy_duplex = 1;
                else
                    cep->phy_duplex = 0;
            }
            else {
                miiphy_read(phyaddr, PHY_BMCR, &rv);
                PPDEBUG("BMCR: %04x\n", rv);
                if (rv & PHY_BMCR_100MB)
                    cep->phy_speed = 100;
                else
                    cep->phy_speed = 10;

                if (rv & PHY_BMCR_DPLX)
                    cep->phy_duplex = 1;
                else
                    cep->phy_duplex = 0;
            }
        }
    } /* if (cep->phy_type) */

    return found_phy;
}

void
mii_discover_phy(void)
{
    /* defined in cpu/ppc4xx/405gp_enet.c */
    //extern void eth_set_phy_addr(unsigned char addr);

    if (mii_phy_init == 0) {
        mii_discover_phy_poll(&mii_priv[phy_id]);
        //eth_set_phy_addr(phyaddr);
        mii_phy_init = 1;
    }

    return;
}

int
mii_check_phy(void)
{
    switch( mii_priv[phy_id].phy_ident ) {
        case PHY_ID_LXT972:
        case PHY_ID_DM9161:
            return 0;

        default:
            printf("Unknown PHY device id %08x\n", mii_priv[phy_id].phy_type);
            return -1;
    }
}

int
mii_get_phy(void)
{
    switch( mii_priv[phy_id].phy_ident ) {
        case PHY_ID_LXT972:
        case PHY_ID_DM9161:
            break;

        default:
            return 0;
    }

    return( mii_priv[phy_id].phy_ident );
}

void
mii_sw_reset(void)
{
    /* software reset bit 0.15 */
    unsigned short cr;

    if (mii_check_phy() < 0) return;

    miiphy_read(phyaddr, PHY_BMCR, &cr);
    cr |= PHY_BMCR_RESET;
    miiphy_write(phyaddr, PHY_BMCR, cr);

    /* don't relinquish control until reset is done */
    do {
        udelay(1);
        miiphy_read(phyaddr, PHY_BMCR, &cr);
    } while (cr & PHY_BMCR_RESET);

    PPDEBUG("autoneg=%d link_options=0x%x\n", mii_priv[phy_id].auto_neg,
          mii_priv[phy_id].link_options);
    if (mii_phy_set_configuration(mii_priv[phy_id].auto_neg,
                                  mii_priv[phy_id].link_options) < 0) {
        printf("Failed to configure!\n");
    }

    return;
}

void
mii_read_all_regs(void)
{
    switch( mii_priv[phy_id].phy_ident ) {
        case PHY_ID_LXT972:
            printf("\nLXT972A Ethernet PHY @ 0x%x\n", phyaddr);
            break;

        case PHY_ID_DM9161:
            printf("\nDM9161 Ethernet PHY @ 0x%x\n", phyaddr);
            break;

        default:
            printf("Unknown PHY device id %08x\n", mii_priv[phy_id].phy_type);
            return;
    }

    mii_phy_read_all_regs();

    return;
}

void
mii_get_status(void)
{
    switch( mii_priv[phy_id].phy_ident ) {
        case PHY_ID_LXT972:
            printf("\nLXT972A Ethernet PHY @ 0x%x\n", phyaddr);
            break;

        case PHY_ID_DM9161:
            printf("\nDM9161 Ethernet PHY @ 0x%x\n", phyaddr);
            break;

        default:
            printf("Unknown PHY device id %08x\n", mii_priv[phy_id].phy_type);
            return;
    }

    mii_phy_get_status();

    return;
}

int
mii_phy_set_configuration(int auto_neg, int link_options)
{
    int result = 0;
    unsigned short anar, cfg = 0; /* auto-neg advertisement register */
    unsigned short cr;            /* control register */
    unsigned char speed_100m = -1;
    unsigned char full_duplex = -1;

    if (mii_check_phy() < 0) return -1;

    if (auto_neg) {
        /* enable auto-negotiation */

        switch (link_options) {
            case LINK_ALL:
                /* 10/100 Mbs Full or Half duplex */
                cfg = PHY_ANAR_TXFD | PHY_ANAR_TX |PHY_ANAR_10FD | PHY_ANAR_10;
                speed_100m = 1;
                full_duplex = 1;
                break;
            case LINK_10_100HD:
                /* 10/100 Mbs Half-duplex only */
                cfg = PHY_ANAR_TX | PHY_ANAR_10;
                speed_100m = 1;
                full_duplex = 0;
                break;
            case LINK_100FD:
                /* 100 Mbs only Full-duplex */
                cfg = PHY_ANAR_TXFD | PHY_ANAR_TX; 
                speed_100m = 1;
                full_duplex = 1;
                break;
            case LINK_100HD:
                /* 100 Mbs only Half-duplex */
                cfg = PHY_ANAR_TX;
                speed_100m = 1;
                full_duplex = 0;
                break;
            case LINK_10FD: /* should this be invalid for lxt972a */
                /* 10 Mbs only Full-duplex */
                cfg = PHY_ANAR_10FD | PHY_ANAR_10;
                speed_100m = 0;
                full_duplex = 1;
                break;
            case LINK_10HD: /* should this be invalid for lxt972a */
                /* 10 Mbs only Half-duplex */
                cfg = PHY_ANAR_10;
                speed_100m = 0;
                full_duplex = 0;
                break;
            default:
                printf("Invalid link advertisement for auto-neg config\n");
                result = -1;
                break;
        }
    }
    else {
        /* disable auto-negotiation */

        switch (link_options) {
            case LINK_100FD:
                /* 100 Mbs only Full-duplex */
                cfg = PHY_ANAR_TXFD | PHY_ANAR_TX; 
                speed_100m = 1;
                full_duplex = 1;
                break;
            case LINK_100HD:
                /* 100 Mbs only Half-duplex */
                cfg = PHY_ANAR_TX;
                speed_100m = 1;
                full_duplex = 0;
                break;
            case LINK_10FD:
                /* 10 Mbs only Full-duplex */
                cfg = PHY_ANAR_10FD | PHY_ANAR_10;
                speed_100m = 0;
                full_duplex = 1;
                break;
            case LINK_10HD:
                /* 10 Mbs only Half-duplex */
                cfg = PHY_ANAR_10;
                speed_100m = 0;
                full_duplex = 0;
                break;
            default:
                printf("Invalid link advertisement for non auto-neg config\n");
                result = -1;
                break;
        }
    }

    if (result == 0) {
        if (auto_neg) {
            /* update auto-neg advertisement register */
            anar = mii_phy_read(PHY_ANAR);
            anar &= ~(PHY_ANAR_TXFD | PHY_ANAR_TX |PHY_ANAR_10FD |PHY_ANAR_10);
            anar |= cfg;
            mii_phy_write(PHY_ANAR, anar);
        }

        /* update control register */
        cr = mii_phy_read(PHY_BMCR);
        cr &= ~(PHY_BMCR_AUTON | PHY_BMCR_100MB |PHY_BMCR_SS6 | PHY_BMCR_DPLX);
        if (auto_neg) {
            cr |= (PHY_BMCR_AUTON | PHY_BMCR_RST_NEG);
        }
        if (speed_100m == 1) {
            cr |= PHY_BMCR_100MB;
        }
        if (full_duplex == 1) {
            cr |= PHY_BMCR_DPLX;
        }
        mii_phy_write(PHY_BMCR, cr);

        mii_priv[phy_id].auto_neg = auto_neg;
        mii_priv[phy_id].link_options = link_options;
    }

    return result;
}

int
mii_phy_get_configuration(int *auto_neg, int *link_options)
{
    /*
     * Get configuration directly from the registers
     */
    int result = 0;
    unsigned short anar; /* auto-neg advertisement register */
    unsigned short cr;   /* control register */
    unsigned char speed_100m = 0;
    unsigned char full_duplex = 0;
    int warning = 0;

    if (mii_check_phy() < 0) return -1;
    if (!auto_neg || !link_options) return -1;

    /* check first if config for auto-negotiation */
    cr = mii_phy_read(PHY_BMCR);
    if (cr & PHY_BMCR_AUTON) {
        *auto_neg = 1;
    }
    else {
        *auto_neg = 0;
    }

    /* full or half duplex */
    if (cr & PHY_BMCR_DPLX) {
        full_duplex = 1;
    }

    /* 10/100 Mbs */
    if (cr & PHY_BMCR_100MB) {
        speed_100m = 1;
    }

    /* check for the link options */
    anar = mii_phy_read(PHY_ANAR);

    /* mask out unnecessary bits */
    anar &= PHY_ANAR_TXFD | PHY_ANAR_TX | PHY_ANAR_10FD | PHY_ANAR_10;
    switch (anar) {
        case (PHY_ANAR_TXFD | PHY_ANAR_TX | PHY_ANAR_10FD | PHY_ANAR_10):
            /* 10/100 Mbs Full or Half duplex */
            *link_options = LINK_ALL;
            if (!speed_100m || !full_duplex) {
                /* give warning for incorrect configuration */
                printf("Configured for 10/100 Mbs Full or Half duplex but "
                       "not set for");
                if (!speed_100m) {
                    printf(" 100 Mbs");
                    warning = 1;
                }
                if (!full_duplex) {
                    printf("%s full duplex!\n", warning?" and":"");
                }
                else {
                    printf("!\n");
                }
            }
            break;
        case (PHY_ANAR_TX | PHY_ANAR_10):
            /* 10/100 Mbs Half-duplex only */
            *link_options = LINK_10_100HD;
            if (full_duplex || !speed_100m) {
                /* give warning for incorrect configuration */
                printf("Configured for 10/100 Mbs Half-duplex only but");
                if (!speed_100m) {
                    printf(" not set for 100 Mbs");
                    warning = 1;
                }
                if (full_duplex) {
                    printf("%s set for full duplex!\n", warning?" and":"");
                }
                else {
                    printf("!\n");
                }
            }
            break;
        case (PHY_ANAR_TXFD | PHY_ANAR_TX):
            /* 100 Mbs only Full-duplex */
            *link_options = LINK_100FD;
            if (!full_duplex || !speed_100m) {
                /* give warning for incorrect configuration */
                printf("Configured for 100 Mbs Full-duplex only but not "
                       "set for");
                if (!speed_100m) {
                    printf(" 100 Mbs");
                    warning = 1;
                }
                if (!full_duplex) {
                    printf("%s full duplex!\n", warning?" and":"");
                }
                else {
                    printf("!\n");
                }
            }
            break;
        case PHY_ANAR_TX:
            /* 100 Mbs only Half-duplex */
            *link_options = LINK_100HD;
            if (full_duplex || !speed_100m) {
                /* give warning for incorrect configuration */
                printf("Configured for 100 Mbs Half-duplex only but");
                if (!speed_100m) {
                    printf(" not set for 100 Mbs");
                    warning = 1;
                }
                if (full_duplex) {
                    printf("%s set for full duplex!\n", warning?" and":"");
                }
                else {
                    printf("!\n");
                }
            }
            break;
        case (PHY_ANAR_10FD | PHY_ANAR_10): /* is this invalid for lxt972a? */
            /* 10 Mbs only Full-duplex */
            *link_options = LINK_10FD;
            if (!full_duplex || speed_100m) {
                /* give warning for incorrect configuration */
                printf("Configured for 10 Mbs Full-duplex only but");
                if (speed_100m) {
                    printf(" set for 100 Mbs");
                    warning = 1;
                }
                if (!full_duplex) {
                    printf("%s not set for full duplex!\n", warning?" and":"");
                }
                else {
                    printf("!\n");
                }
            }
            break;
        case PHY_ANAR_10:                   /* is this invalid for lxt972a? */
            /* 10 Mbs only Half-duplex */
            *link_options = LINK_10HD;
            if (full_duplex || speed_100m) {
                /* give warning for incorrect configuration */
                printf("Configured for 10 Mbs Half-duplex only but set for");
                if (speed_100m) {
                    printf(" 100 Mbs");
                    warning = 1;
                }
                if (full_duplex) {
                    printf("%s full duplex!\n", warning?" and":"");
                }
                else {
                    printf("!\n");
                }
            }
            break;
        default:
            printf("ERROR: Invalid MII PHY configuration! (ANAR=%04x)\n", anar);
            *link_options = 0;
            result = -2;
    }

    return result;
}

void 
mii_phy_print_configuration(int auto_neg, int link_options)
{
    if (auto_neg) {
        printf("Auto-negotiation with ");
    }
    else {
        printf("No auto-negotiation with ");
    }

    switch (link_options) {
        case LINK_ALL:
            printf("10/100 Mbs Full or Half duplex\n");
            break;
        case LINK_10_100HD:
            printf("10/100 Mbs Half-duplex only\n");
            break;
        case LINK_100FD:
            printf("100 Mbs Full-duplex\n");
            break;
        case LINK_100HD:
            printf("100 Mbs Half-duplex only\n");
            break;
        case LINK_10FD: /* should this be invalid for lxt972a */
            printf("10 Mbs Full-duplex\n");
            break;
        case LINK_10HD: /* should this be invalid for lxt972a */
            printf("10 Mbs Half-duplex\n");
            break;
        default:
            printf("Invalid link advertisement for auto-neg config\n");
            break;
    }

    return;
}

int
mii_phy_config_led(ushort led, ushort cfg, ushort option)
{
    int res = 0;

    switch( mii_priv[phy_id].phy_ident ) {
        case PHY_ID_LXT972:
            switch (option) {
                case PHY_LED_OPTION1:
                    res = mii_phy_config_led_disp(led, cfg);
                    break;
                case PHY_LED_OPTION2:
                    res = mii_phy_config_led_freq(cfg);
                    break;
                case PHY_LED_OPTION3:
                    mii_phy_config_led_ps(cfg);
                    break;
                default:
                    printf("Invalid LED configuration option!\n");
                    res = -1;
            }
            break;

        case PHY_ID_DM9161:
            res = mii_phy_config_led_disp(led, cfg);
            break;

        default:
            printf("Unknown PHY device id %08x\n", mii_priv[phy_id].phy_type);
            res = -2;
            break;
    }

    return res;
}
