/*
 * (C) Copyright 2000
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */
/* ------------------------------------------------------------------------+ */
/*
 * This source code has been made available to you by IBM on an AS-IS
 * basis.  Anyone receiving this source is licensed under IBM
 * copyrights to use it in any way he or she deems fit, including
 * copying it, modifying it, compiling it, and redistributing it either
 * with or without modifications.  No license under IBM patents or
 * patent applications is to be implied by the copyright license.
 *
 * Any user of this software should understand that IBM cannot provide
 * technical support for this software and will not be responsible for
 * any consequences resulting from the use of this software.
 *
 * Any person who transfers this source code or any derivative work
 * must include the IBM copyright notice, this paragraph, and the
 * preceding two paragraphs in the transferred software.
 *
 * COPYRIGHT   I B M   CORPORATION 1995
 * LICENSED MATERIAL  -  PROGRAM PROPERTY OF I B M
 */
/* ------------------------------------------------------------------------- */

#include <common.h>
#include <commproc.h>
#include <asm/processor.h>
#include <watchdog.h>
#include <rs485.h>


#if defined(CONFIG_405EP) && defined(CONFIG_KX101)

#define UART0_BASE      0xef600300
#define UART1_BASE      0xef600400
#define UCR0_MASK       0x0000007f
#define UCR1_MASK       0x00007f00
#define UCR0_UDIV_POS   0
#define UCR1_UDIV_POS   8
#define UDIV_MAX        127

/* using serial port 1 as RS485 */
#define ACTING_UART_BASE    UART1_BASE

#define UART_RBR    0x00
#define UART_THR    0x00
#define UART_IER    0x01
#define UART_IIR    0x02
#define UART_FCR    0x02
#define UART_LCR    0x03
#define UART_MCR    0x04
#define UART_LSR    0x05
#define UART_MSR    0x06
#define UART_SCR    0x07
#define UART_DLL    0x00
#define UART_DLM    0x01

/*----------------------------------------------------------------------------+
  | Line Status Register.
  +--------------------------------------------------------------------------*/
/*#define asyncLSRport1           ACTING_UART_BASE+0x05 */
#define asyncLSRDataReady1            0x01
#define asyncLSROverrunError1         0x02
#define asyncLSRParityError1          0x04
#define asyncLSRFramingError1         0x08
#define asyncLSRBreakInterrupt1       0x10
#define asyncLSRTxHoldEmpty1          0x20
#define asyncLSRTxShiftEmpty1         0x40
#define asyncLSRRxFifoError1          0x80

/* define for loopback test if required to send multiple characters */
#undef RS485_LPBK_MULTIPLE_PATTERN

/* for injecting test errors */
#undef RS485_INJECT_DATA_ERROR

/* default transmit timeout time in case it fails */
#define RS485_TX_TIMEOUT              0x00100000

/* flag to really enable RS485 functionality by toggling GPIO7 */
#define ENABLE_RS485

int rs485_serial_init(void)
{
    DECLARE_GLOBAL_DATA_PTR;

    unsigned long reg;
    unsigned long tmp;
    unsigned long clk;
    unsigned long udiv;
    unsigned short bdiv;
    volatile char val;
    unsigned long baudrate = RS485_BAUD_RATE;
    static int init_rs485 = 0;

    if (init_rs485) {
        /* already initialized */
        return 0;
    }

    reg = mfdcr(cpc0_ucr) & ~(UCR0_MASK | UCR1_MASK);
    clk = gd->cpu_clk;
    tmp = CFG_BASE_BAUD * 16;
    udiv = (clk + tmp / 2) / tmp;
    if (udiv > UDIV_MAX)                    /* max. n bits for udiv */
        udiv = UDIV_MAX;
    reg |= (udiv) << UCR0_UDIV_POS;	        /* set the UART divisor */
    reg |= (udiv) << UCR1_UDIV_POS;	        /* set the UART divisor */
    mtdcr (cpc0_ucr, reg);

    tmp = baudrate * udiv * 16;
    bdiv = (clk + tmp / 2) / tmp;

    out8 (ACTING_UART_BASE + UART_LCR, 0x80);	/* set DLAB bit */
    out8 (ACTING_UART_BASE + UART_DLL, bdiv);	/* set baudrate divisor */
    out8 (ACTING_UART_BASE + UART_DLM, bdiv >> 8);/* set baudrate divisor */
    out8 (ACTING_UART_BASE + UART_LCR, 0x03);	/* clear DLAB; set 8 bits, no parity */
    out8 (ACTING_UART_BASE + UART_FCR, 0x00);	/* disable FIFO */
    out8 (ACTING_UART_BASE + UART_MCR, 0x00);	/* no modem control DTR RTS */
    val = in8 (ACTING_UART_BASE + UART_LSR);	/* clear line status */
    val = in8 (ACTING_UART_BASE + UART_RBR);	/* read receive buffer */
    out8 (ACTING_UART_BASE + UART_SCR, 0x00);	/* set scratchpad */
    out8 (ACTING_UART_BASE + UART_IER, 0x00);	/* set interrupt enable reg */

    init_rs485 = 1;

    return (0);
}


void rs485_serial_setbrg(void)
{
    DECLARE_GLOBAL_DATA_PTR;

    unsigned long tmp;
    unsigned long clk;
    unsigned long udiv;
    unsigned short bdiv;

    clk = gd->cpu_clk;

    udiv = ((mfdcr (cpc0_ucr) & UCR0_MASK) >> UCR0_UDIV_POS);
    tmp = gd->baudrate * udiv * 16;
    bdiv = (clk + tmp / 2) / tmp;

    out8 (ACTING_UART_BASE + UART_LCR, 0x80);	/* set DLAB bit */
    out8 (ACTING_UART_BASE + UART_DLL, bdiv);	/* set baudrate divisor */
    out8 (ACTING_UART_BASE + UART_DLM, bdiv >> 8);/* set baudrate divisor */
    out8 (ACTING_UART_BASE + UART_LCR, 0x03);	/* clear DLAB; set 8 bits, no parity */
}


int rs485_serial_putc(const char c)
{
    int i, result = 0;
    char lsr;
    unsigned long timeout = 0x00100000;
    unsigned long counter = 0;
#ifdef ENABLE_RS485
    u32 gpio0_or;
#endif

    /* check THRE bit, wait for transmiter available */
    for (i = 1; i < 3500; i++) {
        if ((in8 (ACTING_UART_BASE + UART_LSR) & 0x20) == 0x20)
            break;
        udelay (100);
    }

#ifdef ENABLE_RS485
    /* to transmit, pull GPIO7 high */
    gpio0_or = in32(GPIO0_OR);
    gpio0_or |= RS485_GPIO7;
    out32(GPIO0_OR, gpio0_or);
#endif

    /* put character out */
    out8 (ACTING_UART_BASE + UART_THR, c);

    /* wait to pull GPIO7 low until the byte is sent */
    do {
        lsr = in8(ACTING_UART_BASE + UART_LSR);

        /* check for timeout */
        if (timeout != RS485_TIMEOUT_NOLIMIT) {
            if (counter++ >= timeout) {
                result = -1;
                break;
            }
        }

        /* check if user wants to halt */
        if (ctrlc()) {
            result = -2;
            break;
        }
    } while((lsr & 0x40) != 0x40);

#ifdef ENABLE_RS485
    udelay(1);

    /* to end transmission, pull GPIO7 low */
    gpio0_or = in32(GPIO0_OR);
    gpio0_or &= ~RS485_GPIO7;
    out32(GPIO0_OR, gpio0_or);
#endif

    return result;
}


void rs485_serial_puts (const char *s)
{
    while (*s) {
        serial_putc (*s++);
    }
}


int rs485_serial_getc(char *byte, unsigned long timeout)
{
    unsigned char status = 0;
    unsigned long counter = 0;
#ifdef ENABLE_RS485
    u32 gpio0_or;

    /* to receive, pull GPIO7 low */
    gpio0_or = in32(GPIO0_OR);
    gpio0_or &= ~RS485_GPIO7;
    out32(GPIO0_OR, gpio0_or);
#endif

    while (1) {
#if defined(CONFIG_HW_WATCHDOG)
        WATCHDOG_RESET ();	/* Reset HW Watchdog, if needed */
#endif	/* CONFIG_HW_WATCHDOG */
        status = in8 (ACTING_UART_BASE + UART_LSR);
        if ((status & asyncLSRDataReady1) != 0x0) {
            break;
        }
        if ((status & ( asyncLSRFramingError1 |
                        asyncLSROverrunError1 |
                        asyncLSRParityError1  |
                        asyncLSRBreakInterrupt1 )) != 0) {
            out8 (ACTING_UART_BASE + UART_LSR,
                  asyncLSRFramingError1 |
                  asyncLSROverrunError1 |
                  asyncLSRParityError1  |
                  asyncLSRBreakInterrupt1);
        }

        /* check for timeout */
        if (timeout != RS485_TIMEOUT_NOLIMIT) {
            if (counter++ >= timeout) {
                return -1;
            }
        }

        /* check if user wants to halt */
        if (ctrlc()) {
            return -2;
        }
    }

    *byte = in8(ACTING_UART_BASE);

    return 0;
}


int rs485_serial_tstc(void)
{
    unsigned char status;

    status = in8 (ACTING_UART_BASE + UART_LSR);
    if ((status & asyncLSRDataReady1) != 0x0) {
        return (1);
    }
    if ((status & ( asyncLSRFramingError1 |
                    asyncLSROverrunError1 |
                    asyncLSRParityError1  |
                    asyncLSRBreakInterrupt1 )) != 0) {
        out8 (ACTING_UART_BASE + UART_LSR,
              asyncLSRFramingError1 |
              asyncLSROverrunError1 |
              asyncLSRParityError1  |
              asyncLSRBreakInterrupt1);

    }

    return 0;
}

int rs485_loopback_test(void)
{
#ifdef RS485_LPBK_MULTIPLE_PATTERN
    char test_pattern[] = {
        0x00, 0x11, 0x22, 0x33,
        0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb,
        0xcc, 0xdd, 0xee, 0xff,
        0xa5, 0x5a, 0xf0, 0x0f,
        0x01, 0x23, 0x45, 0x67,
        0x89, 0xab, 0xcd, 0xef
    };
#else
    char test_pattern[] = {0x55};
#endif
    int idx;
    char rxbyte;
    int err;

    for( idx = 0; idx < sizeof(test_pattern)/sizeof(test_pattern[0]); idx++ ) {
        /* send */
        err = rs485_serial_putc(test_pattern[idx]);
        if (err < 0) {
            if (err == -1) {
                printf("ERROR\n\n");
                printf("Timed out waiting for byte 0x%x to send!\n",
                       test_pattern[idx]);
                return -4;
            }
            else if (err == -2) {
                printf("Test Halted\n");
                return -3;
            }
        }

        /* receive */
#ifdef RS485_INJECT_DATA_ERROR 
        err = 0;
        if (idx == 16)
            rxbyte = ~test_pattern[idx];
        else
            rxbyte = test_pattern[idx];
#else
        err = rs485_serial_getc(&rxbyte, 0x00100000);
#endif
        if (err == 0) {
            if (rxbyte != test_pattern[idx]) {
                printf("ERROR\n\n");
                printf("Expected: 0x%x\tReceived: 0x%x\n",
                       test_pattern[idx], rxbyte);
                return -1;
            }
        }
        else if (err == -1) {
            printf("ERROR\n\n");
            printf("Timed out waiting for byte 0x%x to receive!\n",
                   test_pattern[idx]);
            return -2;
        }
        else if (err == -2) {
            printf("Test Halted\n");
            return -3;
        }
    }

    return 0;
}

#endif	/* CONFIG_405GP || CONFIG_405CR */
