/*****************************************************************************
 *  Utility menu to update bootloader, CONFIG, and JFFS2 images.
 *
 *  FILE: update_menu.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <common.h>
#include <board_type.h>
#include <image_update.h>
#include <update_menu.h>
#include <flash_partition.h>
#include <util.h>


static void print_update_menu( void )
{
    int i = 1;

    printf("\nImage Updates:");
    print_exit();
    printf("\n\t %d: Update Bootloader", i++);
#if defined(CFG_RARITAN_UPDATE_CONFIG)
    printf("\n\t %d: Update CONFIG1 Partition", i++);
    printf("\n\t %d: Update CONFIG2 Partition", i++);
#endif
#if defined(CFG_RARITAN_UPDATE_JFFS2)
    printf("\n\t %d: Update JFFS2 Partition", i++);
#endif
    printf("\n\t %d: Update Whole Flash", i++);
#if defined(CFG_RARITAN_UPDATE_KERNELINFO)
    printf("\n\t %d: Get Current Kernel Info", i++);
#endif
#if defined(CFG_RARITAN_UPDATE_RAMDISKINFO)
    printf("\n\t %d: Get Current Ramdisk Info", i++);
#endif
#if defined(CFG_RARITAN_UPDATE_JFFS2INFO)
    printf("\n\t %d: Get Current JFFS2 Info", i++);
#endif
    printf("\n\n");

    return;
}

int update_menu( int board_type )
{
    unsigned long diag_opt;
    char update_flag = 1;
    int  result = 1; /* result is used for reverting back to U-boot cmds */
    int  opt_res;
    char defname[DEFAULT_NAME_SIZE];
    int  start, end, size, bank;
    char ok_to_update;

    while (update_flag) {
        print_update_menu();

        opt_res = menu_get_user_input(&diag_opt);
        if( opt_res > 0 ) {
            return 0;
        }
        else if( opt_res < 0 ) {
            continue;
        }

        switch( diag_opt ) {
            case UPDATE_MENU_EXIT:
                update_flag = 0;
                break;

            case UPDATE_MENU_BOOT:
                memset(defname, 0, DEFAULT_NAME_SIZE);
                switch( board_type ) {
                    case RARITAN_BOARD_KX101:
                        strncpy(defname, "kx101.bin", 9);
                        bank = UBOOT_FLASH_BANK;
                        start = UBOOT_SECTOR_START;
                        end = UBOOT_SECTOR_END;
                        ok_to_update = 1;
                        break;
                    case RARITAN_BOARD_KXGEN2:
                        strncpy(defname, "u-boot-dkx.bin", 14);
                        bank = UBOOT_FLASH_BANK;
                        start = UBOOT_SECTOR_START;
                        end = UBOOT_SECTOR_END;
                        ok_to_update = 1;
                        break;
                    default:
                        ok_to_update = 0;
                        break;
                }

                if( ok_to_update ) {
                    boot_tftp_update(defname, bank, start, end);
                }
                break;

#if defined(CFG_RARITAN_UPDATE_CONFIG)
            case UPDATE_MENU_CONFIG1:
                config_tftp_update( 0 );
                break;

            case UPDATE_MENU_CONFIG2:
                config_tftp_update( 1 );
                break;
#endif

#if defined(CFG_RARITAN_UPDATE_JFFS2)
            case UPDATE_MENU_JFFS2:
                jffs2_tftp_update();
                break;
#endif

            case UPDATE_MENU_WHOLE_FLASH:
                memset(defname, 0, DEFAULT_NAME_SIZE);
                switch( board_type ) {
                    case RARITAN_BOARD_KX101:
                        strncpy(defname, "kx101rom.bin", 12);
                        start = UBOOT_SECTOR_START;
                        end = UBOOT_SECTOR_END;
                        size = PP_(FLASH_SIZE,PP_HWID);
                        ok_to_update = 1;
                        break;
                    case RARITAN_BOARD_KXGEN2:
                        strncpy(defname, "dkxrom.bin", 10);
                        start = UBOOT_SECTOR_START;
                        end = UBOOT_SECTOR_END;
                        size = PP_(FLASH_SIZE,PP_HWID);
                        ok_to_update = 1;
                        break;
                    default:
                        ok_to_update = 0;
                        break;
                }

                if( ok_to_update ) {
                    tftp_update_whole_flash(defname, size, start, end);
                }
                break;

#if defined(CFG_RARITAN_UPDATE_KERNELINFO)
            case UPDATE_MENU_KERNELINFO:
                get_kernel_image_info();
                break;
#endif

#if defined(CFG_RARITAN_UPDATE_RAMDISKINFO)
            case UPDATE_MENU_RAMDISKINFO:
                get_ramdisk_image_info();
                break;
#endif

#if defined(CFG_RARITAN_UPDATE_JFFS2INFO)
            case UPDATE_MENU_JFFS2INFO:
                get_jffs2_image_info();
                break;
#endif

            default:
                print_invalid();
                break;
        }
    }

    return result;
}
