/*
 * at93c46.c : U-Boot driver for the Atmel AT93C46
 *
 */

/**************************************************************************
*    at93c46.c: 
*
*    This program is free software; you can redistribute it and/or modify
*    it under the terms of the GNU General Public License as published by
*    the Free Software Foundation; either version 2 of the License, or
*    (at your option) any later version.
*
*    This program is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*    GNU General Public License for more details.
*
*    You should have received a copy of the GNU General Public License
*    along with this program; if not, write to the Free Software
*    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*    REVISION HISTORY:
*    ================
*
***************************************************************************/

#include <common.h>
#include <asm/io.h>

#if (CONFIG_COMMANDS & CFG_CMD_NET) && defined(CONFIG_NET_MULTI) && \
	defined(CONFIG_RTL8169)

/*#define DEBUG*/
#define GE_EEPROM_DELAY	250		/* 250 us for half clock cycle */

enum databits {
	EECS = 8,
	EESK = 4,
	EEDI = 2,
	EEDO = 1,
};

/* the following defines are based on 16 bits mode operation */
#define OP_9BITS_NR	9
#define DATA_16BITS_NR	16
#define OP_EWEN		0x130
#define OP_EWDS		0x100
#define OP_READ		0x180
#define OP_WRITE	0x140
#ifdef DEBUG
#define DBG(x...)	printf(x)
#else /* DEBUG */
#define DBG(x...)	do { } while(0)
#endif /* DEBUG */

/* write/read MMIO register */
extern void rtl_signal_eeprom( int cs_high, int sk_high, int di_high );
extern int rtl_rx_eeprom( void );
static int rx_stream( u32 pci_ioaddr, u32 length, u32 *bitstream_ptr, int END );
static int tx_stream( u32 pci_ioaddr, u32 length, u32 bitstream, int END );

u16 at93c46_read( u32 pci_ioaddr, u16 offset );
void at93c46_wr_enable( u32 pci_ioaddr );
void at93c46_wr_disable( u32 pci_ioaddr );
void at93c46_write( u32 pci_ioaddr, u16 offset, u16 data );



u16 at93c46_read( u32 pci_ioaddr, u16 offset )
{
    u32 length, readed, bitstream;
    u16 swapdata;
    
    length = OP_9BITS_NR;
    bitstream = OP_READ;
    bitstream += offset;
    tx_stream( pci_ioaddr, length, bitstream, 0 );
    rx_stream( pci_ioaddr, (u32)16, &readed, 1 );
    DBG("%s(): READED %04x\n", __FUNCTION__, readed);
    writew((u16)readed, &swapdata);
    DBG("%s() 0x%04x 0x%04x\n", __FUNCTION__, (u16)readed, swapdata);
    return (u16)swapdata;
}

static int tx_stream( u32 pci_ioaddr, u32 length, u32 bitstream, int END )
{
	u32 bitstream_idx, bitstream_mask;
	int index;
	unsigned char outbyte;

	if ( length > 32 ) return -1;
	if ( length == 0 ) return -1;

	bitstream_mask = 1<<length;
	bitstream_mask--;
	bitstream_idx = 1<<(length-1);
	
	DBG( "%s() length %d bitstream 0x%08x\n", __FUNCTION__, length, bitstream );
	bitstream &= bitstream_mask;
	DBG( "bitstream 0x%x bitstream_mask 0x%x bitstream_idx 0x%08x\n",
		bitstream, bitstream_mask, bitstream_idx );
	outbyte = EESK | EECS;
	for ( index = 0; index < 400; index++ ) {
	    outbyte ^= EESK;		/* clock */
	    if ( outbyte&EESK ) { /* no change for DI at raising edge */
		if ( outbyte & EECS ) {
		    DBG("[%d]: C%c d%c %02x\n", index, outbyte&EESK?'1':'0',
			    outbyte&EEDI?'1':'0', outbyte);
		}
		else {
		    printf("[%d]: ERROR missing CS %02x\n", index, outbyte);
		}
		rtl_signal_eeprom( outbyte&EECS, outbyte&EESK, outbyte&EEDI );
		udelay(GE_EEPROM_DELAY);
		continue;
	    }
	    if ( bitstream_idx ) {
		if ( bitstream & bitstream_idx )
		    outbyte |= EEDI;
		else
		    outbyte &= (~EEDI);
		bitstream_idx >>= 1;
	    }
	    else break;
	   
	    if ( outbyte & EECS ) {
		DBG("[%d]: c%c d%c %02x\n", index, outbyte&EESK?'1':'0',
			outbyte&EEDI?'1':'0', outbyte);
	    }
	    else {
		printf("[%d]: BAD %02x\n", index, outbyte );
	    }
	    rtl_signal_eeprom( outbyte&EECS, outbyte&EESK, outbyte&EEDI );
	    udelay(GE_EEPROM_DELAY);
	}

	if ( END ) { /* Tx ~EECS for a full clock cycle */
	    outbyte &= (~EECS);
	    outbyte &= (~EEDI);
	    rtl_signal_eeprom( outbyte&EECS, outbyte&EESK, outbyte&EEDI );
	    udelay(GE_EEPROM_DELAY);
	    outbyte ^= EESK;
	    rtl_signal_eeprom( outbyte&EECS, outbyte&EESK, outbyte&EEDI );
	    udelay(GE_EEPROM_DELAY);
	}

	return 0;
}

static int rx_stream( u32 pci_ioaddr, u32 length, u32 *bitstream_ptr, int END )
{
	u32 bitstream_mask;
	u32 bitstream = 0;
	int index;
	unsigned char outbyte;

	if ( length > 32 ) return -1;
	if ( length == 0 ) return -1;

	bitstream_mask = 1<<length;
	bitstream_mask--;
	
	DBG( "%s() length %d bitstream_mask 0x%08x\n",
	    __FUNCTION__, length, bitstream_mask );
	outbyte = EESK | EECS;
	for ( index = 0; index < 400; index++ ) {
	    outbyte ^= EESK;
	    if ( outbyte&EESK ) { /* read DO at raising edge */
		if ( outbyte & EECS ) {
		    DBG("[%d]: C%c d%c %02x\n", index, outbyte&EESK?'1':'0',
			    outbyte&EEDI?'1':'0', outbyte);
		}
		else {
		    printf("[%d]: ERROR missing CS %02x\n", index, outbyte);
		}
		rtl_signal_eeprom( outbyte&EECS, outbyte&EESK, outbyte&EEDI );
		if ( length ) {
		    bitstream <<= 1;
		    if ( rtl_rx_eeprom() ) bitstream |= 1;
		    length--;
		}
		udelay(GE_EEPROM_DELAY);
		continue;
	    }
	    if ( length ) { /* always drive DI as 0 in reading */
		outbyte &= (~EEDI);
	    }
	    else break;
	   
	    if ( outbyte & EECS ) {
		DBG("[%d]: c%c d%c %02x\n", index, outbyte&EESK?'1':'0',
			outbyte&EEDI?'1':'0', outbyte);
	    }
	    else {
		printf("[%d]: BAD %02x\n", index, outbyte );
	    }
	    rtl_signal_eeprom( outbyte&EECS, outbyte&EESK, outbyte&EEDI );
	    udelay(GE_EEPROM_DELAY);
	}
	bitstream &= bitstream_mask;
	DBG( "%s() bitstream 0x%08x\n", __FUNCTION__, bitstream );
	*bitstream_ptr = bitstream;

	if ( END ) { /* Tx ~EECS for a full clock cycle */
	    outbyte &= (~EECS);
	    outbyte &= (~EEDI);
	    rtl_signal_eeprom( outbyte&EECS, outbyte&EESK, outbyte&EEDI );
	    udelay(GE_EEPROM_DELAY);
	    outbyte ^= EESK;
	    rtl_signal_eeprom( outbyte&EECS, outbyte&EESK, outbyte&EEDI );
	    udelay(GE_EEPROM_DELAY);
	}

	return 0;
}

void at93c46_wr_enable( u32 pci_ioaddr )
{
	u32 length, opcode;

	length = OP_9BITS_NR;
	opcode = OP_EWEN;
	tx_stream( pci_ioaddr, length, opcode, 1 );
	DBG( "%s(): EWEN DONE length %d opcode %04x\n",
		__FUNCTION__, length, opcode );
	return;
}

void at93c46_wr_disable( u32 pci_ioaddr )
{
	u32 length, opcode;

	length = OP_9BITS_NR;
	opcode = OP_EWDS;
	tx_stream( pci_ioaddr, length, opcode, 1 );
	DBG( "%s(): EWDS DONE length %d opcode %04x\n",
		__FUNCTION__, length, opcode );
	return;
}

void at93c46_write( u32 pci_ioaddr, u16 offset, u16 data )
{
	u32 length, bitstream;
	u16 swapdata;

	length = OP_9BITS_NR+DATA_16BITS_NR;
	bitstream = OP_WRITE;
	bitstream |= offset;
	bitstream <<= DATA_16BITS_NR;
	writew(data, &swapdata);
	DBG("%s() 0x%04x 0x%04x\n", __FUNCTION__, data, swapdata);
	bitstream |= swapdata;
	tx_stream( pci_ioaddr, length, bitstream, 1 );
	DBG( "%s(): WRITE DONE length %d bitstream %04x\n",
		__FUNCTION__, length, bitstream );
	return;
}



#undef DEBUG
#undef OP_9BITS_NR
#undef DATA_16BITS_NR
#undef OP_EWEN
#undef OP_EWDS
#undef GE_EEPROM_DELAY
#endif
