/*------------------------------------------------------------------------
 . smc91111.c
 . This is a driver for SMSC's 91C111 single-chip Ethernet device.
 .
 . (C) Copyright 2002
 . Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 . Rolf Offermanns <rof@sysgo.de>
 .
 . Copyright (C) 2001 Standard Microsystems Corporation (SMSC)
 .       Developed by Simple Network Magic Corporation (SNMC)
 . Copyright (C) 1996 by Erik Stahlman (ES)
 .
 . This program is free software; you can redistribute it and/or modify
 . it under the terms of the GNU General Public License as published by
 . the Free Software Foundation; either version 2 of the License, or
 . (at your option) any later version.
 .
 . This program is distributed in the hope that it will be useful,
 . but WITHOUT ANY WARRANTY; without even the implied warranty of
 . MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 . GNU General Public License for more details.
 .
 . You should have received a copy of the GNU General Public License
 . along with this program; if not, write to the Free Software
 . Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 .
 . Information contained in this file was obtained from the LAN91C111
 . manual from SMC.  To get a copy, if you really want one, you can find 
 . information under www.smsc.com.
 . 
 .
 . "Features" of the SMC chip:
 .   Integrated PHY/MAC for 10/100BaseT Operation
 .   Supports internal and external MII
 .   Integrated 8K packet memory
 .   EEPROM interface for configuration
 .
 . Arguments:
 . 	io	= for the base address
 .	irq	= for the IRQ
 .
 . author:
 . 	Erik Stahlman				( erik@vt.edu )
 . 	Daris A Nevil				( dnevil@snmc.com )
 .
 .
 . Hardware multicast code from Peter Cammaert ( pc@denkart.be )
 .
 . Sources:
 .    o   SMSC LAN91C111 databook (www.smsc.com)
 .    o   smc9194.c by Erik Stahlman
 .    o   skeleton.c by Donald Becker ( becker@cesdis.gsfc.nasa.gov )
 .
 . History:
 .	10/17/01  Marco Hasewinkel Modify for DNP/1110
 .	07/25/01  Woojung Huh      Modify for ADS Bitsy
 .	04/25/01  Daris A Nevil    Initial public release through SMSC
 .	03/16/01  Daris A Nevil    Modified smc9194.c for use with LAN91C111
 ----------------------------------------------------------------------------*/

#include "kira/porting.h"
#include <asm/u-boot.h>
#include "command.h"
#include "fmac.h"
#include "net.h"


#ifdef CONFIG_DRIVER_FMAC

// Use power-down feature of the chip
#define POWER_DOWN	0


static const char version[] =
	"Faraday FTMAC100 Driver, (Linux Kernel 2.4) 10/18/02 - by Faraday\n";






#define inl(addr) 			(*((volatile u32 *)(addr)))
#define inw(addr)			(*((volatile u16 *)(addr)))
#define outl(value, addr)  	(*((volatile u32 *)(addr)) = value)
#define outb(value, addr)	(*((volatile u8 *)(addr)) = value)


u32 RXADR_REG[RX_REG_NUM] = {RXADR0_REG, RXADR1_REG};
u32 RCSR_REG[RX_REG_NUM] = {RCSR0_REG, RCSR1_REG};
u32 RBUF_PTR[RX_REG_NUM] = {RBUF0_PTR, RBUF1_PTR};

u32	TXADR_REG[TX_REG_NUM] = {TXADR0_REG, TXADR1_REG, TXADR2_REG, TXADR3_REG};
u32 TCSR_REG[TX_REG_NUM] = {TCSR0_REG, TCSR1_REG, TCSR2_REG, TCSR3_REG};

u32 MULTIMS_REG[MULTI_REG_NUM] = {MULTI0MS_REG, MULTI1MS_REG, MULTI2MS_REG, MULTI3MS_REG, 
							  MULTI4MS_REG, MULTI5MS_REG, MULTI6MS_REG, MULTI7MS_REG};
u32 MULTILS_REG[MULTI_REG_NUM] = {MULTI0LS_REG, MULTI1LS_REG, MULTI2LS_REG, MULTI3LS_REG, 
							  MULTI4LS_REG, MULTI5LS_REG, MULTI6LS_REG, MULTI7LS_REG};


struct net_device dev_eth0;
int tx_rx_cnt = 0;

//#define FMAC_DEBUG 	3

/*------------------------------------------------------------------------
 .
 . Configuration options, for the experienced user to change.
 .
 -------------------------------------------------------------------------*/

/*
 . Wait time for memory to be free.  This probably shouldn't be
 . tuned that much, as waiting for this means nothing else happens
 . in the system
*/
#define MEMORY_WAIT_TIME 16

#if (FMAC_DEBUG > 2 )
#define PRINTK3(args...) printf(args)
#else
#define PRINTK3(args...)
#endif

#if FMAC_DEBUG > 1
#define PRINTK2(args...) printf(args)
#else
#define PRINTK2(args...)
#endif

#ifdef FMAC_DEBUG
#define PRINTK(args...) printf(args)
#else
#define PRINTK(args...)
#endif

#define DO_PRINT(args...) printf(args)


/*------------------------------------------------------------------------
 .
 . The internal workings of the driver.  If you are changing anything
 . here with the SMC stuff, you should have the datasheet and know
 . what you are doing.
 .
 -------------------------------------------------------------------------*/
#define CARDNAME "FMAC_AHB"

// Memory sizing constant
#define LAN91C111_MEMORY_MULTIPLIER	(1024*2)

#ifndef CONFIG_SMC91111_BASE
#define CONFIG_SMC91111_BASE 0x20000300
#endif

#define SMC_BASE_ADDRESS CONFIG_SMC91111_BASE 


#define ETH_ZLEN 60

#ifdef  CONFIG_SMC_USE_32_BIT
#define USE_32_BIT  
#else
#undef USE_32_BIT 
#endif
/*-----------------------------------------------------------------
 .
 .  The driver can be entered at any of the following entry points.
 .
 .------------------------------------------------------------------  */

extern int eth_init(bd_t *bd);
extern void eth_halt(void);
extern int eth_rx(void);
extern int eth_send(volatile void *packet, int length);


int initialized = 0;



/*
 . This is called by  register_netdev().  It is responsible for
 . checking the portlist for the FMAC series chipset.  If it finds
 . one, then it will initialize the device, find the hardware information,
 . and sets up the appropriate device parameters.
 . NOTE: Interrupts are *OFF* when this procedure is called.
 .
 . NB:This shouldn't be static since it is referred to externally.
*/
int fmac_init(struct net_device *dev);

/*
 . This is called by  unregister_netdev().  It is responsible for
 . cleaning up before the driver is finally unregistered and discarded.
*/
void fmac_destructor(struct net_device *dev);

/*
 . The kernel calls this function when someone wants to use the net_device,
 . typically 'ifconfig ethX up'.
*/
static int fmac_open(struct net_device *dev);


/*
 . This is called by the kernel in response to 'ifconfig ethX down'.  It
 . is responsible for cleaning up everything that the open routine
 . does, and maybe putting the card into a powerdown state.
*/
static int fmac_close(struct net_device *dev);


/*
 . This is a separate procedure to handle the receipt of a packet, to
 . leave the interrupt code looking slightly cleaner
*/
inline static int fmac_rcv( struct net_device *dev );



/*
 ------------------------------------------------------------
 .
 . Internal routines
 .
 ------------------------------------------------------------
*/



/*
 . A rather simple routine to print out a packet for debugging purposes.
*/
#if FMAC_DEBUG > 2
static void print_packet( byte *, int );
#endif




/* this does a soft reset on the device */
static void fmac_reset( struct net_device *dev );

/* Enable Interrupts, Receive, and Transmit */
static void fmac_enable( struct net_device *dev );

/* this puts the device in an inactive state */
static void fmac_shutdown( unsigned int ioaddr );

/* Routines to Read and Write the PHY Registers across the
   MII Management Interface
*/


void put_mac(int base, char *mac_addr)
{
	int val;
	
	val = ((u32)mac_addr[0])<<8 | (u32)mac_addr[1];
	outl(val, base);
	val = ((((u32)mac_addr[2])<<24)&0xff000000) |
		  ((((u32)mac_addr[3])<<16)&0xff0000) |
		  ((((u32)mac_addr[4])<<8)&0xff00)  |
		  ((((u32)mac_addr[5])<<0)&0xff);
	outl(val, base+4);

}


/*
 . Function: fmac_reset( struct device* dev )
 . Purpose:
 .  	This sets the SMC91111 chip to its normal state, hopefully from whatever
 . 	mess that any other DOS driver has put it in.
 .
 . Maybe I should reset more registers to defaults in here?  SOFTRST  should
 . do that for me.
 .
 . Method:
 .	1.  send a SOFT RESET
 .	2.  wait for it to finish
 .	3.  enable autorelease mode
 .	4.  reset the memory management unit
 .	5.  clear all interrupts
 .
*/
static void fmac_reset( struct net_device *dev )
{
	struct fmac_local *lp 	= (struct fmac_local *)dev->priv;
	unsigned int	ioaddr = dev->base_addr;
	int i;

	PRINTK2("%s:fmac_reset\n", dev->name);

	outl( SW_RST_bit, ioaddr + MACCR_REG );

	/* this should pause enough for the chip to be happy */
	for (; (inl( ioaddr + MACCR_REG ) & SW_RST_bit) != 0; )
	{
		udelay(10);
		PRINTK3("RESET: reset not complete yet\n" );
	}
	

	outl( 0, ioaddr + IMR_REG );			/* Disable all interrupts */

	put_mac(ioaddr + MACMS_REG, dev->dev_addr);
	
	/// ]w receiving
	for (i=0; i<RX_REG_NUM; ++i)
	{
    	outl( lp->rxadr_dma[i], ioaddr + RXADR_REG[i]);
		lp->rxadr_ptr[i] = (int)lp->rxadr_cpu[i];
	}
    
    /// ]w transmitting
    for (i=0; i<TX_REG_NUM; ++i)
    {
    	outl( lp->txadr_dma[i], ioaddr + TXADR_REG[i] );
    }
    lp->rx_idx = 0;
    lp->tx_idx = 0;
	
}


/*
 . Function: fmac_enable
 . Purpose: let the chip talk to the outside work
 . Method:
 .	1.  Enable the transmitter
 .	2.  Enable the receiver
 .	3.  Enable interrupts
*/
static void fmac_enable( struct net_device *dev )
{
	unsigned int ioaddr 	= dev->base_addr;
	int i;
	struct fmac_local *lp 	= (struct fmac_local *)dev->priv;

	PRINTK2("%s:fmac_enable\n", dev->name);

	/// enable trans/recv,...
	outl(lp->maccr_val, ioaddr + MACCR_REG );
    
    for (i=0; i<RX_REG_NUM; ++i)
    {
    	outl( RX_BUF_SIZE | RBUF_EN, ioaddr + RCSR_REG[i]);
    }

	/* now, enable interrupts */
	outl( 
		RPKT_FINISH_bit |     	// R/C DMA receive packet successfully
        RPKT_ERR_bit |        	// R/C DMA receiving channel error
        XPKT_ERR_bit |        	// R/C DMA transmitting channel error
        XPKT_OK_bit |         	// R/C Transmitting packet successfully
        XPKT_LOST_bit |    		// R/C Transmitting packet is lost
        RPKT_SAV_bit |        	// R/C Receiving packet successfully
        RPKT_LOST_bit |     	// R/C Receiving packet is lost
        XHANG_bit |           	// R/C Transmitter is hang
		RBUFF_OUT0_bit |      	// R/C Allocated buffer0 is full
		RBUFF_OUT1_bit     	 	// R/C Allocated buffer1 is full
        ,ioaddr + IMR_REG
        );

}

/*
 . Function: fmac_shutdown
 . Purpose:  closes down the SMC91xxx chip.
 . Method:
 .	1. zero the interrupt mask
 .	2. clear the enable receive flag
 .	3. clear the enable xmit flags
 .
 . TODO:
 .   (1) maybe utilize power down mode.
 .	Why not yet?  Because while the chip will go into power down mode,
 .	the manual says that it will wake up in response to any I/O requests
 .	in the register space.   Empirical results do not show this working.
*/
static void fmac_shutdown( unsigned int ioaddr )
{
	/// ]w interrupt mask register
	outl( 0, ioaddr + IMR_REG );

	/// enable trans/recv,...
	outl( 0, ioaddr + MACCR_REG );
}



static int fmac_send_packet( void *packet, int length, struct net_device *dev )
{
	struct fmac_local *lp = (struct fmac_local *)dev->priv;
	unsigned int ioaddr = dev->base_addr;


	PRINTK("%s:fmac_hardware_send_packet\n", dev->name);

	for (; (inl(ioaddr + TCSR_REG[lp->tx_idx]) & XPKT_AVA_bit) != 0; )			/// SŪ transmit descriptor iHϥ
	{
		DO_PRINT("Transmitting busy\n");
		udelay(10);
	}
	length = ETH_ZLEN < length ? length : ETH_ZLEN;
	if (length > TX_BUF_SIZE)
	{
		DO_PRINT("%s: Far too big packet error. \n", dev->name);
		/// Bz
	}
	else
	{
   		PRINTK3("%s: Trying to xmit packet of length %x\n",	dev->name, length);
#if FMAC_DEBUG > 2
		DO_PRINT("Transmitting Packet\n");
		print_packet( packet, length );
#endif
	
		memcpy(lp->txadr_cpu[lp->tx_idx], packet, length);
		outl( length | XPKT_AVA_bit, ioaddr + TCSR_REG[lp->tx_idx] );
		lp->tx_idx = (lp->tx_idx + 1) % TX_REG_NUM;
	}
	

	PRINTK2("%s: Sent packet of length %d \n", dev->name, length);


	return length;
}

/*-------------------------------------------------------------------------
 |
 | smc_destructor( struct net_device * dev )
 |   Input parameters:
 |	dev, pointer to the device structure
 |
 |   Output:
 |	None.
 |
 ---------------------------------------------------------------------------
*/
void fmac_destructor(struct net_device *dev)
{
	PRINTK3("%s:fmac_destructor\n", dev->name);
}


/*
 * Open and Initialize the board
 *
 * Set up everything, reset the card, etc ..
 *
 */
static int fmac_open(struct net_device *dev)
{
	unsigned int	ioaddr = dev->base_addr;

	PRINTK2("%s:fmac_open\n", dev->name);

#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif

	/* reset the hardware */
	fmac_reset( dev );
	fmac_enable( dev );

	/* set the MAC address */
	put_mac(ioaddr + MACMS_REG, dev->dev_addr);

	return 0;
}

#ifdef USE_32_BIT
void
insl32(r,b,l) 	
{	
   int __i ;  
   dword *__b2;  

	__b2 = (dword *) b;  
	for (__i = 0; __i < l; __i++) 
   {  
		  *(__b2 + __i) = *(dword *)(r+0x10000300);  
	}  
}
#endif

/*-------------------------------------------------------------
 .
 . fmac_rcv -  receive a packet from the card
 .
 . There is ( at least ) a packet waiting to be read from
 . chip-memory.
 .
 . o Read the status
 . o If an error, record it
 . o otherwise, read in the packet
 --------------------------------------------------------------
*/
static int fmac_rcv(struct net_device *dev)
{
	struct fmac_local *lp = (struct fmac_local *)dev->priv;
	unsigned int  ioaddr = dev->base_addr;
	int 	buf_ptr;				/// ثe fmac w쪺m
	u8		*data_ptr;
	int 	packet_length;
	u32		status;
	int 	i;
	int rcv_cnt;

//	PRINTK3("%s:fmac_rcv\n", dev->name);
//	PRINTK("%s:fmac_rcv\n", dev->name);

///	for (i=0; i<RX_REG_NUM; ++i)
	{
		for (rcv_cnt=0; rcv_cnt<1; ++rcv_cnt)
		{
			buf_ptr = inl(ioaddr + RBUF_PTR[lp->rx_idx]);
//			PRINTK2("RCV: lp->rxadr_ptr[%d]=%x >= buf_ptr = %x\n", lp->rx_idx, lp->rxadr_ptr[lp->rx_idx], buf_ptr );
			
			if ( lp->rxadr_ptr[lp->rx_idx] >= buf_ptr)		///  j pX{
			{
				if (buf_ptr != 0 && lp->rxadr_ptr[lp->rx_idx]>buf_ptr)
				{
					DO_PRINT("RCV ptr strange: lp->rxadr_ptr[%d]=%x,  buf_ptr = %x\n", lp->rx_idx, lp->rxadr_ptr[lp->rx_idx], buf_ptr );
				}
				if ( (inl(ioaddr + RCSR_REG[lp->rx_idx]) & RBUF_EN) ==0)			// receive buffer become full ==> re-enable receive buffer
				{
					///DO_PRINT("buffer is going to full\n");
					lp->rxadr_ptr[lp->rx_idx] = lp->rxadr_dma[lp->rx_idx];
					outl( RX_BUF_SIZE | RBUF_EN, ioaddr + RCSR_REG[lp->rx_idx]);
					buf_ptr = inl(ioaddr + RBUF_PTR[lp->rx_idx]);
					///DO_PRINT("RCV adjusted value: lp->rxadr_ptr[%d]=%x,  buf_ptr = %x\n", lp->rx_idx, lp->rxadr_ptr[lp->rx_idx], buf_ptr );
					lp->rx_idx = (lp->rx_idx + 1) % RX_REG_NUM;
					continue;
				}
				break;
			}
			else 			// ( lp->rxadr_ptr[lp->rx_idx] < buf_ptr )
			{
				data_ptr = lp->rxadr_cpu[lp->rx_idx] + (lp->rxadr_ptr[lp->rx_idx] - lp->rxadr_dma[lp->rx_idx]);		
				packet_length = data_ptr[1] + (u32)data_ptr[0]*256;
				lp->rxadr_ptr[lp->rx_idx] += ((packet_length+7)&0xfffffffc);				/// round up to dword align
				status = *(u32 *)data_ptr;
			
				PRINTK2("RCV: STATUS %4x LENGTH 0x%x\n", status, packet_length );

				if (! (status & RS_ERRORS) )
				{
					memcpy( NetRxPackets[0], data_ptr+4, packet_length);
					NetReceive(NetRxPackets[0], packet_length);
#if	FMAC_DEBUG > 4
					DO_PRINT("Receiving Packet\n");
					print_packet( NetRxPackets[0], packet_length );
#endif
					return packet_length;
				}
				else
				{
					return 0;
				}
			}
		}
	}
	return 0;
}



/*----------------------------------------------------
 . fmac_close
 .
 . this makes the board clean up everything that it can
 . and not talk to the outside world.   Caused by
 . an 'ifconfig ethX down'
 .
 -----------------------------------------------------*/
static int fmac_close(struct net_device *dev)
{
	//netif_stop_queue(dev);
	//dev->start = 0;

	PRINTK2("%s:fmac_close\n", dev->name);


	/* clear everything */
	fmac_shutdown( dev->base_addr );

	/* Update the statistics here. */
#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif

	return 0;
}


//---PHY CONTROL AND CONFIGURATION-----------------------------------------



#if FMAC_DEBUG > 2
static void print_packet( byte * buf, int length )
{
#if 1
#if FMAC_DEBUG > 3
	int i;
	int remainder;
	int lines;
#endif

	DO_PRINT("Packet of length %d \n", length );

#if FMAC_DEBUG > 3
	lines = length / 16;
	remainder = length % 16;

	for ( i = 0; i < lines ; i ++ ) {
		int cur;

		for ( cur = 0; cur < 8; cur ++ ) {
			byte a, b;

			a = *(buf ++ );
			b = *(buf ++ );
			DO_PRINT("%02x%02x ", a, b );
		}
		DO_PRINT("\n");
	}
	for ( i = 0; i < remainder/2 ; i++ ) {
		byte a, b;

		a = *(buf ++ );
		b = *(buf ++ );
		DO_PRINT("%02x%02x ", a, b );
	}
	DO_PRINT("\n");
#endif
#endif
}
#endif


static char fmac_mac_addr[] = {0x05, 0x20, 0x10, 0x10, 0x10, 0x10}; 

int eth_init(bd_t *bd) 
{
	struct fmac_local *lp;
	int i;
	
	if (initialized == 0)
	{
		initialized = 1;
	
		dev_eth0.base_addr = CPE_FMAC_BASE;
		/* Initialize the private structure. */
		dev_eth0.priv = (void *)malloc(sizeof(struct fmac_local));
		if (dev_eth0.priv == NULL) 
		{
			DO_PRINT("out of memory\n");
			return 0;
		}


		/// --------------------------------------------------------------------
		///		l fmac_local
		/// --------------------------------------------------------------------
		memset(dev_eth0.priv, 0, sizeof(struct fmac_local));
		strcpy(dev_eth0.name, "eth0");
		lp = (struct fmac_local *)dev_eth0.priv;
		lp->maccr_val = FULLDUP_bit | CRC_APD_bit | MDC_SEL_bit | RCV_EN_bit | XMT_EN_bit  | RDMA_EN_bit	| XDMA_EN_bit ;
		for (i=0; i<RX_REG_NUM; ++i)
		{
			lp->rxadr_cpu[i] = lp->rxadr_dma[i] = (int)malloc(RX_BUF_SIZE);
		}

	    for (i=0; i<TX_REG_NUM; ++i)
    	{
    		lp->txadr_cpu[i] = lp->txadr_dma[i] = (int)malloc(TX_BUF_SIZE);
    	}
    	for (i=0; i<6; ++i)
    	{
    		dev_eth0.dev_addr[i] = fmac_mac_addr[i];
    	}    
	}
	fmac_open(&dev_eth0);
	
	return 0;
}

void eth_halt() 
{
	if (initialized == 1)
	{
		fmac_close(&dev_eth0);
	}
}

int eth_rx() 
{
	return fmac_rcv(&dev_eth0);
}

int eth_send(volatile void *packet, int length) 
{
	return fmac_send_packet(packet, length, &dev_eth0);
}

void fmac_dummy()
{
	printf("enter fmac_dummy\n");
}
#endif 
