#!/usr/bin/perl -w
#
# gimp wrapper for generating menu and buttons
#

use strict;
use Getopt::Std;

my $usage =
	"Usage: $0 [-h] [-b|-e|-t] -i <gimp_inc_file> [-f <spec_file>] [-d <target_dir>]\n".
	"    -h                 show this help\n".
	"    -b                 generate buttons\n".
	"    -m                 generate menu tabs and entries\n".
	"    -d                 target dir for outputting the images\n".
	"    -i <gimp_inc_file> must contain gimp function definitions\n".
	"    -f <spec_file>     contains labels and file names to generate;\n".
	"                       if '-' or omitted, STDIN is used\n".
	"    Format of <spec_file>:\n".
	"        - menu:\n".
	"                       unindented lines specify menu tabs;\n".
	"                       indented lines specify menu entries;\n".
	"                       tabs:    <label>|<basename for tab images>\n".
	"                       entries: <label>|<asp basename>.asp[|<target frame>]\n".
	"                                <asp basename> is also used for the images\n".
	"                                <target frame> is ignored\n".
	"        - buttons:\n".
	"                       lines starting with '#' are comments;\n".
	"                       others have the format:\n".
	"                       <text on button>|<base name for button_* image file >\n".
	"";

my $template_dir = "templates";

my $opt_do_help = 0;
my $opt_do_btns = 0;
my $opt_do_menu = 0;
my $opt_spec_file = "-";    # default: stdin
my $opt_gimp_inc_file;
my $opt_target_dir = ".";

# opens the given file (or stdin for '-')
sub open_file ($) {
	my $fname = shift;
	my $fh;

	# open spec file
	if ($fname eq "-") {
		print STDERR "Using STDIN...\n";
		$fh = *STDIN;
	} else {
		open(FH, "<$fname") || die "ERROR: $0: Could not open file '$fname'!\n";
		$fh = *FH;
	}
	return $fh;
}

# read file FNAME and return contents
sub read_gimp_includes ($) {
	my $fname = shift;
	my $src = "";

	open(FUNCS, "<$fname") || die "ERROR: $0: Could not open file '$fname'!\n";
	while (<FUNCS>) {
		/^\s*;|^\s*$/ && next;  # skip comments and empty lines
		$src .= $_;
	}
	close(FUNCS);

	return $src;
}

# has gimp execute the given SCRIPT
sub run_gimp ($) {
	my $script = shift;
	my $GIMP_CMD = "gimp -i --batch-interpreter plug_in_script_fu_eval -b -";

	open (GIMP, "| $GIMP_CMD >/dev/null");
	# print "$script\n";
	print GIMP "$script";
	close(GIMP);
}


sub gen_buttons ($) {
	my $init_script = shift;

	my $script .= $init_script;
	my $fh;

	# process lines in spec file
	$fh = open_file($opt_spec_file);
	my $lineno = 1;
	while (<$fh>) {
		chomp;
		# skip comments and empty lines
		if (! /^\s*#|^\s*$/) {
		    (my $label, my $basename, my $undef) = split(/\s*\|\s*/);
		    defined($label) && defined($basename) && !defined($undef)
			    || die "ERROR: $0: Malformed line in <spec_file> \"$opt_spec_file\", line $lineno\n";
		    $script .= "(gen-button \"$label\" \"$basename\")\n";
		}
		$lineno++;
	}
	close($fh);
	$script .= "(gimp-quit 0)\n";

	# pass script to gimp
	print "Generating button images in \"$opt_target_dir\" ...\n";
	run_gimp($script);
}

sub die_usage ($) {
	my $text = shift;
	die "\nERROR: $0: $text\n\n$usage";
}

sub main () {
	my $opts = {};
	getopts('hbmf:i:d:', $opts);
	$opt_do_help = defined($opts->{"h"});
	if ($opt_do_help) { print "$usage"; exit; }
	$opt_do_btns = defined($opts->{"b"});
	$opt_do_menu = defined($opts->{"m"});

	defined($opts->{"f"}) && ($opt_spec_file = $opts->{"f"});

	defined($opts->{"i"}) || die_usage "-i <gimp_inc_file> is required!";
	$opt_gimp_inc_file = $opts->{"i"};

	defined($opts->{"d"}) && ($opt_target_dir = $opts->{"d"});

	($opt_do_btns + $opt_do_menu == 1) 
		|| die_usage "Exactly one of -b or -m is required!";

	# prepare gimp commands to set things up
	my $init_script = "";
	$init_script .= "(set! target-dir \"$opt_target_dir\")\n";
	$init_script .= "(set! global-template-dir \"$template_dir\")\n";
	$init_script .= read_gimp_includes($opt_gimp_inc_file);

	# generate buttons
	($opt_do_btns) && gen_buttons($init_script);
	# no menu to generate for this OEM -> ignore -m
	($opt_do_menu) && {};  
}

main();
