/**
 * \file bmc_dev_oem_amd.c
 *
 * Description: BMC AMD/OPMA-OEM Device
 *
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 */

#include <pp/base.h>
#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/host_board_id.h>
#include <pp/bmc/host_hardware.h>
#include <pp/bmc/host_power.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/host_sensors.h>
#include <pp/bmc/smi_adapter.h>

#include "bmc_dev_sensor.h"
#include "bmc_dev_oem_amd.h"

/********************************************************************
 * Device command handlers
 */

#define MCARD_CAPS_LAN_SUPPORT_TYPE_M1			0x01
#define MCARD_CAPS_LAN_SUPPORT_TYPE_M2			0x02
#define MCARD_CAPS_LAN_SUPPORT_TYPE_M3			0x03

#define MCARD_CAPS_LAN_SUPPORT_STATUS_OOB_ENABLED	0x01
#define MCARD_CAPS_LAN_SUPPORT_STATUS_OOB_DISABLED	0x02

#define MCARD_CAPS_CLI_SUPPORT_ENABLED			0x01
#define MCARD_CAPS_CLI_SUPPORT_DISABLED			0x02

#define MCARD_CAPS_IPMI_INTERFACE_KCS0			0x01
#define MCARD_CAPS_IPMI_INTERFACE_KCS1			0x02
#define MCARD_CAPS_IPMI_INTERFACE_BT			0x04
#define MCARD_CAPS_IPMI_INTERFACE_SSIF			0x08
#define MCARD_CAPS_IPMI_INTERFACE_SMIC			0x10

#define MCARD_CAPS_SENSOR_SCANNING_ACTIVE		0x01
#define MCARD_CAPS_SENSOR_SCANNING_INACTIVE		0x02

#define MCARD_CAPS_COMM1_NOT_SUPPORTED			0x00
#define MCARD_CAPS_COMM1_SUPPORTED			0x01 
#define MCARD_CAPS_VFLOPPY_NOT_SUPPORTED		0x00
#define MCARD_CAPS_VFLOPPY_SUPPORTED			0x01

struct oem_amd_get_mcard_capabilities_rs_s {
    u_short opma_version_be16;
    u_char lan_support_type;
    u_char lan_support_status;

    u_char cli_support;
    u_char ipmi_interface;
    u_char sensor_scanning_status;
    u_char virtual_comm1_support;
    u_char virtual_floppy_support;
} __attribute__ ((packed));
typedef struct oem_amd_get_mcard_capabilities_rs_s oem_amd_get_mcard_capabilities_rs_t;

static int oem_amd_get_mcard_capabilities(imsg_t* imsg)
{
    oem_amd_get_mcard_capabilities_rs_t *rs;
    u_short oem_id, impl_id, opma_version;
    pp_bmc_smi_mode_t smi_mode;
    int op_ret;
    
    pp_bmc_log_info("[OEM-AMD] GetmCardCapabilities");
    
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_amd_get_mcard_capabilities_rs_t));

    rs->opma_version_be16 = cpu_to_be16(OPMA_VERSION_SUPPORTED);
    rs->lan_support_type = MCARD_CAPS_LAN_SUPPORT_TYPE_M3;
    rs->lan_support_status = MCARD_CAPS_LAN_SUPPORT_STATUS_OOB_ENABLED;
    rs->cli_support = MCARD_CAPS_CLI_SUPPORT_DISABLED;  // TODO set dynamically!
    rs->virtual_comm1_support = MCARD_CAPS_COMM1_NOT_SUPPORTED;
    rs->virtual_floppy_support = MCARD_CAPS_VFLOPPY_NOT_SUPPORTED;
    
    smi_mode = pp_bmc_smi_get_mode();

    switch(smi_mode) {
      case PP_SMI_MODE_KCS:
	  rs->ipmi_interface = MCARD_CAPS_IPMI_INTERFACE_KCS0;
	  break;
      case PP_SMI_MODE_BT:
	  rs->ipmi_interface = MCARD_CAPS_IPMI_INTERFACE_BT;
	  break;
      case PP_SMI_MODE_SMIC:
	  rs->ipmi_interface = MCARD_CAPS_IPMI_INTERFACE_SMIC;
	  break;
      case PP_SMI_MODE_UNKNOWN:
      case PP_SMI_MODE_LOOPBACK:
      default:
	  rs->ipmi_interface = 0;
	  break;
    }

    op_ret = pp_bmc_host_board_id_get_opma(&oem_id, &impl_id, &opma_version);
    if (op_ret == PP_SUC) {
	rs->sensor_scanning_status = MCARD_CAPS_SENSOR_SCANNING_ACTIVE;
    } else {
	rs->sensor_scanning_status = MCARD_CAPS_SENSOR_SCANNING_INACTIVE;
    }
    
    return pp_bmc_router_send_msg(imsg);
}

struct oem_amd_set_system_type_identifier_rq_s {
    u_short oem_id_le16;
    u_short impl_id_le16;
    u_short opma_version_be16;
} __attribute__ ((packed));
typedef struct oem_amd_set_system_type_identifier_rq_s oem_amd_set_system_type_identifier_rq_t;

static int oem_amd_set_system_type_identifier(imsg_t* imsg)
{
    oem_amd_set_system_type_identifier_rq_t* rq = (void*) imsg->data;
    int op_ret;
    u_char compl_code;

    pp_bmc_log_info("[OEM-AMD] SetSystemTypeID %04x/%04x opma %04x",
		    le16_to_cpu(rq->oem_id_le16), le16_to_cpu(rq->impl_id_le16), be16_to_cpu(rq->opma_version_be16));
    
    op_ret = pp_bmc_host_board_id_set_opma(le16_to_cpu(rq->oem_id_le16),
					   le16_to_cpu(rq->impl_id_le16),
					   be16_to_cpu(rq->opma_version_be16));

    switch (op_ret) {
      case HOST_BOARD_ID_OPMA_VERSION_UNSUPPORTED:
	  compl_code = IPMI_ERR_PARAM_OUT_OF_RANGE; 
	  break;
      case HOST_BOARD_ID_OPMA_BOARD_UNSUPPORTED:
	  compl_code = IPMI_ERR_INVALID_DATA_FIELD;
	  break;	  
      case HOST_BOARD_ID_OPMA_ACCEPTED: 
	  /* fallthrough */
      case HOST_BOARD_ID_OPMA_ALREADY_SET:
	  /* fallthrough */
      default:
	  compl_code = IPMI_ERR_SUCCESS;
	  break;
    }

    return pp_bmc_router_resp_err(imsg, compl_code);    
}
    
struct oem_amd_get_system_type_identifier_rs_s {
    u_short oem_id_le16;
    u_short impl_id_le16;
    u_short opma_version_be16;
} __attribute__ ((packed));
typedef struct oem_amd_get_system_type_identifier_rs_s oem_amd_get_system_type_identifier_rs_t;

static int oem_amd_get_system_type_identifier(imsg_t* imsg)
{
    oem_amd_get_system_type_identifier_rs_t *rs;
    u_short oem_id, impl_id, opma_version;
    int op_ret;
    
    op_ret = pp_bmc_host_board_id_get_opma(&oem_id, &impl_id, &opma_version);
    if (op_ret == PP_ERR) {
	oem_id = impl_id = opma_version = 0x0000;
    }

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_amd_get_system_type_identifier_rs_t));    
    rs->oem_id_le16 = cpu_to_le16(oem_id);
    rs->impl_id_le16 = cpu_to_le16(impl_id);
    rs->opma_version_be16 = cpu_to_be16(opma_version);

    pp_bmc_log_info("[OEM-AMD] GetSystemTypeID %04x/%04x opma %04x",
		    oem_id, impl_id, opma_version);
    
    return pp_bmc_router_send_msg(imsg);
}

struct oem_amd_set_local_access_lockout_state_rq_s {
    BITFIELD2(u_char,
	      unlock : 1,
	      _resv  : 7);
} __attribute__ ((packed));
typedef struct oem_amd_set_local_access_lockout_state_rq_s oem_amd_set_local_access_lockout_state_rq_t;

static int oem_amd_set_local_access_lockout_state(imsg_t* imsg)
{
    oem_amd_set_local_access_lockout_state_rq_t* rq = (void*)imsg->data;

    pp_bmc_log_info("[OEM-AMD] SetLocalLockout unlock=%d", rq->unlock);
    
    if (PP_FAILED(pp_bmc_hardware_frontpanel_hard_lockout_set(rq->unlock ? 0 : 1))) {
	pp_bmc_log_error("[OEM-AMD] SetLocalLockout failed");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

struct oem_amd_get_local_access_lockout_state_rs_s {
    BITFIELD2(u_char,
	      unlocked : 1,
	      _resv    : 7);
} __attribute__ ((packed));
typedef struct oem_amd_get_local_access_lockout_state_rs_s oem_amd_get_local_access_lockout_state_rs_t;

static int oem_amd_get_local_access_lockout_state(imsg_t* imsg)
{
    oem_amd_get_local_access_lockout_state_rs_t* rs;
    int rv;

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_amd_get_local_access_lockout_state_rs_t));

    if ((rv = pp_bmc_hardware_frontpanel_hard_lockout_get()) == PP_ERR) {
	rs->unlocked = 1;
    } else {
	rs->unlocked = rv ? 0 : 1;
    }
   
    pp_bmc_log_info("[OEM-AMD] GetLocalLockout unlocked=%d", rs->unlocked);

    return pp_bmc_router_send_msg(imsg);
}

struct oem_amd_get_supported_host_ids_rs_s {
    u_char count;
    struct {
	u_short oem_id_le16;
	u_short impl_id_le16;
    } __attribute__ ((packed)) board_id[OPMA_MAX_BOARD_COUNT];
} __attribute__ ((packed));
typedef struct oem_amd_get_supported_host_ids_rs_s oem_amd_get_supported_host_ids_rs_t;

static int oem_amd_get_supported_host_ids(imsg_t* imsg)
{
    oem_amd_get_supported_host_ids_rs_t* rs;
    opma_board_id_t *board;
    vector_t *boards;
    u_char count, i;

    boards = pp_bmc_host_board_id_get_supported_opma();
    count = vector_size(boards);
    pp_bmc_log_info("[OEM-AMD] GetSupportedHostIDs, count=%d", count);

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_amd_get_supported_host_ids_rs_t));

    rs->count = count;
    
    for (i = 0; i < count; i++) {
	board = (opma_board_id_t*) vector_get2(boards, i);
	rs->board_id[i].oem_id_le16 = cpu_to_le16(board->oem_id);
	rs->board_id[i].impl_id_le16 = cpu_to_le16(board->impl_id);
    }
    
    return pp_bmc_router_send_msg(imsg);
}

static int oem_amd_clear_cmos(imsg_t* imsg)
{
    pp_bmc_log_info("[OEM-AMD] Clear CMOS");

    if (PP_FAILED(pp_bmc_hardware_clear_cmos())) {
	pp_bmc_log_error("[OEM-AMD] Clear CMOS failed");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

struct oem_qlogic_get_sensor_reading_offset_rs_s {
    char offset;
} __attribute__ ((packed));
typedef struct oem_qlogic_get_sensor_reading_offset_rs_s oem_qlogic_get_sensor_reading_offset_rs_t;

static int oem_qlogic_get_sensor_reading_offset(imsg_t* imsg) {
    pp_tp_ipmi_sens_t* sensor;
    oem_qlogic_get_sensor_reading_offset_rs_t* response;
    signed char offset;

    pp_bmc_log_info("[OEM-AMD] GetSensorReadingOffset, LUN %d, Sens 0x%02x",
		    imsg->data[0], imsg->data[1]);
    
    // we always use LUN 0, so the sensor number == index into sdrr, and it is unique
    assert(imsg->data[0] == 0);
    sensor = (pp_tp_ipmi_sens_t*) get_sensor_by_index(imsg->data[1]);

    if (sensor == NULL) // sensor with this number doesn't exist
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);

    if (!PP_TP_OBJ_IS_TYPE(PP_TP_THRESH_IPMI_SENS, sensor))
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_COMMAND_INVALID_FOR_THIS_LUN);

    response = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 
				sizeof(oem_qlogic_get_sensor_reading_offset_rs_t));
    offset = pp_bmc_tp_ipmi_sens_get_reading_offset(sensor);

    pp_bmc_log_info("[OEM-AMD] GetSensorReadingOffset returning %d",
		    offset);

    response->offset = (unsigned char) offset;

    return pp_bmc_router_send_msg(imsg);
}

static int oem_qlogic_set_sensor_reading_offset(imsg_t* imsg) {  
    pp_bmc_log_info("[OEM-AMD] SetSensorReadingOffset, LUN %d, Sens 0x%02x, Offset %d",
		    imsg->data[0], imsg->data[1], (signed char) imsg->data[2]);
   
    // we always use LUN 0, so the sensor number == index into sdrr, and it is unique
    assert(imsg->data[0] == 0);
    pp_tp_ipmi_sens_t* sensor = (pp_tp_ipmi_sens_t*) get_sensor_by_index(imsg->data[1]);

    if (sensor == NULL) // sensor with this number doesn't exist
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);

    if (!PP_TP_OBJ_IS_TYPE(PP_TP_THRESH_IPMI_SENS, sensor))
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_COMMAND_INVALID_FOR_THIS_LUN);

    pp_bmc_tp_ipmi_sens_set_reading_offset(sensor, imsg->data[2]);

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/********************************************************************
 * Device command table
 */

static const dev_cmd_entry_t oem_amd_cmd_tab[] = {
    {
        .cmd_hndlr = oem_amd_get_mcard_capabilities,
        .netfn = IPMI_NETFN_OEM_AMD,
	.cmd = IPMI_CMD_AMD_GET_MCARD_CAPABILITIES,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = oem_amd_set_system_type_identifier,
        .netfn = IPMI_NETFN_OEM_AMD,
	.cmd = IPMI_CMD_AMD_SET_SYSTEM_TYPE_IDENTIFIER,
        .min_data_size = 6, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_amd_get_system_type_identifier,
        .netfn = IPMI_NETFN_OEM_AMD,
	.cmd = IPMI_CMD_AMD_GET_SYSTEM_TYPE_IDENTIFIER,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_amd_set_local_access_lockout_state,
        .netfn = IPMI_NETFN_OEM_AMD,
	.cmd = IPMI_CMD_AMD_SET_LOCAL_ACCESS_LOCKOUT_STATE,
        .min_data_size = 1, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_amd_get_local_access_lockout_state,
        .netfn = IPMI_NETFN_OEM_AMD,
	.cmd = IPMI_CMD_AMD_GET_LOCAL_ACCESS_LOCKOUT_STATE,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_amd_get_supported_host_ids,
        .netfn = IPMI_NETFN_OEM_AMD,
	.cmd = IPMI_CMD_AMD_GET_SUPPORTED_HOST_IDS,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
	.cmd_hndlr = oem_amd_clear_cmos,
        .netfn = IPMI_NETFN_OEM_AMD,
	.cmd = IPMI_CMD_AMD_CLEAR_CMOS,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },

    {
        .cmd_hndlr = oem_qlogic_set_sensor_reading_offset,
        .netfn = IPMI_NETFN_OEM_QLOGIC,
	.cmd = IPMI_CMD_QLOGIC_SET_SENSOR_READING_OFFSET,
        .min_data_size = 3, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_qlogic_get_sensor_reading_offset,
        .netfn = IPMI_NETFN_OEM_QLOGIC,
	.cmd = IPMI_CMD_QLOGIC_GET_SENSOR_READING_OFFSET,
        .min_data_size = 2, .min_priv_level = IPMI_PRIV_ADMIN
    },
    
    { .cmd_hndlr = NULL }
};


/********************************************************************
 * Device c'tor/d'tor
 */

int pp_bmc_dev_oem_amd_init(void)
{
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(oem_amd_cmd_tab)) return PP_ERR;

    pp_bmc_log_info("[OEM-AMD] device started");
    return PP_SUC;
}

void pp_bmc_dev_oem_amd_cleanup(void)
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(oem_amd_cmd_tab);

    pp_bmc_log_info("[OEM-AMD] device shut down");
}
