/**
 * bmc_dev_oem_icp.c
 *
 * Description: BMC ICP-OEM Device
 *
 * (c) 2005 Peppercon AG, thomas@peppercon.de
 * 
 */

#include <pp/base.h>
#include <pp/i2c.h>

#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/host_sensors.h>
#include <pp/bmc/tp_gpio_sens.h>
#include <pp/bmc/tp_ipmi_sens.h>

#include <pp/hal_icpmmd.h>
#include "bmc_dev_sensor.h"
#include "bmc_dev_oem_icp.h"
#include "bmc_msg_tracker.h"


#define OEM_ICP_IPMI_ERR_SET_STATUS	   0x80
#define OEM_ICP_IPMI_ERR_GET_STATUS	   0x80
#define OEM_ICP_IPMI_ERR_GET_SENSOR_VALUES 0x80
#define OEM_ICP_IPMI_ERR_GET_STATUS_VALUES 0x80

#define ICP_NODE_INDEX_MIN 0
#define ICP_NODE_INDEX_MAX 11

#define OEM_ICP_MODULE_TYPE_CFU		0x0
#define OEM_ICP_MODULE_TYPE_PSU		0x1
#define OEM_ICP_MODULE_TYPE_IBSW	0x2
#define OEM_ICP_MODULE_TYPE_GESW	0x3

#define OEM_ICP_UNIT_TYPE_LED		0x0
#define OEM_ICP_UNIT_TYPE_PWM		0x1

/* IPMI sensor ids */
#define OEM_ICP_IPMI_SENS_CFU0_TACH1	0x00
#define OEM_ICP_IPMI_SENS_CFU0_TACH2	0x01
#define OEM_ICP_IPMI_SENS_CFU0_TEMP	0x02
#define OEM_ICP_IPMI_SENS_CFU1_TACH1	0x04
#define OEM_ICP_IPMI_SENS_CFU1_TACH2	0x05
#define OEM_ICP_IPMI_SENS_CFU1_TEMP	0x06
#define OEM_ICP_IPMI_SENS_CFU2_TACH1	0x08
#define OEM_ICP_IPMI_SENS_CFU2_TACH2	0x09
#define OEM_ICP_IPMI_SENS_CFU2_TEMP	0x0a
#define OEM_ICP_IPMI_SENS_CFU3_TACH1	0x0c
#define OEM_ICP_IPMI_SENS_CFU3_TACH2	0x0d
#define OEM_ICP_IPMI_SENS_CFU3_TEMP	0x0e
#define OEM_ICP_IPMI_SENS_PSU0_TACH1	0x10
#define OEM_ICP_IPMI_SENS_PSU0_TACH2	0x11
#define OEM_ICP_IPMI_SENS_PSU0_TACH3	0x12
#define OEM_ICP_IPMI_SENS_PSU0_5VSB	0x13
#define OEM_ICP_IPMI_SENS_PSU0_12V	0x14
#define OEM_ICP_IPMI_SENS_PSU0_TEMP	0x15
#define OEM_ICP_IPMI_SENS_PSU1_TACH1	0x18
#define OEM_ICP_IPMI_SENS_PSU1_TACH2	0x19
#define OEM_ICP_IPMI_SENS_PSU1_TACH3	0x1a
#define OEM_ICP_IPMI_SENS_PSU1_5VSB	0x1b
#define OEM_ICP_IPMI_SENS_PSU1_12V	0x1c
#define OEM_ICP_IPMI_SENS_PSU1_TEMP	0x1d
#define OEM_ICP_IPMI_SENS_PSU2_TACH1	0x20
#define OEM_ICP_IPMI_SENS_PSU2_TACH2	0x21
#define OEM_ICP_IPMI_SENS_PSU2_TACH3	0x22
#define OEM_ICP_IPMI_SENS_PSU2_5VSB	0x23
#define OEM_ICP_IPMI_SENS_PSU2_12V	0x24
#define OEM_ICP_IPMI_SENS_PSU2_TEMP	0x25
#define OEM_ICP_IPMI_SENS_PSU3_TACH1	0x28
#define OEM_ICP_IPMI_SENS_PSU3_TACH2	0x29
#define OEM_ICP_IPMI_SENS_PSU3_TACH3	0x2a
#define OEM_ICP_IPMI_SENS_PSU3_5VSB	0x2b
#define OEM_ICP_IPMI_SENS_PSU3_12V	0x2c
#define OEM_ICP_IPMI_SENS_PSU3_TEMP	0x2d
#define OEM_ICP_IPMI_SENS_IBSW0_5V	0x30
#define OEM_ICP_IPMI_SENS_IBSW0_12V	0x31
#define OEM_ICP_IPMI_SENS_IBSW0_TEMP	0x32
#define OEM_ICP_IPMI_SENS_IBSW0_TACH1	0x33
#define OEM_ICP_IPMI_SENS_IBSW0_TACH2	0x34
#define OEM_ICP_IPMI_SENS_IBSW1_5V	0x38
#define OEM_ICP_IPMI_SENS_IBSW1_12V	0x39
#define OEM_ICP_IPMI_SENS_IBSW1_TEMP	0x3a
#define OEM_ICP_IPMI_SENS_IBSW1_TACH1	0x3b
#define OEM_ICP_IPMI_SENS_IBSW1_TACH2	0x3c

/* PWM indices */
#define PP_ACTOR_CFU0_PWM		0
#define PP_ACTOR_CFU1_PWM		1
#define PP_ACTOR_CFU2_PWM		2
#define PP_ACTOR_CFU3_PWM		3
#define PP_ACTOR_PSU0_PWM		4
#define PP_ACTOR_PSU1_PWM		5
#define PP_ACTOR_PSU2_PWM		6
#define PP_ACTOR_PSU3_PWM		7

int cfu_fan_ok_led_actor_ids[4][2] = {
    { PP_ACTOR_CFU0_FAN1_OK, PP_ACTOR_CFU0_FAN2_OK },
    { PP_ACTOR_CFU1_FAN1_OK, PP_ACTOR_CFU1_FAN2_OK },
    { PP_ACTOR_CFU2_FAN1_OK, PP_ACTOR_CFU2_FAN2_OK },
    { PP_ACTOR_CFU3_FAN1_OK, PP_ACTOR_CFU3_FAN2_OK }
};

int cfu_pwm_actor_ids[4] = {
    PP_ACTOR_CFU0_PWM,
    PP_ACTOR_CFU1_PWM,
    PP_ACTOR_CFU2_PWM,
    PP_ACTOR_CFU3_PWM
};

int psu_led_actor_ids[4][2] = {
    { PP_ACTOR_PSU0_FAN_OK, PP_ACTOR_PSU0_DC_OK },
    { PP_ACTOR_PSU1_FAN_OK, PP_ACTOR_PSU1_DC_OK },
    { PP_ACTOR_PSU2_FAN_OK, PP_ACTOR_PSU2_DC_OK },
    { PP_ACTOR_PSU3_FAN_OK, PP_ACTOR_PSU3_DC_OK }
};

int psu_ac_ok_sensor_ids[4][2] = {
    { PP_SENSOR_PSU0_AC1_OK, PP_SENSOR_PSU0_AC2_OK },
    { PP_SENSOR_PSU1_AC1_OK, PP_SENSOR_PSU1_AC2_OK },
    { PP_SENSOR_PSU2_AC1_OK, PP_SENSOR_PSU2_AC2_OK },
    { PP_SENSOR_PSU3_AC1_OK, PP_SENSOR_PSU3_AC2_OK }
};

int psu_pwm_actor_ids[4] = {
    PP_ACTOR_PSU0_PWM,
    PP_ACTOR_PSU1_PWM,
    PP_ACTOR_PSU2_PWM,
    PP_ACTOR_PSU3_PWM
};

int ibsw_slot_id_sensor_ids[2] = {
    PP_SENSOR_IBSW0_SLOTID,
    PP_SENSOR_IBSW1_SLOTID
};

int ibsw_power_ok_sensor_ids[2] = {
    PP_SENSOR_IBSW0_POWER_OK,
    PP_SENSOR_IBSW1_POWER_OK
};

int ibsw_fault_led_actor_ids[2] = {
    PP_ACTOR_IBSW0_FAULT_LED,
    PP_ACTOR_IBSW1_FAULT_LED,
};

int ibsw_good_led_actor_ids[2] = {
    PP_ACTOR_IBSW0_GOOD_LED,
    PP_ACTOR_IBSW1_GOOD_LED,
};

int ibsw_reset_actor_ids[2] = {
    PP_ACTOR_IBSW0_RESET,
    PP_ACTOR_IBSW1_RESET
};

/********************************************************************
 * Device command handlers
 */

typedef struct {
    unsigned char module_type;
    unsigned char module_index;
    unsigned char unit_type;
    unsigned char unit_index;
    unsigned char status_value;
} __attribute__ ((packed)) oem_icp_set_config_parameter_rq_t;

static int
oem_icp_cmd_set_config_parameter(imsg_t* imsg)
{
    oem_icp_set_config_parameter_rq_t* rq = (void*)imsg->data;
    pp_tp_gpio_act_t * gpio_actor;
    pp_tp_pwm_act_t * pwm_actor;

    switch (rq->module_type) {
      case OEM_ICP_MODULE_TYPE_CFU:
	  if (rq->module_index > 3) goto bail;
	  switch (rq->unit_type) {
	    case OEM_ICP_UNIT_TYPE_LED:
		if (rq->unit_index > 1) goto bail;
		if (rq->status_value > 1) goto bail;
		gpio_actor = pp_bmc_host_sensor_get_gpio_actor(cfu_fan_ok_led_actor_ids[rq->module_index][rq->unit_index]);
		if (!gpio_actor) goto bail;
		if (PP_FAILED(gpio_actor->set_gpio(gpio_actor, rq->status_value))) goto bail;
		break;
	    case OEM_ICP_UNIT_TYPE_PWM:
		if (rq->unit_index > 0) goto bail;
		//if (rq->status_value != 50 && rq->status_value != 100) goto bail;
		pwm_actor = pp_bmc_host_sensor_get_pwm_actor(cfu_pwm_actor_ids[rq->module_index]);
		if (!pwm_actor) goto bail;
		if (PP_FAILED(pwm_actor->set_duty_cycle(pwm_actor, rq->status_value))) goto bail;
		break;
	    default:
		goto bail;
	  }
	  break;
      case OEM_ICP_MODULE_TYPE_PSU:
	  if (rq->module_index > 3) goto bail;
	  switch (rq->unit_type) {
	    case OEM_ICP_UNIT_TYPE_LED:
		if (rq->unit_index > 1) goto bail;
		if (rq->status_value > 1) goto bail;
		gpio_actor = pp_bmc_host_sensor_get_gpio_actor(psu_led_actor_ids[rq->module_index][rq->unit_index]);
		if (!gpio_actor) goto bail;
		if (PP_FAILED(gpio_actor->set_gpio(gpio_actor, rq->status_value))) goto bail;
		break;
	    case OEM_ICP_UNIT_TYPE_PWM:
		if (rq->unit_index > 0) goto bail;
		if (rq->status_value != 50 && rq->status_value != 100) goto bail;
		pwm_actor = pp_bmc_host_sensor_get_pwm_actor(psu_pwm_actor_ids[rq->module_index]);
		if (!pwm_actor) goto bail;
		if (PP_FAILED(pwm_actor->set_duty_cycle(pwm_actor, rq->status_value))) goto bail;
		break;
	    default:
		goto bail;
	  }
	  break;
      case OEM_ICP_MODULE_TYPE_IBSW:
	  if (rq->module_index > 1) goto bail;
	  switch (rq->unit_type) {
	    case OEM_ICP_UNIT_TYPE_LED:
		if (rq->status_value > 1) goto bail;
		if (rq->unit_index == 2) {
		    pp_tp_gpio_act_t * fault_led_actor = pp_bmc_host_sensor_get_gpio_actor(ibsw_fault_led_actor_ids[rq->module_index]);
		    pp_tp_gpio_act_t * good_led_actor = pp_bmc_host_sensor_get_gpio_actor(ibsw_good_led_actor_ids[rq->module_index]);
		    if (!fault_led_actor || !good_led_actor) goto bail;
		    if (PP_FAILED(fault_led_actor->set_gpio(fault_led_actor, rq->status_value))) goto bail;
		    if (PP_FAILED(good_led_actor->set_gpio(good_led_actor, rq->status_value))) goto bail;
		} else if (rq->unit_index == 3) {
		    gpio_actor = pp_bmc_host_sensor_get_gpio_actor(ibsw_reset_actor_ids[rq->module_index]);
		    if (!gpio_actor) goto bail;
		    if (PP_FAILED(gpio_actor->set_gpio(gpio_actor, rq->status_value))) goto bail;
		} else {
		    goto bail;
		}
		break;
	    default:
		goto bail;
	  }
	  break;
      default:
	  goto bail;
    }

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);

 bail:
    return pp_bmc_router_resp_err(imsg, OEM_ICP_IPMI_ERR_SET_STATUS);
}

/* ------------------------------ */
typedef struct {
    unsigned char module_type;
    unsigned char module_index;
    unsigned char unit_type;
    unsigned char unit_index;
} __attribute__ ((packed)) oem_icp_get_config_parameter_rq_t;

typedef struct {
    unsigned char status_value;
} __attribute__ ((packed)) oem_icp_get_config_parameter_rs_t;

static int
oem_icp_cmd_get_config_parameter(imsg_t* imsg)
{
    oem_icp_get_config_parameter_rq_t* rq = (void*)imsg->data;
    oem_icp_get_config_parameter_rs_t* rs;
    pp_tp_sensdev_t * sensor;
    pp_tp_gpio_act_t * gpio_actor;
    pp_tp_pwm_act_t * pwm_actor;
    u_char duty_cycle;
    int value = -1;

    switch (rq->module_type) {
      case OEM_ICP_MODULE_TYPE_CFU:
	  if (rq->module_index > 3) goto bail;
	  switch (rq->unit_type) {
	    case OEM_ICP_UNIT_TYPE_LED:
		if (rq->unit_index > 1) goto bail;
		gpio_actor = pp_bmc_host_sensor_get_gpio_actor(cfu_fan_ok_led_actor_ids[rq->module_index][rq->unit_index]);
		if (!gpio_actor) goto bail;
		value = gpio_actor->get_gpio(gpio_actor);
		break;
	    case OEM_ICP_UNIT_TYPE_PWM:
		if (rq->unit_index > 0) goto bail;
		pwm_actor = pp_bmc_host_sensor_get_pwm_actor(cfu_pwm_actor_ids[rq->module_index]);
		if (!pwm_actor) goto bail;
		if (PP_FAILED(pwm_actor->get_duty_cycle(pwm_actor, &duty_cycle))) goto bail;
		value = duty_cycle;
		break;
	    default:
		goto bail;
	  }
	  break;
      case OEM_ICP_MODULE_TYPE_PSU:
	  if (rq->module_index > 3) goto bail;
	  switch (rq->unit_type) {
	    case OEM_ICP_UNIT_TYPE_LED:
		if (rq->unit_index < 2) {
		    gpio_actor = pp_bmc_host_sensor_get_gpio_actor(psu_led_actor_ids[rq->module_index][rq->unit_index]);
		    if (!gpio_actor) goto bail;
		    value = gpio_actor->get_gpio(gpio_actor);
		} else if (rq->unit_index < 4) {
		    sensor = pp_bmc_host_sensor_get_sensor(psu_ac_ok_sensor_ids[rq->module_index][rq->unit_index-2]);
		    if (!sensor) goto bail;
		    value = pp_tp_sensdev_get_reading(sensor);
		} else {
		    goto bail;
		}
		break;
	    case OEM_ICP_UNIT_TYPE_PWM:
		if (rq->unit_index > 0) goto bail;
		pwm_actor = pp_bmc_host_sensor_get_pwm_actor(psu_pwm_actor_ids[rq->module_index]);
		if (!pwm_actor) goto bail;
		if (PP_FAILED(pwm_actor->get_duty_cycle(pwm_actor, &duty_cycle))) goto bail;
		value = duty_cycle;
		break;
	    default:
		goto bail;
	  }
	  break;
      case OEM_ICP_MODULE_TYPE_IBSW:
	  if (rq->module_index > 1) goto bail;
	  switch (rq->unit_type) {
	    case OEM_ICP_UNIT_TYPE_LED:
		if (rq->unit_index == 0) {
		    sensor = pp_bmc_host_sensor_get_sensor(ibsw_slot_id_sensor_ids[rq->module_index]);
		    if (!sensor) goto bail;
		    value = pp_tp_sensdev_get_reading(sensor);
		} else if (rq->unit_index == 1) {
		    sensor = pp_bmc_host_sensor_get_sensor(ibsw_power_ok_sensor_ids[rq->module_index]);
		    if (!sensor) goto bail;
		    value = pp_tp_sensdev_get_reading(sensor);
		} else if (rq->unit_index == 2) {
		    gpio_actor = pp_bmc_host_sensor_get_gpio_actor(ibsw_fault_led_actor_ids[rq->module_index]);
		    if (!gpio_actor) goto bail;
		    value = gpio_actor->get_gpio(gpio_actor);
		} else {
		    goto bail;
		}
		break;
	    default:
		goto bail;
	  }
	  break;
      case OEM_ICP_MODULE_TYPE_GESW:
	  if (rq->module_index > 1 || rq->unit_type > 0 || rq->unit_index > 3) goto bail;
	  pp_tp_i2c_comdev_t * i2c_dev;
	  int i2c_addr = rq->module_index == 0 ? 0x28 : 0x29;
	  i2c_dev = pp_bmc_host_get_i2c_comdev(PP_BMC_I2C_COMDEV_GESW);
	  if (!i2c_dev) goto bail;
	  if (PP_SUC == pp_tp_i2c_comdev_pre_com(i2c_dev, i2c_addr)) {
	      u_char ipbuf[4];
	      int i2c_handle = pp_tp_i2c_comdev_root_handle(i2c_dev);
	      if (i2c_handle < 0) goto bail;
	      int err, len = pp_i2c_rx_burst_core(i2c_handle, i2c_addr, ipbuf, 4, &err);
	      pp_tp_i2c_comdev_post_com(i2c_dev, i2c_addr);
	      if (err != PP_I2C_NO_ERROR || len != 4) goto bail;
	      value = ipbuf[rq->unit_index];
	  }
	  break;
      default:
	  goto bail;
    }

    if (value >= 0 && value <= 255) {
	rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_icp_get_config_parameter_rs_t));
	rs->status_value = value;
	return pp_bmc_router_send_msg(imsg);
    }

 bail:
    return pp_bmc_router_resp_err(imsg, OEM_ICP_IPMI_ERR_GET_STATUS);
}

/* ------------------------------ */
typedef struct {
    unsigned char reading;
    unsigned char status;
    unsigned char treshold_comp_status;
} __attribute__ ((packed)) sensor_reading_threshold_t;

static int
oem_icp_cmd_get_sensor_values_internal(int sensor_ids[], sensor_reading_threshold_t sensor_vals[], size_t sensor_cnt)
{
    u_int i;

    for (i = 0; i < sensor_cnt; ++i) {
	u_char rsp_code;
	ipmi_sensor_get_reading_rs_t rsp_data;
	/* we always use LUN 0, so the sensor number == index into sdrr, and it is unique */
	pp_tp_ipmi_sens_t* sensor = (pp_tp_ipmi_sens_t*) get_sensor_by_index(sensor_ids[i]);
	if (!sensor) return PP_ERR;
	if (PP_FAILED(pp_tp_ipmi_sens_get_reading(sensor, &rsp_code, &rsp_data))
	    || rsp_code != IPMI_ERR_SUCCESS) {
	    return PP_ERR;
	}
	memcpy(&sensor_vals[i], &rsp_data.rt, sizeof(sensor_reading_threshold_t));
    }
    return PP_SUC;
}

/* ------------------------------ */
#define OEM_ICP_CFU_SENSOR_VAL_CNT 3

typedef struct {
    unsigned char index;
} __attribute__ ((packed)) oem_icp_get_cfu_sensor_values_rq_t;

typedef struct {
    sensor_reading_threshold_t sensor_vals[OEM_ICP_CFU_SENSOR_VAL_CNT];
} __attribute__ ((packed)) oem_icp_get_cfu_sensor_values_rs_t;

static int
oem_icp_cmd_get_cfu_sensor_values(imsg_t* imsg)
{
    oem_icp_get_cfu_sensor_values_rq_t* rq = (void*)imsg->data;
    oem_icp_get_cfu_sensor_values_rs_t* rs;
    sensor_reading_threshold_t sensor_vals[OEM_ICP_CFU_SENSOR_VAL_CNT];
    int sensor_ids[] = {
	OEM_ICP_IPMI_SENS_CFU0_TACH1,
	OEM_ICP_IPMI_SENS_CFU0_TACH2,
	OEM_ICP_IPMI_SENS_CFU0_TEMP,
	OEM_ICP_IPMI_SENS_CFU1_TACH1,
	OEM_ICP_IPMI_SENS_CFU1_TACH2,
	OEM_ICP_IPMI_SENS_CFU1_TEMP,
	OEM_ICP_IPMI_SENS_CFU2_TACH1,
	OEM_ICP_IPMI_SENS_CFU2_TACH2,
	OEM_ICP_IPMI_SENS_CFU2_TEMP,
	OEM_ICP_IPMI_SENS_CFU3_TACH1,
	OEM_ICP_IPMI_SENS_CFU3_TACH2,
	OEM_ICP_IPMI_SENS_CFU3_TEMP,
    };

    if (rq->index > 3) goto bail;
    if (PP_SUCCED(oem_icp_cmd_get_sensor_values_internal(&sensor_ids[rq->index * OEM_ICP_CFU_SENSOR_VAL_CNT],
							 sensor_vals, OEM_ICP_CFU_SENSOR_VAL_CNT))) {
	rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_icp_get_cfu_sensor_values_rs_t));
	memcpy(&rs->sensor_vals, &sensor_vals, sizeof(rs->sensor_vals));
	return pp_bmc_router_send_msg(imsg);
    }
    
 bail:
    return pp_bmc_router_resp_err(imsg, OEM_ICP_IPMI_ERR_GET_SENSOR_VALUES);
}

/* ------------------------------ */
#define OEM_ICP_PSU_SENSOR_VAL_CNT 6

typedef struct {
    unsigned char index;
} __attribute__ ((packed)) oem_icp_get_psu_sensor_values_rq_t;

typedef struct {
    sensor_reading_threshold_t sensor_vals[OEM_ICP_PSU_SENSOR_VAL_CNT];
} __attribute__ ((packed)) oem_icp_get_psu_sensor_values_rs_t;

static int
oem_icp_cmd_get_psu_sensor_values(imsg_t* imsg)
{
    oem_icp_get_psu_sensor_values_rq_t* rq = (void*)imsg->data;
    oem_icp_get_psu_sensor_values_rs_t* rs;
    sensor_reading_threshold_t sensor_vals[OEM_ICP_PSU_SENSOR_VAL_CNT];
    int sensor_ids[] = {
	OEM_ICP_IPMI_SENS_PSU0_TACH1,
	OEM_ICP_IPMI_SENS_PSU0_TACH2,
	OEM_ICP_IPMI_SENS_PSU0_TACH3,
	OEM_ICP_IPMI_SENS_PSU0_5VSB,
	OEM_ICP_IPMI_SENS_PSU0_12V,
	OEM_ICP_IPMI_SENS_PSU0_TEMP,
	OEM_ICP_IPMI_SENS_PSU1_TACH1,
	OEM_ICP_IPMI_SENS_PSU1_TACH2,
	OEM_ICP_IPMI_SENS_PSU1_TACH3,
	OEM_ICP_IPMI_SENS_PSU1_5VSB,
	OEM_ICP_IPMI_SENS_PSU1_12V,
	OEM_ICP_IPMI_SENS_PSU1_TEMP,
	OEM_ICP_IPMI_SENS_PSU2_TACH1,
	OEM_ICP_IPMI_SENS_PSU2_TACH2,
	OEM_ICP_IPMI_SENS_PSU2_TACH3,
	OEM_ICP_IPMI_SENS_PSU2_5VSB,
	OEM_ICP_IPMI_SENS_PSU2_12V,
	OEM_ICP_IPMI_SENS_PSU2_TEMP,
	OEM_ICP_IPMI_SENS_PSU3_TACH1,
	OEM_ICP_IPMI_SENS_PSU3_TACH2,
	OEM_ICP_IPMI_SENS_PSU3_TACH3,
	OEM_ICP_IPMI_SENS_PSU3_5VSB,
	OEM_ICP_IPMI_SENS_PSU3_12V,
	OEM_ICP_IPMI_SENS_PSU3_TEMP
    };

    if (rq->index > 3) goto bail;
    if (PP_SUCCED(oem_icp_cmd_get_sensor_values_internal(&sensor_ids[rq->index * OEM_ICP_PSU_SENSOR_VAL_CNT],
							 sensor_vals, OEM_ICP_PSU_SENSOR_VAL_CNT))) {
	rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_icp_get_psu_sensor_values_rs_t));
	memcpy(&rs->sensor_vals, &sensor_vals, sizeof(rs->sensor_vals));
	return pp_bmc_router_send_msg(imsg);
    }

 bail:
    return pp_bmc_router_resp_err(imsg, OEM_ICP_IPMI_ERR_GET_SENSOR_VALUES);
}

/* ------------------------------ */
#define OEM_ICP_IBSW_SENSOR_VAL_CNT 5

typedef struct {
    unsigned char index;
} __attribute__ ((packed)) oem_icp_get_ibsw_sensor_values_rq_t;

typedef struct {
    sensor_reading_threshold_t sensor_vals[OEM_ICP_IBSW_SENSOR_VAL_CNT];
} __attribute__ ((packed)) oem_icp_get_ibsw_sensor_values_rs_t;

static int
oem_icp_cmd_get_ibsw_sensor_values(imsg_t* imsg)
{
    oem_icp_get_ibsw_sensor_values_rq_t* rq = (void*)imsg->data;
    oem_icp_get_ibsw_sensor_values_rs_t* rs;
    sensor_reading_threshold_t sensor_vals[OEM_ICP_IBSW_SENSOR_VAL_CNT];
    int sensor_ids[] = {
	OEM_ICP_IPMI_SENS_IBSW0_5V,
	OEM_ICP_IPMI_SENS_IBSW0_12V,
	OEM_ICP_IPMI_SENS_IBSW0_TEMP,
	OEM_ICP_IPMI_SENS_IBSW0_TACH1,
	OEM_ICP_IPMI_SENS_IBSW0_TACH2,
	OEM_ICP_IPMI_SENS_IBSW1_5V,
	OEM_ICP_IPMI_SENS_IBSW1_12V,
	OEM_ICP_IPMI_SENS_IBSW1_TEMP,
	OEM_ICP_IPMI_SENS_IBSW1_TACH1,
	OEM_ICP_IPMI_SENS_IBSW1_TACH2
    };

    if (rq->index > 3) goto bail;
    if (PP_SUCCED(oem_icp_cmd_get_sensor_values_internal(&sensor_ids[rq->index * OEM_ICP_IBSW_SENSOR_VAL_CNT],
							 sensor_vals, OEM_ICP_IBSW_SENSOR_VAL_CNT))) {
	rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_icp_get_ibsw_sensor_values_rs_t));
	memcpy(&rs->sensor_vals, &sensor_vals, sizeof(rs->sensor_vals));
	return pp_bmc_router_send_msg(imsg);
    }

 bail:
    return pp_bmc_router_resp_err(imsg, OEM_ICP_IPMI_ERR_GET_SENSOR_VALUES);
}

/* ------------------------------ */
#define OEM_ICP_CFU_STATUS_VAL_CNT 12

typedef struct {
    unsigned char status_vals[OEM_ICP_CFU_STATUS_VAL_CNT];
} __attribute__ ((packed)) oem_icp_get_cfu_status_values_rs_t;

static int
oem_icp_cmd_get_cfu_status_values(imsg_t* imsg)
{
    oem_icp_get_cfu_status_values_rs_t* rs;
    unsigned char status_vals[OEM_ICP_CFU_STATUS_VAL_CNT];
    int gpio_actor_ids[] = {
	PP_ACTOR_CFU0_FAN1_OK,
	PP_ACTOR_CFU0_FAN2_OK,
	PP_ACTOR_CFU1_FAN1_OK,
	PP_ACTOR_CFU1_FAN2_OK,
	PP_ACTOR_CFU2_FAN1_OK,
	PP_ACTOR_CFU2_FAN2_OK,
	PP_ACTOR_CFU3_FAN1_OK,
	PP_ACTOR_CFU3_FAN2_OK,
    };
    int pwm_actor_ids[] = {
	PP_ACTOR_CFU0_PWM,
	PP_ACTOR_CFU1_PWM,
	PP_ACTOR_CFU2_PWM,
	PP_ACTOR_CFU3_PWM
    };
    u_int i, j, k = 0;

    for (j = 0; j < 4; ++j) {
	pp_tp_pwm_act_t * pwm_actor = pp_bmc_host_sensor_get_pwm_actor(pwm_actor_ids[j]);
	if (!pwm_actor) goto bail;
	if (PP_FAILED(pwm_actor->get_duty_cycle(pwm_actor, &status_vals[k]))) {
	    status_vals[k] = -1;
	}
	++k;
	for (i = j*2; i < j*2+2; ++i) {
	    pp_tp_gpio_act_t * gpio_actor = pp_bmc_host_sensor_get_gpio_actor(gpio_actor_ids[i]);
	    if (!gpio_actor) goto bail;
	    status_vals[k++] = gpio_actor->get_gpio(gpio_actor);
	}
    }

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_icp_get_cfu_status_values_rs_t));
    memcpy(&rs->status_vals, &status_vals, sizeof(rs->status_vals));
    return pp_bmc_router_send_msg(imsg);

 bail:
    return pp_bmc_router_resp_err(imsg, OEM_ICP_IPMI_ERR_GET_STATUS_VALUES);
}

/* ------------------------------ */
#define OEM_ICP_PSU_STATUS_VAL_CNT 20

typedef struct {
    unsigned char status_vals[OEM_ICP_PSU_STATUS_VAL_CNT];
} __attribute__ ((packed)) oem_icp_get_psu_status_values_rs_t;

static int
oem_icp_cmd_get_psu_status_values(imsg_t* imsg)
{
    oem_icp_get_psu_status_values_rs_t* rs;
    unsigned char status_vals[OEM_ICP_PSU_STATUS_VAL_CNT];
    int gpio_sensor_ids[] = {
	PP_SENSOR_PSU0_AC1_OK,
	PP_SENSOR_PSU0_AC2_OK,
	PP_SENSOR_PSU1_AC1_OK,
	PP_SENSOR_PSU1_AC2_OK,
	PP_SENSOR_PSU2_AC1_OK,
	PP_SENSOR_PSU2_AC2_OK,
	PP_SENSOR_PSU3_AC1_OK,
	PP_SENSOR_PSU3_AC2_OK
    };
    int gpio_actor_ids[] = {
	PP_ACTOR_PSU0_FAN_OK,
	PP_ACTOR_PSU0_DC_OK,
	PP_ACTOR_PSU1_FAN_OK,
	PP_ACTOR_PSU1_DC_OK,
	PP_ACTOR_PSU2_FAN_OK,
	PP_ACTOR_PSU2_DC_OK,
	PP_ACTOR_PSU3_FAN_OK,
	PP_ACTOR_PSU3_DC_OK
    };
    int pwm_actor_ids[] = {
	PP_ACTOR_PSU0_PWM,
	PP_ACTOR_PSU1_PWM,
	PP_ACTOR_PSU2_PWM,
	PP_ACTOR_PSU3_PWM
    };
    u_int i, j, k = 0;

    for (j = 0; j < 4; ++j) {
	pp_tp_pwm_act_t * pwm_actor = pp_bmc_host_sensor_get_pwm_actor(pwm_actor_ids[j]);
	if (!pwm_actor) goto bail;
	if (PP_FAILED(pwm_actor->get_duty_cycle(pwm_actor, &status_vals[k]))) {
	    status_vals[k] = -1;
	}
	++k;
	for (i = j*2; i < j*2+2; ++i) {
	    pp_tp_gpio_act_t * gpio_actor = pp_bmc_host_sensor_get_gpio_actor(gpio_actor_ids[i]);
	    if (!gpio_actor) goto bail;
	    status_vals[k++] = gpio_actor->get_gpio(gpio_actor);
	}
	for (i = j*2; i < j*2+2; ++i) {
	    pp_tp_sensdev_t * sensor = pp_bmc_host_sensor_get_sensor(gpio_sensor_ids[i]);
	    if (!sensor) goto bail;
	    status_vals[k++] = pp_tp_sensdev_get_reading(sensor);
	}
    }

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_icp_get_psu_status_values_rs_t));
    memcpy(&rs->status_vals, &status_vals, sizeof(rs->status_vals));
    return pp_bmc_router_send_msg(imsg);

 bail:
    return pp_bmc_router_resp_err(imsg, OEM_ICP_IPMI_ERR_GET_STATUS_VALUES);
}

/* ------------------------------ */
#define OEM_ICP_IBSW_STATUS_VAL_CNT 8

typedef struct {
    unsigned char status_vals[OEM_ICP_IBSW_STATUS_VAL_CNT];
} __attribute__ ((packed)) oem_icp_get_ibsw_status_values_rs_t;

static int
oem_icp_cmd_get_ibsw_status_values(imsg_t* imsg)
{
    oem_icp_get_ibsw_status_values_rs_t* rs;
    unsigned char status_vals[OEM_ICP_IBSW_STATUS_VAL_CNT];
    int gpio_sensor_ids[] = {
	PP_SENSOR_IBSW0_SLOTID,
	PP_SENSOR_IBSW0_POWER_OK,
	PP_SENSOR_IBSW1_SLOTID,
	PP_SENSOR_IBSW1_POWER_OK
    };
    int gpio_actor_ids[] = {
	PP_ACTOR_IBSW0_FAULT_LED,
	PP_ACTOR_IBSW0_GOOD_LED,
	PP_ACTOR_IBSW1_FAULT_LED,
	PP_ACTOR_IBSW1_GOOD_LED
    };
    u_int i, j, k = 0;

    for (j = 0; j < 2; ++j) {
	for (i = j*2; i < j*2+2; ++i) {
	    pp_tp_sensdev_t * sensor = pp_bmc_host_sensor_get_sensor(gpio_sensor_ids[i]);
	    if (!sensor) goto bail;
	    status_vals[k++] = pp_tp_sensdev_get_reading(sensor);
	}
	for (i = j*2; i < j*2+2; ++i) {
	    pp_tp_gpio_act_t * gpio_actor = pp_bmc_host_sensor_get_gpio_actor(gpio_actor_ids[i]);
	    if (!gpio_actor) goto bail;
	    status_vals[k++] = gpio_actor->get_gpio(gpio_actor);
	}
    }

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_icp_get_ibsw_status_values_rs_t));
    memcpy(&rs->status_vals, &status_vals, sizeof(rs->status_vals));
    return pp_bmc_router_send_msg(imsg);

 bail:
    return pp_bmc_router_resp_err(imsg, OEM_ICP_IPMI_ERR_GET_STATUS_VALUES);
}

/* ------------------------------ */
typedef struct {
    unsigned char node_index;
    unsigned char param_select;
    unsigned char param_data[0];
} __attribute__ ((packed)) oem_icp_set_node_config_param_rq_t;

typedef struct {
    BITFIELD2(unsigned char, channel : 4, rsvd : 4);
    unsigned char param_select;
    unsigned char param_data[0];
} __attribute__ ((packed)) set_lan_config_param_rq_t;

/* node 0..5:  addr 0x10..0x15, chan IPMI_CHAN_PRIMARY_IPMB *
 * node 6..11: addr 0x10..0x15, chan IPMI_CHAN_SECDARY_IPMB */
static int oem_icp_node_get_addr(unsigned char nidx, unsigned char* ipmb_chan,
				 unsigned char* ipmb_addr) {
    switch(nidx / 6) {
      case 0: *ipmb_chan = IPMI_CHAN_PRIMARY_IPMB; break;
      case 1: *ipmb_chan = IPMI_CHAN_SECDARY_IPMB; break;
      default: return PP_ERR;
    }
    *ipmb_addr = 0x10 + (nidx % 6);
    return PP_SUC;
}


static int oem_icp_set_node_config_mtrack_cb(imsg_t* imsg, void* ctx) {
    imsg_t* omsg = ctx;
    /* send response code back to originater */
    pp_bmc_imsg_resp(omsg, imsg->data[0], 0);
    pp_bmc_imsg_delete(imsg);
    return pp_bmc_router_send_msg(omsg);
}    

static int oem_icp_node_config_mtrack_to_cb(void* ctx) {
    imsg_t* omsg = ctx;
    return pp_bmc_router_resp_err(omsg, IPMI_ERR_TIMEOUT);
}

static int oem_icp_cmd_set_node_config_parameter(imsg_t* imsg) {
    oem_icp_set_node_config_param_rq_t* rq = (void*)imsg->data;
    set_lan_config_param_rq_t* n_rq;
    imsg_t* nmsg;
    unsigned int param_data_size;
    unsigned char ipmb_addr, ipmb_chan;

    /* check and find node address */
    if (oem_icp_node_get_addr(rq->node_index, &ipmb_chan, &ipmb_addr)
	== PP_ERR) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    /* build message to be send over IPMB */
    param_data_size = imsg->data_size
	- sizeof(oem_icp_set_node_config_param_rq_t);
    nmsg = pp_bmc_imsg_new(param_data_size +sizeof(set_lan_config_param_rq_t));

    /* msg header */
    nmsg->chan       = ipmb_chan;
    nmsg->rs_addr    = ipmb_addr << 1;
    nmsg->rs_lun     = 0;
    nmsg->netfn      = IPMI_NETFN_TRANSPORT;
    //nmsg->rq_addr    = 0; /* filled in by channel adapter */
    //nmsg->rq_lun     = 0; /* filled in by channel adapter */
    //nmsg->rq_seq     = 0; /* filled in by msg_tracker     */
    nmsg->cmd        = IPMI_CMD_SET_LAN_CONFIGURATION;

    /* nmsg data, SET_LAN_CONFIGURATION cmd */
    n_rq = (set_lan_config_param_rq_t*)nmsg->data;
    n_rq->channel      = IPMI_CHAN_LAN; /* default, not specified in oem cmd */
    n_rq->param_select = rq->param_select;
    memcpy(n_rq->param_data, rq->param_data, param_data_size);
    
    /* send message and track response */
    if (PP_ERR == pp_bmc_mtrack_send_msg(nmsg,
					 oem_icp_set_node_config_mtrack_cb,
					 oem_icp_node_config_mtrack_to_cb,
					 imsg)) {
	pp_bmc_log_error("[OEM-ICP]: set_node_config: "
			 "pp_bmc_mtrack_send_msg failed");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_DEST_UNAVAILABLE);
    }
    return PP_SUC;
}

/* ------------------------------ */
typedef struct {
    unsigned char node_index;
    unsigned char param_select;
} __attribute__ ((packed)) oem_icp_get_node_config_param_rq_t;

typedef struct {
    BITFIELD3(unsigned char,
	      channel  : 4,
	      rsvd     : 3,
	      rev_only : 1);
    unsigned char param_select;
    unsigned char set_select;
    unsigned char block_select;
} __attribute__ ((packed)) get_lan_config_param_rq_t;

static int oem_icp_get_node_config_mtrack_cb(imsg_t* imsg, void* ctx) {
    
    imsg_t* omsg = ctx;
    unsigned char* omsg_resp_data;
    unsigned int omsg_resp_size = imsg->data_size - 1 /* resp-code */;
    /* send response code back to originater */
    omsg_resp_data = pp_bmc_imsg_resp(omsg, imsg->data[0], omsg_resp_size);
    memcpy(omsg_resp_data, &imsg->data[1], omsg_resp_size);
    pp_bmc_imsg_delete(imsg);
    return pp_bmc_router_send_msg(omsg);
}    

static int oem_icp_cmd_get_node_config_parameter(imsg_t* imsg) {
    oem_icp_get_node_config_param_rq_t* rq = (void*)imsg->data;
    get_lan_config_param_rq_t* n_rq;
    imsg_t* nmsg;
    unsigned char ipmb_addr, ipmb_chan;

    /* check and find node address */
    if (oem_icp_node_get_addr(rq->node_index, &ipmb_chan, &ipmb_addr)
	== PP_ERR) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    /* build message to be send over IPMB */
    nmsg = pp_bmc_imsg_new(sizeof(get_lan_config_param_rq_t));

    /* msg header */
    nmsg->chan       = ipmb_chan;
    nmsg->rs_addr    = ipmb_addr << 1;
    nmsg->rs_lun     = 0;
    nmsg->netfn      = IPMI_NETFN_TRANSPORT;
    //nmsg->rq_addr    = 0; /* filled in by channel adapter */
    //nmsg->rq_lun     = 0; /* filled in by channel adapter */
    //nmsg->rq_seq     = 0; /* filled in by msg_tracker     */
    nmsg->cmd        = IPMI_CMD_GET_LAN_CONFIGURATION;

    /* nmsg data, GET_LAN_CONFIGURATION cmd */
    n_rq = (get_lan_config_param_rq_t*)nmsg->data;
    n_rq->channel      = IPMI_CHAN_LAN; /* default, not specified in oem cmd */
    n_rq->rev_only     = 0;             /* default, not specified in oem cmd */
    n_rq->param_select = rq->param_select;
    n_rq->set_select   = 0;             /* default, not specified in oem cmd */
    n_rq->block_select = 0;             /* default, not specified in oem cmd */
    
    /* send message and track response */
    if (PP_ERR == pp_bmc_mtrack_send_msg(nmsg,
					 oem_icp_get_node_config_mtrack_cb,
					 oem_icp_node_config_mtrack_to_cb,
					 imsg)) {
	pp_bmc_log_error("[OEM-ICP]: get_lan_config: "
			 "pp_bmc_mtrack_send_msg failed");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_DEST_UNAVAILABLE);
    }
    return PP_SUC;
}

/* ------------------------------ */
typedef struct {
    unsigned char node_index;
    unsigned char asrt; /* 1 -> do assert, 0 -> nothing (strange?) */
    unsigned char asrt_duration; /* in s, 0 -> 250 ms */
} __attribute__ ((packed)) oem_icp_assert_power_rq_t;

static int oem_icp_cmd_assert_power(imsg_t* imsg) {
    oem_icp_assert_power_rq_t* rq = (void*)imsg->data;
    int ret;

    if (rq->node_index > ICP_NODE_INDEX_MAX) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    if (rq->asrt) {
	if (rq->asrt_duration == 0) {
	    ret = pp_hal_icpmmd_power_blade(rq->node_index, 250, 0);
	} else {
	    /* non blocking call */
	    ret = pp_hal_icpmmd_power_blade(rq->node_index,
					    1000 * rq->asrt_duration, 1);
	}
	if (ret == PP_ERR) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_NODE_BUSY);
	}
    }
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/* ------------------------------ */
typedef struct oem_icp_assert_reset_rq_s {
    unsigned char node_index;
    unsigned char asrt;
    unsigned char asrt_duration; /* in 100 ms, 0 -> 250 ms */
} __attribute__ ((packed)) oem_icp_assert_reset_rq_t;

static int oem_icp_cmd_assert_reset(imsg_t* imsg) {
    oem_icp_assert_reset_rq_t* rq = (void*)imsg->data;
    int ret;

    if (rq->node_index > ICP_NODE_INDEX_MAX) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    if (rq->asrt) {
	if (rq->asrt_duration == 0) {
	    ret = pp_hal_icpmmd_reset_blade(rq->node_index, 250, 0);
	} else if (rq->asrt_duration < 3) {
	    ret = pp_hal_icpmmd_reset_blade(rq->node_index,
					    100 * rq->asrt_duration, 0);
	} else {
	    /* non blocking call */
	    ret = pp_hal_icpmmd_reset_blade(rq->node_index,
					    100 * rq->asrt_duration, 1);
	}
	if (ret == PP_ERR) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_NODE_BUSY);
	}
    }
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/* ------------------------------ */
typedef struct oem_icp_assert_bmc_reset_rq_s {
    unsigned char node_index;
} __attribute__ ((packed)) oem_icp_assert_bmc_reset_rq_t;

static int oem_icp_cmd_assert_bmc_reset(imsg_t* imsg) {
    oem_icp_assert_bmc_reset_rq_t* rq = (void*)imsg->data;

    if (rq->node_index > ICP_NODE_INDEX_MAX) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    if (PP_ERR == pp_hal_icpmmd_reset_bmc(rq->node_index)) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NODE_BUSY);
    }
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/* ------------------------------ */
typedef struct {
    unsigned char node_index;
} __attribute__ ((packed)) oem_icp_cmd_set_kvm_status_rq_t;

static int
oem_icp_cmd_set_kvm_status(imsg_t* imsg)
{
    oem_icp_cmd_set_kvm_status_rq_t* rq = (void*)imsg->data;

    if (rq->node_index > ICP_NODE_INDEX_MAX) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    if (PP_ERR == pp_hal_icpmmd_switch_kvm(rq->node_index)) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NODE_BUSY);
    }
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/* ------------------------------ */
typedef struct {
    unsigned char selected_node;
} __attribute__ ((packed)) oem_icp_cmd_get_kvm_status_rs_t;

static int
oem_icp_cmd_get_kvm_status(imsg_t* imsg)
{
    oem_icp_cmd_get_kvm_status_rs_t* rs;
    unsigned char sel_node;

    if (PP_ERR == pp_hal_icpmmd_get_selected_blade(&sel_node)) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NODE_BUSY);
    }
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
			  sizeof(oem_icp_cmd_get_kvm_status_rs_t));
    rs->selected_node = sel_node;
    return pp_bmc_router_send_msg(imsg);
}

/*
 * default: error
 * ----------------
static int oem_icp_cmd_default(imsg_t* imsg)
{
    pp_bmc_log_warn("[OEM-ICP] unsupported OEM cmd called: "
		    "netfn/cmd=0x%02x/0x%02x", imsg->netfn, imsg->cmd);
    pp_bmc_imsg_dump(PP_BMC_LOG_WARN, "[OEM-ICP]", imsg);
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_CMD);
}
 */

/********************************************************************
 * Device command table
 */

static const dev_cmd_entry_t oem_icp_cmd_tab[] = {
    {
        .cmd_hndlr = oem_icp_cmd_set_config_parameter,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_SET_CONFIG_PARAMETER,
        .min_data_size = 5,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_get_config_parameter,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_GET_CONFIG_PARAMETER,
        .min_data_size = 4,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_get_cfu_sensor_values,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_GET_CFU_SENSOR_VALUES,
        .min_data_size = 1,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_get_psu_sensor_values,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_GET_PSU_SENSOR_VALUES,
        .min_data_size = 1,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_get_ibsw_sensor_values,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_GET_IBSW_SENSOR_VALUES,
        .min_data_size = 1,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_get_cfu_status_values,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_GET_CFU_STATUS_VALUES,
        .min_data_size = 1,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_get_psu_status_values,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_GET_PSU_STATUS_VALUES,
        .min_data_size = 1,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_get_ibsw_status_values,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_GET_IBSW_STATUS_VALUES,
        .min_data_size = 1,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_set_node_config_parameter,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_SET_NODE_CONFIG_PARAMETERS,
        .min_data_size = 3,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_get_node_config_parameter,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_GET_NODE_CONFIG_PARAMETERS,
        .min_data_size = 2,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_assert_power,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_SET_NODE_POWER,
        .min_data_size = 2,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_assert_reset,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_SET_NODE_RESET,
        .min_data_size = 2,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_assert_bmc_reset,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_SET_NODE_BMC_RESET,
        .min_data_size = 1,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_set_kvm_status,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_SET_KVM_STATUS,
        .min_data_size = 1,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_icp_cmd_get_kvm_status,
        .netfn = IPMI_NETFN_OEM_ICP,
	.cmd = IPMI_CMD_ICP_GET_KVM_STATUS,
        .min_data_size = 1,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    { .cmd_hndlr = NULL }
};

/********************************************************************
 * Device c'tor/d'tor
 */

int pp_bmc_dev_oem_icp_init(void)
{
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(oem_icp_cmd_tab)) return PP_ERR;

    pp_bmc_log_info("[OEM-ICP] device started");
    return PP_SUC;
}

void pp_bmc_dev_oem_icp_cleanup(void)
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(oem_icp_cmd_tab);

    pp_bmc_log_info("[OEM-ICP] device shut down");
}
