/**
 * bmc_dev_oem_msi.c
 *
 * Description: BMC MSI-OEM Device
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 * 
 * Note: the MSI 9247 hack is still integrated in this code
 */

#include <pp/base.h>
#include <pp/cfg.h>

#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/host_power.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/host_sensors.h>
#include <pp/bmc/tp_static_cond.h>

#include "bmc_dev_oem_msi.h"

struct {
    unsigned char speed;
    unsigned char type;
} cpu_info[4];

struct {
    unsigned char rack_type;
    unsigned char model;
    unsigned char system;
} system_info;

/********************************************************************
 * Device command handlers
 */

/**
 * Disable serial port sharing. Map to channel access mode enable/disable.
 */
static int oem_msi_cmd_disable_serial_port_sharing(imsg_t* imsg)
{
    pp_bmc_router_chan_adapter_config_t* serial_config;

    pp_bmc_log_info("[OEM-MSI] disable serial port sharing");
    serial_config = pp_bmc_router_channel_get_config(IPMI_CHAN_SERIAL);
    if (serial_config == NULL) {
        return IPMI_ERR_UNSPECIFIED;
    }
    
    serial_config->access_mode = IPMI_CHAN_ACCESS_DISABLE;
    pp_bmc_router_channel_config_updated(IPMI_CHAN_SERIAL);
    
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/**
 * Enable serial port sharing. Map to channel access mode enable/disable.
 */
static int oem_msi_cmd_enable_serial_port_sharing(imsg_t* imsg)
{
    pp_bmc_router_chan_adapter_config_t* serial_config;

    pp_bmc_log_info("[OEM-MSI] enable serial port sharing");
    serial_config = pp_bmc_router_channel_get_config(IPMI_CHAN_SERIAL);
    if (serial_config == NULL) {
        return IPMI_ERR_UNSPECIFIED;
    }
    
    serial_config->access_mode = IPMI_CHAN_ACCESS_SHARED;
    pp_bmc_router_channel_config_updated(IPMI_CHAN_SERIAL);

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Write BMC default config. Performs a 'reset to defaults'.
 * This is a very crude command but highly effective. Do not
 * use lightly.
 */
static int oem_msi_cmd_write_default_config(imsg_t* imsg)
{
    pp_bmc_log_warn("[OEM-MSI] Write default config - reseting device to defaults");
    
    /* might not be sent due to reset */
    pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
    usleep(1000);
    
    /* "touch" the reset to defaults file and quit. eric.sh will do the rest */
    system("echo \"\" | cat > /tmp/reset_to_defaults");
    exit(0);
    /* any errors here will be ignored */
    
    return PP_SUC;
}

/**
 * Set the bmc system guid.
 */
static int oem_msi_cmd_set_system_guid(imsg_t* imsg)
{
    /* store in config system so that the value can easily be used in other parts */
    if ((pp_cfg_set_binary(imsg->data, 16, "bmc_oem_msi.device_guid") == PP_ERR) ||
        (pp_cfg_save(DO_FLUSH) == PP_ERR) ) 
    {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static int oem_msi_cmd_set_cpu_info(imsg_t* imsg)
{
    int cpu_num;
    int cpu_cond_id;
    pp_tp_cond_t* cond;

    pp_bmc_log_info("[OEM-MSI] CPU%d: %d.%dGHz, Type %d",
		    imsg->data[0], imsg->data[1] >> 4, imsg->data[1] & 0x0f,
		    imsg->data[2]);

    if (imsg->data[0] >= 1 && imsg->data[0] <= 4) {
	cpu_num = imsg->data[0] - 1;
        cpu_info[cpu_num].speed = imsg->data[1];
        cpu_info[cpu_num].type = imsg->data[2];

	/* set cpu presence condition for sensors */
	cpu_cond_id = cpu_num + PP_BMC_COND_CPU_PRESENCE_BASE;
	if   ( (cpu_cond_id <= PP_BMC_COND_CPU_PRESENCE_LAST)
	    && (NULL != (cond = pp_bmc_host_get_condition(cpu_cond_id)))
	    && (PP_TP_OBJ_IS_TYPE(PP_TP_STATIC_COND, cond)) )
        {
	    pp_bmc_tp_static_cond_set((pp_tp_static_cond_t*)cond, imsg->data[1]);
	}
	
    } else {
        pp_bmc_log_warn("[OEM-MSI] CPU%d not supported (only 1..4), info ignored", imsg->data[0]);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static int oem_msi_cmd_set_system_info(imsg_t* imsg)
{
    pp_bmc_log_info("[OEM-MSI] %s Rack Type, %s Model, %s System",
        imsg->data[0] == 0 ? "1U"            : imsg->data[0] == 1 ? "2U"        : "unknown",
        imsg->data[1] == 0 ? "SCSI"          : imsg->data[1] == 1 ? "SATA"      : "unknown",
        imsg->data[2] == 0 ? "Non-redundant" : imsg->data[2] == 1 ? "Redundant" : "unknown");

    system_info.rack_type = imsg->data[0];
    system_info.model = imsg->data[1];
    system_info.system = imsg->data[2];
    
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/**
 * Get smi request info. Returns what type of smi request the bmc
 * returns to the bios. We dont return any interrupts yet, so we
 * return 0 hardwired.
 */
static int oem_msi_cmd_get_smi_request(imsg_t* imsg) {
    unsigned char* c;
    c = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 1);
    *c = 0;
    return pp_bmc_router_send_msg(imsg);
}

/**
 * Get the serial port status
 */
static int oem_msi_cmd_get_serial_port_status(imsg_t* imsg) {
    unsigned char* ret;
    pp_bmc_router_chan_adapter_config_t* serial_config;

    serial_config = pp_bmc_router_channel_get_config(IPMI_CHAN_SERIAL);
    if (serial_config == NULL) {
        return IPMI_ERR_UNSPECIFIED;
    }
    
    ret = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 1);
    if (serial_config->access_mode == IPMI_CHAN_ACCESS_DISABLE) {
        *ret = 2;  // disabled
    } else {
        *ret = 0;  // enabled (shared)
    }
    
    return pp_bmc_router_send_msg(imsg);
}

/**
 * Ready for hardware monitor command. Sent by BIOS to indicate that
 * the bmc can use the i2c bus.
 */
static int oem_msi_cmd_ready_for_hardware_monitor(imsg_t* imsg) {
    /* We cant implement anything here until we know  *
     * what this command should really do.            */   
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * set/get pwm
 * ---------------
 */
static pp_tp_pwm_act_t* oem_msi_get_pwm_actor(int pwmid) {
    pp_tp_pwm_act_t* o;

    if (NULL == (o = pp_bmc_host_sensor_get_pwm_actor(pwmid))) {
	pp_bmc_log_warn("[OEM-MSI] pwm-actor (%d) not registered", pwmid);
	return NULL;
    }
    return o;
}

/*
 * in case pwm is not found in hardware lib it is a configuration
 * error. in this case oem_msi_get_pwm_actor will issue a warning
 * and we'll return dummy values
 */
static int oem_msi_get_pwm_info(int pwmid, pp_tp_pwm_mode_t* mode_ptr,
			   unsigned char* dc_ptr) {
    pp_tp_pwm_act_t* pwm;

    *dc_ptr = 0;
    *mode_ptr = PP_TP_PWM_MANUAL;
    if (NULL != (pwm = oem_msi_get_pwm_actor(pwmid))) {
	if (PP_ERR == pp_tp_pwm_get_mode(pwm, mode_ptr)) {
	    return PP_ERR;
	}
	if (*mode_ptr == PP_TP_PWM_MANUAL) {
	    if (PP_ERR == pp_tp_pwm_get_duty_cycle(pwm, dc_ptr)) {
		return PP_ERR;
	    }
	}
    }
    return PP_SUC; /* ignore PWMs that are not registered */
}

struct oem_msi_get_pwm_rs_s {
    unsigned char pwm1;
    unsigned char pwm2;
    unsigned char pwm3;
} __attribute__ ((packed));
typedef struct oem_msi_get_pwm_rs_s oem_msi_get_pwm_rs_t;

static int oem_msi_cmd_get_pwm(imsg_t* imsg) {
    oem_msi_get_pwm_rs_t* rs;
    unsigned char duty_cycle1, duty_cycle2, duty_cycle3;
    pp_tp_pwm_mode_t mode1, mode2, mode3;

    if (PP_ERR == oem_msi_get_pwm_info(1, &mode1, &duty_cycle1) ||
	PP_ERR == oem_msi_get_pwm_info(2, &mode2, &duty_cycle2) ||
	PP_ERR == oem_msi_get_pwm_info(3, &mode3, &duty_cycle3)) {
	return pp_bmc_router_resp_err(imsg, 0x82 /* MSI I2C bus error */);
    } else if (mode1 != PP_TP_PWM_MANUAL || mode2 != PP_TP_PWM_MANUAL ||
	       mode3 != PP_TP_PWM_MANUAL) {
	return pp_bmc_router_resp_err(imsg,
				      IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE);
    } else {
	rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_msi_get_pwm_rs_t));
	rs->pwm1 = duty_cycle1;
	rs->pwm2 = duty_cycle2;
	rs->pwm3 = duty_cycle3;
    }
    return pp_bmc_router_send_msg(imsg);
}

static int oem_msi_set_pwm_mode(int pwmid, pp_tp_pwm_mode_t mode) {
    pp_tp_pwm_act_t* pwm;
    if (NULL != (pwm = oem_msi_get_pwm_actor(pwmid))) {
	return pp_tp_pwm_set_mode(pwm, mode);
    }
    return PP_SUC; // ignore pwm's that are not registered
}

static int oem_msi_set_pwm_duty_cycle(int pwmid, unsigned char duty_cycle) {
    pp_tp_pwm_act_t* pwm;
    if (NULL != (pwm = oem_msi_get_pwm_actor(pwmid))) {
	return pp_tp_pwm_set_duty_cycle(pwm, duty_cycle);
    }
    return PP_SUC; // ignore pwm's that are not registered
}

struct oem_mse_set_pwm_rq_s {
    unsigned char target; /* 00: manual/auto, 01: pwm1, 02 pwm2, 03 pwm3 */
    unsigned char value;  /* on/off for target 00, pwm duty cycle else */
} __attribute__ ((packed));
typedef struct oem_mse_set_pwm_rq_s oem_mse_set_pwm_rq_t;

static int oem_msi_cmd_set_pwm(imsg_t* imsg) {
    oem_mse_set_pwm_rq_t* rq = (void*)imsg->data;
    pp_tp_pwm_mode_t mode;
    int ret = PP_SUC;

    switch (rq->target) {
      case 0x0:
	  if (rq->value == 0x00) {
	      mode = PP_TP_PWM_MANUAL;
	  } else {
	      mode = PP_TP_PWM_AUTO;
	  }
	  if (PP_ERR == oem_msi_set_pwm_mode(1, mode) ||
	      PP_ERR == oem_msi_set_pwm_mode(2, mode) ||
              PP_ERR == oem_msi_set_pwm_mode(3, mode) ||
              PP_ERR == oem_msi_set_pwm_mode(4, mode) ||
              PP_ERR == oem_msi_set_pwm_mode(5, mode) ||
	      PP_ERR == oem_msi_set_pwm_mode(6, mode))
          {
	      ret = PP_ERR;
	  }
	  break;
      case 0x1:
      case 0x2:
      case 0x3:
      case 0x4:
      case 0x5:
      case 0x6:
	  ret = oem_msi_set_pwm_duty_cycle(rq->target, rq->value);
	  break;
      default:
	  return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    if (ret == PP_ERR) {
	return pp_bmc_router_resp_err(imsg, 0x82 /* MSI I2C bus error */);
    }
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * default: error
 * ----------------
 */
static int oem_msi_cmd_default(imsg_t* imsg)
{
    pp_bmc_log_warn("[OEM-MSI] unsupported OEM cmd called: "
		    "netfn/cmd=0x%02x/0x%02x", imsg->netfn, imsg->cmd);
    pp_bmc_imsg_dump(PP_BMC_LOG_WARN, "[OEM-MSI]", imsg);
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_CMD);
}

/* MSI9247 sometimes stucks at start-up.
 * In that situation, the BIOS send always
 * the unknown OEM netfn/cmd 0x30/0x28.
 * To workaround this, we just make a hard reset here.
 * The second start-up usualy succeeds.
 */
static int oem_msi_reboot_hack(imsg_t* imsg)
{
    pp_bmc_log_error("[BMCcore] ,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,");
    pp_bmc_log_error("[BMCcore] frozen MSI9247 detected: force quick power cycle!!!");
    pp_bmc_log_error("[BMCcore] ```````````````````````````````````````````````````");

    pp_bmc_host_power_control(HOST_POWER_CONTROL_OFF,
                              HOST_POWER_CONTROL_SOURCE_CHASSIS_CONTROL,
                              imsg->chan);
    sleep(10);
    pp_bmc_host_power_control(HOST_POWER_CONTROL_ON,
                              HOST_POWER_CONTROL_SOURCE_CHASSIS_CONTROL,
                              imsg->chan);

    pp_bmc_imsg_delete(imsg); /* dont send reply since host is in power cycle */
    return PP_SUC;
}

/********************************************************************
 * Device command table
 */

static const dev_cmd_entry_t oem_msi_cmd_tab[] = {
    {
        .cmd_hndlr = oem_msi_cmd_disable_serial_port_sharing,
        .netfn = IPMI_NETFN_OEM_MSI,
	.cmd = IPMI_CMD_MSI_DISABLE_SERIAL_PORT_SHARING,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_enable_serial_port_sharing,
        .netfn = IPMI_NETFN_OEM_MSI,
	.cmd = IPMI_CMD_MSI_ENABLE_SERIAL_PORT_SHARING,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_enable_serial_port_sharing,
        .netfn = IPMI_NETFN_OEM_MSI,
	.cmd = IPMI_CMD_MSI_ENABLE_SERIAL_PORT_SHARING2,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_write_default_config,
        .netfn = IPMI_NETFN_OEM_MSI,
	.cmd = IPMI_CMD_MSI_WRITE_BMC_DEFAULT_CONFIG,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_set_system_guid,
        .netfn = IPMI_NETFN_OEM_MSI, .cmd = IPMI_CMD_MSI_SET_DEV_GUID,
        .min_data_size = 16, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_set_cpu_info,
        .netfn = IPMI_NETFN_OEM_MSI, .cmd = IPMI_CMD_MSI_SET_CPU_INFO,
        .min_data_size = 3, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_default,
        .netfn = IPMI_NETFN_OEM_MSI, .cmd = IPMI_CMD_MSI_DIMM_FAULT,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_set_system_info,
        .netfn = IPMI_NETFN_OEM_MSI, .cmd = IPMI_CMD_MSI_SET_SYSTEM_INFO,
        .min_data_size = 3, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_get_smi_request,
        .netfn = IPMI_NETFN_OEM_MSI, .cmd = IPMI_CMD_MSI_GET_SMI_REQUEST,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_get_serial_port_status,
        .netfn = IPMI_NETFN_OEM_MSI,
	.cmd = IPMI_CMD_MSI_READ_SERIAL_PORT_STATUS,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_ready_for_hardware_monitor,
        .netfn = IPMI_NETFN_OEM_MSI,
	.cmd = IPMI_CMD_MSI_READY_FOR_HARDWARE_MONITOR,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },

    {
        .cmd_hndlr = oem_msi_cmd_default,
        .netfn = IPMI_NETFN_OEM_MSI2, .cmd = IPMI_CMD_MSI_LED_CONTROL,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_get_pwm,
        .netfn = IPMI_NETFN_OEM_MSI2,
	.cmd = IPMI_CMD_MSI_GET_PWM_OF_FAN_SPEED_CONTROL,
        .min_data_size = 0,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_cmd_set_pwm,
        .netfn = IPMI_NETFN_OEM_MSI2,
	.cmd = IPMI_CMD_MSI_SET_PWM_OF_FAN_SPEED_CONTROL,
        .min_data_size = sizeof(oem_mse_set_pwm_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_msi_reboot_hack,
        .netfn = 0x30, .cmd = 0x28,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
        // don't worry about privlevel here, always called from system interface
    },

    { .cmd_hndlr = NULL }
};

/********************************************************************
 * Device c'tor/d'tor
 */

int pp_bmc_dev_oem_msi_init()
{
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(oem_msi_cmd_tab)) return PP_ERR;

    pp_bmc_log_info("[OEM-MSI] device started");
    return PP_SUC;
}

void pp_bmc_dev_oem_msi_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(oem_msi_cmd_tab);

    pp_bmc_log_info("[OEM-MSI] device shut down");
}
