/**
 * bmc_dev_oem_pp.c
 *
 * Description: BMC Peppercon OEM commands
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 *                        Ingo van Lil <inva@peppercon.de>
 */

#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <zlib.h>

#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/firmware.h>
#if defined (PP_FEAT_MASS_STORAGE)
#include <pp/usb.h>
#endif
#include <liberic_config.h>
#include <liberic_misc.h>

#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/host_sensors.h>
#include <pp/bmc/utils.h>
#include <pp/bmc/user_manager.h>

#include "bmc_dev_oem_pp.h"

/*
 * Static Vars
 */

/* Firmware upgrade */
static reserv_t *fw_reserv = NULL;
static pp_firmware_ctx_t fw_ctx = 0;
static unsigned int fw_size;
static unsigned int fw_cur_size;
static const char *fw_update_err = NULL;

pthread_t flash_thread;
pthread_mutex_t flash_state_mtx = PTHREAD_MUTEX_INITIALIZER;
typedef enum {
    FLASH_IDLE,
    FLASH_IN_PROGRESS,
    FLASH_FINISHED,
    FLASH_ERROR
} flash_state_t;
static volatile flash_state_t flash_state = FLASH_IDLE;
static int flash_dont_reboot = 0;

static inline flash_state_t get_flash_state(void)
{
    flash_state_t ret;
    pthread_mutex_lock(&flash_state_mtx);
    ret = flash_state;
    pthread_mutex_unlock(&flash_state_mtx);
    return ret;
}

static inline void set_flash_state(flash_state_t state)
{
    pthread_mutex_lock(&flash_state_mtx);
    flash_state = state;
    pthread_mutex_unlock(&flash_state_mtx);
}

/* Config file backup/restore */
#define CFG_BACKUP_TEMP_FILE "/tmp/cfg_backup"
static reserv_t *cfg_reserv = NULL;
static enum { CFG_IDLE, CFG_BACKUP, CFG_RESTORE } cfg_mode = CFG_IDLE;
static int cfg_fd;
static ssize_t cfg_total_size;
static ssize_t cfg_current_size;
static unsigned int cfg_crc_expected, cfg_crc_actual;

/* Virtual Media */
#ifdef PP_FEAT_MASS_STORAGE
struct vmedia_data_s {
    int reserv_active;
    reserv_t *reserv;
    pp_usb_vfup_ctx_t *vfup_ctx;
    pp_usb_smbmnt_ctx_t *smbmnt_ctx;
};

static struct vmedia_data_s vmedia_data[PP_FEAT_USB_MASS_STORAGE_NO];
#endif

/*
 * set/get pwm
 * ---------------
 */
static pp_tp_pwm_act_t* oem_pp_get_pwm_actor(int pwmid) {
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    pp_tp_pwm_act_t* o;
    if (NULL == (o = pp_bmc_host_sensor_get_pwm_actor(pwmid))) {
	pp_bmc_log_warn("[OEM-PP] pwm-actor (%d) not registered", pwmid);
	return NULL;
    }
    return o;
#else /* PP_FEAT_BMC_OEMCMDS_ONLY */
    (void)pwmid;
    return NULL;
#endif /* PP_FEAT_BMC_OEMCMDS_ONLY */
}

static int oem_pp_get_pwm_info(int pwmid, pp_tp_pwm_mode_t* mode_ptr,
			   unsigned char* dc_ptr) {
    pp_tp_pwm_act_t* pwm;

    *dc_ptr = 0;
    *mode_ptr = PP_TP_PWM_MANUAL;
    if (NULL != (pwm = oem_pp_get_pwm_actor(pwmid))) {
	if (PP_ERR == pp_tp_pwm_get_mode(pwm, mode_ptr)) {
	    return PP_ERR;
	}
	if (*mode_ptr == PP_TP_PWM_MANUAL) {
	    if (PP_ERR == pp_tp_pwm_get_duty_cycle(pwm, dc_ptr)) {
		return PP_ERR;
	    }
	}
    }
    return PP_SUC; /* ignore PWMs that are not registered */
}

/*
 * Get PWM
 */
struct oem_pp_get_pwm_rs_s {
    unsigned char pwm1;
    unsigned char pwm2;
    unsigned char pwm3;
} __attribute__ ((packed));
typedef struct oem_pp_get_pwm_rs_s oem_pp_get_pwm_rs_t;

static int oem_pp_cmd_get_pwm(imsg_t* imsg) {
    oem_pp_get_pwm_rs_t* rs;
    unsigned char duty_cycle1, duty_cycle2, duty_cycle3;
    pp_tp_pwm_mode_t mode1, mode2, mode3;

    if (PP_ERR == oem_pp_get_pwm_info(1, &mode1, &duty_cycle1) ||
	PP_ERR == oem_pp_get_pwm_info(2, &mode2, &duty_cycle2) ||
	PP_ERR == oem_pp_get_pwm_info(3, &mode3, &duty_cycle3)) {
	return pp_bmc_router_resp_err(imsg, 0x82 /* MSI I2C bus error */);
    } else if (mode1 != PP_TP_PWM_MANUAL || mode2 != PP_TP_PWM_MANUAL ||
	       mode3 != PP_TP_PWM_MANUAL) {
	return pp_bmc_router_resp_err(imsg,
				      IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE);
    } else {
	rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_pp_get_pwm_rs_t));
	rs->pwm1 = duty_cycle1;
	rs->pwm2 = duty_cycle2;
	rs->pwm3 = duty_cycle3;
    }
    return pp_bmc_router_send_msg(imsg);
}

static int oem_pp_set_pwm_mode(int pwmid, pp_tp_pwm_mode_t mode) {
    pp_tp_pwm_act_t* pwm;
    if (NULL != (pwm = oem_pp_get_pwm_actor(pwmid))) {
	return pp_tp_pwm_set_mode(pwm, mode);
    }
    return PP_SUC; // ignore pwm's that are not registered
}

static int oem_pp_set_pwm_duty_cycle(int pwmid, unsigned char duty_cycle) {
    pp_tp_pwm_act_t* pwm;
    if (NULL != (pwm = oem_pp_get_pwm_actor(pwmid))) {
	return pp_tp_pwm_set_duty_cycle(pwm, duty_cycle);
    }
    return PP_SUC; // ignore pwm's that are not registered
}

/*
 * Set PWM
 */
struct oem_pp_set_pwm_rq_s {
    unsigned char target; /* 00: manual/auto, 01: pwm1, 02 pwm2, 03 pwm3 */
    unsigned char value;  /* on/off for target 00, pwm duty cycle else */
} __attribute__ ((packed));
typedef struct oem_pp_set_pwm_rq_s oem_pp_set_pwm_rq_t;

static int oem_pp_cmd_set_pwm(imsg_t* imsg) {
    oem_pp_set_pwm_rq_t* rq = (void*)imsg->data;
    pp_tp_pwm_mode_t mode;
    int ret = PP_SUC;

    switch (rq->target) {
      case 0x0:
	  if (rq->value == 0x00) {
	      mode = PP_TP_PWM_MANUAL;
	  } else {
	      mode = PP_TP_PWM_AUTO;
	  }
	  if (PP_ERR == oem_pp_set_pwm_mode(1, mode) ||
	      PP_ERR == oem_pp_set_pwm_mode(2, mode) ||
	      PP_ERR == oem_pp_set_pwm_mode(3, mode)) {
	      ret = PP_ERR;
	  }
	  break;
      case 0x1:
      case 0x2:
      case 0x3:
	  ret = oem_pp_set_pwm_duty_cycle(rq->target, rq->value);
	  break;
      default:	
	  return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    if (ret == PP_ERR) {
	return pp_bmc_router_resp_err(imsg, 0x82 /* MSI I2C bus error */);
    }
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}



/*
 * Firmware Get Version
 */
struct oem_pp_firmware_get_version_rs_s {
    int ver_le32[4]; // major, minor, subminor, build#
    unsigned char hwid;
    char tag_oem[0]; // tag and oem string separated by '\0'
} __attribute__ ((packed));
typedef struct oem_pp_firmware_get_version_rs_s oem_pp_firmware_get_version_rs_t;

static int oem_pp_cmd_firmware_get_version(imsg_t* imsg)
{
    const char *oem = PP_STRINGIFY_EXPANDED(PP_OEM);
    int sz = strlen(pp_firmware_erla_tag) + 1 + strlen(oem) + 1;
    oem_pp_firmware_get_version_rs_t* rs = pp_bmc_imsg_resp(imsg,
        IPMI_ERR_SUCCESS, sizeof(oem_pp_firmware_get_version_rs_t) + sz);

    rs->ver_le32[0] = cpu_to_le32(pp_firmware_erla_version_major);
    rs->ver_le32[1] = cpu_to_le32(pp_firmware_erla_version_minor);
    rs->ver_le32[2] = cpu_to_le32(pp_firmware_erla_version_subminor);
    rs->ver_le32[3] = cpu_to_le32(pp_firmware_erla_build_nr);

    rs->hwid = eric_misc_get_hardware_rev_int();

    snprintf(rs->tag_oem, sz, "%s%c%s", pp_firmware_erla_tag, '\0', oem);

    pp_bmc_log_info("[OEM-PP] Firmware Get Version: ver=%d.%d.%d.%d hwid=%02x tag=%s oem=%s",
        pp_firmware_erla_version_major, pp_firmware_erla_version_minor,
        pp_firmware_erla_version_subminor, pp_firmware_erla_build_nr,
        rs->hwid, pp_firmware_erla_tag, oem);

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Firmware Start Upgrade
 */
struct oem_pp_firmware_start_upgrade_rq_s {
    unsigned int size_le32;
} __attribute__ ((packed));
typedef struct oem_pp_firmware_start_upgrade_rq_s oem_pp_firmware_start_upgrade_rq_t;

struct oem_pp_firmware_start_upgrade_rs_s {
    unsigned short res_id_le16;
    unsigned int max_chunk_size_le32;
} __attribute__ ((packed));
typedef struct oem_pp_firmware_start_upgrade_rs_s oem_pp_firmware_start_upgrade_rs_t;

#define MAX_CHUNK_SIZE 1000

static int oem_pp_cmd_firmware_start_upgrade(imsg_t* imsg)
{
    oem_pp_firmware_start_upgrade_rq_t *rq = (void*)imsg->data;
    oem_pp_firmware_start_upgrade_rs_t *rs;
    int size = le32_to_cpu(rq->size_le32);

    flash_state_t state = get_flash_state();
    if (state == FLASH_IN_PROGRESS) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NODE_BUSY);
    } else if (state == FLASH_ERROR) {
	// reap the luckless flash thread so we can start all over
	pthread_join(flash_thread, NULL);
    }

    // cancel old upgrade process
    if (fw_ctx) {
        pp_bmc_log_warn("[OEM-PP] Start Firmware Upgrade: cancel another firmware upgrade in progress");
        pp_firmware_erla_ctx_free(fw_ctx);
        reserv_cancel(fw_reserv);
        // TODO stop old timer
    }

    // create firmware upgrade context
    fw_ctx = pp_firmware_erla_ctx_new(size, 1);
    if (!fw_ctx) {
        pp_bmc_log_warn("[OEM-PP] Start Firmware Upgrade: unable to init firmware upgrade");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }

    // init global values
    fw_size = size;
    fw_cur_size = 0;

    // start timer
    // TODO start timer

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_pp_firmware_start_upgrade_rs_t));
    rs->res_id_le16 = cpu_to_le16(reserv_make(fw_reserv, imsg));
    rs->max_chunk_size_le32 = cpu_to_le32(MAX_CHUNK_SIZE);
    return pp_bmc_router_send_msg(imsg);
}

/*
 * Firmware Upload
 */
struct oem_pp_firmware_upload_rq_s {
    unsigned short res_id_le16;
    unsigned int offs_le32;
    unsigned char junk[0];
} __attribute__ ((packed));
typedef struct oem_pp_firmware_upload_rq_s oem_pp_firmware_upload_rq_t;

static int oem_pp_cmd_firmware_upload(imsg_t* imsg)
{
    oem_pp_firmware_upload_rq_t *rq = (void*)imsg->data;
    size_t len = imsg->data_size - sizeof(oem_pp_firmware_upload_rq_t);
    unsigned int offs = le32_to_cpu(rq->offs_le32);
    flash_state_t state = get_flash_state();

    // check reservation
    if (PP_FAILED(reserv_check(fw_reserv, le16_to_cpu(rq->res_id_le16), 1, imsg))) {
        pp_bmc_log_warn("[OEM-PP] Upload Firmware: bad reservation");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }
    // TODO retrigger timeout

    if (state != FLASH_IDLE) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NODE_BUSY);
    }

    // check size and junk idx
    if (fw_cur_size + len > fw_size || offs != fw_cur_size) {
        pp_bmc_log_warn("[OEM-PP] Upload Firmware: chunk size too big or unexpected chunk offset "
                        "(fw: size=%d offs=%d, chunk: size=%d offs=%d)",
                        fw_size, fw_cur_size, len, offs);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    // store junk
    if (PP_FAILED(pp_firmware_erla_data_set(fw_ctx, rq->junk, fw_cur_size, len))) {
        pp_bmc_log_warn("[OEM-PP] Upload Firmware: failed to process chunk");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    fw_cur_size += len;

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static void *flash_thread_func(void *arg UNUSED)
{
    if (PP_FAILED(pp_firmware_erla_update(fw_ctx))) {
	set_flash_state(FLASH_ERROR);
	pp_bmc_log_warn("[OEM-PP] Flash Firmware: update failed: %s", fw_update_err);
	fw_update_err = pp_error_string(errno);
    } else if (!flash_dont_reboot) {
	set_flash_state(FLASH_FINISHED);
	pp_bmc_log_warn("[OEM-PP] Finalize Firmware Flash: update successful, rebooting device in 5 seconds ...");
	sleep(5);
	pp_firmware_erla_reset_device(0);
	// not reached
    } else {
	set_flash_state(FLASH_FINISHED);
	pp_bmc_log_warn("[OEM-PP] Finalize Firmware Flash: update successful, not rebooting ...");
    }
    return NULL;
}

/*
 * Firmware Flash
 */
struct oem_pp_firmware_flash_rq_s {
    unsigned short res_id_le16;
    BITFIELD4(unsigned char, validate_only : 1, cross_oem : 1, cross_hwid : 1, dont_reboot : 1);
} __attribute__ ((packed));
typedef struct oem_pp_firmware_flash_rq_s oem_pp_firmware_flash_rq_t;

#define IPMI_ERR_FW_INCOMPLETE  0x80
#define IPMI_ERR_FW_BAD_CRC     0x81
#define IPMI_ERR_FW_INVALID     0x82

static int oem_pp_cmd_firmware_flash(imsg_t* imsg)
{
    oem_pp_firmware_flash_rq_t *rq = (void*)imsg->data;
    char *rs;
    int sz;
    flash_state_t state = get_flash_state();

    // check reservation
    if (PP_FAILED(reserv_check(fw_reserv, le16_to_cpu(rq->res_id_le16), 1, imsg))) {
        pp_bmc_log_warn("[OEM-PP] Flash Firmware: bad reservation");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }
    // TODO retrigger timeout

    if (state != FLASH_IDLE) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NODE_BUSY);
    }

    // check if firmware was completely uploaded
    if (fw_cur_size != fw_size) {
        pp_bmc_log_warn("[OEM-PP] Flash Firmware: size mismatch (size=%d uploaded=%d)",
                        fw_size, fw_cur_size);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_FW_INCOMPLETE);
    }

    // check firmware combatibility
    if (PP_FAILED(pp_firmware_erla_validate(fw_ctx, rq->cross_oem, rq->cross_hwid))) {
        const char *errstr = pp_error_string(errno);
        pp_bmc_log_warn("[OEM-PP] Flash Firmware: validation failed: %s", errstr);
        sz = strlen(errstr);
        rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_FW_INVALID, sz);
        strncpy(rs, errstr, sz);
        return pp_bmc_router_send_msg(imsg);
    }
	
    if (!rq->validate_only) {
        // flash the firmware
	set_flash_state(FLASH_IN_PROGRESS);
	flash_dont_reboot = rq->dont_reboot;
	if (pthread_create(&flash_thread, NULL, flash_thread_func, NULL) != 0) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	};
    }

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

#define IPMI_ERR_FW_FAILED      0x83

/*
 * Firmware Finalize Upgrade (Get flash status and reboot)
 */
struct oem_pp_firmware_finalize_upgrade_rq_s {
    unsigned short res_id_le16;
} __attribute__ ((packed));
typedef struct oem_pp_firmware_finalize_upgrade_rq_s oem_pp_firmware_finalize_upgrade_rq_t;

static int oem_pp_cmd_firmware_finalize_upgrade(imsg_t* imsg)
{
    oem_pp_firmware_finalize_upgrade_rq_t *rq = (void*)imsg->data;
    char *rs;
    int sz;
    flash_state_t state = get_flash_state();

    // check reservation
    if (PP_FAILED(reserv_check(fw_reserv, le16_to_cpu(rq->res_id_le16), 1, imsg))) {
        pp_bmc_log_warn("[OEM-PP] Finalize Firmware Flash: bad reservation");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }

    if (state == FLASH_IDLE) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE);
    } else if (state == FLASH_IN_PROGRESS) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NODE_BUSY);
    } else if (state == FLASH_FINISHED) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
    }

    sz = fw_update_err ? strlen(fw_update_err) : 0;
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_FW_FAILED, sz);
    strncpy(rs, fw_update_err, sz);
    pthread_join(flash_thread, NULL);
    set_flash_state(FLASH_IDLE);
    return pp_bmc_router_send_msg(imsg);
}

/*
 * Firmware Cancel Upgrade
 */
struct oem_pp_firmware_cancel_upgrade_rq_s {
    unsigned short res_id_le16;
} __attribute__ ((packed));
typedef struct oem_pp_firmware_cancel_upgrade_rq_s oem_pp_firmware_cancel_upgrade_rq_t;

static int oem_pp_cmd_firmware_cancel_upgrade(imsg_t* imsg)
{
    oem_pp_firmware_upload_rq_t *rq = (void*)imsg->data;
    flash_state_t state = get_flash_state();

    // check reservation
    if (PP_FAILED(reserv_check(fw_reserv, le16_to_cpu(rq->res_id_le16), 1, imsg))) {
        pp_bmc_log_warn("[OEM-PP] Cancel Firmware Upgrade: bad reservation");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }
    // TODO release timer

    if (state != FLASH_IDLE) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NODE_BUSY);
    }

    reserv_cancel(fw_reserv);
    pp_firmware_erla_ctx_free(fw_ctx);
    fw_ctx = 0;

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}


/*
 * Config Get
 */
struct oem_pp_config_get_rq_s {
    BITFIELD1(unsigned char, no_default : 1);
    char key[0];
} __attribute__ ((packed));
typedef struct oem_pp_config_get_rq_s oem_pp_config_get_rq_t;

static int oem_pp_cmd_config_get(imsg_t* imsg)
{
    oem_pp_config_get_rq_t* rq = (void*)imsg->data;
    char *rs;
    char *key, *val;
    int n = imsg->data_size - sizeof(oem_pp_config_get_rq_t);
    int res;

    // make null-terminated key string
    key = malloc(n + 1);
    strncpy(key, rq->key, n);
    key[n] = '\0';

    // get param
    if (rq->no_default) {
        res = pp_cfg_get_nodflt(&val, key);
    } else {
        res = pp_cfg_get(&val, key);
    }

    if (PP_FAILED(res)) {
        pp_bmc_log_error("[OEM-PP] Config Get: failed to read key '%s'", key);
        free(key);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }

    // copy result
    pp_bmc_log_info("[OEM-PP] Config Get: '%s'='%s'", key, val);
    n = strlen(val);
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, n);
    strncpy(rs, val, n);
    free(val);
    free(key);
    return pp_bmc_router_send_msg(imsg);
}

/*
 * Config Set
 */
static int oem_pp_cmd_config_set(imsg_t* imsg)
{
    char *rq = (void*)imsg->data;
    char *sep, *val;
    int n, res;

    // find '\0' separating key and val
    sep = memchr(rq, '\0', imsg->data_size);
    if (!sep) {
        pp_bmc_log_warn("[OEM-PP] Config Set: needs key and value seperated by 0");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    // make null-terminated val string
    sep++; // points now to val
    n = imsg->data_size - (sep - rq);
    val = malloc(n + 1);
    strncpy(val, sep, n);
    val[n] = '\0';

    // set param
    res = pp_cfg_set(val, rq);
    if (PP_FAILED(res)) {
        pp_bmc_log_warn("[OEM-PP] Config Set: failed to write val '%s' to key '%s'", val, rq);
        free(val);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    pp_cfg_save(DO_FLUSH);

    pp_bmc_log_info("[OEM-PP] Config Set: '%s'='%s'", rq, val);
    free(val);
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}


static int crc32_fd(int fd, unsigned int *result)
{
    char buffer[512];
    uLong crc = 0;
    if (lseek(fd, 0, SEEK_SET) < 0) perror("lseek");
    for (;;) {
	ssize_t len = read(fd, buffer, 512);
	if (len < 0) {
	    perror("read");
	    return PP_ERR;
	} else if (len == 0) {
	    break;
	}
	crc = crc32(crc, buffer, len);
    }
    *result = crc;
    return PP_SUC;
}
    	  	 
/*
 * Start Config Backup
 */
struct oem_pp_start_config_backup_rs_s {
    unsigned short res_id_le16;
    unsigned int size_le32;
    unsigned int crc_le32;
    unsigned int max_chunk_size_le32;
} __attribute__ ((packed));
typedef struct oem_pp_start_config_backup_rs_s oem_pp_start_config_backup_rs_t;
    	  	 
static int oem_pp_cmd_start_config_backup(imsg_t *imsg)
{
    oem_pp_start_config_backup_rs_t *rs;
    unsigned int crc;
    struct stat st;
    	  	 
    if (cfg_mode != CFG_IDLE) {
	reserv_cancel(cfg_reserv);
	close(cfg_fd);
	cfg_mode = CFG_IDLE;
    }
    	  	 
    eric_config_file_copy("!config", CFG_BACKUP_TEMP_FILE);
    cfg_fd = open(CFG_BACKUP_TEMP_FILE, O_RDONLY);
    if (cfg_fd < 0) return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    if (fstat(cfg_fd, &st) != 0) return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    if (PP_FAILED( crc32_fd(cfg_fd, &crc) )) return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    cfg_total_size = st.st_size;
    cfg_mode = CFG_BACKUP;
    	  	 
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_pp_start_config_backup_rs_t));
    rs->res_id_le16 = cpu_to_le16(reserv_make(cfg_reserv, imsg));
    rs->size_le32 = cpu_to_le32(cfg_total_size);
    rs->crc_le32 = cpu_to_le32(crc);
    rs->max_chunk_size_le32 = cpu_to_le32(MAX_CHUNK_SIZE);
    return pp_bmc_router_send_msg(imsg);
}
    	  	 
/*
 * Backup Config
 */
struct oem_pp_backup_config_rq_s {
    unsigned short res_id_le16;
    unsigned int offset_le32;
    unsigned int size_le32;
} __attribute__ ((packed));
typedef struct oem_pp_backup_config_rq_s oem_pp_backup_config_rq_t;
    	  	 
struct oem_pp_backup_config_rs_s {
    unsigned char chunk[0];
} __attribute__ ((packed));
typedef struct oem_pp_backup_config_rs_s oem_pp_backup_config_rs_t;
    	  	 
static int oem_pp_cmd_backup_config(imsg_t *imsg)
{
    oem_pp_backup_config_rq_t *rq = (void *)imsg->data;
    oem_pp_backup_config_rs_t *rs;
    ssize_t ofs = le32_to_cpu(rq->offset_le32);
    ssize_t len = le32_to_cpu(rq->size_le32);
    unsigned char buffer[MAX_CHUNK_SIZE];
    	  	 
    if (PP_FAILED(reserv_check(cfg_reserv, le16_to_cpu(rq->res_id_le16), 1, imsg))
	|| cfg_mode != CFG_BACKUP) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    } else if (ofs + len > cfg_total_size || len > MAX_CHUNK_SIZE) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    	  	 
    lseek(cfg_fd, ofs, SEEK_SET);
    if (read(cfg_fd, buffer, len) != len) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    	  	 
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_pp_backup_config_rs_t) + len);
    memcpy(rs->chunk, buffer, len);
    return pp_bmc_router_send_msg(imsg);
}
    	  	 
/*
 * Finish Config Backup
 */
struct oem_pp_finish_config_backup_rq_s {
    unsigned short res_id_le16;
} __attribute__ ((packed));
typedef struct oem_pp_finish_config_backup_rq_s oem_pp_finish_config_backup_rq_t;
    	  	 
static int oem_pp_cmd_finish_config_backup(imsg_t *imsg)
{
    oem_pp_finish_config_backup_rq_t *rq = (void *)imsg->data;
    	  	 
    if (PP_FAILED(reserv_check(cfg_reserv, le16_to_cpu(rq->res_id_le16), 1, imsg))
	|| cfg_mode != CFG_BACKUP) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }
    	  	 
    reserv_cancel(cfg_reserv);
    close(cfg_fd);
    unlink(CFG_BACKUP_TEMP_FILE);
    cfg_mode = CFG_IDLE;
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);;
}
    	  	 
/*
 * Start Config Restore
 */
struct oem_pp_start_config_restore_rq_s {
    unsigned int size_le32;
    unsigned int crc_le32;
} __attribute__ ((packed));
typedef struct oem_pp_start_config_restore_rq_s oem_pp_start_config_restore_rq_t;
    	  	 
struct oem_pp_start_config_restore_rs_s {
    unsigned short res_id_le16;
    unsigned int max_chunk_size_le32;
} __attribute__ ((packed));
typedef struct oem_pp_start_config_restore_rs_s oem_pp_start_config_restore_rs_t;
    	  	 
static int oem_pp_cmd_start_config_restore(imsg_t *imsg)
{
    oem_pp_start_config_restore_rq_t *rq = (void *)imsg->data;
    oem_pp_start_config_restore_rs_t *rs;
    	  	 
    if (cfg_mode != CFG_IDLE) {
	reserv_cancel(cfg_reserv);
	close(cfg_fd);
	cfg_mode = CFG_IDLE;
    }
    	  	 
    cfg_fd = creat(CFG_BACKUP_TEMP_FILE, S_IREAD|S_IWRITE);
    if (cfg_fd < 0) return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    cfg_mode = CFG_RESTORE;
    cfg_total_size = le32_to_cpu(rq->size_le32);
    cfg_current_size = 0;
    cfg_crc_expected = le32_to_cpu(rq->crc_le32);
    cfg_crc_actual = 0;
    	  	 
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_pp_start_config_restore_rs_t));
    rs->res_id_le16 = cpu_to_le16(reserv_make(cfg_reserv, imsg));
    rs->max_chunk_size_le32 = cpu_to_le32(MAX_CHUNK_SIZE);
    return pp_bmc_router_send_msg(imsg);
}
    	  	 
/*
 * Restore Config
 */
struct oem_pp_restore_config_rq_s {
    unsigned short res_id_le16;
    unsigned int offset_le32;
    unsigned char chunk[0];
} __attribute__ ((packed));
typedef struct oem_pp_restore_config_rq_s oem_pp_restore_config_rq_t;
    	  	 
static int oem_pp_cmd_restore_config(imsg_t *imsg)
{
    oem_pp_restore_config_rq_t *rq = (void *)imsg->data;
    ssize_t ofs = le32_to_cpu(rq->offset_le32);
    ssize_t len = imsg->data_size - sizeof(oem_pp_restore_config_rq_t);
    	  	 
    if (PP_FAILED(reserv_check(cfg_reserv, le16_to_cpu(rq->res_id_le16), 1, imsg))
	|| cfg_mode != CFG_RESTORE) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    } else if (ofs != cfg_current_size || ofs + len > cfg_total_size) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    	  	 
    if (lseek(cfg_fd, ofs, SEEK_SET) < 0 ) perror("lseek");
    if (write(cfg_fd, rq->chunk, len) != len) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    cfg_current_size = ofs + len;
    cfg_crc_actual = crc32(cfg_crc_actual, rq->chunk, len);
    	  	 
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}
    	  	 
#define IPMI_ERR_CFG_INCOMPLETE 0x80
#define IPMI_ERR_CFG_BAD_CRC    0x81
    	  	 
/*
 * Finish Config Restore
 */
struct oem_pp_finish_config_restore_rq_s {
    unsigned short res_id_le16;
    unsigned char opcode;
} __attribute__ ((packed));
typedef struct oem_pp_finish_config_restore_rq_s oem_pp_finish_config_restore_rq_t;
    	  	 
static int oem_pp_cmd_finish_config_restore(imsg_t *imsg)
{
    oem_pp_finish_config_restore_rq_t *rq = (void *)imsg->data;
    	  	 
    if (PP_FAILED(reserv_check(cfg_reserv, le16_to_cpu(rq->res_id_le16), 1, imsg))
	|| cfg_mode != CFG_RESTORE) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    } else if (rq->opcode != 0 && rq->opcode != 1) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    } else if (cfg_current_size != cfg_total_size) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_CFG_INCOMPLETE);
    }
    	  	 
    if (rq->opcode == 1) {
	if (cfg_crc_actual != cfg_crc_expected) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_CFG_BAD_CRC);
	}
        pp_bmc_log_info("[OEM-PP] Restoring config file and rebooting");
        pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
	eric_config_file_replace_and_reboot("!config", CFG_BACKUP_TEMP_FILE);
	// this will reboot the device
    }
    	  	 
    reserv_cancel(cfg_reserv);
    close(cfg_fd);
    unlink(CFG_BACKUP_TEMP_FILE);
    cfg_mode = CFG_IDLE;
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}	

/*
 * Reset To Factory Defaults
 */
static int oem_pp_cmd_reset_to_factory_defaults(imsg_t* imsg)
{
    pp_bmc_log_info("[OEM-PP] Reseting device to factory defaults and rebooting");

    /* might not be sent due to reset */
    pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
    usleep(1000);

    /* "touch" the reset to defaults file and quit. eric.sh will do the rest */
    system("echo \"\" | cat > /tmp/reset_to_defaults");
    exit(0);
    /* any errors here will be ignored */

    return PP_SUC;
}


/*
 * Get Serial Number
 */
#define IPMI_ERR_GSN_NOT_SET    0x80

static int oem_pp_cmd_get_serial_number(imsg_t* imsg)
{
    char *rs, *val;
    int n, res;

    // get config value
    res = pp_cfg_get(&val, "serial");
    if (PP_FAILED(res)) {
        pp_bmc_log_warn("[OEM-PP] Get Serial Number: failed to read serial number from config sys");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_GSN_NOT_SET);
    }

    // copy result
    pp_bmc_log_info("[OEM-PP] Get Serial Number: sn='%s'", val);
    n = strlen(val);
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, n);
    strncpy(rs, val, n);
    free(val);
    return pp_bmc_router_send_msg(imsg);
}

/*
 * Set Serial Number
 */
#define IPMI_ERR_SSN_ALREADY_SET    0x80

static int oem_pp_cmd_set_serial_number(imsg_t* imsg)
{
    char *rq = (void*)imsg->data, *val;
    int n, res;

    // get config value
    res = pp_cfg_get_nodflt(&val, "serial");
    if (PP_SUCCED(res)) {
        pp_bmc_log_error("[OEM-PP] Set Serial Number: serial number already set, cannot be changed");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_SSN_ALREADY_SET);
    }

    // make null-terminated val string
    n = imsg->data_size;
    val = malloc(n + 1);
    strncpy(val, rq, n);
    val[n] = '\0';

    // set param
    res = pp_cfg_set_at_layer(PP_PROFILE_SYSTEM, val, "serial");
    if (PP_FAILED(res)) {
        pp_bmc_log_warn("[OEM-PP] Set Serial Number: failed to write serial number to config sys");
        free(val);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    pp_cfg_save_layer(PP_PROFILE_SYSTEM, DO_FLUSH);

    pp_bmc_log_info("[OEM-PP] Set Serial Number: sn='%s'", val);
    free(val);
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Set extended user parameters
 */

#define IPMI_OEM_PP_EXT_USER_DEVICE_UID     0x01

#define IPMI_ERR_PARAMETER_NOT_SUPPORTED    0x80
#define IPMI_ERR_WRITING_READ_ONLY_PARAM    0x82

struct set_extended_user_parameters_rq_s {
    BITFIELD2(unsigned char, uid : 6, _rsv : 2);
    unsigned char param;
    char data[0];
} __attribute__ ((packed));
typedef struct set_extended_user_parameters_rq_s set_extended_user_parameters_rq_t;

static int oem_pp_cmd_set_extended_user_parameters(imsg_t* imsg) {
    int ierr = IPMI_ERR_SUCCESS;
    set_extended_user_parameters_rq_t *rq = (set_extended_user_parameters_rq_t *)imsg->data;

    switch (rq->param) {
        case IPMI_OEM_PP_EXT_USER_DEVICE_UID:
            pp_bmc_log_warn("[OEM-PP] Set extended user data failed, cannot write read-only parameter %u", rq->param);
            ierr = IPMI_ERR_WRITING_READ_ONLY_PARAM;
            break;
        default:
            pp_bmc_log_warn("[OEM-PP] Set extended user data failed, parameter %u not supported", rq->param);
            ierr = IPMI_ERR_PARAMETER_NOT_SUPPORTED;
            break;
    }
    
    // this will return SUCCESS in case nothing bad happened :-)
    return pp_bmc_router_resp_err(imsg, ierr);
}

/*
 * Get extended user parameters
 */

struct get_extended_user_parameters_rq_s {
    BITFIELD2(unsigned char, uid : 6, _rsv : 2);
    unsigned char param;
    unsigned char set_sel;
    unsigned char block_sel;
} __attribute__ ((packed));
typedef struct get_extended_user_parameters_rq_s get_extended_user_parameters_rq_t;

static int oem_pp_cmd_get_extended_user_parameters(imsg_t* imsg) {
    int ierr = IPMI_ERR_SUCCESS;
    get_extended_user_parameters_rq_t *rq = (get_extended_user_parameters_rq_t *)imsg->data;

    unsigned char *resp;
    unsigned char data[64];
    int data_len = 0;

    switch (rq->param) {
        case IPMI_OEM_PP_EXT_USER_DEVICE_UID:
            {
                int pp_uid = pp_bmc_user_get_pp_uid(rq->uid);
                pp_bmc_log_info("[OEM-PP] Get extended user data - user ID: %u, device UID %i",
                    rq->uid, pp_uid);
                *((uint32_t *)data) = cpu_to_le32(pp_uid);
                data_len = sizeof(uint32_t);
            }
            break;
            
        default:
            pp_bmc_log_warn("[OEM-PP] Get extended user data failed, parameter %u not supported", rq->param);
            ierr = IPMI_ERR_PARAMETER_NOT_SUPPORTED;
            break;
    }

    if (ierr != IPMI_ERR_SUCCESS) {
        return pp_bmc_router_resp_err(imsg, ierr);
    }

    resp = pp_bmc_imsg_resp(imsg, ierr, data_len);
    memcpy(resp, data, data_len);

    return pp_bmc_router_send_msg(imsg);
}


#ifdef PP_FEAT_MASS_STORAGE
/********************************************************************
 * Virtual Media Management Commands
 */

#define IPMI_ERR_VMEDIA_NO_EMPTY_DEVICE    0x80
#define IPMI_ERR_VMEDIA_INVALID_FILE_SIZE  0x81
#define IPMI_ERR_VMEDIA_SMB_IN_PROGRESS    0x82
#define IPMI_ERR_VMEDIA_INCOMPLETE_UPLOAD  0x83
#define IPMI_ERR_VMEDIA_ACCESS_FAILED      0x84

/*
 * Get Virtual Media Status
 */
struct oem_pp_get_virtual_media_status_rq_s {
    unsigned char device_id;
} __attribute__ ((packed));
typedef struct oem_pp_get_virtual_media_status_rq_s oem_pp_get_virtual_media_status_rq_t;

struct oem_pp_get_virtual_media_status_rs_s {
    unsigned char device_count;
    unsigned char ipmi_status;
    unsigned char device_status;
    unsigned int size_le32;
    unsigned char unit;
    unsigned char rw;
    unsigned char filename[0];
} __attribute__ ((packed));
typedef struct oem_pp_get_virtual_media_status_rs_s oem_pp_get_virtual_media_status_rs_t;

static int oem_pp_cmd_get_virtual_media_status(imsg_t *imsg)
{
    oem_pp_get_virtual_media_status_rq_t *rq = (void *)imsg->data;
    oem_pp_get_virtual_media_status_rs_t *rs;
    int dev_id, n, rw;
    char *filename = NULL;
    const char *tmp;
    pp_usb_image_type_t type;
    int status, unit;
    off_t size;

    if (imsg->data_size != 1) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_INVALID);
    }

    if (rq->device_id == 0) {
	// return device count only
	rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 1);
	rs->device_count = PP_FEAT_USB_MASS_STORAGE_NO;
	return pp_bmc_router_send_msg(imsg);
    } else if (rq->device_id > PP_FEAT_USB_MASS_STORAGE_NO) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    dev_id = rq->device_id - 1;

    type = pp_usb_ms_get_image_type(dev_id);
    switch (type) {
	case PP_USB_MS_IMAGE_MEM:
	    status = 1;
	    filename = strdup(pp_usb_ms_get_image_info(dev_id, "name"));
	    rw = 1;
	    break;
	case PP_USB_MS_IMAGE_SMB:
	    {
		const char *host, *share, *path;
		status = 2;
		host = pp_usb_ms_get_image_info(dev_id, "host");
		share = pp_usb_ms_get_image_info(dev_id, "share");
		path = pp_usb_ms_get_image_info(dev_id, "path");
		asprintf(&filename, "//%s/%s/%s\n", host, share, path);
		rw = 0;
		break;
	    }
	case PP_USB_MS_IMAGE_MSP:
	    status = 3;
	    filename = strdup(pp_usb_ms_get_image_info(dev_id, "redir_ip"));
	    tmp = pp_usb_ms_get_image_info(dev_id, "redir_ro");
	    rw = (strcmp(tmp, "read-only") != 0);
	    break;
	case PP_USB_MS_IMAGE_NONE:
	default:
	    status = 0;
	    filename = NULL;
	    rw = 0;
    }

    n = sizeof(oem_pp_get_virtual_media_status_rs_t);
    if (filename) n += strlen(filename);
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, n);
    rs->device_count = PP_FEAT_USB_MASS_STORAGE_NO;
    rs->ipmi_status = (vmedia_data[dev_id].reserv_active)? 0x01 : 0;
    if (vmedia_data[dev_id].smbmnt_ctx &&
	    pp_usb_smbmnt_get_status(vmedia_data[dev_id].smbmnt_ctx)
	    == PP_USB_SMBMNT_STATUS_BUSY) {
	rs->ipmi_status |= 0x02;
    }
    rs->device_status = status;

    size = pp_usb_ms_get_image_size(dev_id);
    unit = 0;
    while (size>>32 > 0) {
	size >>= 10;
	unit++;
    }
    rs->size_le32 = cpu_to_le32((unsigned int)size);
    rs->unit = unit;
    rs->rw = rw;
    if (filename) {
	memcpy(rs->filename, filename, strlen(filename));
    }

    free(filename);
    return pp_bmc_router_send_msg(imsg);
}

/*
 * Close Virtual Media Session
 */
struct oem_pp_close_virtual_media_session_rq_s {
    unsigned char device_id;
} __attribute__ ((packed));
typedef struct oem_pp_close_virtual_media_session_rq_s oem_pp_close_virtual_media_session_rq_t;

static int oem_pp_cmd_close_virtual_media_session(imsg_t *imsg)
{
    oem_pp_close_virtual_media_session_rq_t *rq = (void *)imsg->data;
    int dev_id = rq->device_id - 1;
    int err, ret;

    if (imsg->data_size != 1) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_INVALID);
    }

    if (rq->device_id == 0 || rq->device_id > PP_FEAT_USB_MASS_STORAGE_NO) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    ret = pp_usb_ms_unset_image(dev_id, &err);

    return pp_bmc_router_resp_err(imsg,
	    PP_SUCCED(ret)? IPMI_ERR_SUCCESS : IPMI_ERR_UNSPECIFIED);
}

/*
 * Start Floppy Image Upload
 */
struct oem_pp_start_floppy_image_upload_rq_s {
    unsigned char device_id;
    unsigned int size_le32;
    unsigned char filename[0];
} __attribute__ ((packed));
typedef struct oem_pp_start_floppy_image_upload_rq_s oem_pp_start_floppy_image_upload_rq_t;

struct oem_pp_start_floppy_image_upload_rs_s {
    unsigned char device_id;
    unsigned short res_id_le16;
    unsigned int max_chunk_size_le32;
} __attribute__ ((packed));
typedef struct oem_pp_start_floppy_image_upload_rs_s oem_pp_start_floppy_image_upload_rs_t;

static int oem_pp_cmd_start_floppy_image_upload(imsg_t *imsg)
{
    oem_pp_start_floppy_image_upload_rq_t *rq = (void *)imsg->data;
    oem_pp_start_floppy_image_upload_rs_t *rs;
    int len = imsg->data_size - sizeof(oem_pp_start_floppy_image_upload_rq_t);
    int size = le32_to_cpu(rq->size_le32);
    int dev_id;
    char *name;

    if (size > (1440 << 10) || size % 512) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_VMEDIA_INVALID_FILE_SIZE);
    }

    if (rq->device_id > PP_FEAT_USB_MASS_STORAGE_NO) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    } else if (rq->device_id == 0) {
	for (dev_id = 0; dev_id < PP_FEAT_USB_MASS_STORAGE_NO; dev_id++) {
	    if (!vmedia_data[dev_id].reserv_active &&
		    pp_usb_ms_get_image_type(dev_id) == PP_USB_MS_IMAGE_NONE) break;
	}
	if (dev_id == PP_FEAT_USB_MASS_STORAGE_NO) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_VMEDIA_NO_EMPTY_DEVICE);
	}
    } else {
	dev_id = rq->device_id - 1;
    }

    // Try to close an active smbmount helper; this may fail
    if (vmedia_data[dev_id].smbmnt_ctx) {
	if (PP_FAILED(pp_usb_smbmnt_close(vmedia_data[dev_id].smbmnt_ctx))) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_VMEDIA_SMB_IN_PROGRESS);
	}
	vmedia_data[dev_id].smbmnt_ctx = NULL;
    }

    // Cancel an active upload for this device
    if (vmedia_data[dev_id].vfup_ctx) {
	pp_bmc_log_warn("[OEM-PP] IPMI Floppy upload (dev #%d) canceled.", dev_id);
	pp_usb_vfup_delete(vmedia_data[dev_id].vfup_ctx);
	vmedia_data[dev_id].vfup_ctx = NULL;
    }

    // Cancel an active reservaion
    if (vmedia_data[dev_id].reserv_active) {
	reserv_cancel(vmedia_data[dev_id].reserv);
	vmedia_data[dev_id].reserv_active = 0;
    }

    // Create upload context in libpp_usb
    name = malloc(len + 1);
    memcpy(name, rq->filename, len);
    name[len] = '\0';
    vmedia_data[dev_id].vfup_ctx = pp_usb_vfup_start(dev_id,
	    le32_to_cpu(rq->size_le32), name);
    free(name);
    if (vmedia_data[dev_id].vfup_ctx == NULL) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_OUT_OF_SPACE);
    }

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_pp_start_floppy_image_upload_rs_t));
    rs->device_id = dev_id + 1;
    rs->res_id_le16 = cpu_to_le16(reserv_make(vmedia_data[dev_id].reserv, imsg));
    vmedia_data[dev_id].reserv_active = 1;
    rs->max_chunk_size_le32 = cpu_to_le32(MAX_CHUNK_SIZE);

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Upload Floppy Image
 */
struct oem_pp_upload_floppy_image_rq_s {
    unsigned char device_id;
    unsigned short res_id_le16;
    unsigned int offset_le32;
    unsigned char chunk[0];
} __attribute__ ((packed));
typedef struct oem_pp_upload_floppy_image_rq_s oem_pp_upload_floppy_image_rq_t;

static int oem_pp_cmd_upload_floppy_image(imsg_t *imsg)
{
    oem_pp_upload_floppy_image_rq_t *rq = (void *)imsg->data;
    int len = imsg->data_size - sizeof(oem_pp_upload_floppy_image_rq_t);
    int dev_id = rq->device_id - 1;

    if (rq->device_id == 0 || rq->device_id > PP_FEAT_USB_MASS_STORAGE_NO) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    // Check reservation
    if (PP_FAILED(reserv_check(vmedia_data[dev_id].reserv,
		    le16_to_cpu(rq->res_id_le16), 1, imsg))) {
	pp_bmc_log_warn("[OEM-PP] Upload Floppy Image: bad reservation");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }

    // Check context
    if (!vmedia_data[dev_id].vfup_ctx) {
	pp_bmc_log_warn("[OEM-PP] Upload Floppy Image: No valid context");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE);
    }

    // Pass chunk to libpp_usb
    if (PP_FAILED(pp_usb_vfup_add_chunk(vmedia_data[dev_id].vfup_ctx,
		    le32_to_cpu(rq->offset_le32), rq->chunk, len))) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    };

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Finalize Floppy Image Upload
 */
struct oem_pp_finalize_floppy_image_upload_rq_s {
    unsigned char device_id;
    unsigned short res_id_le16;
    unsigned char opcode;
} __attribute__ ((packed));
typedef struct oem_pp_finalize_floppy_image_upload_rq_s oem_pp_finalize_floppy_image_upload_rq_t;

static int oem_pp_cmd_finalize_floppy_image_upload(imsg_t *imsg)
{
    oem_pp_finalize_floppy_image_upload_rq_t *rq = (void *)imsg->data;
    int dev_id = rq->device_id - 1;
    int ret = IPMI_ERR_SUCCESS;

    if (rq->device_id == 0 || rq->device_id > PP_FEAT_USB_MASS_STORAGE_NO) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    // Check reservation
    if (PP_FAILED(reserv_check(vmedia_data[dev_id].reserv,
		    le16_to_cpu(rq->res_id_le16), 1, imsg))) {
	pp_bmc_log_warn("[OEM-PP] Finalize Floppy Image Upload: bad reservation");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }

    // Check context
    if (!vmedia_data[dev_id].vfup_ctx) {
	pp_bmc_log_warn("[OEM-PP] Finalize Floppy Image: No valid context");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE);
    }

    switch (rq->opcode) {
	case 0: // cancel upload
	    pp_usb_vfup_delete(vmedia_data[dev_id].vfup_ctx);
	    vmedia_data[dev_id].vfup_ctx = NULL;
	    reserv_cancel(vmedia_data[dev_id].reserv);
	    vmedia_data[dev_id].reserv_active = 0;
	    break;
	case 1: // activate image
	    if (!pp_usb_vfup_complete(vmedia_data[dev_id].vfup_ctx)) {
		ret = IPMI_ERR_VMEDIA_INCOMPLETE_UPLOAD;
	    } else if (PP_FAILED(
			pp_usb_vfup_activate(vmedia_data[dev_id].vfup_ctx))) {
		ret = IPMI_ERR_UNSPECIFIED;
	    } else {
		vmedia_data[dev_id].vfup_ctx = NULL;
		reserv_cancel(vmedia_data[dev_id].reserv);
		vmedia_data[dev_id].reserv_active = 0;
	    }
	    break;
	default:
	    ret = IPMI_ERR_PARAM_OUT_OF_RANGE;
    }

    return pp_bmc_router_resp_err(imsg, ret);
}

/*
 * Start SMB Image Mount
 */
struct oem_pp_start_smb_image_mount_rq_s {
    unsigned char device_id;
} __attribute__ ((packed));
typedef struct oem_pp_start_smb_image_mount_rq_s oem_pp_start_smb_image_mount_rq_t;

struct oem_pp_start_smb_image_mount_rs_s {
    unsigned char device_id;
    unsigned short res_id_le16;
} __attribute__ ((packed));
typedef struct oem_pp_start_smb_image_mount_rs_s oem_pp_start_smb_image_mount_rs_t;

static int oem_pp_cmd_start_smb_image_mount(imsg_t *imsg)
{
    oem_pp_start_smb_image_mount_rq_t *rq = (void *)imsg->data;
    oem_pp_start_smb_image_mount_rs_t *rs;
    int dev_id;

    if (rq->device_id > PP_FEAT_USB_MASS_STORAGE_NO) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    } else if (rq->device_id == 0) {
	for (dev_id = 0; dev_id < PP_FEAT_USB_MASS_STORAGE_NO; dev_id++) {
	    if (!vmedia_data[dev_id].reserv_active &&
		    pp_usb_ms_get_image_type(dev_id) == PP_USB_MS_IMAGE_NONE) break;
	}
	if (dev_id == PP_FEAT_USB_MASS_STORAGE_NO) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_VMEDIA_NO_EMPTY_DEVICE);
	}
    } else {
	dev_id = rq->device_id - 1;
    }

    // Try to close an active smbmount helper; this may fail
    if (vmedia_data[dev_id].smbmnt_ctx) {
	if (PP_FAILED(pp_usb_smbmnt_close(vmedia_data[dev_id].smbmnt_ctx))) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_VMEDIA_SMB_IN_PROGRESS);
	}
	vmedia_data[dev_id].smbmnt_ctx = NULL;
    }

    // Cancel an active upload for this device
    if (vmedia_data[dev_id].vfup_ctx) {
	pp_bmc_log_warn("[OEM-PP] IPMI Floppy upload (dev #%d) canceled.", dev_id);
	pp_usb_vfup_delete(vmedia_data[dev_id].vfup_ctx);
	vmedia_data[dev_id].vfup_ctx = NULL;
    }

    // Cancel an active reservaion
    if (vmedia_data[dev_id].reserv_active) {
	reserv_cancel(vmedia_data[dev_id].reserv);
	vmedia_data[dev_id].reserv_active = 0;
    }

    // Create smbmnt context in libpp_usb
    vmedia_data[dev_id].smbmnt_ctx = pp_usb_smbmnt_start(dev_id);
    if (vmedia_data[dev_id].smbmnt_ctx == NULL) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(oem_pp_start_smb_image_mount_rs_t));
    rs->device_id = dev_id + 1;
    rs->res_id_le16 = cpu_to_le16(reserv_make(vmedia_data[dev_id].reserv, imsg));

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Set SMB Image Parameter
 */
struct oem_pp_set_smb_image_parameter_rq_s {
    unsigned char device_id;
    unsigned short res_id_le16;
    unsigned char param;
    unsigned char value[0];
} __attribute__ ((packed));
typedef struct oem_pp_set_smb_image_parameter_rq_s oem_pp_set_smb_image_parameter_rq_t;

static int oem_pp_cmd_set_smb_image_parameter(imsg_t *imsg)
{
    oem_pp_set_smb_image_parameter_rq_t *rq = (void *)imsg->data;
    int len = imsg->data_size - sizeof(oem_pp_set_smb_image_parameter_rq_t);
    int dev_id = rq->device_id - 1;
    int param;
    char *value;
    int status;

    if (rq->device_id == 0 || rq->device_id > PP_FEAT_USB_MASS_STORAGE_NO) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    // Check reservation
    if (PP_FAILED(reserv_check(vmedia_data[dev_id].reserv,
		    le16_to_cpu(rq->res_id_le16), 1, imsg))) {
	pp_bmc_log_warn("[OEM-PP] Set SMB Image Parameter: bad reservation");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }

    // Check context
    if (!vmedia_data[dev_id].smbmnt_ctx) {
	pp_bmc_log_warn("[OEM-PP] Set SMB Image Parameter: No valid context");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE);
    }

    status = pp_usb_smbmnt_get_status(vmedia_data[dev_id].smbmnt_ctx);
    if (status != PP_USB_SMBMNT_STATUS_IDLE) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE);
    }

    switch (rq->param) {
	case 0:
	    param = PP_USB_SMBMNT_PARAM_HOST;
	    break;
	case 1:
	    param = PP_USB_SMBMNT_PARAM_SHARE;
	    break;
	case 2:
	    param = PP_USB_SMBMNT_PARAM_PATH;
	    break;
	case 3:
	    param = PP_USB_SMBMNT_PARAM_USER;
	    break;
	case 4:
	    param = PP_USB_SMBMNT_PARAM_PASS;
	    break;
	default:
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    value = malloc(len + 1);
    memcpy(value, rq->value, len);
    value[len] = '\0';
    pp_usb_smbmnt_set_param(vmedia_data[dev_id].smbmnt_ctx, param, value);
    free(value);

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Finalize SMB Image Mount
 */
struct oem_pp_finalize_smb_image_mount_rq_s {
    unsigned char device_id;
    unsigned short res_id_le16;
    unsigned char opcode;
} __attribute__ ((packed));
typedef struct oem_pp_finalize_smb_image_mount_rq_s oem_pp_finalize_smb_image_mount_rq_t;

static int oem_pp_cmd_finalize_smb_image_mount(imsg_t *imsg)
{
    oem_pp_finalize_smb_image_mount_rq_t *rq = (void *)imsg->data;
    int dev_id = rq->device_id - 1;
    int ret = IPMI_ERR_SUCCESS;
    int status;

    if (rq->device_id == 0 || rq->device_id > PP_FEAT_USB_MASS_STORAGE_NO) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    // Check reservation
    if (PP_FAILED(reserv_check(vmedia_data[dev_id].reserv,
		    le16_to_cpu(rq->res_id_le16), 1, imsg))) {
	pp_bmc_log_warn("[OEM-PP] Finalize SMB Image Mount: bad reservation");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }

    // Check context
    if (!vmedia_data[dev_id].smbmnt_ctx) {
	pp_bmc_log_warn("[OEM-PP] Finalize SMB Image Mount: No valid context");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE);
    }

    status = pp_usb_smbmnt_get_status(vmedia_data[dev_id].smbmnt_ctx);
    if (status == PP_USB_SMBMNT_STATUS_BUSY) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_VMEDIA_SMB_IN_PROGRESS);
    }

    switch (rq->opcode) {
	case 0: // cancel sequence
	    ret = IPMI_ERR_SUCCESS;
	    goto cancel_sequence;
	case 1: // activate image
	    if (PP_FAILED(pp_usb_smbmnt_activate(
			    vmedia_data[dev_id].smbmnt_ctx))) {
		ret = IPMI_ERR_UNSPECIFIED;
	    } else {
		ret = IPMI_ERR_SUCCESS;
	    }
	    break;
	case 2: // check mount status
	    switch (status) {
		case PP_USB_SMBMNT_STATUS_SUCCESS:
		    ret = IPMI_ERR_SUCCESS;
		    goto cancel_sequence;
		case PP_USB_SMBMNT_STATUS_FAILED:
		    ret = IPMI_ERR_VMEDIA_ACCESS_FAILED;
		    goto cancel_sequence;
		case PP_USB_SMBMNT_STATUS_IDLE:
		    ret = IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE;
		    break;
		default:
		    ret = IPMI_ERR_UNSPECIFIED;
	    }
	    break;
	default:
	    ret = IPMI_ERR_PARAM_OUT_OF_RANGE;
    }
    return pp_bmc_router_resp_err(imsg, ret);

cancel_sequence:
    // Cancel sequence, invalidate reservation
    pp_usb_smbmnt_close(vmedia_data[dev_id].smbmnt_ctx);
    vmedia_data[dev_id].smbmnt_ctx = NULL;
    reserv_cancel(vmedia_data[dev_id].reserv);
    vmedia_data[dev_id].reserv_active = 0;
    return pp_bmc_router_resp_err(imsg, ret);
}

#endif

/********************************************************************
 * Device command table
 */

static const dev_cmd_entry_t oem_pp_cmd_tab[] = {
    {
        .cmd_hndlr = oem_pp_cmd_get_pwm,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_GET_PWM_OF_FAN_SPEED_CONTROL,
        .min_data_size = 0,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_cmd_set_pwm,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_SET_PWM_OF_FAN_SPEED_CONTROL,
        .min_data_size = sizeof(oem_pp_set_pwm_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },

    {
        .cmd_hndlr = oem_pp_cmd_firmware_get_version,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_FIRMWARE_GET_VERSION,
        .min_data_size = 0,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_cmd_firmware_start_upgrade,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_FIRMWARE_START_UPGRADE,
        .min_data_size = sizeof(oem_pp_firmware_start_upgrade_rq_t),
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_cmd_firmware_upload,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_FIRMWARE_UPLOAD,
        .min_data_size = sizeof(oem_pp_firmware_upload_rq_t),
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_cmd_firmware_flash,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_FIRMWARE_FLASH,
        .min_data_size = sizeof(oem_pp_firmware_flash_rq_t),
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_cmd_firmware_finalize_upgrade,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_FIRMWARE_FINALIZE_UPGRADE,
        .min_data_size = sizeof(oem_pp_firmware_finalize_upgrade_rq_t),
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_cmd_firmware_cancel_upgrade,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_FIRMWARE_CANCEL_UPGRADE,
        .min_data_size = sizeof(oem_pp_firmware_cancel_upgrade_rq_t),
        .min_priv_level = IPMI_PRIV_ADMIN
    },

    {
        .cmd_hndlr = oem_pp_cmd_config_get,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_CONFIG_GET,
        .min_data_size = sizeof(oem_pp_config_get_rq_t),
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_cmd_config_set,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_CONFIG_SET,
        .min_data_size = 1,
        .min_priv_level = IPMI_PRIV_ADMIN
    },

    {
	.cmd_hndlr = oem_pp_cmd_start_config_backup,
	.netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_START_CONFIG_BACKUP,
	.min_data_size = 0,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
	.cmd_hndlr = oem_pp_cmd_backup_config,
	.netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_BACKUP_CONFIG,
	.min_data_size = sizeof(oem_pp_backup_config_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
	.cmd_hndlr = oem_pp_cmd_finish_config_backup,
	.netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_FINISH_CONFIG_BACKUP,
	.min_data_size = sizeof(oem_pp_finish_config_backup_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
	.cmd_hndlr = oem_pp_cmd_start_config_restore,
	.netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_START_CONFIG_RESTORE,
	.min_data_size = sizeof(oem_pp_start_config_restore_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
	.cmd_hndlr = oem_pp_cmd_restore_config,
	.netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RESTORE_CONFIG,
	.min_data_size = sizeof(oem_pp_restore_config_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
	.cmd_hndlr = oem_pp_cmd_finish_config_restore,
	.netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_FINISH_CONFIG_RESTORE,
	.min_data_size = sizeof(oem_pp_finish_config_restore_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },

    {
        .cmd_hndlr = oem_pp_cmd_reset_to_factory_defaults,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_RESET_TO_FACTORY_DEFAULTS,
        .min_data_size = 0,
        .min_priv_level = IPMI_PRIV_ADMIN
    },

    {
        .cmd_hndlr = oem_pp_cmd_get_serial_number,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_GET_SERIAL_NUMBER,
        .min_data_size = 0,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_cmd_set_serial_number,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_SET_SERIAL_NUMBER,
        .min_data_size = 0,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_cmd_set_extended_user_parameters,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_SET_EXTENDED_USER_PARAMETERS,
        .min_data_size = 0,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_cmd_get_extended_user_parameters,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_GET_EXTENDED_USER_PARAMETERS,
        .min_data_size = 0,
        .min_priv_level = IPMI_PRIV_ADMIN
    },

#ifdef PP_FEAT_MASS_STORAGE
    {
        .cmd_hndlr = oem_pp_cmd_get_virtual_media_status,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_GET_VIRTUAL_MEDIA_STATUS,
        .min_data_size = sizeof(oem_pp_get_virtual_media_status_rq_t),
        .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = oem_pp_cmd_close_virtual_media_session,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_CLOSE_VIRTUAL_MEDIA_SESSION,
        .min_data_size = sizeof(oem_pp_close_virtual_media_session_rq_t),
        .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = oem_pp_cmd_start_floppy_image_upload,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_START_FLOPPY_IMAGE_UPLOAD,
        .min_data_size = sizeof(oem_pp_start_floppy_image_upload_rq_t),
        .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = oem_pp_cmd_upload_floppy_image,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_UPLOAD_FLOPPY_IMAGE,
        .min_data_size = sizeof(oem_pp_upload_floppy_image_rq_t),
        .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = oem_pp_cmd_finalize_floppy_image_upload,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_FINALIZE_FLOPPY_IMAGE_UPLOAD,
        .min_data_size = sizeof(oem_pp_finalize_floppy_image_upload_rq_t),
        .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = oem_pp_cmd_start_smb_image_mount,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_START_SMB_IMAGE_MOUNT,
        .min_data_size = sizeof(oem_pp_start_smb_image_mount_rq_t),
        .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = oem_pp_cmd_set_smb_image_parameter,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_SET_SMB_IMAGE_PARAMETER,
        .min_data_size = sizeof(oem_pp_set_smb_image_parameter_rq_t),
        .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = oem_pp_cmd_finalize_smb_image_mount,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_FINALIZE_SMB_IMAGE_MOUNT,
        .min_data_size = sizeof(oem_pp_finalize_smb_image_mount_rq_t),
        .min_priv_level = IPMI_PRIV_OPERATOR
    },
#endif

    { .cmd_hndlr = NULL }
};

/********************************************************************
 * Device c'tor/d'tor
 */

int pp_bmc_dev_oem_pp_init()
{
    fw_reserv = reserv_new();
    cfg_reserv = reserv_new();

#ifdef PP_FEAT_MASS_STORAGE
    int i;
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	memset(&vmedia_data[i], 0, sizeof(struct vmedia_data_s));
	vmedia_data[i].reserv = reserv_new();
    }
#endif

    /* register all entries of cmd tab */
    if (PP_FAILED(pp_bmc_core_reg_cmd_tab(oem_pp_cmd_tab))) return PP_ERR;

    pp_bmc_log_info("[OEM-PP] device started");
    return PP_SUC;
}

void pp_bmc_dev_oem_pp_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(oem_pp_cmd_tab);

    reserv_delete(fw_reserv); fw_reserv = NULL;
    reserv_delete(cfg_reserv); cfg_reserv = NULL;

#ifdef PP_FEAT_MASS_STORAGE
    int i;
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	reserv_delete(vmedia_data[i].reserv);
    }
#endif

    pp_bmc_log_info("[OEM-PP] device shut down");
}

