/**
 * bmc_dev_pef_nv.c
 *
 * Platform event filter related non volatile storage routines,
 * data structure definitions. May implement caching and rollback.
 * 
 * Implements data structures and access functions for
 * - event filter table
 * - alert policy table
 * - alert string table
 * - (alert queue entry)
 * - configuration
 *
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef __BMC_DEV_PEF_NV_H__
#define __BMC_DEV_PEF_NV_H__

#include "pp/base.h"
#include "pp/bmc/utils.h"

/**
 * Some global definitions
 */
#define EFT_SIZE 32                  // event filter table
#define APT_SIZE 20                  // alert policy table, min ChanSupported, max 15
#define AST_SIZE 15+1                // alert string table
#define ALERT_STRING_BLOCK_CNT  64   // max length of alert string (in blocks)
// caveat: the volatile alert string 0 is also stored in the alert string table, so
// in the real AlertStringTable has size AST_SIZE+1

/**
 * Event Filter Table Entry
 */
struct event_filter_entry_s {
    // according to table 17-2
    BITFIELD3(unsigned, \
             record_type:5, \
             configuration_type:2, \
             enable:1);
        
    struct {
        BITFIELD8(unsigned,  \
                 alert:1, \
                 poweroff:1, \
                 reset:1, \
                 power_cycle:1, \
                 OEM:1, \
                 diagnostic_interrupt:1, \
                 group_control:1, \
                 dummy2:1);
    } __attribute__ ((packed)) action;

    BITFIELD3(unsigned,
             alert_policy_number:4, \
             alert_policy_group_control_selector:3, \
             dummy3:1);
    
    unsigned char event_severity;
    unsigned char generatorID1;
    unsigned char generatorID2;
    unsigned char sensor_type;
    unsigned char sensor_number;
    unsigned char event_trigger;
    
    unsigned offset_mask:16;
    struct {
        unsigned char mask;
        unsigned char exact;
        unsigned char data;
    } __attribute__ ((packed)) data_filter[3];
} __attribute__ ((packed));
typedef struct event_filter_entry_s event_filter_entry_t;

/**
 * get/set an entry of the event filter table. The set operation is
 * directly written to nv_storage, the get operation may use a memory cache
 * @returns PP_ERR or PP_SUC
 */
int pef_get_event_filter(unsigned char idx, event_filter_entry_t* filter);
int pef_set_event_filter(unsigned char idx, event_filter_entry_t* filter);



/**
 * Alert Policy Table Entry
 */
struct alert_policy_entry_s {
    BITFIELD3(unsigned, \
             policy:3, \
             enable:1, \
             number:4);
    
    BITFIELD2(unsigned, \
             destination:4, \
             channel:4);
    
    BITFIELD2(unsigned, \
             string_set:7, \
             string_type:1);   // indicates if string_set is a key or an index to an entry
} __attribute__ ((packed));
typedef struct alert_policy_entry_s alert_policy_entry_t;

/**
 * get/set an entry of the alert policy table. The set operation is
 * directly written to nv_storage, the get operation may use a memory cache
 * @returns PP_ERR or PP_SUC
 */
int pef_get_alert_policy(unsigned char idx,
			 alert_policy_entry_t* alert_policy);
int pef_set_alert_policy(unsigned char idx,
			 const alert_policy_entry_t* alert_policy);



/**
 * Alert String Table Entry
 */
struct alert_string_entry_s {
    BITFIELD2(unsigned, \
             event_filter_number:7, \
             dummy1:1);
    BITFIELD2(unsigned, \
             alert_string_set:7, \
             dummy2:1);
    char alertString[16*ALERT_STRING_BLOCK_CNT];  // 256 blocks * 16 Bytes per block
} __attribute__ ((packed));
typedef struct alert_string_entry_s alert_string_entry_t;

/**
 * get/set an entry of the alert string table. The set operation is
 * directly written to nv_storage, the get operation may use a memory cache
 * @returns PP_ERR or PP_SUC
 */
int pef_get_alert_string(unsigned char idx, alert_string_entry_t* alert_string);
int pef_set_alert_string(unsigned char idx, alert_string_entry_t* alert_string);



/**
 * Global PEF configuration options
 */
struct pef_configuration_s {
    BITFIELD5(unsigned, \
             enable_pef:1, \
             enable_action_events:1,   \
             disable_startup_delay:1,      \
             disable_alert_startup_delay:1, \
             dummy1:4);
    BITFIELD7(unsigned, \
             enable_alert:1, \
             enable_poweroff:1, \
             enable_reset:1,     \
             enable_power_cycle:1,\
             enable_oem:1,     \
             enable_diagnostic_interrupt:1, \
             dummy2:2);
    
    unsigned char startup_delay;
    unsigned char alert_startup_delay;

    // last processed events, stored intel-style little endian
    unsigned short lpe_bmc;
    unsigned short lpe_sms;
    unsigned short next_unprocessed_event; // event to process if timer is up

    BITFIELD2(unsigned, \
             enable_system_guid:1, \
             dummy3:7);
    unsigned char system_guid[16];  // alternate system guid, can be used instead of GetSystemGUID()
    
} __attribute__ ((packed));
typedef struct pef_configuration_s pef_configuration_t;

/**
 * get/set the pef configuration. The set operation is
 * directly written to nv_storage, the get operation may use a memory cache
 * @returns PP_ERR or PP_SUC
 */
int pef_get_cfg(pef_configuration_t* config);
int pef_set_cfg(pef_configuration_t* config);



/**
 * Initialize PEF. Create a file with the necessary header and
 * initialize each section with the passed data structure.
 * (Table entries are reused)
 */
int pef_initialize(pef_configuration_t* config,
                   event_filter_entry_t* filter,
                   alert_policy_entry_t* alert_policy,
                   alert_string_entry_t* alert_string);

/**
 * Performs a complete check on the pef. (File exists, file magic
 * correct, file length correct). If this returns PP_SUC, all future
 * operations on the file should not fail. If PP_ERR is returned,
 * pef_initialized() must be called to create a correct pef.
 */
int pef_complete_check(void);

#endif
