/**
 * bmc_dev_sel_nv.c
 *
 * Provides a non volatile storage for SEL entries. The entries are ordered
 * as a linked list and can be retrieved by their id. The id does not have to
 * be sequential. Internally the entries are stored in an array and the
 * next id is calculated by retrieving the id of the next element of the
 * array.
 * To simplify iterations through the list, the last accessed index (last_index)
 * is stored in a global variable and searching an element usually starts at
 * this element index to speed up sequential access to the linked list.
 * 
 * Note: The SEL disable flag is stored in this file, but is checked/applied
 *       by the event_receiver as the SEL knows nothing of channels.
 * 
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <string.h>
#include "pp/base.h"

#include "pp/bmc/debug.h"
#include "pp/bmc/utils.h"
#include "pp/bmc/bmc_nv_storage.h"

#include "bmc_dev_sel_nv.h"
#include "bmc_dev_event_msg_buff.h"

void dump_sel_nv(void);

typedef struct {
    unsigned short sel_size;
} sel_config_t;

typedef struct {
    char magic[16];
    unsigned int ver;
} sel_file_header_t;
const unsigned int selHeaderVersion = 0x00100001; // Version 001.00.001
const char* selHeaderMagic = "bmc_sel_magic";

static sel_listener_hndl_t sel_listener_evt_r = { NULL, NULL, NULL };
static sel_listener_hndl_t sel_listener_sel_dev = { NULL, NULL, NULL };

static unsigned short last_index = 0;
static unsigned short cur_id = 0;

/**
 * File layout of file 'sel'
 * - header
 * - configuration
 * - sel_nv_entry_t[SEL_MAX_SIZE]
 */

/*******************************
 * Internal helper functions 
 ***/

/**
 * Open the sel nv_storage and check if the file header is correct.
 * Internal file-offset will be set after the header.
 * @param fd pointer to filedescriptor (set to -1 on error)
 * @returns PP_SUC if file is a correct sel
 * @returns PP_ERR if file is not a correct sel storage or could not be opened
 */
static int open_check(int* fd) {
    if (PP_FAILED(bmc_nv_open(fd, "sel"))) {
        pp_bmc_log_debug("[SEL] could not open persistent storage");
        goto failed;
    }

    sel_file_header_t header;
    if (bmc_nv_read(*fd, &header, sizeof(sel_file_header_t)) != sizeof(sel_file_header_t)) {
        pp_bmc_log_debug("[SEL] could not read file header from persistent storage");
        goto failed;
    }

    if (header.ver != selHeaderVersion) {
        pp_bmc_log_debug("[SEL] could not read file, version does not match sel version");
        goto failed;
    }

    if (strncmp(header.magic ,selHeaderMagic, 16) != 0) {
        pp_bmc_log_debug("[SEL] could not read file, wrong magic number");
        goto failed;
    }

    return PP_SUC;

failed:
    if (*fd >= 0) {
        bmc_nv_close(*fd);
        *fd = -1;
    }
    return PP_ERR;
}

/** read the configuration */
static int get_config(int fd, sel_config_t* config) {
    bmc_nv_lseek(fd, sizeof(sel_file_header_t), SEEK_SET);
    
    if (bmc_nv_read(fd, config, sizeof(sel_config_t)) != sizeof(sel_config_t))  {
        pp_bmc_log_debug("[SEL] could not read configuration");
        return PP_ERR;
    }
    
    return PP_SUC;
}

/** write the configuration */
static int set_config(int fd, sel_config_t* config) {
    bmc_nv_lseek(fd, sizeof(sel_file_header_t), SEEK_SET);
    
    if (bmc_nv_write(fd, config, sizeof(sel_config_t)) != sizeof(sel_config_t))  {
        pp_bmc_log_debug("[SEL] could not write configuration");
        return PP_ERR;
    }
    
    return PP_SUC;
}

/** get an element */
static int get_element(int fd, int idx, sel_entry_t* entry) {
    bmc_nv_lseek(fd, sizeof(sel_file_header_t) + sizeof(sel_config_t) + (idx * sizeof(sel_entry_t)), SEEK_SET);
    
    if (bmc_nv_read(fd, entry, sizeof(sel_entry_t)) != sizeof(sel_entry_t)) {
        pp_bmc_log_debug("[SEL] could not read entry %d from persistent storage", idx);
        return PP_ERR;
    }
    return PP_SUC;
}

/** set an element */
static int set_element(int fd, int idx, sel_entry_t* entry) {
    bmc_nv_lseek(fd, sizeof(sel_file_header_t) + sizeof(sel_config_t) + (idx * sizeof(sel_entry_t)), SEEK_SET);

    if (bmc_nv_write(fd, entry, sizeof(sel_entry_t)) != sizeof(sel_entry_t)) {
        pp_bmc_log_debug("[SEL] add_event - could not write entry %d", idx);
        return PP_ERR;
    }
    return PP_SUC;
}

#include <pp/hash.h>
static pp_hash_i_t *id_hash = NULL;

static void id_hash_add(unsigned short id, int idx)
{
    // add idx + 1 since value 0 is reserved
    pp_hash_set_entry_i(id_hash, id, (void*)(idx + 1), NULL);
}

static void id_hash_del(unsigned short id)
{
    pp_hash_delete_entry_i(id_hash, id);
}

static void id_hash_clear(void)
{
    pp_hash_clear_i(id_hash);
}

static int id_hash_get_idx(unsigned short id)
{
    // (0 - 1) if not found
    return (int)pp_hash_get_entry_i_as_int(id_hash, id) - 1;
}

static int id_hash_init(void)
{
    int fd;
    sel_config_t config;
    int i;

    id_hash = pp_hash_create_i(32);

    if (PP_FAILED(open_check(&fd))) goto failed;
    if (PP_FAILED(get_config(fd, &config))) goto failed;

    // initially fill hash from persistent SEL storage
    for (i = 0; i < config.sel_size; i++) {
        sel_entry_t entry;
        if (PP_FAILED(get_element(fd, i, &entry))) {
            pp_bmc_log_error("[SEL] error creating search index at entry index %d", i);
            goto failed;
        }
        id_hash_add(le16_to_cpu(entry.id_le16), i);
    }
    bmc_nv_close(fd);

    return PP_SUC;

failed:
    if (fd >= 0) bmc_nv_close(fd);
    return PP_ERR;
}

static void id_hash_cleanup(void)
{
    pp_hash_delete_i(id_hash);
    id_hash = NULL;
}

/**************************
 * High level commands
 */

int sel_nv_init(void)
{
    id_hash_init(); // ignore return value
    return PP_SUC;
}

void sel_nv_cleanup(void)
{
    id_hash_cleanup();
}

/**
 * Get the current size of the SEL.
 */
int sel_nv_get_size() {
    int fd;
    sel_config_t config;

    if (PP_FAILED(open_check(&fd))) {
        return 0;
    }
    if (PP_FAILED(get_config(fd, &config))) {
        bmc_nv_close(fd);
        return 0;
    }

    bmc_nv_close(fd);

    return config.sel_size;
}

unsigned short sel_nv_get_first_event_id() {
    sel_entry_t entry;
    sel_config_t config;
    int fd;

    if (PP_FAILED(open_check(&fd))) goto failed;

    if (PP_FAILED(get_config(fd, &config))) goto failed;

    if (config.sel_size == 0) goto failed;

    if (PP_FAILED(get_element(fd, 0, &entry))) goto failed;

    bmc_nv_close(fd);
    return le16_to_cpu(entry.id_le16);

failed:
    if (fd >= 0) bmc_nv_close(fd);
    return 0xffff;
}

/**
 * Get the id of the last (newest) sel record.
 */
unsigned short sel_nv_get_last_event_id() {
    sel_entry_t entry;
    sel_config_t config;
    int fd;

    if (PP_FAILED(open_check(&fd))) goto failed;

    if (PP_FAILED(get_config(fd, &config))) goto failed;

    if (config.sel_size == 0) goto failed;

    if (PP_FAILED(get_element(fd, config.sel_size-1, &entry))) goto failed;

    bmc_nv_close(fd);
    return le16_to_cpu(entry.id_le16);

failed:
    if (fd >= 0) bmc_nv_close(fd);
    return 0xffff;
}

/**
 * Get an unused SEL id for new events.
 */
unsigned short sel_nv_get_next_entry_id()
{
    while (1) {
	cur_id++;

        // special id, must not be used
        if (cur_id == IPMI_SEL_FIRST_REC) continue;
        if (cur_id == IPMI_SEL_LAST_REC) continue;

        // check if the selected id is in use
        if (id_hash_get_idx(cur_id) < 0) {
	    break; // id is not in use - leave loop to pick this id
	}
    }

    return cur_id;
}

/**
 * Retrieve the event with the specified id from the SEL.
 * The event will be copied to the specified address. The
 * caller must provide the necessary memory.
 * 
 * If next_id is not NULL, the id of the next element will also
 * be returned (0xffff if the current event is the last event)
 * 
 * @returns PP_SUC if the event could be retrieved 
 * @returns PP_ERR if the event with the specified id does not exist
 */
int sel_nv_get_event(unsigned short id, sel_entry_t* entry, unsigned short* next_id) {
    int fd;
    sel_config_t config;
    sel_entry_t next_entry;
    int idx;

    if (PP_FAILED(open_check(&fd))) goto failed;

    if (PP_FAILED(get_config(fd, &config))) goto failed;

    if (config.sel_size == 0) goto failed;

    if (last_index >= config.sel_size) {
        last_index = config.sel_size - 1;
    }

    idx = id_hash_get_idx(id);
    if (idx < 0) goto failed;

    get_element(fd, idx, entry);

    if (next_id != NULL) {
        if (idx + 1 < config.sel_size) {
            // there is a next element
            if (PP_FAILED(get_element(fd, idx + 1, &next_entry))) goto failed;
            *next_id = le16_to_cpu(next_entry.id_le16);
        }
        else
            *next_id = 0xffff;
    }

    bmc_nv_close(fd);
    return PP_SUC;

failed:
    if (fd >= 0) bmc_nv_close(fd);
    return PP_ERR;
}

/**
 * Add an event to the SEL. It will be inserted at the end of the
 * linked list. The id of the entry must already be set.
 * 
 * @returns the id of the newly created SEL entry
 * @returns 0xFFFF if the event could not be added
 */
unsigned short sel_nv_add_event(sel_entry_t* entry) {
    int fd;
    sel_config_t config;

    if (PP_FAILED(open_check(&fd))) goto failed;

    if (PP_FAILED(get_config(fd, &config))) goto failed;

    if (config.sel_size >= SEL_MAX_SIZE) goto failed;

    if (PP_FAILED(set_element(fd, config.sel_size, entry))) goto failed;

    // fixup id hash
    id_hash_add(le16_to_cpu(entry->id_le16), config.sel_size);

    config.sel_size++;
    if (PP_FAILED(set_config(fd, &config))) goto failed;

    bmc_nv_close(fd);

    if (sel_listener_evt_r.add != NULL) {
        sel_listener_evt_r.add(le16_to_cpu(entry->id_le16));
    }
    if (sel_listener_sel_dev.add != NULL) {
        sel_listener_sel_dev.add(le16_to_cpu(entry->id_le16));
    }
    
    return le16_to_cpu(entry->id_le16);

failed:
    if (fd >= 0) bmc_nv_close(fd);
    return 0xFFFF;
}

/**
 * Delete the event with the specified id from the SEL.
 * @returns PP_SUC if successful
 * @returns PP_ERR if the event with the specified id does not exist.
 */
int sel_nv_delete_event(unsigned short id) {
    int fd;
    sel_config_t config;
    sel_entry_t entry;
    unsigned short next_id;
    int idx, i;

    if (PP_FAILED(open_check(&fd))) goto failed;

    if (PP_FAILED(get_config(fd, &config))) goto failed;

    idx = id_hash_get_idx(id);
    if (idx < 0) goto failed;

    // fixup id hash
    id_hash_del(id);

    // delete this event by copying all following elements to their previous element
    i = idx + 1;
    next_id = 0xffff;
    while (i < config.sel_size) {
        // this must not fail, we could not recover
        get_element(fd, i, &entry);

        // store id of next element after deleted entry
        if (i == idx + 1) next_id = le16_to_cpu(entry.id_le16);

        // this must not fail, we could not recover
        set_element(fd, i - 1, &entry);

        // fixup id hash
        id_hash_add(le16_to_cpu(entry.id_le16), i - 1);

        i++;
    }

    config.sel_size--;
    set_config(fd, &config);

    bmc_nv_close(fd);
    
    if (sel_listener_evt_r.del != NULL) {
        sel_listener_evt_r.del(id, next_id);
    }
    if (sel_listener_sel_dev.del != NULL) {
        sel_listener_sel_dev.del(id, next_id);
    }
    
    return PP_SUC;

failed:
    if (fd >= 0) bmc_nv_close(fd);
    return PP_ERR;
}

/**
 * Clear the SEL and delete all entries.
 * @returns PP_SUC if successful
 * @returns PP_ERR if the SEL could not be cleared
 */
int sel_nv_clear_sel() {
    int fd;

    sel_config_t config;
    if (PP_FAILED(open_check(&fd))) return PP_ERR;

    config.sel_size = 0;
    set_config(fd, &config);

    id_hash_clear();

    bmc_nv_close(fd);

    if (sel_listener_sel_dev.clear != NULL) {
        sel_listener_sel_dev.clear();
    }
    if (sel_listener_evt_r.clear != NULL) {
        sel_listener_evt_r.clear();
    }

    return PP_SUC;
}


int sel_nv_complete_check() {
    int fd;
    sel_entry_t se;

    if (PP_FAILED(open_check(&fd)))
        return PP_ERR;
    
    bmc_nv_lseek(fd, (sizeof(sel_config_t) + SEL_MAX_SIZE-1 * sizeof(sel_entry_t)), SEEK_CUR);
    
    if (bmc_nv_read(fd, &se, sizeof(sel_entry_t)) != sizeof(sel_entry_t)) {
        pp_bmc_log_debug("[SEL] could not read entry %d from persistent storage", SEL_MAX_SIZE-1);
        bmc_nv_close(fd);
        return PP_ERR;
    }
    
    bmc_nv_close(fd);
    return PP_SUC;
}

void sel_set_listener_evt_r(sel_listener_hndl_t handlers) {
    sel_listener_evt_r.add = handlers.add;
    sel_listener_evt_r.del = handlers.del;
    sel_listener_evt_r.clear = handlers.clear;
}

void sel_set_listener_sel_dev(sel_listener_hndl_t handlers) {
    sel_listener_sel_dev.add = handlers.add;
    sel_listener_sel_dev.del = handlers.del;
    sel_listener_sel_dev.clear = handlers.clear;
}

int sel_nv_create() {
    int fd;
    int i;
    sel_config_t config;
    sel_entry_t entry;
    
    memset(&config, 0, sizeof(sel_config_t));
    memset(&entry, 0, sizeof(sel_entry_t));

    if (PP_FAILED(bmc_nv_open(&fd, "sel"))) {
        pp_bmc_log_debug("[SEL] could not open persistent storage");
        goto failed;
    }

    sel_file_header_t header;
    header.ver = selHeaderVersion;
    strncpy(header.magic,selHeaderMagic, 16);
    if (bmc_nv_write(fd, &header, sizeof(sel_file_header_t)) != sizeof(sel_file_header_t)) {
        pp_bmc_log_debug("[SEL] could not write file header to persistent storage");
        goto failed;
    }

    if (bmc_nv_write(fd, &config, sizeof(sel_config_t)) != sizeof(sel_config_t)) {
        pp_bmc_log_debug("[SEL] could not initialize configuration");
        goto failed;
    }

    for (i=0; i<SEL_MAX_SIZE; i++) {
        if (bmc_nv_write(fd, &entry, sizeof(sel_entry_t)) != sizeof(sel_entry_t)) {
            pp_bmc_log_debug("[SEL] could not initialize entry %d", i);
            goto failed;
        }
    }

    bmc_nv_close(fd);
    pp_bmc_log_debug("[SEL] new SEL storage initialized");
    return PP_SUC;

failed:
    if (fd >= 0) bmc_nv_close(fd);
    return PP_ERR;
}

void dump_sel_nv(void)
{
    int fd;
    sel_config_t config;
    int i;

    if (PP_FAILED(open_check(&fd))) goto failed;
    if (PP_FAILED(get_config(fd, &config))) goto failed;
    pp_bmc_log_notice("[SEL] dumping nv SEL (%d entries)...", config.sel_size);

    for (i = 0; i < config.sel_size; i++) {
        sel_entry_t entry;
        if (PP_FAILED(get_element(fd, i, &entry))) goto failed;
        pp_bmc_log_notice("[SEL] entry %d: id=%d", i, le16_to_cpu(entry.id_le16));
    }
failed:
    if (fd >= 0) bmc_nv_close(fd);
}
