/**
 * bmc_imsg.c
 *
 * Description: BMC internal IPMI Message Structure and Methods
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <malloc.h>
#include <memory.h>
#include <pp/base.h>
#include <pp/selector.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_str.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/ipmi_sess.h>
 
imsg_t* pp_bmc_imsg_new(unsigned int data_size)
{
    imsg_t *imsg = malloc(sizeof(imsg_t));
    memset(imsg, 0, sizeof(imsg_t));

    imsg->data = malloc(data_size);
    imsg->data_size = data_size;
    memset(imsg->data, 0, data_size);

    imsg->session = NULL;
    imsg->buf = NULL;
    imsg->priv_level = IPMI_PRIV_UNSPEC;
    
    return imsg;
}

void pp_bmc_imsg_delete(imsg_t *imsg_ptr)
{
    if (imsg_ptr->data) free(imsg_ptr->data);
    if (imsg_ptr->buf) free(imsg_ptr->buf);
    if (imsg_ptr->session) pp_bmc_session_delete(imsg_ptr->session);
    free(imsg_ptr);
}

imsg_session_t* pp_bmc_session_new() {
    imsg_session_t* session;
    session = malloc(sizeof(imsg_session_t));
    memset (session, 0, sizeof(imsg_session_t));
    session->ref_count = 1;
    session->inactivity_timeout = -1;
    return session;
}

void pp_bmc_session_clear_last_message(imsg_session_t* session) {
    assert(session);
    if (session->last_message_data) {
       free(session->last_message_data);
       session->last_message_data = NULL;
       session->last_message_len = -1;
    }
}

void pp_bmc_session_delete(imsg_session_t* session) {
    if (session == NULL)
        return;
    session->ref_count --;
    if (session->ref_count <= 0) {
        pp_bmc_session_clear_last_message(session);
        if (session->password != NULL)
            free(session->password);
        if (session->challenge != NULL)
            free(session->challenge);
        if (session->sik != NULL)
            free(session->sik);
        if (session->k1 != NULL)
            free(session->k1);
        if (session->k2 != NULL)
            free(session->k2);
        if (session->manager_random != NULL)
            free(session->manager_random);
        if (session->username != NULL)
            free(session->username);
        if (session->addr != NULL)
            free(session->addr);
        if (session->inactivity_timeout != -1) {
            pp_select_remove_to(session->inactivity_timeout);
        }

        free(session);
    }
}


imsg_t* pp_bmc_imsg_copy(imsg_t* imsg, int flags) {
    imsg_t* nmsg = malloc(sizeof(imsg_t));
    *nmsg = *imsg;
    nmsg->buf = NULL;
    if ((flags & IMSG_COPY_DATA) != 0) {
        nmsg->data = malloc(imsg->data_size);
        nmsg->data_size = imsg->data_size;
	memcpy(nmsg->data, imsg->data, imsg->data_size);
    } else {
        nmsg->data = NULL;
        nmsg->data_size = 0;
    }
    return nmsg;
}

void* pp_bmc_imsg_resp(imsg_t *imsg,
                unsigned char compl_code,
                unsigned int resp_data_size)
{
    assert (!IPMI_IS_RESPONSE(imsg->netfn));
    if (IPMI_IS_RESPONSE(imsg->netfn)) return NULL;
    
    /* re-alloc space */
    if (imsg->data) free(imsg->data);
    imsg->data = malloc(resp_data_size + 1);
    imsg->data_size = resp_data_size + 1; /* add space for completion code */

    /* make to response */
    imsg->netfn |= 1;
    imsg->data[0] = compl_code;
    if (resp_data_size > 0) memset(&imsg->data[1], 0, resp_data_size);

    return imsg->data + 1;
}

void pp_bmc_imsg_err(imsg_t* imsg, unsigned char compl_code)
{
    assert (IPMI_IS_RESPONSE(imsg->netfn));
    imsg->data[0] = compl_code;
    imsg->data_size = 1;
}

char* pp_bmc_imsg_to_str(const imsg_t *imsg, char* buf, int buf_size)
{
    char s[1024];
    int p = 0;
    p += snprintf(buf + p, buf_size - p, "%s",
        ipmi_get_netfn_string(imsg->netfn, s, sizeof(s)));

    p += snprintf(buf + p, buf_size - p, "/%s",
        ipmi_get_command_string(imsg->netfn, imsg->cmd, s, sizeof(s)));

    p += snprintf(buf + p, buf_size - p, " rs/lun=%d/%d",
        imsg->rs_addr, imsg->rs_lun);
                    
    p += snprintf(buf + p, buf_size - p, " rq/lun=%d/%d",
        imsg->rq_addr, imsg->rq_lun);

    p += snprintf(buf + p, buf_size - p, " seq#=%d",
        imsg->rq_seq);

    p += snprintf(buf + p, buf_size - p, " chan=%d priv=%s",
        imsg->chan,
        ipmi_get_privilege_level_string(imsg->priv_level));
        
    if (imsg->session == NULL)
        p += snprintf(buf + p, buf_size-p, " sid=NULL");
    else
        p += snprintf(buf + p, buf_size-p, " sid=%u", imsg->session->session_id);
            
    if (IPMI_IS_RESPONSE(imsg->netfn))
        p += snprintf(buf + p, buf_size - p, " cc=%s",
            imsg->data_size > 0 ? ipmi_get_completion_code_string(imsg->data[0]) : "<missing>");
        
    p += snprintf(buf + p, buf_size - p, " size=%d",
        imsg->data_size);

    return buf;
}


#if PP_BMC_DEBUG_LEVEL>0
/*
 * Dump IPMI messages if debug_level = debug.
 * Message dumping can be suppressed by defining PP_IPMI_NO_MSG_DUMP.
 */
# ifdef PP_IPMI_NO_MSG_DUMP
void pp_bmc_imsg_dump(int dbg_level UNUSED, const char* originator UNUSED, 
               const imsg_t *imsg UNUSED) {
}
# else 
void pp_bmc_imsg_dump(int dbg_level, const char* originator, const imsg_t *imsg) {
    if (PP_BMC_DEBUG_LEVEL >= dbg_level)
    {
        char s[1024];

         {
            pp_bmc_imsg_to_str(imsg, s, sizeof(s));
            pp_bmc_log(dbg_level, "%s %s", originator, s);

            /*
            // Debug only
            int i;
            printf("[MSGDUMP] <");
            for (i=0; i<(imsg->data_size); i++) {
                printf("%2x ", imsg->data[i]);
            }
            printf(">\n");
            */
         }
    }
}
# endif
#endif
